import { ProductSearchCriteria, ProductSite, Supplier } from '@sage/x3-master-data-api';
import { dialogConfirmation, dialogMessage } from '@sage/x3-master-data/lib/client-functions/dialogs';
import { getSelectedStockSite } from '@sage/x3-master-data/lib/client-functions/get-selected-stock-site';
import { onGoto } from '@sage/x3-master-data/lib/client-functions/on-goto';
import {
    GraphApi,
    ProductSupplierView,
    PurchaseEntryTransaction,
    PurchaseEntryTransactionInput,
    PurchaseOrder,
    PurchaseOrderQuantityLinesSupplierView,
    PurchaseReceiptLineInput,
    PurchaseShipmentLine,
} from '@sage/x3-purchasing-api';
import { receipt } from '@sage/x3-stock-data/build/lib/menu-items/receipt';
import { Site } from '@sage/x3-system-api';
import {
    AsyncVoidFunction,
    DictionaryFieldSupported,
    VoidFunction,
} from '@sage/x3-system/lib/client-functions/screen-management-gs-1';
import { SupportServiceManagementGs1Page } from '@sage/x3-system/lib/client-functions/support-service-management-gs-1-page';
import { DataTitle } from '@sage/x3-system/lib/shared-functions/parsed-element';
import { Filter, extractEdges } from '@sage/xtrem-client';
import { DateValue } from '@sage/xtrem-date-time';
import { BusinessRuleError } from '@sage/xtrem-shared';
import * as ui from '@sage/xtrem-ui';
import {
    PurchaseReceiptDetailsRecordManager,
    PurchaseReceiptSession,
    validate,
} from '../client-functions/purchase-receipt-details-control';
import { readParameterValue } from '../client-functions/read-parameter';

// Key to use with Composite Data Gs1 for this application
export const mobileApplicationGs1Key = 'MobilePurchaseReceiptGs1Key';
const QTY_DECIMAL_PLACES: number = 8;
const CONVERSION_RATE_DECIMAL_PLACES: number = 6;

interface receiptUnit {
    code: string;
    numberOfDecimals: number;
}

@ui.decorators.page<MobilePurchaseReceipt>({
    title: 'Purchase receipt',
    module: 'x3-purchasing',
    mode: 'default',
    menuItem: receipt,
    priority: 100,
    isTransient: true,
    isTitleHidden: true,
    authorizationCode: 'CWSPTH',
    access: { node: '@sage/x3-purchasing/PurchaseReceipt' },
    async onLoad() {
        if (!(await this._init())) {
            this._disablePage();
        }
    },
    businessActions() {
        return [this.createButton];
    },
})
export class MobilePurchaseReceipt extends SupportServiceManagementGs1Page<GraphApi> {
    /*
     *
     *  Technical properties
     *
     */
    private _purchaseReceiptRecordManager: PurchaseReceiptDetailsRecordManager; // to store the receipt to create from session data
    private _purchaseSession: PurchaseReceiptSession;
    private _entryTransactionMap: Map<string, PurchaseEntryTransactionInput>;
    private _productSearchCriteria1: ProductSearchCriteria;
    private _productSearchCriteria2: ProductSearchCriteria;
    private _purchaseShipment: string | undefined;
    private _purchaseShipmentLine: string | undefined;
    private _shipmentLines: any[] = [];
    private _purchaseShipmentIds: string[] = [];
    private _isLocationPreloaded: boolean;
    /* @internal */
    private _globalTradeItemNumber: string | null = null;

    @ui.decorators.textField<MobilePurchaseReceipt>({
        isHidden: true,
    })
    stockSite: ui.fields.Text;

    /*
     *
     *  Page Actions
     *
     */

    @ui.decorators.pageAction<MobilePurchaseReceipt>({
        title: 'Create',
        buttonType: 'primary',
        shortcut: ['f2'],
        isDisabled: true,
        async onClick() {
            // Check if the purchase receipt to create contains at least one valid line
            if (!this._purchaseSession) {
                this.$.showToast(
                    ui.localize(
                        '@sage/x3-purchasing/notification-error-purchase-receipt-no-products',
                        `Enter at least one product.`,
                    ),
                    { type: 'error', timeout: 5000 },
                );
                return;
            }

            // Only validate receipt date is populated
            const validateError = (await this.receiptDate.validate()) || (await this.supplierPackingSlip.validate());
            if (validateError) {
                this.$.removeToasts();
                this.$.showToast(
                    `${ui.localize(
                        '@sage/x3-purchasing/notification-error-purchase-receipt-error',
                        'Error',
                    )}: ${validateError}`,
                    { type: 'error', timeout: 30000 },
                );
                return;
            }
            this.createButton.isDisabled = true; // to prevent extreme case of duplicate receipts by rapidly clicking on create button multiple times
            this.$.loader.isHidden = false;
            const result = await this._callCreationAPI();
            this.$.loader.isHidden = true;

            // Special case unable to connect check type of error
            this.supplierPackingSlip.isDirty = false;
            if (!result || result instanceof Error) {
                this.$.loader.isHidden = true;
                const options: ui.dialogs.DialogOptions = {
                    acceptButton: {
                        text: ui.localize('@sage/x3-purchasing/button-goback', 'Go back'),
                    },
                    cancelButton: {
                        text: ui.localize('@sage/x3-purchasing/button-cancel', 'Cancel'),
                    },
                    size: 'small',
                    mdContent: true,
                };

                let message = '';

                if (!result?.message) {
                    message = `${ui.localize(
                        '@sage/x3-purchasing/pages_creation_error_connexion_webservice_contact_administrator',
                        'An error has occurred (connection or webservice error). Please contact your administrator.',
                    )}`;
                } else {
                    const _messages = <string[]>[];
                    const _results = <any>result;
                    let _diagnoses = _results?.diagnoses;
                    if (_diagnoses?.length > 1) {
                        _diagnoses = _diagnoses.splice(0, _diagnoses.length - 1);
                    }

                    (
                        (_results?.errors
                            ? _results.errors[0]?.extensions?.diagnoses
                            : (_results?.innerError?.errors[0]?.extensions?.diagnoses ??
                              _results.extensions?.diagnoses ??
                              _diagnoses)) ?? []
                    )
                        .filter((d: { severity: number; message: any }) => d.severity > 2 && d.message)
                        .forEach((d: { message: any }) => {
                            const _message = d.message.split(`\n`);
                            _messages.push(..._message);
                        });

                    const _result = _messages.length ? <string[]>_messages : <string[]>result.message.split(`\n`);

                    message = `**${ui.localize(
                        '@sage/x3-purchasing/dialog-error-purchase-receipt-creation',
                        'An error has occurred',
                    )}**\n\n`;

                    if (_result.length === 1) {
                        message += `${_result[0]}`;
                    } else {
                        message += _result.map(item => `* ${item}`).join('\n');
                    }
                }
                await this.$.sound.error();

                if (
                    await dialogConfirmation(
                        this,
                        'error',
                        ui.localize('@sage/x3-purchasing/dialog-error-title', 'Error'),
                        message,
                        options,
                    )
                ) {
                    await this.$.router.refresh();
                } else {
                    this._purchaseReceiptRecordManager.clearSession();
                    await this.$.router.emptyPage();
                }
            } else {
                this._purchaseReceiptRecordManager.clearSession();

                const options: ui.dialogs.DialogOptions = {
                    acceptButton: {
                        text: ui.localize('@sage/x3-purchasing/button-accept-ok', 'OK'),
                    },
                    fullScreen: true,
                };

                await this.$.sound.success();

                await dialogMessage(
                    this,
                    'success',
                    ui.localize('@sage/x3-purchasing/dialog-success-title', 'Success'),
                    ui.localize(
                        '@sage/x3-purchasing/dialog-success-purchase-receipt-creation',
                        'Document no. {{receiptId}} created.',
                        { receiptId: result.id },
                    ),
                    options,
                );
                await this.$.router.emptyPage();
            }
        },
    })
    createButton: ui.PageAction;

    /*
     *
     *  Sections
     *
     */

    @ui.decorators.section<MobilePurchaseReceipt>({
        isTitleHidden: true,
    })
    mainSection: ui.containers.Section;

    /*
     *
     *  Blocks
     *
     */

    @ui.decorators.block<MobilePurchaseReceipt>({
        parent() {
            return this.mainSection;
        },
        isTitleHidden: true,
    })
    productBlock: ui.containers.Block;

    @ui.decorators.block<MobilePurchaseReceipt>({
        parent() {
            return this.mainSection;
        },
        title: 'Receipt to create',
        isHidden: true,
    })
    receiptBlock: ui.containers.Block;

    /*
     *
     *  Fields
     *
     */
    @ui.decorators.dateField<MobilePurchaseReceipt>({
        parent() {
            return this.productBlock;
        },
        title: 'Receipt date',
        isTransient: true,
        isMandatory: true,
        maxDate: DateValue.today().toString(), // restricts to current date or earlier
        onChange() {
            if (this._purchaseSession && this.receiptDate.value) {
                this._purchaseSession.purchaseReceipt.receiptDate = this.receiptDate.value;
                this._purchaseReceiptRecordManager.purchaseSession = this._purchaseSession; // trigger session manager's setter to save receipt date into session
            }
        },
    })
    receiptDate: ui.fields.Date;

    @ui.decorators.dropdownListField<MobilePurchaseReceipt>({
        parent() {
            return this.productBlock;
        },
        title: 'Transaction',
        isMandatory: true,
        isTransient: true,
    })
    transaction: ui.fields.DropdownList;

    @ui.decorators.labelField<MobilePurchaseReceipt>({
        isTransient: true,
        isHidden: true,
    })
    legalCompany: ui.fields.Label;

    // X3-184652 TODO: Issue: Unable to scroll down in the lookup to view more results (only up to 10 records can be viewed)
    @ui.decorators.referenceField<MobilePurchaseReceipt, PurchaseOrder>({
        parent() {
            return this.productBlock;
        },
        title: 'Purchase order',
        placeholder: 'Scan or select...',
        node: '@sage/x3-purchasing/PurchaseOrder',
        valueField: 'id',
        helperTextField: { orderFromSupplier: { code: { code: true } } },
        isHelperTextHidden: true,
        canFilter: false,
        filter() {
            const filter: Filter<PurchaseOrder> = {
                isClosed: { _eq: false },
                isIntersite: { _eq: false },
                // (X3-227355) TODO: Issue: Cannot use the less verbose _in operator instead of individual _or filter criterion
                _and: [
                    {
                        _or: [
                            {
                                signatureStatus: 'inFull',
                            },
                            {
                                signatureStatus: 'notManaged',
                            },
                            {
                                signatureStatus: 'automatic',
                            },
                        ],
                    },
                    {
                        _or: [
                            {
                                receiptStatus: 'no',
                            },
                            {
                                receiptStatus: 'inPart',
                            },
                        ],
                    },
                ],
                // (X3-227172) TODO: Issue: Using _id instead of code in order to properly filter for 'null' values, however removed because this will break lookup Search bar
                // receiptSite: { _id: { _in: [this.stockSite.value, null] } },
                purchaseSite: { legalCompany: { code: this.legalCompany.value ?? undefined } },
                // filtering on PO's lines to exclude PO that have nothing to receive
                purchaseOrderLines: {
                    _atLeast: 1,
                    //(X3-191778) TODO: Issue: Not possible to filter on a collection's reference property
                    // product: {
                    //     productStatus: { _eq: 'active' },
                    //     isReceived: true as any,
                    // },
                    // _or: [
                    //     { purchaseOrder: { isIntersite: { _eq: true as any } } },
                    //     {
                    //         purchaseOrder: { isIntersite: { _eq: false as any } },
                    //         product: {
                    //             _or: [{ isManufactured: false as any }, { isPhantom: false as any }],
                    //         },
                    //     },
                    // ],
                    isClosed: { _eq: false },
                    receiptSite: { code: this.stockSite.value ?? undefined },
                    sourceRequest: { _ne: 'directOrder' },
                    productType: { _ne: 'subcontract' },
                    lineType: { _eq: 'normal' },
                },
            };

            if (this.supplier.value) {
                filter.orderFromSupplier = { code: this.supplier.value.code };
            }

            return filter;
        },
        async onChange() {
            this._showWorkflowLayout(!!this.purchaseOrder.value);
            // perform another screen initialization for current workFlow enabled
            // TODO: This operation deletes the composite data, see if you want to transfer them here
            if (!(await this._updateControlManagerGs1())) {
                // TODO: What action should be taken in the event of a fatal error ?
            }
            this.supplierPackingSlipLink.isHidden = false;
        },
        isTransient: true,
        isFullWidth: true,
        isAutoSelectEnabled: true,
        columns: [
            ui.nestedFields.text({
                bind: 'id',
                title: 'Order no.',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-system/Site',
                bind: 'purchaseSite',
                valueField: 'code',
                title: 'Order site',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/Supplier',
                bind: 'orderFromSupplier',
                valueField: { code: { code: true } },
                title: 'Supplier',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'internalOrderReference',
                title: 'Internal or order reference',
                isReadOnly: true,
            }),
            // (X3-227347) TODO: Obsolete: Having to specify & hide fields used in filter that don't need to be displayed
            ui.nestedFields.checkbox({
                bind: 'isClosed',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'signatureStatus',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'receiptStatus',
                isHidden: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-system/Site',
                bind: 'receiptSite',
                valueField: '_id',
                isHidden: true,
            }),
            ui.nestedFields.reference<MobilePurchaseReceipt, PurchaseOrder, Site>({
                node: '@sage/x3-system/Site',
                bind: 'purchaseSite',
                valueField: { legalCompany: { code: true } },
                isHidden: true,
            }),
        ],
    })
    purchaseOrder: ui.fields.Reference;

    @ui.decorators.referenceField<MobilePurchaseReceipt, Supplier>({
        parent() {
            return this.productBlock;
        },
        title: 'Supplier',
        placeholder: 'Scan or select...',
        node: '@sage/x3-master-data/Supplier',
        valueField: { code: { code: true } },
        canFilter: false,
        filter: {
            isActive: { _eq: true },
        },
        onChange() {
            this.productSupplier.isDisabled = !this.supplier.value;

            if (this.supplier.value) {
                this.productSupplier.focus();
            }
        },
        isTransient: true,
        isFullWidth: true,
        isAutoSelectEnabled: true,
        shouldSuggestionsIncludeColumns: true,
        columns: [
            ui.nestedFields.reference({
                bind: 'code',
                title: 'Supplier',
                isReadOnly: true,
                node: '@sage/x3-master-data/BusinessPartner',
                valueField: 'code',
            }),
            ui.nestedFields.text({
                bind: 'shortCompanyName',
                title: 'Short Description',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'companyName1',
                title: 'Company name',
                isReadOnly: true,
            }),
            // (X3-227347) TODO: Obsolete: Having to specify & hide fields used in filter that don't need to be displayed
            ui.nestedFields.checkbox({
                bind: 'isActive',
                isHidden: true,
            }),
        ],
    })
    supplier: ui.fields.Reference;

    @ui.decorators.linkField<MobilePurchaseReceipt>({
        parent() {
            return this.productBlock;
        },
        isTransient: true,
        isTitleHidden: true,
        size: 'small',
        isFullWidth: true,
        width: 'small',
        map() {
            return ui.localize('@sage/x3-purchasing/select-supplier-packing-slip', 'Add a supplier packing slip');
        },
        async onClick() {
            this.supplierPackingSlip.isHidden = false;
            this.supplierPackingSlipLink.isHidden = true;
            this.supplierPackingSlip.focus();
        },
    })
    supplierPackingSlipLink: ui.fields.Link;

    @ui.decorators.textField<MobilePurchaseReceipt>({
        parent() {
            return this.productBlock;
        },
        title: 'Supplier packing slip',
        placeholder: 'Scan or enter...',
        isTransient: true,
        isMandatory: false,
        isFullWidth: true,
        validation: /^$|^[^|]+$/,
        maxLength: 20,
        isHidden: true,
        onChange() {
            if (this.supplierPackingSlip.value) this.supplierPackingSlip.getNextField(true)?.focus();
            if (this._purchaseSession) {
                this._purchaseSession.purchaseReceipt.supplierPackingSlip = this.supplierPackingSlip.value ?? undefined;
                this._purchaseReceiptRecordManager.purchaseSession = this._purchaseSession;
            }
        },
    })
    supplierPackingSlip: ui.fields.Text;

    @ui.decorators.referenceField<MobilePurchaseReceipt, ProductSupplierView>({
        parent() {
            return this.productBlock;
        },
        title: 'Product',
        placeholder: 'Scan or select...',
        node: '@sage/x3-purchasing/ProductSupplierView',
        valueField: { product: true },
        helperTextField: 'upc',
        canFilter: false,
        filter() {
            return {
                stockSite: this.stockSite.value ?? undefined,
                supplier: this.supplier.value?.code.code ?? undefined,
                productStatus: 'active',
                isReceived: { _eq: true },
                _or: [
                    { isPurchased: { _eq: true } },
                    { isSubcontracted: { _eq: true } },
                    { isService: { _eq: true } },
                    { isPhantom: { _eq: true } },
                    { isDeliverable: { _eq: true } },
                    { isSold: { _eq: true } },
                ],
            };
        },
        onError(error: any, originScreenId: string, originElementId: string) {
            ui.console.warn(`Error on ${originScreenId} ${originElementId}: ${error.message || error}`);
        },
        async onInputValueChange(this, rawData: string): Promise<void> {
            await this.scanBarCode(this.productSupplier, rawData);
        },
        async onChange() {
            await this._onChange_productSupplier();
        },
        isTransient: true,
        isMandatory: true,
        isFullWidth: true,
        isDisabled: true,
        isAutoSelectEnabled: true,
        shouldSuggestionsIncludeColumns: true,
        columns: [
            ui.nestedFields.text({
                bind: 'product',
                title: 'Product',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'upc',
                title: 'UPC',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'globalTradeItemNumber',
                title: 'GTIN',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'description1',
                title: 'Description',
                isReadOnly: true,
                isHidden: false,
            }),
            ui.nestedFields.text({
                bind: 'supplierUpc',
                title: 'Supplier UPC',
                isReadOnly: true,
                isHidden: false,
            }),
            ui.nestedFields.text({
                bind: 'supplierProduct',
                title: 'Supplier product',
                isReadOnly: true,
                isHidden: false,
            }),
            ui.nestedFields.text({
                bind: 'supplierProductDescription',
                title: 'Supplier product description',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'productCategory',
                title: 'Product category',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'productStatus',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isReceived',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isPurchased',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isSubcontracted',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isService',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isPhantom',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isDeliverable',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isSold',
                isHidden: true,
            }),
        ],
    })
    productSupplier: ui.fields.Reference;

    @ui.decorators.referenceField<MobilePurchaseReceipt, PurchaseOrderQuantityLinesSupplierView>({
        parent() {
            return this.productBlock;
        },
        title: 'Product',
        placeholder: 'Scan or select...',
        node: '@sage/x3-purchasing/PurchaseOrderQuantityLinesSupplierView',
        valueField: { product: true },
        helperTextField: { upc: true },
        canFilter: false,
        filter() {
            return {
                productStatus: 'active',
                isReceived: { _eq: true },
                _or: [
                    { isIntersite: { _eq: true } },
                    { isIntersite: { _eq: false }, isManufactured: { _eq: false } },
                    { isIntersite: { _eq: false }, isPhantom: { _eq: false } },
                ],
                isClosed: { _eq: false },
                sourceRequest: { _ne: 'directOrder' },
                productType: { _ne: 'subcontract' },
                lineType: 'normal',
                receiptSite: this.stockSite.value ?? undefined,
                purchaseOrder: this.purchaseOrder.value?.id,
            };
        },
        onError(error: any, originScreenId: string, originElementId: string) {
            console.warn(`Error on ${originScreenId} ${originElementId}: ${error.message || error}`);
        },
        async onInputValueChange(this, rawData: string): Promise<void> {
            await this.scanBarCode(this.purchaseOrderLinesSupplier, rawData);
        },
        async onChange() {
            await this._onChange_purchaseOrderLinesSupplier();
        },
        isTransient: true,
        isMandatory: true,
        isFullWidth: true,
        isHidden: true,
        isDisabled: true,
        isAutoSelectEnabled: true,
        shouldSuggestionsIncludeColumns: true,
        columns: [
            ui.nestedFields.text({
                bind: 'product',
                title: 'Product',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'lineNumber',
                title: 'Order line',
                prefix: 'Line no.',
                isReadOnly: true,
                isHidden: false,
            }),
            ui.nestedFields.text({
                bind: 'supplierUpc',
                title: 'Supplier upc',
                isReadOnly: true,
                canFilter: true,
                isHidden() {
                    return this._productSearchCriteria1 !== 'supplierUpcCode';
                },
            }),
            ui.nestedFields.text({
                bind: 'supplierProduct',
                title: 'Supplier product',
                isReadOnly: true,
                canFilter: true,
                isHidden() {
                    return this._productSearchCriteria1 !== 'supplierProductCode';
                },
            }),
            ui.nestedFields.text({
                bind: 'upc',
                title: 'UPC',
                isReadOnly: true,
                isHidden() {
                    return this._productSearchCriteria1 !== 'x3UpcCode';
                },
            }),
            ui.nestedFields.text({
                bind: 'globalTradeItemNumber',
                title: 'GTIN',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'description1',
                title: 'Description',
                isReadOnly: true,
                isHidden() {
                    return this._productSearchCriteria1 !== 'x3ProductDescription';
                },
            }),
            ui.nestedFields.text({
                bind: 'supplierUpc',
                title: 'Supplier upc',
                isReadOnly: true,
                canFilter: true,
                isHidden() {
                    return this._productSearchCriteria2 !== 'supplierUpcCode';
                },
            }),
            ui.nestedFields.text({
                bind: 'supplierProduct',
                title: 'Supplier product',
                isReadOnly: true,
                canFilter: true,
                isHidden() {
                    return this._productSearchCriteria2 !== 'supplierProductCode';
                },
            }),
            ui.nestedFields.text({
                bind: 'upc',
                title: 'UPC',
                isReadOnly: true,
                isHidden() {
                    return this._productSearchCriteria2 !== 'x3UpcCode';
                },
            }),
            ui.nestedFields.text({
                bind: 'description1',
                title: 'Description',
                isReadOnly: true,
                isHidden() {
                    return this._productSearchCriteria2 !== 'x3ProductDescription';
                },
            }),
            ui.nestedFields.numeric({
                bind: 'purchaseUnitToStockUnitConversionFactor',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'quantityInPurchaseUnitOrdered',
                title: 'Ordered qty',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'quantityInPurchaseUnitReceived',
                title: 'Received qty',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.label({
                title: 'Ordered',
                prefix: 'Ordered',
                isFullWidth: false,
                postfix(value, rowData) {
                    return rowData?.purchaseUnit;
                },
                isTitleHidden: false,
                isTransient: true,
                bind: '_id',
                map(value, rowData) {
                    return rowData.quantityInPurchaseUnitOrdered.toString();
                },
            }),
            ui.nestedFields.label({
                title: 'Expected',
                prefix: 'Expected',
                isFullWidth: false,
                postfix(value, rowData) {
                    return rowData?.purchaseUnit;
                },
                isTitleHidden: false,
                isTransient: true,
                bind: '_id',
                isHidden: false,
                map(value, rowData) {
                    let quantityInPurchaseUnitReceived = 0;
                    let quantityInPurchaseUnitToReceive = Math.max(
                        Number(rowData.quantityInPurchaseUnitOrdered) - Number(rowData.quantityInPurchaseUnitReceived),
                        0,
                    );

                    if (this._purchaseSession?.purchaseReceipt?.lines) {
                        this._purchaseSession?.purchaseReceipt?.lines
                            .filter(
                                (line: PurchaseReceiptLineInput) =>
                                    line.purchaseOrder === rowData.purchaseOrder &&
                                    Number(line.purchaseOrderLineNumber) === Number(rowData.lineNumber) &&
                                    line.product === rowData.product,
                            )
                            .map(
                                (line: PurchaseReceiptLineInput) =>
                                    (quantityInPurchaseUnitReceived +=
                                        (Number(line.quantityInReceiptUnitReceived) *
                                            Number(line.receiptUnitToStockUnitConversionFactor)) /
                                        rowData.purchaseUnitToStockUnitConversionFactor),
                            );
                        if (quantityInPurchaseUnitReceived > 0) {
                            quantityInPurchaseUnitToReceive = Math.max(
                                Number(rowData.quantityInPurchaseUnitOrdered) -
                                    Number(rowData.quantityInPurchaseUnitReceived) -
                                    Number(quantityInPurchaseUnitReceived),
                                0,
                            );
                        }
                    }

                    return Number.isInteger(Number(quantityInPurchaseUnitToReceive))
                        ? quantityInPurchaseUnitToReceive.toString()
                        : quantityInPurchaseUnitToReceive.toFixed(QTY_DECIMAL_PLACES);
                },
            }),
            ui.nestedFields.numeric({
                bind: 'sequenceNumber',
                isTitleHidden: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'purchaseUnit',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'productStatus',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isReceived',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isIntersite',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isManufactured',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isPhantom',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isClosed',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'sourceRequest',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'workInProgressStatus',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'productType',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'lineType',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'receiptSite',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'purchaseOrder',
                isHidden: true,
            }),
        ],
    })
    purchaseOrderLinesSupplier: ui.fields.Reference;

    @ui.decorators.tableField<MobilePurchaseReceipt>({
        parent() {
            return this.receiptBlock;
        },
        //title: 'Product(s)',
        isTitleHidden: true,
        isTransient: true,
        canSelect: false,
        canFilter: false,
        columns: [
            ui.nestedFields.text({
                bind: 'product',
                title: 'Product',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'quantityAndStockUnit',
                title: 'Quantity',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'purchaseOrderLineNumber',
                title: 'Line No.',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'purchaseShipment',
                title: 'Shipment',
                isReadOnly: true,
            }),
        ],
        dropdownActions: [
            {
                icon: 'delete',
                title: 'Delete',
                async onClick(rowId: any, data: any) {
                    this._purchaseSession?.purchaseReceipt?.lines?.splice(rowId, 1);
                    if (!this._purchaseSession?.purchaseReceipt?.lines?.length) {
                        this._purchaseReceiptRecordManager.clearSession();
                        this.createButton.isDisabled = true;
                        onGoto(this, `@sage/x3-purchasing/${this.$.page.id}`);
                    } else {
                        this.receiptToCreate.value = await this._mapPurchaseReceipt(
                            this._purchaseSession.purchaseReceipt.lines ?? [],
                        );
                        // don't forget to update session storage or deleted lines will reappear if user refreshes the page
                        this._purchaseReceiptRecordManager.purchaseSession = {
                            purchaseEntryTransaction: this._purchaseSession.purchaseEntryTransaction,
                            purchaseReceipt: this._purchaseSession.purchaseReceipt,
                            orderUnitToPurchaseUnitConversionFactor: 1,
                        };
                    }
                },
            },
        ],
    })
    receiptToCreate: ui.fields.Table<any>;

    /**
     *
     * Methods section
     *
     *
     */

    private _showWorkflowLayout(isPOWorkflow: boolean = true) {
        this.purchaseOrderLinesSupplier.isHidden = !isPOWorkflow;
        this.purchaseOrderLinesSupplier.isDisabled = !isPOWorkflow || !this.purchaseOrder.value;
        this.productSupplier.isHidden = isPOWorkflow;
        this.productSupplier.isDisabled =
            isPOWorkflow || !this.supplier.value || !this._purchaseSession?.purchaseReceipt?.supplier;
        this.supplier.isReadOnly = isPOWorkflow;
        // product is to be disabled, if it is hidden or supplier field is empty

        // Logic for whether supplier field should be readonly or not. If it becomes readonly, then auto-populate it based on selected PO
        if (!this._purchaseSession?.purchaseReceipt?.supplier) {
            // supplier field is only modifiable when there is no purchase receipt session yet
            this.supplier.isReadOnly = isPOWorkflow;
            if (this.supplier.isReadOnly) {
                // this.supplier.value = this.purchaseOrder.value.orderFromSupplier._id;
                this.supplier.value = this.purchaseOrder.value?.orderFromSupplier;
            }
            //Supplier will be disabled if current user is not authorized to process direct receipts.
            //So clear supplier value when clearing the PO value.
            if (!isPOWorkflow) this.supplier.value = null;
        }

        if (this.supplier.isDisabled && !isPOWorkflow) this.productSupplier.isDisabled = true;
        if (isPOWorkflow) this.purchaseOrderLinesSupplier.focus();
        this.supplierPackingSlip.isHidden = true;
        // this.$.commitValueAndPropertyChanges();
    }

    /**
     * Initialize the page
     * @returns false when page must be disabled
     */
    /** @internal */
    private async _init(): Promise<boolean> {
        //Populate the receipt so far if provided in the session data
        this._purchaseReceiptRecordManager = new PurchaseReceiptDetailsRecordManager(this, true);
        this._purchaseSession = this._purchaseReceiptRecordManager.purchaseSession;
        const purchaseOrderId = this.$.queryParameters.purchaseOrderId;
        const supplierCode = this.$.queryParameters.supplierCode;
        let fromInquiries: boolean = false;

        // if this is the first time user is on the main page of Purchase Receipt
        if (!this._purchaseSession) {
            // Initialize Receipt Date to the current date
            this.receiptDate.value = DateValue.today().toString();

            try {
                // Populate Default Site & Legal company & Entry Transaction fields
                this.stockSite.value = await getSelectedStockSite(
                    this,
                    ui.localize('@sage/x3-purchasing/dialog-error-title', 'Error'),
                    ui.localize(
                        '@sage/x3-purchasing/dialog-error-location-inquiry-set-site',
                        'Define a default stock site on the user function profile.',
                    ),
                );
                if (!this.stockSite.value) {
                    return false;
                }
                await this._getProductSearch();
                await this._initTransaction();
            } catch (error) {
                this.$.showToast(error.message, { type: 'error' });
                return false;
            }

            // Preloading purchaseOrderId and supplierCode into the purchaseOrder and supplier code fields
            if (purchaseOrderId && supplierCode) {
                fromInquiries = true;
                this.purchaseOrder.value = {
                    id: purchaseOrderId,
                    orderFromSupplier: {
                        code: { code: supplierCode },
                    },
                };

                this._showWorkflowLayout(true);
                await this.$.commitValueAndPropertyChanges();
            }
        } else {
            // else display user's selected entry transaction & date (both as disabled) & selected products to add stored from session storage
            const purchaseReceipt = this._purchaseSession.purchaseReceipt;

            // Populate Default Site
            this.stockSite.value = purchaseReceipt.receiptSite ?? null; // TODO: Issue: User can change site via sticker. This can a runtime error because Enter Detail page always use the site from the sticker. (To reproduce error, add a PO line from site FR022 and then change site to NA021, refresh the main page, and notice the site is still set to FR022. When selecting a product from FR022, a runtime error during loading Enter Detail page will occur because it is using the wrong site NA021 from the sticker to retrieve a product that is from FR022)
            await this._getProductSearch();
            // Populate Entry Transaction, Receipt Date, Receipt lines to create, and Supplier fields based on session storage

            try {
                await this._initTransaction();
            } catch (e) {
                ui.console.error(e);
            }

            this.transaction.options = [this._purchaseSession?.purchaseEntryTransaction?.code ?? ''];
            this.transaction.value = this._purchaseSession.purchaseEntryTransaction.code ?? null;

            // not allowed to be changed unless user creates successfully or remove all receipt lines

            this.receiptDate.value = purchaseReceipt.receiptDate ?? null;
            this.supplierPackingSlip.value = purchaseReceipt.supplierPackingSlip ?? null;

            if (this.supplierPackingSlip.value) {
                this.supplierPackingSlip.isHidden = false;
                this.supplierPackingSlipLink.isHidden = true;
            }

            this.receiptToCreate.value = await this._mapPurchaseReceipt(purchaseReceipt.lines ?? []);
            this.receiptBlock.isHidden = false;
            this.supplier.value = { code: { code: purchaseReceipt.supplier } };

            this.supplier.isReadOnly = true;
            this.productSupplier.isDisabled = false; // because supplier is now poulated & cannot be changed til the entire session is cleared out
            // auto-populate PO lookup with most recently selected PO or empty if most recently is a direct receipt line (i.e. Supplier->product workflow)
            if (purchaseReceipt.lines && purchaseReceipt.lines[purchaseReceipt.lines.length - 1].purchaseOrder) {
                this.purchaseOrder.value = {
                    id: purchaseReceipt.lines[purchaseReceipt.lines.length - 1].purchaseOrder,
                }; // TODO: Verify: Is this bad practice of auto-populating reference field
                this.purchaseOrderLinesSupplier.isDisabled = this.purchaseOrderLinesSupplier.isHidden = false;
                this.productSupplier.isDisabled = this.productSupplier.isHidden = true;
                this.purchaseOrderLinesSupplier.focus();
            }

            this.createButton.isDisabled = false;
        }
        // required for Purchase Order lookup (TODO: Verify if legalCompany can be stored in session to avoid a GraphQL call)
        this.legalCompany.value = await this._getLegalCompany(this.stockSite.value ?? '');

        // Disallow direct receipts based on PTHDIR setting. Storing value in local storage so only need to read once.
        const valPTHDIR = await readParameterValue('PTHDIR', this.$.userCode ?? '', this.stockSite.value ?? '', this);
        this.supplier.isDisabled = valPTHDIR !== 2;

        if (fromInquiries) {
            this.purchaseOrder.focus();
            this.purchaseOrderLinesSupplier.focus();
        }

        // perform screen initialization for current workFlow enabled
        if (!(await this._initControlManagerGs1(this.stockSite.value ?? ''))) {
            return false;
        }

        return true;
    }

    /**
     * Initialize ControlManagerGs1
     * @returns true when ControlManagerGs1 has usable
     */
    private async _initControlManagerGs1(site: string): Promise<boolean> {
        const isPOWorkflow = !!this.purchaseOrder.value;
        return await this.createAndInitServiceGs1(site, mobileApplicationGs1Key, {
            [DataTitle.gtin]: {
                mainField: isPOWorkflow ? this.purchaseOrderLinesSupplier : this.productSupplier,
                onChangeMainField: isPOWorkflow
                    ? this._onChange_purchaseOrderLinesSupplier
                    : this._onChange_productSupplier,
            },
        } as DictionaryFieldSupported);
    }

    /**
     * Update fields mappings
     * @returns true when ControlManagerGs1 has usable
     */
    /** @internal */
    private async _updateControlManagerGs1(): Promise<boolean> {
        const isPOWorkflow = !!this.purchaseOrder.value;
        return await this.setScreenFieldSupported({
            [DataTitle.gtin]: {
                mainField: isPOWorkflow ? this.purchaseOrderLinesSupplier : this.productSupplier,
                onChangeMainField: isPOWorkflow
                    ? this._onChange_purchaseOrderLinesSupplier
                    : this._onChange_productSupplier,
            },
        } as DictionaryFieldSupported);
    }

    private goToDetailsPage(line: PurchaseReceiptLineInput, globalTradeItemNumber: string | null) {
        const purchaseReceiptLine: PurchaseReceiptLineInput = {
            receiptSite: this.stockSite.value ?? undefined,
            ...line,
        };

        // Store globalTradeItemNumber for postDone action
        this._globalTradeItemNumber = globalTradeItemNumber;

        if (!this._purchaseSession) {
            // if no session, create a new session object
            const selectedEntryTransaction = this._entryTransactionMap.get(this.transaction.value ?? '');
            this._purchaseReceiptRecordManager.purchaseSession = {
                purchaseReceipt: {
                    receiptSite: this.stockSite.value ?? undefined,
                    receiptDate: this.receiptDate.value ?? undefined,
                    supplier: this.supplier.value?.code.code,
                    stockAutomaticJournal: selectedEntryTransaction?.stockAutomaticJournal,
                    stockMovementCode: selectedEntryTransaction?.stockMovementCode ?? '',
                    transaction: selectedEntryTransaction?.code,
                    destination: this.$.storage.get('mobile-label-destination')?.toString() ?? '',
                    document: this.$.storage.get('mobile-document-destination')?.toString() ?? '',
                    supplierPackingSlip: this.supplierPackingSlip.value ?? undefined,
                    supplierPackingSlipDate: DateValue.today().toString(),
                    lines: [
                        {
                            ...purchaseReceiptLine,
                            stockMovementGroup: selectedEntryTransaction?.defaultStockMovementGroup ?? '',
                            purchaseShipment: this._purchaseShipment,
                            purchaseShipmentLine: this._purchaseShipmentLine,
                        },
                    ],
                },
                purchaseEntryTransaction: {
                    code: this.transaction.value ?? undefined,
                    isLotCustomField1Allowed: selectedEntryTransaction?.isLotCustomField1Allowed,
                    isLotCustomField2Allowed: selectedEntryTransaction?.isLotCustomField2Allowed,
                    isLotCustomField3Allowed: selectedEntryTransaction?.isLotCustomField3Allowed,
                    isLotCustomField4Allowed: selectedEntryTransaction?.isLotCustomField4Allowed,
                    identifier1Entry: selectedEntryTransaction?.identifier1Entry,
                    identifier1Detail: selectedEntryTransaction?.identifier1Detail,
                    identifier2Entry: selectedEntryTransaction?.identifier2Entry,
                    identifier2Detail: selectedEntryTransaction?.identifier2Detail,
                    isLotPotencyAllowed: selectedEntryTransaction?.isLotPotencyAllowed,
                    isLotExpirationDateAllowed: selectedEntryTransaction?.isLotExpirationDateAllowed,
                    stockMovementCode: selectedEntryTransaction?.stockMovementCode,
                },
                orderUnitToPurchaseUnitConversionFactor: 1,
            };
        } else {
            this._purchaseSession.purchaseReceipt.lines?.push({
                ...purchaseReceiptLine,
                stockMovementGroup:
                    this._purchaseSession.purchaseReceipt.lines[this._purchaseSession.purchaseReceipt.lines.length - 1]
                        .stockMovementGroup, // copy stockMovementGroup from the previous most line
                purchaseShipment: this._purchaseShipment,
                purchaseShipmentLine: this._purchaseShipmentLine,
            });
            this._purchaseReceiptRecordManager.purchaseSession = {
                purchaseReceipt: this._purchaseSession.purchaseReceipt,
                purchaseEntryTransaction: this._purchaseSession.purchaseEntryTransaction,
                orderUnitToPurchaseUnitConversionFactor: 1,
            };
        }

        // saving data and aborting dispatch is not enough :
        // the page has changed before dispatching has aborted !
        this.saveCompositeData();

        // If we are in the process of dispatching, we must delay the change of page
        if (this.controlManagerGs1.isDispatchInProgress) {
            this.controlManagerGs1.abortDispatch(this._postDoneDetailPage);
        } else if (!this.controlManagerGs1.isInitializationInProgress) {
            this._postDoneDetailPage();
        }
    }

    /** @internal */
    private readonly _postDoneDetailPage: VoidFunction = () => {
        this.$.removeToasts();
        this.$.setPageClean();
        if (this._purchaseShipmentIds?.length > 1) {
            this.$.router.goTo('@sage/x3-purchasing/MobilePurchaseReceiptSelectAShipment', {
                purchaseOrderId: `${this.purchaseOrder?.value?.id}`,
                purchaseOrderLineNumber: `${this.purchaseOrderLinesSupplier?.value?.lineNumber}`,
                globalTradeItemNumber: `${this._globalTradeItemNumber ?? ''}`,
                shipmentToLoad: `${JSON.stringify(this._purchaseShipmentIds)}`,
                isLocationPreloaded: `${this._isLocationPreloaded ? '1' : '0'}`,
            });
        } else {
            this.$.router.goTo('@sage/x3-purchasing/MobilePurchaseReceiptEnterReceiptDetail', {
                globalTradeItemNumber: `${this._globalTradeItemNumber ?? ''}`,
                isLocationPreloaded: `${this._isLocationPreloaded ? '1' : '0'}`,
            });
        }
    };

    private async _getLegalCompany(defaultSite: string): Promise<string> {
        const response = await this.$.graph
            .node('@sage/x3-system/Site')
            .read(
                {
                    _id: true, // TODO: Remove unneeded property
                    legalCompany: { code: true },
                },
                defaultSite,
            )
            .execute();

        if (!response) {
            // This should never occur as long as the user function profile has a valid site set up
            throw new Error(
                ui.localize(
                    '@sage/x3-purchasing/error-purchase-receipt-no-legal-company',
                    'The invalid site {{site}} has no legal company.',
                    {
                        site: defaultSite,
                    },
                ),
            );
        }

        return response.legalCompany.code;
    }

    private async _getEntryTransactionOptions(): Promise<string[]> {
        const response = extractEdges(
            await this.$.graph
                // with 'provides' property defined in accessCode of this node, should automatically return only transactions that are accessible for the current user
                .node('@sage/x3-purchasing/PurchaseEntryTransaction')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            _id: true,
                            code: true,
                            stockMovementCode: {
                                code: true,
                            },
                            defaultStockMovementGroup: {
                                code: true,
                            },
                            stockAutomaticJournal: {
                                code: true,
                            },
                            isLotCustomField1Allowed: true,
                            isLotCustomField2Allowed: true,
                            isLotCustomField3Allowed: true,
                            isLotCustomField4Allowed: true,
                            identifier1Entry: true,
                            identifier1Detail: true,
                            identifier2Entry: true,
                            identifier2Detail: true,
                            isLotPotencyAllowed: true,
                            isLotExpirationDateAllowed: true,
                        },
                        {
                            filter: {
                                entryTransactionType: 'receipt',
                                isActive: true,
                            },
                        },
                    ),
                )
                .execute(),
        ) as PurchaseEntryTransaction[];

        this._entryTransactionMap = new Map<string, PurchaseEntryTransactionInput>(); // TODO: Put this in a better place

        // transform Entry Transaction response into a string array
        return response.map((entryTransaction: PurchaseEntryTransaction) => {
            this._entryTransactionMap.set(entryTransaction.code, {
                code: entryTransaction.code,
                _id: entryTransaction._id,
                stockAutomaticJournal: entryTransaction.stockAutomaticJournal?.code,
                stockMovementCode: entryTransaction.stockMovementCode?.code,
                defaultStockMovementGroup: entryTransaction.defaultStockMovementGroup?.code,
                isLotCustomField1Allowed: entryTransaction.isLotCustomField1Allowed,
                isLotCustomField2Allowed: entryTransaction.isLotCustomField2Allowed,
                isLotCustomField3Allowed: entryTransaction.isLotCustomField3Allowed,
                isLotCustomField4Allowed: entryTransaction.isLotCustomField4Allowed,
                identifier1Entry: entryTransaction.identifier1Entry,
                identifier1Detail: entryTransaction.identifier1Detail,
                identifier2Entry: entryTransaction.identifier2Entry,
                identifier2Detail: entryTransaction.identifier2Detail,
                isLotPotencyAllowed: entryTransaction.isLotPotencyAllowed,
                isLotExpirationDateAllowed: entryTransaction.isLotExpirationDateAllowed,
            });
            return entryTransaction.code;
        });
    }

    private async _callCreationAPI(): Promise<any | Error> {
        const _purchaseReceiptArgs = this._purchaseSession?.purchaseReceipt ?? {};
        let result: any;

        try {
            result = await this.$.graph
                .node('@sage/x3-purchasing/PurchaseReceipt')
                .create(
                    {
                        id: true,
                    },
                    {
                        data: _purchaseReceiptArgs,
                    },
                )
                .execute();
            if (!result) {
                throw new BusinessRuleError(
                    ui.localize(
                        '@sage/x3-purchasing/dialog-error-purchase-receipt-no-create-results',
                        'No results for the created purchase receipt',
                    ),
                );
            }
        } catch (error) {
            return error;
        }
        return result;
    }

    private async _getReceiptUnits(units: (string | undefined)[]): Promise<receiptUnit[]> {
        const response = await this.$.graph
            .node('@sage/x3-master-data/UnitOfMeasure')
            .query(
                ui.queryUtils.edgesSelector(
                    {
                        code: true,
                        numberOfDecimals: true,
                    },
                    {
                        filter: {
                            code: { _in: units },
                        },
                    },
                ),
            )
            .execute();
        return response.edges.map(edge => {
            return {
                code: edge.node.code,
                numberOfDecimals: edge.node.numberOfDecimals,
            };
        });
    }

    private async _mapPurchaseReceipt(receipt: Partial<PurchaseReceiptLineInput>[]) {
        let rowCount = 0;
        const receiptUnits = await this._getReceiptUnits(
            receipt.map((line: PurchaseReceiptLineInput) => line.receiptUnit),
        );
        return receipt.map((line: PurchaseReceiptLineInput) => {
            return {
                _id: String(rowCount++), // this defines the rowId parameter in dropdownActions onClick() event
                purchaseOrderLineNumber:
                    line.purchaseOrder && line.purchaseOrderLineNumber
                        ? `${line.purchaseOrder} ${line.purchaseOrderLineNumber}`
                        : '',
                product: line.product,
                quantityAndStockUnit: `${Number(line.quantityInReceiptUnitReceived).toFixed(receiptUnits.find(unit => unit.code === line.receiptUnit)?.numberOfDecimals ?? 0)} ${line.receiptUnit}`,
                purchaseShipment:
                    line.purchaseShipment && line.purchaseShipmentLine
                        ? `${line.purchaseShipment} ${line.purchaseShipmentLine}`
                        : '',
            };
        });
    }

    private _disablePage(): void {
        this.purchaseOrder.isDisabled = true;
        this.supplier.isDisabled = true;
        this.receiptDate.isDisabled = true;
        this.transaction.isDisabled = true;
        this.supplierPackingSlip.isDisabled = true;
    }

    private async _getProductSearch() {
        const responseSite = await this.$.graph
            .node('@sage/x3-master-data/MobileAutomationSetup')
            .query(
                ui.queryUtils.edgesSelector(
                    {
                        criteria1: true,
                        criteria2: true,
                        isLocationPreloaded: true,
                    },
                    {
                        filter: {
                            site: { code: this.stockSite.value },
                        },
                    },
                ),
            )
            .execute();
        if (responseSite.edges.length !== 0) {
            responseSite.edges.some(edge => {
                this._productSearchCriteria1 = edge.node.criteria1;
                this._productSearchCriteria2 = edge.node.criteria2;
                this._isLocationPreloaded = edge.node.isLocationPreloaded;
            });
        } else {
            const response = await this.$.graph
                .node('@sage/x3-master-data/MobileAutomationSetup')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            criteria1: true,
                            criteria2: true,
                            isLocationPreloaded: true,
                        },
                        {
                            filter: {
                                site: null,
                            },
                        },
                    ),
                )
                .execute();
            if (response.edges.length !== 0) {
                response.edges.some(edge => {
                    this._productSearchCriteria1 = edge.node.criteria1;
                    this._productSearchCriteria2 = edge.node.criteria2;
                    this._isLocationPreloaded = edge.node.isLocationPreloaded;
                });
            } else {
                this._productSearchCriteria1 = 'none';
                this._productSearchCriteria2 = 'none';
                this._isLocationPreloaded = true;
            }
        }
    }

    private async _initTransaction() {
        this.transaction.options = await this._getEntryTransactionOptions();
        switch (this.transaction.options.length) {
            case 0:
                this._disablePage();
                throw new Error('No transaction, cannot continue');
            case 1:
                this.transaction.value = this.transaction.options[0];
                this.transaction.isHidden = true;
                break;
            default:
                this.transaction.value = this.transaction.options[0];
                if (this._purchaseSession) {
                    this.transaction.isDisabled = true;
                }
                break;
        }
    }

    /**
     * OnChange readonly process
     *
     * Used both decorator and bar code manager.
     * @returns Promise<void>
     */

    /** @internal */
    private readonly _onChange_productSupplier: AsyncVoidFunction = async () => {
        this._globalTradeItemNumber = null;

        if (!this.productSupplier.value) return; // if user closes the lookup instead of selecting a record
        if (!(await validate(this))) {
            this.productSupplier.value = null;
            return;
        }

        this._purchaseShipment = undefined;
        this._purchaseShipmentLine = undefined;

        const response = await this.$.graph
            .node('@sage/x3-master-data/ProductSite')
            .query(
                ui.queryUtils.edgesSelector(
                    {
                        isBeingCounted: true,
                    },
                    {
                        filter: {
                            product: {
                                code: this.productSupplier?.value?.product,
                            },
                            stockSite: {
                                code: this.stockSite.value ?? undefined,
                            },
                        },
                    },
                ),
            )
            .execute();

        if (response.edges.length !== 0 && response.edges[0].node.isBeingCounted === true) {
            if (
                !(await dialogConfirmation(
                    this,
                    'warn',
                    ui.localize('@sage/x3-purchasing/dialog-warning-title', 'Warning'),
                    ui.localize(
                        '@sage/x3-purchasing/product-blocked-by-count-continue',
                        'Product blocked by count. Do you want to continue?',
                    ),
                    {
                        acceptButton: {
                            text: ui.localize('@sage/x3-purchasing/button-accept-yes', 'Yes'),
                        },
                        cancelButton: {
                            text: ui.localize('@sage/x3-purchasing/button-cancel-no', 'No'),
                        },
                    },
                ))
            ) {
                //cancel button clicked - promise rejected
                this.productSupplier.value = null;
                this.productSupplier.focus();
                return;
            }
        }

        this.goToDetailsPage(
            { product: this.productSupplier?.value?.product },
            this.productSupplier?.value?.globalTradeItemNumber ?? null,
        );
    };

    /** @internal */
    private readonly _onChange_purchaseOrderLinesSupplier: AsyncVoidFunction = async () => {
        this._globalTradeItemNumber = null;

        this._purchaseShipment = undefined;
        this._purchaseShipmentLine = undefined;

        // if user closes the lookup instead of selecting a record
        if (!this.purchaseOrderLinesSupplier.value) return;

        if (!(await validate(this))) {
            this.purchaseOrderLinesSupplier.value = null;
            return;
        }

        this._shipmentLines = await this._readShipmentLines();
        this._cleanShipmentLines();

        if (this._shipmentLines?.length === 1) {
            this._purchaseShipment = this._shipmentLines[0].purchaseShipment?.id;
            this._purchaseShipmentLine = this._shipmentLines[0].lineNumber?.toString();
        }

        const response = await this.$.graph
            .node('@sage/x3-master-data/ProductSite')
            .query(
                ui.queryUtils.edgesSelector<ProductSite>(
                    {
                        isBeingCounted: true,
                    },
                    {
                        filter: {
                            product: {
                                code: this.purchaseOrderLinesSupplier.value.product,
                            },
                            stockSite: {
                                code: this.stockSite.value ?? undefined,
                            },
                        },
                    },
                ),
            )
            .execute();

        if (response.edges.length !== 0 && response.edges[0].node.isBeingCounted === true) {
            if (
                !(await dialogConfirmation(
                    this,
                    'warn',
                    ui.localize('@sage/x3-purchasing/dialog-warning-title', 'Warning'),
                    ui.localize(
                        '@sage/x3-purchasing/product-blocked-by-count-continue',
                        'Product blocked by count. Do you want to continue?',
                    ),
                    {
                        acceptButton: {
                            text: ui.localize('@sage/x3-purchasing/button-accept-yes', 'Yes'),
                        },
                        cancelButton: {
                            text: ui.localize('@sage/x3-purchasing/button-cancel-no', 'No'),
                        },
                    },
                ))
            ) {
                //cancel button clicked - promise rejected
                this.purchaseOrderLinesSupplier.value = null;
                this.purchaseOrderLinesSupplier.focus();
                return;
            }
        }

        this.goToDetailsPage(
            {
                product: this.purchaseOrderLinesSupplier?.value?.product,
                purchaseOrder: this.purchaseOrder?.value?.id,
                purchaseOrderLineNumber: this.purchaseOrderLinesSupplier?.value?.lineNumber,
                purchaseOrderLine: this.purchaseOrderLinesSupplier?.value?.sequenceNumber,
            },
            this.purchaseOrderLinesSupplier?.value?.globalTradeItemNumber ?? null,
        );
    };

    private async _readShipmentLines() {
        return extractEdges(
            await this.$.graph
                .node('@sage/x3-purchasing/PurchaseShipmentLine')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            purchaseShipment: { id: true },
                            lineNumber: true,
                            product: { code: true },
                            purchaseOrderNumber: { id: true },
                            purchaseOrderLineNumber: true,
                            quantityInPurchaseUnitShipped: true,
                            quantityInPurchaseUnitReceived: true,
                            quantityInPurchaseUnitPreReceived: true,
                            quantityInOrderUnitShipped: true,
                            quantityInStockUnitShipped: true,
                            isClosed: true,
                            isPreReceived: true,
                        },
                        {
                            filter: {
                                purchaseOrderNumber: { id: this.purchaseOrder.value?.id },
                                purchaseOrderLineNumber: this.purchaseOrderLinesSupplier?.value?.lineNumber,
                                purchaseShipment: {
                                    shipmentStatus: {
                                        _nin: ['received', 'closed'],
                                    },
                                },
                                isClosed: { _eq: false },
                            },
                        },
                    ),
                )
                .execute(),
        ) as PurchaseShipmentLine[];
    }

    private _cleanShipmentLines() {
        this._shipmentLines.forEach((element: any) => {
            if (
                (element.isPreReceived &&
                    Number(element.quantityInPurchaseUnitPreReceived) >
                        Number(element.quantityInPurchaseUnitReceived)) ||
                (!element.isPreReceived &&
                    Number(element.quantityInPurchaseUnitShipped) > Number(element.quantityInPurchaseUnitReceived))
            ) {
                this._purchaseShipmentIds.push(element?.purchaseShipment?.id);
            } else {
                const resultIndex = this._shipmentLines.indexOf(element);
                this._shipmentLines.splice(resultIndex, 1);
            }
        });

        if (this._purchaseSession?.purchaseReceipt?.lines) {
            this._shipmentLines.forEach(lineShip => {
                if (
                    lineShip.purchaseOrderNumber === this.purchaseOrder.value?.id &&
                    lineShip.purchaseOrderLineNumber === this.purchaseOrderLinesSupplier?.value?.lineNumber
                ) {
                    let QuantityInStockUnitReceived = 0;
                    let quantityInStockUnitExpected = Number(lineShip.quantityInStockUnitShipped);

                    this._purchaseSession?.purchaseReceipt?.lines?.forEach(line => {
                        if (line.purchaseShipment) {
                            if (
                                line.purchaseShipment === lineShip.purchaseShipment?.id &&
                                Number(line.purchaseShipmentLine) === Number(lineShip.lineNumber)
                            ) {
                                QuantityInStockUnitReceived +=
                                    Number(line.quantityInReceiptUnitReceived) *
                                    Number(line.receiptUnitToStockUnitConversionFactor);
                            }
                        }
                    });

                    if (Number(quantityInStockUnitExpected) - Number(QuantityInStockUnitReceived) <= 0) {
                        let resultIndex = this._purchaseShipmentIds.indexOf(lineShip.purchaseShipment?.id);
                        this._purchaseShipmentIds.splice(resultIndex, 1);
                        resultIndex = this._shipmentLines.indexOf(lineShip);
                        this._shipmentLines.splice(resultIndex, 1);
                    }
                }
            });
        }
    }
}
