import { GraphApi, PurchaseReceiptLineInput, PurchaseShipmentLine } from '@sage/x3-purchasing-api';
import * as ui from '@sage/xtrem-ui';
import {
    PurchaseReceiptDetailsRecordManager,
    PurchaseReceiptSession,
} from '../client-functions/purchase-receipt-details-control';

const QTY_DECIMAL_PLACES: number = 8;
const CONVERSION_RATE_DECIMAL_PLACES: number = 6;
@ui.decorators.page<MobilePurchaseReceiptSelectAShipment, PurchaseShipmentLine>({
    title: 'Purchase receipt',
    module: 'x3-purchasing',
    isTitleHidden: true,
    mode: 'default',
    node: '@sage/x3-purchasing/PurchaseShipmentLine',
    skipDirtyCheck: true,
    navigationPanel: {
        canFilter: false,
        isHeaderHidden: true,
        isAutoSelectEnabled: true,
        listItem: {
            title: ui.nestedFields.reference({
                node: '@sage/x3-master-data/Product',
                bind: 'product',
                valueField: 'code',
            }),
            line2: ui.nestedFields.reference({
                node: '@sage/x3-purchasing/PurchaseShipment',
                bind: 'purchaseShipment',
                valueField: 'id',
            }),
            line2Right: ui.nestedFields.text({
                bind: 'lineNumber',
            }),
            line3: ui.nestedFields.label({
                title: 'Shipped qty',
                prefix: 'Shipped qty.',
                isFullWidth: false,
                postfix(value, rowData) {
                    return rowData?.purchaseUnit?.code;
                },
                isTitleHidden: false,
                isTransient: true,
                bind: '_id',
                map(value, rowData) {
                    return Number.isInteger(Number(rowData.quantityInPurchaseUnitShipped))
                        ? rowData.quantityInPurchaseUnitShipped.toString()
                        : rowData.quantityInPurchaseUnitShipped.toFixed(QTY_DECIMAL_PLACES);
                },
            }),
            line3Right: ui.nestedFields.label({
                title: 'Expected qty',
                prefix: 'Expected qty.',
                isFullWidth: false,
                postfix(value, rowData) {
                    return rowData?.purchaseUnit?.code;
                },
                isTitleHidden: false,
                isTransient: true,
                bind: '_id',
                map(value, rowData) {
                    let quantityInStockUnitReceived = 0;
                    let quantityInPurchaseUnitExpected = rowData.quantityInPurchaseUnitExpected;

                    if (this.purchaseSession?.purchaseReceipt?.lines) {
                        let quantityInStockUnitExpected = Number(rowData.quantityInStockUnitShipped);
                        this.purchaseSession?.purchaseReceipt?.lines
                            ?.filter(
                                (line: PurchaseReceiptLineInput) =>
                                    line.purchaseShipment &&
                                    line.purchaseShipment === rowData.purchaseShipment?.id &&
                                    Number(line.purchaseShipmentLine) === Number(rowData.lineNumber),
                            )
                            .map(
                                (line: PurchaseReceiptLineInput) =>
                                    (quantityInStockUnitReceived +=
                                        Number(line.quantityInReceiptUnitReceived) *
                                        Number(line.receiptUnitToStockUnitConversionFactor)),
                            );

                        if (quantityInStockUnitReceived > 0) {
                            quantityInPurchaseUnitExpected = Math.max(
                                (Number(quantityInStockUnitExpected) - Number(quantityInStockUnitReceived)) /
                                    Number(rowData.product.purchaseUnitToStockUnitConversionFactor),
                                0,
                            );
                        }
                    }

                    return Number.isInteger(Number(quantityInPurchaseUnitExpected))
                        ? quantityInPurchaseUnitExpected.toString()
                        : quantityInPurchaseUnitExpected.toFixed(QTY_DECIMAL_PLACES);
                },
            }),
            line4: ui.nestedFields.numeric({
                bind: 'quantityInPurchaseUnitShipped',
                title: 'Shipped qty',
                isHidden: true,
            }),
            line4Right: ui.nestedFields.numeric({
                bind: 'quantityInPurchaseUnitExpected',
                title: 'Expected qty',
                isHidden: true,
            }),
            line5: ui.nestedFields.reference({
                node: '@sage/x3-master-data/UnitOfMeasure',
                bind: 'purchaseUnit',
                valueField: 'code',
                isHidden: true,
            }),
            line5Right: ui.nestedFields.numeric({
                bind: 'quantityInOrderUnitShipped',
                isHidden: true,
            }),
            line6: ui.nestedFields.numeric({
                bind: 'quantityInStockUnitShipped',
                isHidden: true,
            }),
            line6Right: ui.nestedFields.numeric({
                bind: 'orderUnitToPurchaseUnitConversionFactor',
                isHidden: true,
            }),
            line7: ui.nestedFields.reference({
                node: '@sage/x3-master-data/UnitOfMeasure',
                bind: 'stockUnit',
                valueField: 'code',
                isHidden: true,
            }),
            line7Right: ui.nestedFields.reference({
                node: '@sage/x3-master-data/UnitOfMeasure',
                bind: 'orderUnit',
                valueField: 'code',
                isHidden: true,
            }),
            line8: ui.nestedFields.numeric({
                bind: { product: { purchaseUnitToStockUnitConversionFactor: true } },
                isReadOnly: true,
                isHidden: true,
            }),
        },
        orderBy: {
            purchaseShipment: { id: 1 },
            lineNumber: 1,
        },
        optionsMenu: [
            {
                title: '',
                graphQLFilter: (storage: any, queryParameters: any) => ({
                    purchaseOrderNumber: { id: queryParameters?.purchaseOrderId },
                    purchaseOrderLineNumber: Number(queryParameters?.purchaseOrderLineNumber),
                    purchaseShipment: {
                        id: { _in: JSON.parse(queryParameters.shipmentToLoad) as any },
                    },
                }),
            },
        ],

        onSelect(listItemValue: any) {
            const currentLineNumber = this.purchaseSession?.purchaseReceipt?.lines.length - 1;
            const currentLine = this.purchaseSession?.purchaseReceipt?.lines[currentLineNumber];
            if (currentLine) {
                currentLine.purchaseShipment = String(listItemValue.purchaseShipment.id);
                currentLine.purchaseShipmentLine = String(listItemValue.lineNumber);
            }
            this._purchaseReceiptRecordManager.purchaseSession = this.purchaseSession;

            this.$.router.goTo('@sage/x3-purchasing/MobilePurchaseReceiptEnterReceiptDetail', {
                globalTradeItemNumber: `${(this.$.queryParameters.globalTradeItemNumber as string) ?? ''}`,
            });
            return true;
        },
    },

    onLoad() {
        this._purchaseReceiptRecordManager = new PurchaseReceiptDetailsRecordManager(this);
        this.purchaseSession = this._purchaseReceiptRecordManager.purchaseSession;
        if (!this.purchaseSession) {
            return;
        }
    },
})
export class MobilePurchaseReceiptSelectAShipment extends ui.Page<GraphApi> {
    private _purchaseReceiptRecordManager: PurchaseReceiptDetailsRecordManager;
    public purchaseSession: PurchaseReceiptSession;

    @ui.decorators.section<MobilePurchaseReceiptSelectAShipment>({
        isHidden: true,
    })
    mainSection: ui.containers.Section;

    @ui.decorators.block<MobilePurchaseReceiptSelectAShipment>({
        parent() {
            return this.mainSection;
        },
    })
    mainBlock: ui.containers.Block;
}
