import { Container, ProductSite, ProductVersion } from '@sage/x3-master-data-api';
import { Product } from '@sage/x3-master-data-api-partial';
import { dialogConfirmation, dialogMessage } from '@sage/x3-master-data/lib/client-functions/dialogs';
import { onGoto } from '@sage/x3-master-data/lib/client-functions/on-goto';
import { GraphApi } from '@sage/x3-purchasing-api';
import { LicensePlateNumber, Location, Lot, LotsSites, MajorVersionStatus } from '@sage/x3-stock-data-api';
import {
    AsyncCompositeAllowed,
    AsyncVoidFunction,
    DictionaryDataComposite,
    DictionaryFieldSupported,
} from '@sage/x3-system/lib/client-functions/screen-management-gs-1';
import { SupportServiceManagementGs1Page } from '@sage/x3-system/lib/client-functions/support-service-management-gs-1-page';
import { DataTitle } from '@sage/x3-system/lib/shared-functions/parsed-element';
import { ExtractEdges, Filter, extractEdges } from '@sage/xtrem-client';
import { DateValue } from '@sage/xtrem-date-time';
import * as ui from '@sage/xtrem-ui';
import { expirationDateDefaultValue, useByDateDefaultValue } from '../client-functions/defaultValue';
import {
    //validate,
    PurchaseReceiptDetailsRecordManager,
    PurchaseReceiptSession,
    controlLotReceipt,
} from '../client-functions/purchase-receipt-details-control';
import { PurchaseReceiptDetailsInitializer } from '../client-functions/purchase-receipt-details-initializer';
import { readParameterValue } from '../client-functions/read-parameter';
import { findDefaultLocation, getLocationPreloaded } from '../client-functions/stock-management-rules';
import { mobileApplicationGs1Key } from './mobile-purchase-receipt';

export type packingUnit = {
    node: {
        packingUnit: {
            code: string;
            numberOfDecimals: number;
            name: string;
        };
        packingUnitToStockUnitConversionFactor: string;
        isPackingFactorEntryAllowed: boolean;
        isPackingUnit: boolean;
    };
};

const closePoLineOptions: string[] = [
    ui.localize('@sage/x3-purchasing/Yes', 'Yes'),
    ui.localize('@sage/x3-purchasing/No', 'No'),
];

function getPackingUnitFromName(name: string, packingUnits: packingUnit[]): number {
    return packingUnits.findIndex(packingUnit => packingUnit.node.packingUnit.name === name);
}

@ui.decorators.page<MobilePurchaseReceiptEnterReceiptDetail>({
    title: 'Purchase receipt',
    subtitle: 'Enter details',
    module: 'x3-purchasing',
    mode: 'default',
    isTransient: true,
    isTitleHidden: true,
    headerCard() {
        return {
            title: this.headerTitleLeft,
            titleRight: this.headerTitleRight,
            line2: this.headerLine2Left,
            line2Right: this.headerLine2Right,
            line3: this.headerLine3Left,
            line3Right: this.headerLine3Right,
        };
    },
    businessActions() {
        return [this.addDetails, this.addProduct];
    },
    async onLoad() {
        this.isLocationPreloaded = this.$.queryParameters?.isLocationPreloaded === '1';

        this._purchaseReceiptRecordManager = new PurchaseReceiptDetailsRecordManager(this);
        this.purchaseSession = this._purchaseReceiptRecordManager.purchaseSession;
        if (!this.purchaseSession) {
            this.addDetails.isHidden = true;
            this.addProduct.isHidden = true;
            this.blockProduct.isHidden = true;
            return;
        }

        if (!(await PurchaseReceiptDetailsInitializer.initializePage(this as any, closePoLineOptions))) {
            this.clearAllCompositeDataAndStorage(mobileApplicationGs1Key);
            // TODO: Return a non-developer error message to user about invalid session/routing?
            onGoto(this, '@sage/x3-purchasing/MobilePurchaseReceipt');
            return;
        }

        // Retrieve current product globalTradeItemNumber from the main page
        this._globalTradeItemNumber = (this.$.queryParameters?.globalTradeItemNumber as string) ?? '';

        // Allow reception greater than the order based on RCPORD setting. Storing value in local storage so only need to read once.
        this._isOverReceiptAllowed =
            (await readParameterValue('RCPORD', this.$.userCode ?? '', this.site.value ?? '', this)) === 2;

        this._initDirty();
        await this._focusToNextField(this.receiptUnit);

        if (!(await this._initControlManagerGs1(this.site.value ?? ''))) {
            this.clearAllCompositeDataAndStorage(mobileApplicationGs1Key);
            // TODO: What action should be taken in the event of a fatal error ?
            onGoto(this, '@sage/x3-purchasing/MobilePurchaseReceipt');
            return;
        }

        // Composite data having been dispatched during the initialization phase, can now be deleted from the service
        // because we don't need it anymore.
        this.clearCompositeData();
    },
})
export class MobilePurchaseReceiptEnterReceiptDetail extends SupportServiceManagementGs1Page<GraphApi> {
    /**
     *
     * Technical composite data properties
     *
     */

    /** This value is initialized only by main page
     * and used only for control composite data block
     */
    /* @internal */
    private _globalTradeItemNumber: string | null = null;

    /**
     * This value is initialized only when dispatching
     * and reset when user change lot or expiration date.
     */
    /* @internal */
    private _batchLot: string | null = null;

    /**
     * This value is initialized only when dispatching
     * and reset when user change lot or expiration date.
     */
    /* @internal */
    private _expiryDate: string | null = null;

    /*
     *
     *  Technical properties
     *
     */

    private _purchaseReceiptRecordManager: PurchaseReceiptDetailsRecordManager; // to store the receipt to create from session data
    private _effectiveDate = DateValue.today().toString();
    public purchaseSession: PurchaseReceiptSession;
    private _isOverReceiptAllowed: boolean | undefined;

    receiptUnits: packingUnit[] = [];
    productSite: ProductSite | undefined;
    productNode: Product;
    purchaseOrderNumber: string;
    purchaseOrderLineNumber: string | number | boolean;
    isPackingUnit: boolean;
    isExpirationManaged: boolean;
    quantityToReceiveNoRounded: number;
    quantityToReceiveRounded: number;
    remainingQuantityNoRoundedInPackingUnit: number;
    remainingQuantityRoundedInPackingUnit: number;
    isAddQuantity: boolean = false;
    previousPackingUnitToStockUnitConversionFactor: number;
    remainingQuantityInitialInStockUnit: number;
    remainingQuantityInitialInOrderUnit: number;
    orderUnit: {
        code: string;
    };
    isLocationPreloaded: boolean;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        isTransient: true,
    })
    headerTitleLeft: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        isTransient: true,
    })
    headerTitleRight: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        isTransient: true,
    })
    headerLine2Left: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        isTransient: true,
    })
    headerLine2Right: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        isTransient: true,
    })
    headerLine3Left: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        isTransient: true,
    })
    headerLine3Right: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        title: 'Product',
        isReadOnly: true,
        size: 'large',
    })
    product: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        title: 'Description',
        isReadOnly: true,
        size: 'small',
    })
    description: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        title: 'Site',
        isReadOnly: true,
        isTransient: true,
    })
    site: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        title: 'Supplier',
        isReadOnly: true,
        isTransient: true,
    })
    supplier: ui.fields.Text;

    /*
     *
     *  Page Actions
     *
     */

    @ui.decorators.pageAction<MobilePurchaseReceiptEnterReceiptDetail>({
        title: 'Add quantity',
        buttonType: 'secondary',
        shortcut: ['f7'],
        async onClick() {
            if (await this._isEmptyLpnAndContainerValues()) {
                return;
            }
            if (await this._validate()) {
                const haveLicensePlateNumber: boolean = !!this.licensePlateNumber.value?.code;
                this._purchaseReceiptRecordManager.loadStockDetails(this, this.productNode);
                this.closePoLine.value = closePoLineOptions[0]; //Yes
                const containerValue = this.container.value ?? undefined;
                if (!this.container.isHidden) {
                    if (containerValue) {
                        this.container.value = containerValue;
                        this.container.isDisabled = true;
                        if (!haveLicensePlateNumber) {
                            this.licensePlateNumber.isDisabled = true;
                            this.licensePlateNumber.value = null;
                        }
                    } else if (this.licensePlateNumber.value) {
                        this.container.isDisabled = true;
                    } else {
                        if (this.productSite?.defaultInternalContainer) {
                        this.container.value = this.productSite?.defaultInternalContainer;
                        }
                        else {
                        this.container.value = null}
                        this.container.isDisabled = true;
                    }
                }
                this.isAddQuantity = true;
                this.previousPackingUnitToStockUnitConversionFactor =
                    this.packingUnitToStockUnitConversionFactor.value ?? 0;

                this.$.showToast(
                    ui.localize(
                        '@sage/x3-purchasing/dialog-notification-purchase-receipt-details-quantity-added',
                        'Quantity added',
                    ),
                    { type: 'success', timeout: 3000 },
                );
                this.$.setPageClean();
                this._resetValues();
                this._initDirty();
                await this._focusToNextField(this.receiptUnit);
            }
        },
    })
    addDetails: ui.PageAction;

    @ui.decorators.pageAction<MobilePurchaseReceiptEnterReceiptDetail>({
        title: 'Next',
        buttonType: 'primary',
        shortcut: ['f3'],
        async onClick() {
            if (await this._isEmptyLpnAndContainerValues()) {
                return;
            }
            this.addProduct.isDisabled = true;
            if (this.remainingQuantityInStockUnit.value === 0 && this.quantityToReceive.value === 0) {
                this._purchaseReceiptRecordManager.createLine(this, this.productNode);
                this.$.setPageClean();
                this.addProduct.isDisabled = false;
                this.$.router.goTo('@sage/x3-purchasing/MobilePurchaseReceipt');
            } else if (await this._validate()) {
                this._purchaseReceiptRecordManager.loadStockDetails(this, this.productNode);
                this._purchaseReceiptRecordManager.createLine(this, this.productNode);
                this.$.setPageClean();
                this.addProduct.isDisabled = false;
                this.$.router.goTo('@sage/x3-purchasing/MobilePurchaseReceipt');
            } else {
                this.addProduct.isDisabled = false;
            }
        },
    })
    addProduct: ui.PageAction;

    /*
     *
     *  Sections
     *
     */

    @ui.decorators.section<MobilePurchaseReceiptEnterReceiptDetail>({
        isTitleHidden: true,
    })
    sectionHeader: ui.containers.Section;

    /*
     *
     *  Blocks
     *
     */

    @ui.decorators.block<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.sectionHeader;
        },
        isTitleHidden: false,
    })
    blockProduct: ui.containers.Block;

    @ui.decorators.block<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.sectionHeader;
        },
    })
    blockClosePoLine: ui.containers.Block;

    /*
     *
     *  Fields
     *
     */

    @ui.decorators.selectField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Unit',
        placeholder: 'Enter...',
        options: ['UN'],
        isMandatory: true,
        // (X3-198508) TODO: Issue: Does not get triggered when value is changed to 1st option via manual typing
        onChange() {
            let packingUnitIndex;
            if (
                // To handle scenario of user manually inputting a non-existent value or clearing out value
                !this.receiptUnit.value ||
                (packingUnitIndex = getPackingUnitFromName(this.receiptUnit.value, this.receiptUnits)) === -1
            ) {
                // revert back to previous value that is saved in a hidden label
                this.receiptUnit.value = this.previousReceiptUnit.value;
                return;
            }

            const selectedUnit = this.receiptUnits[packingUnitIndex].node;
            const previousUnit =
                this.receiptUnits[getPackingUnitFromName(this.previousReceiptUnit.value ?? '', this.receiptUnits)].node;
            //if purchase order: calcul the conversion factor with the purchase order quantities
            if (this.purchaseOrderNumber && this.purchaseOrderLineNumber) {
                if (this.remainingQuantityInitialInStockUnit) {
                    // calcul packUnitTostockUnittoStockUnitConveriosnFactor : QTYSTU/QTYUOM
                    if (this.remainingQuantityInitialInOrderUnit) {
                        this.packingUnitToStockUnitConversionFactor.value =
                            this.remainingQuantityInitialInStockUnit / this.remainingQuantityInitialInOrderUnit;
                        if (selectedUnit.packingUnit.code === this.orderUnit.code)
                            selectedUnit.packingUnitToStockUnitConversionFactor =
                                this.packingUnitToStockUnitConversionFactor.value.toString();
                        if (previousUnit.packingUnit.code === this.orderUnit.code)
                            previousUnit.packingUnitToStockUnitConversionFactor =
                                this.packingUnitToStockUnitConversionFactor.value.toString();
                    }
                }
            }

            let selectedUnitConversionFactor: string;
            if (Number(selectedUnit.packingUnitToStockUnitConversionFactor) !== 0) {
                selectedUnitConversionFactor = (
                    Number(previousUnit.packingUnitToStockUnitConversionFactor) /
                    Number(selectedUnit.packingUnitToStockUnitConversionFactor)
                ).toString();
            } else {
                selectedUnitConversionFactor = Number(previousUnit.packingUnitToStockUnitConversionFactor).toString();
            }

            const tmp = Math.pow(10, selectedUnit?.packingUnit.numberOfDecimals);
            if (this.quantityToReceive.value) {
                this.quantityToReceiveNoRounded *= Number(selectedUnitConversionFactor);
                if (tmp !== 0) {
                    this.quantityToReceiveRounded = Math.round(this.quantityToReceiveNoRounded * tmp) / tmp;
                }
                this.quantityToReceive.value = this.quantityToReceiveRounded;
            }
            if (this.purchaseOrderNumber && this.purchaseOrderLineNumber) {
                if (this.remainingQuantityInitialInStockUnit && this.packingUnitToStockUnitConversionFactor.value !== 0)
                    this.remainingQuantityNoRoundedInPackingUnit = Number(
                        this.remainingQuantityInitialInStockUnit /
                            Number(this.packingUnitToStockUnitConversionFactor.value),
                    );
                if (tmp !== 0) {
                    this.remainingQuantityRoundedInPackingUnit =
                        Math.round(this.remainingQuantityNoRoundedInPackingUnit * tmp) / tmp;
                } else {
                    this.remainingQuantityNoRoundedInPackingUnit *= Number(
                        this.packingUnitToStockUnitConversionFactor.value,
                    );
                    if (tmp !== 0) {
                        this.remainingQuantityRoundedInPackingUnit =
                            Math.round(this.quantityToReceiveNoRounded * tmp) / tmp;
                    }
                }
                this.remainingQuantityInPackingUnit.value = this.remainingQuantityRoundedInPackingUnit;
            }
            this.quantityToReceive.focus(); // this moves cursor to quantityToReceive and auto-round & update # of decimal places allowed in its current value
            this.previousReceiptUnit.value = this.receiptUnit.value;

            this.receiptUnitCode.value = selectedUnit.packingUnit.code;
            this.packingUnitToStockUnitConversionFactor.value = Number(
                selectedUnit.packingUnitToStockUnitConversionFactor,
            );
            this.packingUnitToStockUnitConversionFactor.isDisabled = !selectedUnit.isPackingFactorEntryAllowed; // TODO: Obsolete isPackingFactorEntryAllowed because packingUnitToStockUnitConversionFactor is now a hidden field
            this.isPackingUnit = selectedUnit.isPackingUnit;
            this.quantityToReceive.scale = selectedUnit.packingUnit.numberOfDecimals;
        },
    })
    receiptUnit: ui.fields.Select;

    @ui.decorators.labelField<MobilePurchaseReceiptEnterReceiptDetail>({
        isHidden: true,
        isTransient: true,
    })
    receiptUnitCode: ui.fields.Label;

    @ui.decorators.labelField<MobilePurchaseReceiptEnterReceiptDetail>({
        isHidden: true,
        isTransient: true,
    })
    previousReceiptUnit: ui.fields.Label;

    @ui.decorators.numericField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Quantity',
        placeholder: 'Enter...',
        isMandatory: true,
        validation(value: number) {
            const regex = /^([1-9][0-9]*(\.[0-9]+)?|[0]+\.[0-9]*[1-9][0-9]*)$/; // reg ex for any positive numbers (integers or decimals) excluding 0
            if ((value.toString().match(regex)?.length ?? 0) === 0) {
                return ui.localize('@sage/x3-purchasing/validate-error-invalid-value', 'invalid value');
            }
            if (
                this.purchaseOrderNumber &&
                this.purchaseOrderLineNumber &&
                ((this.quantityToReceive.value ?? 0) * (this.packingUnitToStockUnitConversionFactor.value ?? 0) ?? 0) >
                    (this.remainingQuantityInStockUnit.value ?? 0) &&
                !this._isOverReceiptAllowed
            ) {
                return ui.localize(
                    '@sage/x3-purchasing/validation-error-purchase-receipt-details-quantity-received-is-greater-than-the-ordered-quantity',
                    'The quantity received is greater than the ordered quantity.',
                );
            }
            return undefined;
        },
        min: 0,
        async onChange() {
            if (this.purchaseOrderNumber && this.purchaseOrderLineNumber) {
                const tmp = Math.pow(10, this.quantityToReceive.scale ?? 1);
                if (
                    Math.round(
                        (((this.quantityToReceive.value ?? 0) *
                            (this.packingUnitToStockUnitConversionFactor.value ?? 0) ?? 0) *
                            tmp) /
                            tmp,
                    ) > (this.remainingQuantityInStockUnit.value ?? 0)
                ) {
                    if (!this._isOverReceiptAllowed) {
                        await dialogMessage(
                            this,
                            'error',
                            ui.localize('@sage/x3-purchasing/dialog-error-title', 'Error'),
                            ui.localize(
                                '@sage/x3-purchasing/dialog-error-purchase-receipt-details-quantity-received-is-greater-than-the-ordered-quantity',
                                `The quantity received is greater than the ordered quantity.`,
                            ),
                            {
                                fullScreen: false,
                                acceptButton: {
                                    text: ui.localize('@sage/x3-purchasing/button-accept-ok', 'OK'),
                                },
                            },
                        );
                        this.quantityToReceive.focus();
                        return;
                    } else {
                        if (
                            !(await dialogConfirmation(
                                this,
                                'warn',
                                ui.localize('@sage/x3-purchasing/dialog-warning-title', 'Warning'),
                                ui.localize(
                                    '@sage/x3-purchasing/dialog-warn-purchase-receipt-details-quantity-message',
                                    'Quantity larger than expected. Validate ?',
                                ),
                                {
                                    fullScreen: false,
                                    acceptButton: {
                                        text: ui.localize('@sage/x3-purchasing/button-accept-yes', 'Yes'),
                                    },
                                    cancelButton: {
                                        text: ui.localize('@sage/x3-purchasing/button-cancel-no', 'No'),
                                    },
                                }, //
                            ))
                        ) {
                            this.quantityToReceive.value = 0;
                            await this.$.commitValueAndPropertyChanges();
                            this.quantityToReceive.focus();
                        }
                    }
                }

                if (
                    ((this.quantityToReceive.value ?? 0) * (this.packingUnitToStockUnitConversionFactor.value ?? 0) ??
                        0) === this.remainingQuantityInStockUnit.value
                ) {
                    this.closePoLine.value = closePoLineOptions[0]; //Yes
                } else if (
                    ((this.quantityToReceive.value ?? 0) * (this.packingUnitToStockUnitConversionFactor.value ?? 0) ??
                        0) < (this.remainingQuantityInStockUnit.value ?? 0)
                ) {
                    this.closePoLine.value = closePoLineOptions[1]; //No
                }
            }
            if (!this.quantityToReceive.value) {
                this.quantityToReceive.value = 0;
            }
            if (this.quantityToReceive.value !== this.quantityToReceiveRounded) {
                this.quantityToReceiveNoRounded = this.quantityToReceive.value;
            }
            this.quantityToReceiveRounded = this.quantityToReceive.value;
        },
    })
    quantityToReceive: ui.fields.Numeric;

    @ui.decorators.numericField<MobilePurchaseReceiptEnterReceiptDetail>({
        // parent() {
        //     return this.blockQuantity;
        // },
        // title: 'Remaining quantity',
        // placeholder: 'Qty',
        isReadOnly: true,
        isHidden: true, // (X3-184844) TODO: Temporarily hidden until further discussion
    })
    remainingQuantityInPackingUnit: ui.fields.Numeric;

    @ui.decorators.numericField<MobilePurchaseReceiptEnterReceiptDetail>({
        // parent() {
        //     return this.blockQuantity;
        // },
        // title: 'Remaining quantity',
        // placeholder: 'Qty',
        isReadOnly: true,
        isHidden: true, // (X3-184844) TODO: Temporarily hidden until further discussion
    })
    remainingQuantityInStockUnit: ui.fields.Numeric;

    @ui.decorators.numericField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Conversion factor',
        isDisabled: true,
        isHidden: false,
        scale: 6,
        onChange() {
            if (
                this.isAddQuantity &&
                this.packingUnitToStockUnitConversionFactor.value !==
                    this.previousPackingUnitToStockUnitConversionFactor
            ) {
                this.packingUnitToStockUnitConversionFactor.value = this.previousPackingUnitToStockUnitConversionFactor;
                this.packingUnitToStockUnitConversionFactor.focus();
                this.$.showToast(
                    `${ui.localize(
                        '@sage/x3-purchasing/notification-The-conversion-factor-can-t-be-different-as-the-one-entered-previously-on-the-same-stock-line',
                        "The conversion factor can't be different as the one entered previously on the same stock line.",
                    )}`,
                    { type: 'error', timeout: 5000 },
                );
                this.packingUnitToStockUnitConversionFactor.value = this.previousPackingUnitToStockUnitConversionFactor;
                this.packingUnitToStockUnitConversionFactor.focus();
            }
        },
    })
    packingUnitToStockUnitConversionFactor: ui.fields.Numeric;

    @ui.decorators.selectField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Status',
        placeholder: 'Scan or select...',
        isMandatory: true,
        onChange() {
            this.status.getNextField(true)?.focus();
        },
    })
    status: ui.fields.Select;

    @ui.decorators.referenceField<MobilePurchaseReceiptEnterReceiptDetail, Container>({
        parent() {
            return this.blockProduct;
        },
        title: 'Container',
        node: '@sage/x3-master-data/Container',
        valueField: 'code',
        isAutoSelectEnabled: true,
        placeholder: 'Scan or select...',
        width: 'large',
        isTransient: true,
        canFilter: false,
        filter() {
            return {
                isInternal: { _eq: true },
                isActive: { _eq: true },
            };
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Code',
            }),
            ui.nestedFields.text({
                bind: 'containerType',
                title: 'Type',
            }),
            ui.nestedFields.text({
                bind: 'description',
                title: 'Description',
            }),
        ],
    })
    container: ui.fields.Reference;

    // (X3-227172) TODO: Issue: Run-time error occurs when clearing a search bar value while location field is populated
    @ui.decorators.referenceField<MobilePurchaseReceiptEnterReceiptDetail, LicensePlateNumber>({
        parent() {
            return this.blockProduct;
        },
        title: 'License plate number',
        placeholder: 'Scan or select...',
        isFullWidth: true,
        node: '@sage/x3-stock-data/LicensePlateNumber',
        isAutoSelectEnabled: true,
        valueField: 'code',
        canFilter: false,
        filter() {
            // TODO: Issue: Depending on the selected site, it can be LPN managed but contains no LPN results to select from (for ex. FR022)
            let licensePlateNumberFilter: Filter<LicensePlateNumber> = {
                stockSite: { code: this.site.value ?? undefined },
                isActive: { _eq: true },
                _or: [
                    {
                        isSingleProduct: { _eq: true },
                        stock: { _atLeast: 1, product: { product: { code: this.product.value ?? undefined } } },
                    },
                    { isSingleProduct: { _eq: true }, stock: { _none: true } },
                    { isSingleProduct: { _eq: false } },
                ],
            };

            // TODO: LPN must also be associated with container (not available yet) if populated
            if (this.location.value?.code) {
                licensePlateNumberFilter._or = [
                    {
                        location: { code: this.location.value.code },
                    },
                    {
                        // to also include entries without location that have 'free' status
                        _and: [{ location: { code: undefined } }, { status: 'free' }],
                    },
                ];
            }

            if (this.container.value) {
                licensePlateNumberFilter = {
                    ...licensePlateNumberFilter,
                    container: { code: (this.container.value as Container)?.code },
                };
            }

            return licensePlateNumberFilter;
        },
        async onChange() {
            if (!this.licensePlateNumber.value) {
                this.location.isDisabled = false;
                this.location.value = null;
                this.container.isDisabled = this.isAddQuantity;
                await this.$.commitValueAndPropertyChanges(); // without this, when you clear out LPN and then, without tabbing out, click Location's lookup button directly, nothing will happen
                return;
            }

            // LPN can be associated with or without a location
            // if location is NOT popualted manually by the user

            if (!!this.licensePlateNumber.value.location?.code) {
                this.location.isDisabled = true;
                this.location.value = this.licensePlateNumber.value.location;
                if ((this.licensePlateNumber.value as LicensePlateNumber).container) {
                    this.container.value = (this.licensePlateNumber.value as LicensePlateNumber).container;
                    this.container.isDisabled = true;
                } else {
                    this.container.isDisabled = false;
                }
            } else {
                if (this.location.isDisabled) {
                    this.location.value = null;
                    this.location.isDisabled = false;
                }
                this.container.isDisabled = this.isAddQuantity;
            }

            await this.$.commitValueAndPropertyChanges();
            this.licensePlateNumber.getNextField(true)?.focus();
        },
        isTransient: true,
        isMandatory: false,
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'License Plate Number',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-stock-data/Location',
                bind: 'location',
                valueField: 'code',
                title: 'Location',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/Container',
                bind: 'container',
                valueField: 'code',
                title: 'Container',
                isReadOnly: true,
            }),
            ui.nestedFields.label({
                bind: 'status',
                title: 'Status',
                map(value: any, rowData: any) {
                    switch (value) {
                        case 'free':
                            return 'Free';
                        case 'inStock':
                            return 'In Stock';
                        default:
                            return '';
                    }
                },
                borderColor: ui.tokens.colorsYang100,
                optionType: '@sage/x3-stock-data/ContainerStatus',
            }),
            // (X3-227347) TODO: Obsolete: Having to specify & hide fields used in filter that don't need to be displayed
            ui.nestedFields.reference({
                node: '@sage/x3-system/Site',
                bind: 'stockSite',
                valueField: 'code',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isActive',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isSingleProduct',
                isHidden: true,
            }),
        ],
    })
    licensePlateNumber: ui.fields.Reference;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Suggested Location',
        //node: '@sage/xtrem-x3-stock/Location',
        //valueField: 'code',
        isReadOnly: true,
        width: 'large',
        isTransient: true,
    })
    suggestedLocation: ui.fields.Text;

    @ui.decorators.referenceField<MobilePurchaseReceiptEnterReceiptDetail, Location>({
        parent() {
            return this.blockProduct;
        },
        title: 'Location',
        placeholder: 'Scan or select...',
        node: '@sage/x3-stock-data/Location',
        isAutoSelectEnabled: true,
        valueField: 'code',
        canFilter: false,
        filter() {
            return {
                stockSite: { code: this.site.value ?? undefined },
            };
        },
        onChange() {
            if (this.location.value) this.location.getNextField(true)?.focus();
            this.warehouse.value = !this.location.value?.warehouse ? '' : this.location.value.warehouse.code;
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Code',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'type',
                title: 'Type',
                isReadOnly: true,
            }),
            // (X3-227347) TODO: Obsolete: Having to specify & hide fields used in filter that don't need to be displayed
            ui.nestedFields.reference({
                node: '@sage/x3-system/Site',
                bind: 'stockSite',
                valueField: 'code',
                isHidden: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-stock-data/Warehouse',
                bind: 'warehouse',
                valueField: 'code',
                isHidden: true,
            }),
        ],
        minLookupCharacters: 1,
        isFullWidth: true,
        isMandatory: true,
    })
    location: ui.fields.Reference;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        isHidden: true,
        isReadOnly: true,
    })
    warehouse: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Serial number',
        placeholder: 'Scan...',
        isFullWidth: true,
        validation: /^$|^[^|]+$/,
        isMandatory: false,
        async onInputValueChange(this, rawData: string): Promise<void> {
            await this.scanBarCode(this.serialNumber, rawData);
        },
        async onChange() {
            await this._onChange_serialNumber();
        },
    })
    serialNumber: ui.fields.Text;

    // TODO: Make reference field after X3-193258 is resolved to add custom values on reference fields
    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Supplier lot',
        placeholder: 'Scan...',
        isFullWidth: true,
        validation: /^$|^[^|]+$/,
        isMandatory: false,
        async onChange() {
            const supplierLot = this.supplierLot.value?.replace(/\s/g, '')?.toUpperCase() ?? '';
            if (supplierLot !== this.supplierLot.value) {
                this.supplierLot.value = supplierLot;
                await this.$.commitValueAndPropertyChanges();
            }

            if (this.supplierLot.value && this.purchaseSession?.purchaseStockManagementRules) {
                if (this.purchaseSession.purchaseStockManagementRules.lotByDefault === 'supplierLot') {
                    if (this.purchaseSession.purchaseStockManagementRules.lotEntry === 'newLot') {
                        if (
                            (await controlLotReceipt(
                                this,
                                this.supplierLot.value,
                                this.productNode.code,
                                '6',
                                this.site.value ?? '',
                            )) === false
                        ) {
                            this.supplierLot.value = null;
                            this.supplierLot.focus();
                        }
                    }

                    /**
                     *  Only when previous lot value has not same and has editable :
                     * - Assign new lot value.
                     * - If composite control block become invalid, reset it.
                     * - Commit page change.
                     * - If lot has editable, perform explicit lot change event (no _getLotValues) and stay in this field
                     * - Overwise next focus to next field focusable
                     */
                    if (this.supplierLot.value && this.supplierLot.value != this.lot.value) {
                        this.lot.value = this.supplierLot.value;
                        if (this._batchLot && this.lot.value !== this._batchLot) {
                            // this._resetCompositeDataControl();
                        }
                        await this.$.commitValueAndPropertyChanges();
                        if (!this.lot.isHidden && !this.lot.isDisabled && !this.lot.isReadOnly) {
                            // await this._getLotValues();
                            await this._onChange_lot();
                            this.lot.focus();
                        } else {
                            this.lot.getNextField(true)?.focus();
                        }
                    } else {
                        await this.$.commitValueAndPropertyChanges();
                    }
                }
            }
        },
        // node: '@sage/x3-stockData/Stock',
        // valueField: 'supplierLot',
        // filter() {
        //     return {
        //         stockSite: { $eq: this.site.value },
        //         product: { $eq: this.product.value },
        //         //supplierLot: { $ne: null },
        //     };
        // },
        // isTransient: true,
        // isMandatory: false,
        // columns: [
        //     ui.nestedFields.text({
        //         bind: 'supplierLot',
        //         title: 'Supplier lot',
        //         isReadOnly: true,
        //     }),
        // ],
    })
    supplierLot: ui.fields.Text;

    @ui.decorators.filterSelectField<MobilePurchaseReceiptEnterReceiptDetail, LotsSites>({
        parent() {
            return this.blockProduct;
        },
        title: 'Lot',
        placeholder: 'Scan or select...',
        node: '@sage/x3-stock-data/LotsSites',
        valueField: 'lot',
        validation: /^$|^[^|]+$/,
        isMandatory: false,
        isTransient: true,
        isNewEnabled: true,
        minLookupCharacters: 1,
        canFilter: false,
        filter() {
            return {
                product: { code: this.product.value ?? undefined },
                storageSite: { code: this.site.value ?? undefined },
                stock: { _atLeast: 1 },
            };
        },
        columns: [
            ui.nestedFields.text({
                bind: 'lot',
                title: 'Lot',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'sublot',
                title: 'Sublot',
                isReadOnly: true,
            }),
        ],
        async onInputValueChange(this, rawData: string): Promise<void> {
            await this.scanBarCode(this.lot, rawData);
        },
        async onChange() {
            await this._onChange_lot();
        },
    })
    lot: ui.fields.FilterSelect<Lot>;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        isHidden: true,
        isTransient: true,
    })
    svgLot: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Sublot',
        placeholder: 'Scan...',
        isMandatory: true,
        validation: /^$|^[^|]+$/,
        async onInputValueChange(this, rawData: string): Promise<void> {
            this.expirationDate.isDisabled = true;
            this.useByDate.isDisabled = true;
        },
        async onChange() {
            this.sublot.value ? (this.sublot.value = this.sublot.value.toUpperCase()) : '';
            await this._getLotValues();
            this._getNextField(this.sublot)?.focus();
        },
    })
    sublot: ui.fields.Text;

    @ui.decorators.referenceField<MobilePurchaseReceiptEnterReceiptDetail, MajorVersionStatus>({
        parent() {
            return this.blockProduct;
        },
        title: 'Major version',
        placeholder: 'Scan or select...',
        node: '@sage/x3-stock-data/MajorVersionStatus',
        isAutoSelectEnabled: true,
        valueField: 'code',
        canFilter: false,
        filter() {
            const filter: Filter<MajorVersionStatus> = {
                product: { code: this.product.value ?? '' },
                // (X3-227355) TODO: Issue: Cannot use the less verbose _in operator instead of individual _or filter criterion
                _or: [{ status: 'prototypeVersion' }, { status: 'activeVersion' }, { status: 'stoppedVersion' }],
            };

            // if majorAndMinor managed, filter down further for major version with available minor versions
            if (!this.minorVersion.isHidden) {
                filter.minorVersions = {
                    _atLeast: 1,
                    type: { _eq: 'stock' },
                    useStatus: { _eq: 'availableToUse' },
                };
            }

            return filter;
        },
        async onChange() {
            // if not major & minor version managed
            if (this.minorVersion.isHidden) {
                if (this.majorVersion.value) {
                    this.majorVersion.getNextField(true)?.focus();
                }
            }
            // if major version is cleared out, clear out minor version if any as well
            else if ((this.minorVersion.isDisabled = !this.majorVersion.value)) {
                this.minorVersion.value = null;
            } else {
                // Auto-populate minor version based on the last minor version available
                const minorVersions = await this._getMinorVersions(this.majorVersion.value.code);
                this.minorVersion.value = minorVersions[minorVersions.length - 1];
                this._getNextField(this.majorVersion)?.focus();
            }
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Major version',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'status',
                title: 'Status',
            }),
            // (X3-227347) TODO: Obsolete: Having to specify & hide fields used in filter that don't need to be displayed
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/Product',
                bind: 'product',
                valueField: 'code',
                isHidden: true,
            }),
        ],
        isMandatory: true,
    })
    majorVersion: ui.fields.Reference;

    @ui.decorators.referenceField<MobilePurchaseReceiptEnterReceiptDetail, ProductVersion>({
        parent() {
            return this.blockProduct;
        },
        title: 'Minor version',
        placeholder: 'Scan or select...',
        node: '@sage/x3-master-data/ProductVersion',
        isAutoSelectEnabled: true,
        valueField: 'minorVersion',
        canFilter: false,
        filter() {
            return {
                product: { code: this.product.value ?? '' },
                majorVersion: this.majorVersion.value?.code,
                type: 'stock',
                useStatus: 'availableToUse',
            };
        },
        onChange() {
            if (this.minorVersion.value) this._getNextField(this.minorVersion)?.focus();
        },
        columns: [
            ui.nestedFields.text({
                bind: 'minorVersion',
                title: 'Minor version',
                isReadOnly: true,
            }),
            // (X3-227347) TODO: Obsolete: Having to specify & hide fields used in filter that don't need to be displayed
            ui.nestedFields.reference({
                bind: 'product',
                node: '@sage/x3-master-data/Product',
                valueField: 'code',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'majorVersion',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'type',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'useStatus',
                isHidden: true,
            }),
        ],
        isMandatory: true,
    })
    minorVersion: ui.fields.Reference;

    @ui.decorators.numericField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        placeholder: 'Scan...',
        title: 'Potency %',
        isMandatory: true,
        scale: 4,
        validation: /^([1-9][0-9]*(\.[0-9]+)?|[0]+\.[0-9]*[1-9][0-9]*)$/, // reg ex for any positive numbers (integers or decimals) excluding 0
        min: 0,
        max: 100,
    })
    potency: ui.fields.Numeric;

    @ui.decorators.dateField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Expiration date',
        placeholder: 'Enter...',
        isMandatory: true,
        async onInputValueChange(this, rawData: string): Promise<void> {
            await this.scanBarCode(this.expirationDate, rawData);
        },
        async onChange() {
            await this._onChange_expirationDate();
        },
    })
    expirationDate: ui.fields.Date;

    @ui.decorators.dateField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        placeholder: 'Enter...',
        title: 'Use-by date',
        isMandatory: true,
    })
    useByDate: ui.fields.Date;

    @ui.decorators.dateField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Expiration date',
        isDisabled: true,
        isHidden: true,
    })
    calculatedExpirationDate: ui.fields.Date;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        placeholder: 'Scan...',
        title: 'Lot custom field 1',
        isMandatory: false,
        validation: /^$|^[^|]+$/,
        maxLength: 20,
    })
    lotCustomField1: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        placeholder: 'Scan...',
        title: 'Lot custom field 2',
        isMandatory: false,
        validation: /^$|^[^|]+$/,
        maxLength: 10,
    })
    lotCustomField2: ui.fields.Text;

    @ui.decorators.numericField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        placeholder: 'Scan...',
        title: 'Lot custom field 3',
        isMandatory: false,
    })
    lotCustomField3: ui.fields.Numeric;

    @ui.decorators.dateField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        placeholder: 'Enter...',
        title: 'Lot custom field 4',
        isMandatory: false,
    })
    lotCustomField4: ui.fields.Date;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Identifier 1',
        placeholder: 'Scan...',
        isMandatory: false,
        validation: /^$|^[^|]+$/,
        maxLength: 10,
    })
    identifier1: ui.fields.Text;

    @ui.decorators.textField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Identifier 2',
        placeholder: 'Scan...',
        validation: /^$|^[^|]+$/,
        isMandatory: false,
        maxLength: 10,
    })
    identifier2: ui.fields.Text;

    @ui.decorators.selectField<MobilePurchaseReceiptEnterReceiptDetail>({
        parent() {
            return this.blockProduct;
        },
        title: 'Close PO line',
        placeholder: 'Select...',
        isFullWidth: true,
        // options: ['Yes', 'No'],
    })
    closePoLine: ui.fields.Select;

    /**
     *
     * Technical composite data methods
     *
     */

    /**
     * Initialize ControlManagerGs1
     * @returns true when ControlManagerGs1 has usable
     */
    private async _initControlManagerGs1(site: string): Promise<boolean> {
        this._resetCompositeDataControl();
        return await this.createAndInitServiceGs1(
            site,
            mobileApplicationGs1Key,
            {
                [DataTitle.serial]: {
                    mainField: this.serialNumber,
                    onChangeMainField: this._onChange_serialNumber,
                },
                [DataTitle.batchLot]: {
                    mainField: this.lot,
                    onChangeMainField: this._onChange_lot,
                },
                [DataTitle.expiryDate]: {
                    mainField: this.expirationDate,
                    onChangeMainField: this._onChange_expirationDate,
                },
            } as DictionaryFieldSupported,
            undefined,
            this._checkCompositeDataAllowed,
        );
    }

    /**
     * Reinitialize all control data block
     */
    /** @internal */
    private _resetCompositeDataControl(): void {
        this._batchLot = null;
        this._expiryDate = null;
    }

    /**
     * This asynchronous readonly function check if current composite data is valid or not
     * @param dictionaryDataComposite : dictionary of composite data block.
     * @returns false when data must be discarded
     */
    /** @internal */
    private readonly _checkCompositeDataAllowed: AsyncCompositeAllowed = async (
        dictionaryDataComposite: DictionaryDataComposite,
    ) => {
        const product = dictionaryDataComposite[DataTitle.gtin];

        if (
            !!this.product?.value &&
            ((!product && Object.keys(dictionaryDataComposite).length !== 0) ||
                (product && (this._globalTradeItemNumber ?? '') !== product.data))
        ) {
            await this.$.sound.error();
            await dialogMessage(
                this,
                'error',
                ui.localize('@sage/x3-purchasing/dialog-error-title', 'Error'),
                ui.localize(
                    '@sage/x3-purchasing/dialog-error-gs-1-data-is-not-related-to-product',
                    'GS1 data is not related to product {{ product }}.',
                    { product: this.product?.value },
                ),
                {
                    fullScreen: false,
                },
            );
            return false;
        }
        return true;
    };

    /**
     *
     * Private methods
     *
     */

    /**
     * OnChange readonly process
     *
     * Used both decorator and bar code manager.
     * @returns Promise<void>
     */

    /** @internal */
    private readonly _onChange_serialNumber: AsyncVoidFunction = async () => {};

    /** @internal */
    private readonly _onChange_lot: AsyncVoidFunction = async () => {
        this.lot.value ? (this.lot.value = this.lot.value.toUpperCase()) : '';
        if (this.lot.value && this.purchaseSession?.purchaseStockManagementRules?.lotEntry === 'newLot') {
            if (
                !(await controlLotReceipt(
                    this,
                    this.lot.value,
                    this.productNode.code,
                    '6',
                    this.site.value ?? '',
                ))
            ) {
                this.lot.value = null;
                this.lot.focus();
            }
        }

        /**
         * This bloc initialize current batch / lot during scanning,
         * any change after invalidate the entire control data block
         * */
        if (this.controlManagerGs1.isDispatchInProgress) {
            this._batchLot = this.lot.value;
        } else if (this._batchLot !== this.lot.value) {
            this._resetCompositeDataControl();
        }

        this.sublot.value && !this.sublot.isHidden && this.svgLot.value !== this.lot.value
            ? (this.sublot.value = null)
            : null;

        await this._getLotValues();

        this.svgLot.value = this.lot.value;

        if (this.lot.value) {
            this._getNextField(this.lot)?.focus();
        }
    };

    /** @internal */
    private readonly _onChange_expirationDate: AsyncVoidFunction = async () => {
        /**
         * This bloc initialize current batch / lot during scanning,
         * any change after invalidate the entire control data block
         * */
        if (this.controlManagerGs1.isDispatchInProgress) {
            this._expiryDate = this.expirationDate.value;
        } else if (this._expiryDate !== this.expirationDate.value) {
            this._resetCompositeDataControl();
        }

        this.useByDate.value = useByDateDefaultValue(
            this.expirationDate.value ?? '',
            this._effectiveDate,
            this.productNode.expirationManagementMode,
            Number(this.productNode.useByDateCoefficient),
        );
        this.useByDate.maxDate = this.expirationDate.value ?? undefined;
    };

    private _getNextField(field: any) {
        let _nextField = field.getNextField(true);
        if (this.productNode.expirationManagementMode !== 'manualEntry') {
            while (_nextField === this.expirationDate || _nextField === this.useByDate) {
                _nextField = _nextField.getNextField(true);
            }
        }
        return _nextField;
    }

    private async _getLotValues(): Promise<void> {
        if (this.lot.value && (this.sublot.isHidden || this.sublot.value)) {
            try {
                const result = extractEdges(
                    await this.$.graph
                        .node('@sage/x3-stock-data/Lot')
                        .query(
                            ui.queryUtils.edgesSelector(
                                {
                                    code: true,
                                    product: {
                                        code: true,
                                    },
                                    sublot: true,
                                    supplierLot: true,
                                    expirationDate: true,
                                    lotCustomField1: true,
                                    lotCustomField2: true,
                                    lotCustomField3: true,
                                    lotCustomField4: true,
                                    majorVersion: {
                                        code: true,
                                    },
                                    minorVersion: true,
                                    potency: true,
                                    useByDate: true,
                                },
                                {
                                    filter: {
                                        product: this.product.value,
                                        code: this.lot.value,
                                        sublot: this.sublot.value === null ? '' : this.sublot.value,
                                    },
                                },
                            ),
                        )
                        .execute(),
                ) as ExtractEdges<Lot>[];
                if (result.length !== 0) {
                    await this._setLotValues(result);
                    // this.$.commitValueAndPropertyChanges();
                } else {
                    await this._initLotValues();
                }
            } catch (e) {
                await dialogMessage(
                    this,
                    'error',
                    ui.localize('@sage/x3-purchasing/dialog-error-title', 'Error'),
                    ui.localize(
                        '@sage/x3-purchasing/dialog-error-set-expdate',
                        'Error while reading the lot information',
                    ) + String(e),
                );
                return;
            }
        } else {
            await this._initLotValues();
        }
    }

    private async _setLotValues(lotValues: ExtractEdges<Lot>[]): Promise<void> {
        if (this.isExpirationManaged) {
            this.expirationDate.value = lotValues[0].expirationDate;
        }
        this.useByDate.value = lotValues[0].useByDate;
        this.supplierLot.value = lotValues[0].supplierLot;
        this.lotCustomField1.value = lotValues[0].lotCustomField1;
        this.lotCustomField2.value = lotValues[0].lotCustomField2;
        this.lotCustomField3.value = Number(lotValues[0].lotCustomField3);
        this.lotCustomField4.value = lotValues[0].lotCustomField4;
        this.majorVersion.value = null;
        this.minorVersion.value = null;
        this.$.commitValueAndPropertyChanges();
        this.majorVersion.value = { code: lotValues[0].majorVersion ? lotValues[0].majorVersion.code : undefined };
        this.minorVersion.value = {
            minorVersion: lotValues[0].minorVersion,
        };
        this.potency.value = Number(lotValues[0].potency);
        this.supplierLot.isDisabled = true;
        this.lotCustomField1.isDisabled = true;
        this.lotCustomField2.isDisabled = true;
        this.lotCustomField3.isDisabled = true;
        this.lotCustomField4.isDisabled = true;
        this.majorVersion.isDisabled = true;
        this.minorVersion.isDisabled = true;
        this.potency.isDisabled = true;
        this.expirationDate.isDisabled = true;
        this.useByDate.isDisabled = true;
    }

    private async _initLotValues(): Promise<void> {
        const expirationManagement = this.productNode.expirationManagementMode;
        if (this.isExpirationManaged) {
            this.expirationDate.value = this.useByDate.value = null;
        }

        /**
         * When dispatching has finished, lot and expiry date must have data :
         * in this case, our may using saved date.
         * Otherwise, user have make some change OR composite information have missing,
         * and all controls data must be cleared before to continue with standard value.
         */
        if (!this.controlManagerGs1.isDispatchInProgress) {
            if (this._batchLot !== null && this._expiryDate !== null) {
                this.expirationDate.value = this._expiryDate;
            } else {
                this._resetCompositeDataControl();
            }
        }

        // Initialize only expiration date when is not received before
        if (!this.expirationDate.value) {
            this.expirationDate.value = expirationDateDefaultValue(
                this.productNode.expirationManagementMode,
                this.productNode.expirationLeadTime,
                this.productNode.expirationTimeUnit,
                this._effectiveDate,
            );
        }

        this.useByDate.value = useByDateDefaultValue(
            this.expirationDate.value,
            this._effectiveDate,
            this.productNode.expirationManagementMode,
            Number(this.productNode.useByDateCoefficient),
        );

        this.useByDate.maxDate = this.expirationDate.value ?? undefined;

        //disable expirationDate field
        this.expirationDate.isDisabled = false;

        //disable useByDate field
        this.useByDate.isDisabled = ['roundingBeginningMonth1', 'roundingMonthEnd'].includes(expirationManagement);

        this.lotCustomField1.value = null;
        this.lotCustomField2.value = null;
        this.lotCustomField3.value = null;
        this.lotCustomField4.value = null;
        this.potency.value = Number(this.productNode.defaultPotencyInPercentage);
        this.supplierLot.isDisabled = false;
        this.lotCustomField1.isDisabled = false;
        this.lotCustomField2.isDisabled = false;
        this.lotCustomField3.isDisabled = false;
        this.lotCustomField4.isDisabled = false;
        this.majorVersion.isDisabled = false;
        this.minorVersion.isDisabled = !this.majorVersion.value;
        this.potency.isDisabled = false;
    }

    private async _resetValues() {
        // Reinitialize all control composite data before to continue
        this._resetCompositeDataControl();
        if (!this.purchaseSession?.purchaseStockManagementRules) {
            return;
        }
        //location and suggested location
        this.location.value = null;
        if (this.productSite && !this.suggestedLocation.isHidden) {
            const suggestedLocation = await findDefaultLocation(
                this.productSite,
                this.purchaseSession.purchaseStockManagementRules,
                this,
            );
            if (!suggestedLocation) this.suggestedLocation.isHidden = true;
            else {
                this.suggestedLocation.value = suggestedLocation;
                if (this.isLocationPreloaded && this.site.value) {
                    this.location.value = await getLocationPreloaded(this.site.value, suggestedLocation, this);
                }
            }
        }
        //lot and sublot
        const lotManagementRule = this.purchaseSession.purchaseStockManagementRules.lotEntry;
        const lotByDefault = this.purchaseSession.purchaseStockManagementRules.lotByDefault;
        this.lot.value = null;
        if (!this.lot.isHidden) {
            if (lotManagementRule === 'no') {
                if (this.sublot.isHidden === false) {
                    this.sublot.value = '00001';
                }
            } else if (lotManagementRule === 'newLot') {
                if (this.sublot.isHidden === false) this.sublot.value = '00001';
            } else if (lotManagementRule === 'free' && lotByDefault === 'documentNumber') {
                if (this.sublot.isHidden === false) this.sublot.value = '00001';
            } else {
                if (this.sublot.isHidden === false) this.sublot.value = null;
            }
        }
        //expiration date and use by date
        const expirationManagement = this.productNode.expirationManagementMode;
        const effectiveDate = DateValue.today().toString();
        if (!this.expirationDate.isHidden && expirationManagement !== 'manualEntry') {
            this.expirationDate.value = expirationDateDefaultValue(
                this.productNode.expirationManagementMode,
                this.productNode.expirationLeadTime,
                this.productNode.expirationTimeUnit,
                effectiveDate,
            );

            this.useByDate.value = useByDateDefaultValue(
                this.expirationDate.value,
                effectiveDate,
                this.productNode.expirationManagementMode,
                Number(this.productNode.useByDateCoefficient),
            );
            this.useByDate.maxDate = this.expirationDate.value ?? undefined;
        } else {
            this.expirationDate.value = null;
            this.useByDate.value = null;
        }
        this.supplierLot.value = null;
        this.lotCustomField1.value = null;
        this.lotCustomField2.value = null;
        this.lotCustomField3.value = null;
        this.lotCustomField4.value = null;
        this.majorVersion.value = null;
        this.minorVersion.value = null;
        this.status.value = this.purchaseSession.purchaseStockManagementRules.defaultStatus;
        if (!this.potency.isHidden) {
            this.potency.value = Number(this.productNode.defaultPotencyInPercentage);
        }
        this.identifier1.value = null;
        this.identifier2.value = null;
        //if (!this.container.isHidden) this.container.value = this.productSite.defaultInternalContainer;
    }

    // (X3-226769) TODO: Issue: Need better more dynamic way to know which next field to focus, rather than hard-coding every possible scenario
    private async _focusToNextField(
        field: ui.fields.FilterSelect | ui.fields.Text | ui.fields.Reference | ui.fields.Select,
    ) {
        switch (field?.id) {
            default:
                if (
                    !this.quantityToReceive.isHidden &&
                    !this.quantityToReceive.isDisabled &&
                    !this.quantityToReceive.isReadOnly
                ) {
                    this.quantityToReceive.focus();
                    break;
                }
            case this.receiptUnit.id:
                if (
                    !this.quantityToReceive.isHidden &&
                    !this.quantityToReceive.isDisabled &&
                    !this.quantityToReceive.isReadOnly
                ) {
                    this.quantityToReceive.focus();
                    break;
                }
            case this.quantityToReceive.id:
                if (!this.status.isHidden && !this.status.isDisabled && !this.status.isReadOnly) {
                    this.status.focus();
                    break;
                }
            case this.status.id:
                if (!this.container.isHidden && !this.container.isDisabled && !this.container.isReadOnly) {
                    this.container.focus();
                    break;
                }
            case this.container.id:
                if (
                    !this.licensePlateNumber.isHidden &&
                    !this.licensePlateNumber.isDisabled &&
                    !this.licensePlateNumber.isReadOnly
                ) {
                    this.licensePlateNumber.focus();
                    break;
                }
            case this.licensePlateNumber.id:
                if (!this.location.isHidden && !this.location.isDisabled && !this.location.isReadOnly) {
                    this.location.focus();
                    break;
                }
            case this.location.id:
                if (!this.serialNumber.isHidden && !this.serialNumber.isDisabled && !this.serialNumber.isReadOnly) {
                    this.serialNumber.focus();
                    break;
                }
            case this.serialNumber.id:
                if (!this.supplierLot.isHidden && !this.supplierLot.isDisabled && !this.supplierLot.isReadOnly) {
                    this.supplierLot.focus();
                    break;
                }
            case this.supplierLot.id:
                if (!this.lot.isHidden && !this.lot.isDisabled && !this.lot.isReadOnly) {
                    this.lot.focus();
                    break;
                }
            case this.lot.id:
                if (!this.sublot.isHidden && !this.sublot.isDisabled && !this.sublot.isReadOnly) {
                    this.sublot.focus();
                    break;
                }
            case this.sublot.id:
                if (!this.majorVersion.isHidden && !this.majorVersion.isDisabled && !this.majorVersion.isReadOnly) {
                    this.majorVersion.focus();
                    break;
                }
            case this.majorVersion.id:
                if (!this.minorVersion.isHidden && !this.minorVersion.isDisabled && !this.minorVersion.isReadOnly) {
                    this.minorVersion.focus();
                    break;
                }
            case this.minorVersion.id:
                if (!this.potency.isHidden && !this.potency.isDisabled && !this.potency.isReadOnly) {
                    this.potency.focus();
                    break;
                }
            case this.potency.id:
                if (
                    !this.expirationDate.isHidden &&
                    !this.expirationDate.isDisabled &&
                    !this.expirationDate.isReadOnly
                ) {
                    this.expirationDate.focus();
                    break;
                }
            case this.expirationDate.id:
                if (!this.useByDate.isHidden && !this.useByDate.isDisabled && !this.useByDate.isReadOnly) {
                    this.useByDate.focus();
                    break;
                }
            case this.useByDate.id:
                if (
                    !this.calculatedExpirationDate.isHidden &&
                    !this.calculatedExpirationDate.isDisabled &&
                    !this.calculatedExpirationDate.isReadOnly
                ) {
                    this.calculatedExpirationDate.focus();
                    break;
                }
            case this.calculatedExpirationDate.id:
                if (
                    !this.lotCustomField1.isHidden &&
                    !this.lotCustomField1.isDisabled &&
                    !this.lotCustomField1.isReadOnly
                ) {
                    this.lotCustomField1.focus();
                    break;
                }
            case this.lotCustomField1.id:
                if (
                    !this.lotCustomField2.isHidden &&
                    !this.lotCustomField2.isDisabled &&
                    !this.lotCustomField2.isReadOnly
                ) {
                    this.lotCustomField2.focus();
                    break;
                }
            case this.lotCustomField2.id:
                if (
                    !this.lotCustomField3.isHidden &&
                    !this.lotCustomField3.isDisabled &&
                    !this.lotCustomField3.isReadOnly
                ) {
                    this.lotCustomField3.focus();
                    break;
                }
            case this.lotCustomField3.id:
                if (
                    !this.lotCustomField4.isHidden &&
                    !this.lotCustomField4.isDisabled &&
                    !this.lotCustomField4.isReadOnly
                ) {
                    this.lotCustomField4.focus();
                    break;
                }
            case this.lotCustomField4.id:
                if (!this.identifier1.isHidden && !this.identifier1.isDisabled && !this.identifier1.isReadOnly) {
                    this.identifier1.focus();
                    break;
                }
            case this.identifier1.id:
                if (!this.identifier2.isHidden && !this.identifier2.isDisabled && !this.identifier2.isReadOnly) {
                    this.identifier2.focus();
                    break;
                }
            case this.identifier2.id:
                if (!this.closePoLine.isHidden && !this.closePoLine.isDisabled && !this.closePoLine.isReadOnly) {
                    this.closePoLine.focus();
                    break;
                }
            case this.closePoLine.id: // do nothing, cannot focus on a button in xtreem
        }
    }

    // (X3-201046, X3-203885) TODO: Obsolete: Hard-coded way to create a comprehensive validation error message for this page
    private async _validate(): Promise<boolean> {
        const fieldErrors = new Array<string>();
        if (await this._isFieldInvalid(this.receiptUnit)) fieldErrors.push(this.receiptUnit.title ?? '');
        if (await this._isFieldInvalid(this.quantityToReceive)) fieldErrors.push(this.quantityToReceive.title ?? '');
        if (await this._isFieldInvalid(this.status)) fieldErrors.push(this.status.title ?? '');
        if (await this._isFieldInvalid(this.licensePlateNumber)) fieldErrors.push(this.licensePlateNumber.title ?? '');
        if (await this._isFieldInvalid(this.location)) fieldErrors.push(this.location.title ?? '');
        if (await this._isFieldInvalid(this.serialNumber)) fieldErrors.push(this.serialNumber.title ?? '');
        if (await this._isFieldInvalid(this.supplierLot)) fieldErrors.push(this.supplierLot.title ?? '');
        if (await this._isFieldInvalid(this.lot)) fieldErrors.push(this.lot.title ?? '');
        if (await this._isFieldInvalid(this.sublot)) fieldErrors.push(this.sublot.title ?? '');
        if (await this._isFieldInvalid(this.majorVersion)) fieldErrors.push(this.majorVersion.title ?? '');
        if (await this._isFieldInvalid(this.minorVersion)) fieldErrors.push(this.minorVersion.title ?? '');
        if (await this._isFieldInvalid(this.potency)) fieldErrors.push(this.potency.title ?? '');
        if (await this._isFieldInvalid(this.expirationDate)) fieldErrors.push(this.expirationDate.title ?? '');
        if (await this._isFieldInvalid(this.useByDate)) fieldErrors.push(this.useByDate.title ?? '');
        if (await this._isFieldInvalid(this.calculatedExpirationDate))
            fieldErrors.push(this.calculatedExpirationDate.title ?? '');
        if (await this._isFieldInvalid(this.lotCustomField1)) fieldErrors.push(this.lotCustomField1.title ?? '');
        if (await this._isFieldInvalid(this.lotCustomField2)) fieldErrors.push(this.lotCustomField2.title ?? '');
        if (await this._isFieldInvalid(this.lotCustomField3)) fieldErrors.push(this.lotCustomField3.title ?? '');
        if (await this._isFieldInvalid(this.lotCustomField4)) fieldErrors.push(this.lotCustomField4.title ?? '');
        if (await this._isFieldInvalid(this.identifier1)) fieldErrors.push(this.identifier1.title ?? '');
        if (await this._isFieldInvalid(this.identifier2)) fieldErrors.push(this.identifier2.title ?? '');
        if (await this._isFieldInvalid(this.closePoLine)) fieldErrors.push(this.closePoLine.title ?? '');
        if (await this._isFieldInvalid(this.container)) fieldErrors.push(this.container.title ?? '');
        if (fieldErrors.length > 0) {
            this.$.removeToasts();
            this.$.showToast(
                ui.localize(
                    '@sage/x3-purchasing/pages__utils__notification__invalid_entry_error',
                    `Please check your entry for {{#each fieldNames}}\n - {{this}}{{/each}}`,
                    { fieldNames: fieldErrors },
                ),
                { type: 'error', timeout: 5000 },
            );
            await this.$.page.validate(); // (X3-201046, X3-203885) TODO: Obsolete: To trigger highlighting problematic fields with red borders
        }

        return fieldErrors.length === 0;
    }

    // (X3-201046, X3-203885) TODO: Obsolete: Hard-coded way to create a comprehensive validation error message for this page
    private async _isFieldInvalid(
        field:
            | ui.fields.FilterSelect
            | ui.fields.Text
            | ui.fields.Reference
            | ui.fields.Numeric
            | ui.fields.Select
            | ui.fields.Date,
    ): Promise<boolean> {
        return !field.isHidden && ((field.isMandatory && !field.value) || !!(await field.validate()));
    }

    private async _getMinorVersions(majorVersion: string): Promise<ExtractEdges<ProductVersion>[]> {
        return extractEdges(
            await this.$.graph
                .node('@sage/x3-master-data/ProductVersion')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            _id: true,
                            minorVersion: true,
                        },
                        {
                            filter: {
                                product: this.product.value,
                                majorVersion: majorVersion,
                                type: { _eq: 'stock' },
                                useStatus: { _eq: 'availableToUse' },
                            },
                            orderBy: {
                                minorVersion: 1,
                            },
                        },
                    ),
                )
                .execute(),
        ) as ExtractEdges<ProductVersion>[];
    }

    private _initDirty(): void {
        // Ignore empty new row / current row
        if ((this.purchaseSession?.purchaseReceipt.lines ?? []).length > 1) {
            this.product.isDirty = true;
        }
    }

    private async _isEmptyLpnAndContainerValues(): Promise<boolean> {
        if (
            !this.container.isHidden &&
            !this.licensePlateNumber.isHidden &&
            (!this.container.value || this.container.value?.code === '') &&
            (!this.licensePlateNumber.value || this.licensePlateNumber.value?.code === '')
        ) {
            await dialogMessage(
                this,
                'error',
                ui.localize('@sage/x3-purchasing/dialog-error-title', 'Error'),
                ui.localize(
                    '@sage/x3-purchasing/pages__adc_purchase_receipt_enter_detail_container_license_plate_number_mandatory_error',
                    'Container or license plate number is mandatory',
                ),
            );
            return true;
        } else {
            return false;
        }
    }

    /*
     *
     *  record management functions
     *
     */

    // (X3-218180) TODO: Implement: To be migrated
    // private createDetail() {
    //     const manager = new PurchaseReceiptDetailsRecordManager();
    //     manager.createLine(this.$, this, this.productSite);
    // }
}
