import {
    ExpirationManagement,
    SerialNumberManagement,
    StockManagement,
    SupplierProduct,
} from '@sage/x3-master-data-api';
import { dialogMessage } from '@sage/x3-master-data/lib/client-functions/dialogs';
import {
    PurchaseEntryTransactionInput,
    PurchaseReceiptLineInput,
    GraphApi as PurchasingGraphApi,
} from '@sage/x3-purchasing-api';
import { ExpirationLeadTimeUnits, LotManagementMode, StockVersionMode } from '@sage/x3-stock-data-api';
import { getRegExp } from '@sage/x3-system/lib/shared-functions/pat-converter';
import { extractEdges } from '@sage/xtrem-client';
import { DateValue } from '@sage/xtrem-date-time';
import * as ui from '@sage/xtrem-ui';
import { packingUnit } from '../pages/mobile-purchase-receipt-enter-receipt-detail';
import { expirationDateDefaultValue, useByDateDefaultValue } from './defaultValue';
import { ProductSiteData } from './product-site-data';
import { PurchaseReceiptSession } from './purchase-receipt-details-control';
import { findDefaultLocation, findStockManagementRules, getLocationPreloaded } from './stock-management-rules';

/*
 * Page Interfaces
 * These interfaces allow the inputs to the helper functions to be faked for testing.
 */
interface PageWithStorageAndSite {
    site: { value: string };
    purchaseSession: PurchaseReceiptSession;
}

interface PageWithProductSite {
    productSite: {
        product: {
            stockUnit: {
                code: string;
            };
            productCategory: { code: string };
            code: string;
            purchaseUnit: {
                code: string;
            };
            purchaseUnitToStockUnitConversionFactor: string;
            useByDateCoefficient: string;
            expirationLeadTime: number;
            expirationTimeUnit: ExpirationLeadTimeUnits;
            lotSequenceNumber: string;
        };
        stockSite: { code: string };
    } | undefined;
    productNode: {
        stockUnit: {
            code: string;
        };
        productCategory: { code: string };
        code: string;
        purchaseUnit: {
            code: string;
        };
        purchaseUnitToStockUnitConversionFactor: string;
        useByDateCoefficient: string;
        expirationLeadTime: number;
        expirationTimeUnit: ExpirationLeadTimeUnits;
        lotSequenceNumber: string;
    };
}
interface PageWithHeaderFields {
    product: { value: string };
    description: { value: string };
    purchaseOrderLineNumber: string | number | boolean;
    purchaseOrderNumber: string;
    supplier: { value: string };
    headerTitleLeft: { value: string };
    headerTitleRight: { value: string; prefix?: string };
    headerLine2Left: { value: string };
    headerLine2Right: { value: string; prefix?: string };
    headerLine3Left: { value: string };
    headerLine3Right: { value: string; prefix?: string };
}
interface PageWithVersionFields {
    majorVersion: {
        isHidden: boolean;
        isMandatory: boolean;
        isDisabled: boolean;
        value: { code?: string };
    };
    minorVersion: {
        isHidden: boolean;
        isMandatory: boolean;
        isDisabled: boolean;
        value: { minorVersion?: string };
    };
}
interface PageWithLotFields {
    lot: { isHidden: boolean; isMandatory: boolean; value: string; isDisabled: boolean };
    supplierLot: { isHidden: boolean };
    sublot: { isHidden: boolean; isMandatory: boolean; value: string; isDisabled: boolean };
    lotCustomField1: { isHidden: boolean };
    lotCustomField2: { isHidden: boolean };
    lotCustomField3: { isHidden: boolean };
    lotCustomField4: { isHidden: boolean };
    lotSequenceNumber: { value: string };
}
interface PageWithPackUnitFields {
    receiptUnit: {
        options?: string[];
        isReadOnly: boolean;
        isDisabled: boolean;
        value: string;
    };
    receiptUnitCode: { value: string };
    previousReceiptUnit: {
        value: string;
    };
    receiptUnits: Array<{
        node: {
            packingUnit: {
                code: string;
                numberOfDecimals: number;
                name: string;
            };
            packingUnitToStockUnitConversionFactor: string;
            isPackingFactorEntryAllowed: boolean;
            isPackingUnit: boolean;
        };
    }>;
    packingUnitToStockUnitConversionFactor: {
        value: number;
        isDisabled: boolean;
    };
    isPackingUnit: boolean;
}
interface PageWithPurchaseReceiptMiscProperties {
    status: {
        value: string;
        options?: string[];
    };
    potency: {
        isHidden: boolean;
        value: number;
    };
    serialNumber: {
        isHidden: boolean;
        isMandatory: boolean;
    };
    licensePlateNumber: {
        isHidden: boolean;
    };
    location: {
        isHidden: boolean;
        isMandatory: boolean;
        value: {
            code: string;
            type: string;
            stockSite: { code: string };
            warehouse: { code: string };
        };
    };
    suggestedLocation: {
        isHidden: boolean;
        value: string;
    };
    expirationDate: {
        isHidden: boolean;
        isDisabled: boolean;
        value: string;
    };
    useByDate: {
        isHidden: boolean;
        isDisabled: boolean;
        isMandatory: boolean;
        value: string;
        maxDate: string;
    };
    remainingQuantityInStockUnit: {
        value: number;
        isHidden: boolean;
    };
    remainingQuantityInPackingUnit: {
        value: number;
        isHidden: boolean;
    };
    quantityToReceive: {
        value: number;
        scale?: number;
    };
    closePoLine: {
        value: string;
        isHidden: boolean;
        options?: string[];
    };
    lot: { isHidden: boolean; value: string };
    isExpirationManaged: boolean;
    identifier1: { isHidden: boolean; isDisabled: boolean };
    identifier2: { isHidden: boolean; isDisabled: boolean };
    container: {
        value: { code?: string };
        isHidden: boolean;
    };
    quantityToReceiveNoRounded: number;
    quantityToReceiveRounded: number;
    remainingQuantityNoRounded: number;
    remainingQuantityRounded: number;
    remainingQuantityInitialInStockUnit: number;
    remainingQuantityInitialInOrderUnit: number;
    orderUnit: {
        code: string;
    };
    isLocationPreloaded: boolean;
}

interface PageWithShipmentLine {
    purchaseShipment: string | undefined;
    purchaseShipmentLine: number;
}
//interface PageWithPurchaseReceiptMiscProperties {}

export type PurchaseReceiptsDetailPage = ui.Page<PurchasingGraphApi> &
    PageWithStorageAndSite &
    PageWithHeaderFields &
    PageWithVersionFields &
    PageWithLotFields &
    PageWithPackUnitFields &
    PageWithPurchaseReceiptMiscProperties &
    PageWithProductSite &
    PageWithShipmentLine;

type PurchaseOrderNodeResponse = {
    orderUnit: {
        code: string;
    };
    purchaseUnit: {
        code: string;
        numberOfDecimals: number;
    };
    quantityInStockUnitOrdered: string;
    quantityInStockUnitReceived: string;
    quantityInOrderUnitOrdered: string;
    quantityInPurchaseUnitOrdered: string;
    quantityInPurchaseUnitReceived: string;
    orderUnitToPurchaseUnitConversionFactor: string;
    majorProductVersion: {
        code: string;
    };
    minorProductVersion: string;
};

type NodeWithUnitData = {
    stockUnit: {
        code: string;
        numberOfDecimals: number;
    };
    purchaseUnit: {
        code: string;
        numberOfDecimals: number;
    };
    packingUnits: {
        query: {
            edges: packingUnit[];
        };
    };
    purchaseUnitToStockUnitConversionFactor: string;
 };

type NodeWithMiscellaneousData = {
    product: {
        serialNumberManagementMode: SerialNumberManagement;
        expirationManagementMode: ExpirationManagement;
        stockManagementMode: StockManagement;
        defaultPotencyInPercentage: string;
        serialSequenceNumber: string;
        expirationLeadTime: number;
        expirationTimeUnit: ExpirationLeadTimeUnits;
        useByDateCoefficient: string;
    };
    isLicensePlateNumberManaged: boolean;
    isLocationManaged: boolean;
    defaultInternalContainer: {
        code: string;
    };
    /*    internalContainer: {
        code: string;
    }; */
};

type NodeProductWithMiscellaneousData = {
    serialNumberManagementMode: SerialNumberManagement;
    expirationManagementMode: ExpirationManagement;
    stockManagementMode: StockManagement;
    defaultPotencyInPercentage: string;
    serialSequenceNumber: string;
    expirationLeadTime: number;
    expirationTimeUnit: ExpirationLeadTimeUnits;
    useByDateCoefficient: string;
};

type shipmentLineNodeResponse = {
    purchaseShipment: {
        id: string;
    };
    lineNumber: number;
    product: {
        code: string;
    };
    quantityInPurchaseUnitExpected: string;
    quantityInOrderUnitShipped: string;
    quantityInPurchaseUnitShipped: string;
    quantityInStockUnitShipped: string;
    orderUnitToPurchaseUnitConversionFactor: string;
    orderUnit: {
        code: string;
    };
    stockUnit: {
        code: string;
    };
};

/*
 * Initializer helper class for the Purchase Receipts Detail page.
 */
export class PurchaseReceiptDetailsInitializer {
    private static readonly QTY_DECIMAL_PLACES: number = 8; // this precision is based on how classic Sage X3 4GL calculates remaining quantity in ENVADC.src
    private static readonly CONVERSION_RATE_DECIMAL_PLACES: number = 6; //Display and precision length is defined by the X3 datatype which we do not have access to currently.  Setting to 6 decimal places for now as this is the precisions of the conversion rate data type.

    static async initializePage(page: PurchaseReceiptsDetailPage, closePoLineOptions: string[]): Promise<boolean> {
        // Checking for required parameters
        var errMsg;
        if (!page.purchaseSession) {
            return false;
        }

        // Some local variables to reduce code verbosity

        const currReceipt = page.purchaseSession.purchaseReceipt;
        if (!currReceipt || !currReceipt.lines) {
            return false;
        }
        const currLine = currReceipt.lines[currReceipt.lines.length - 1]; // get the latest line user wants to add
        const transactionEntry: PurchaseEntryTransactionInput = page.purchaseSession.purchaseEntryTransaction;

        if (!currReceipt.receiptSite) {
            errMsg = ui.localize('@sage/x3-purchasing/notification-error-missing-site', 'Stock site not defined');
        } else if (!currReceipt.supplier) {
            errMsg = ui.localize('@sage/x3-purchasing/notification-error-missing-supplier', 'Supplier not defined');
        } else if (!currLine.product) {
            errMsg = ui.localize(
                '@sage/x3-purchasing/notification-error-purchase-receipt-details-invalid-product-site',
                `The product could not be retreived for the {{ siteCode }} site.`,
                { siteCode: currReceipt.receiptSite },
            );
        }

        if (errMsg) {
            page.$.showToast(errMsg, { type: 'error', timeout: 5000 });
            return false;
        }

        page.site.value = currReceipt.receiptSite ?? '';
        page.supplier.value = currReceipt.supplier ?? '';

        if (!currLine.product || !page.site.value) {
            return false;
        }

        const productSiteData = new ProductSiteData(currLine.product, page.site.value, page);
        const productPromise = await productSiteData.product;
        const productSitePromise = productPromise?.stockManagementMode !== 'notManaged' ? await productSiteData.productSite : undefined;
        let purchaseOrderLine: PurchaseOrderNodeResponse | null;
        purchaseOrderLine = null;
        let supplierProductUnits: SupplierProduct;
        let statuses: string[];
        let shipmentLine: shipmentLineNodeResponse | null;
        shipmentLine = null;
        // if this receipt line to create is based on a PO line, additionally retrieve information for calculating remaining quantity & setting default unit of measure
        if (currLine.purchaseOrder && currLine.purchaseOrderLineNumber && currLine.purchaseOrderLine) {
            page.purchaseOrderNumber = currLine.purchaseOrder;
            page.purchaseOrderLineNumber = currLine.purchaseOrderLineNumber;
            page.purchaseShipment = currLine.purchaseShipment;
            page.purchaseShipmentLine = Number(currLine.purchaseShipmentLine);

            try {
                purchaseOrderLine = await page.$.graph
                    .node('@sage/x3-purchasing/PurchaseOrderLine')
                    .read(
                        {
                            orderUnit: {
                                code: true,
                            },
                            purchaseUnit: {
                                code: true,
                                numberOfDecimals: true,
                            },
                            quantityInStockUnitOrdered: true,
                            quantityInStockUnitReceived: true,
                            quantityInOrderUnitOrdered: true,
                            quantityInPurchaseUnitOrdered: true,
                            quantityInPurchaseUnitReceived: true,
                            orderUnitToPurchaseUnitConversionFactor: true,
                            majorProductVersion: { code: true },
                            minorProductVersion: true,
                        },
                        // TODO: find a better way if possible (instead of having to pass PO seq number, pass in _id)
                        `${page.purchaseOrderNumber}|${page.purchaseOrderLineNumber}|${currLine.purchaseOrderLine}`, // TODO Verify: need sequence number?
                    )
                    .execute();
            } catch (e) {
                await dialogMessage(
                    page,
                    'error',
                    ui.localize('@sage/x3-purchasing/error-loading-purchase-order', 'Error loading purchase order'),
                    String(e),
                ); // (X3-218179) TODO Issue: Better error handling: String(e) calculates to "Error: testing"
                return false;
            }

            try {
                const shipmentLineReponse = await page.$.graph
                    .node('@sage/x3-purchasing/PurchaseShipmentLine')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                purchaseShipment: { id: true },
                                lineNumber: true,
                                product: { code: true },
                                quantityInPurchaseUnitExpected: true,
                                quantityInOrderUnitShipped: true,
                                quantityInPurchaseUnitShipped: true,
                                quantityInStockUnitShipped: true,
                                orderUnitToPurchaseUnitConversionFactor: true,
                                orderUnit: { code: true },
                                stockUnit: { code: true },
                            },
                            {
                                filter: {
                                    purchaseShipment: {
                                        id: page.purchaseShipment,
                                    },
                                    lineNumber: page.purchaseShipmentLine,
                                },
                            },
                        ),
                    )
                    .execute();

                if (shipmentLineReponse.edges.length === 1) {
                    shipmentLine = shipmentLineReponse.edges[0].node;
                }
            } catch (e) {
                await dialogMessage(
                    page,
                    'error',
                    ui.localize(
                        '@sage/x3-purchasing/error-loading-purchase-shipment-line',
                        'Error loading purchase shipment line',
                    ),
                    String(e),
                );
                return false;
            }
        }

        //Add supplier product units
        try {
            supplierProductUnits = (await page.$.graph
                .node('@sage/x3-master-data/SupplierProduct')
                .read(
                    {
                        purchaseUnit: {
                            code: true,
                            numberOfDecimals: true,
                        },
                        purchaseUnitToStockUnitConversionFactor: true,
                        packingUnit: {
                            code: true,
                            numberOfDecimals: true,
                        },
                        packingUnitToPurchaseUnitConversionFactor: true,
                    },
                    // TODO: Better way to specify id for a Sage X3 record?
                    `${currLine.product}|${currReceipt.supplier}`,
                )
                .execute()) as SupplierProduct;
        } catch (e) {
            await dialogMessage(
                page,
                'error',
                ui.localize('@sage/x3-purchasing/error-loading-supplier-product', 'Error loading supplier product'),
                String(e),
            ); // (X3-218179) TODO Issue: Better error handling: String(e) calculates to "Error: testing"
            return false;
        }

        const productSite = await productSitePromise;
        const product = await productPromise;
        //read stock rules management
        if (!product) {
            return false;
        }
        page.product.value = product.code;
        page.purchaseSession.purchaseStockManagementRules = await findStockManagementRules(
            page.site.value,
            product.productCategory.code,
            '3',
            transactionEntry.stockMovementCode ?? null,
            page,
        );
        //Add available statuses
        if (!page.purchaseSession.purchaseStockManagementRules) {
            return false;
        }

        const selectedStatus: { _regex: string }[] = [];
        page.purchaseSession.purchaseStockManagementRules.authorizedSubstatus.split(',').forEach(function (status) {
            selectedStatus.push({ _regex: getRegExp(status).source });
        });
        try {
            statuses = extractEdges<{ code: string }>(
                await page.$.graph
                    .node('@sage/x3-stock-data/StockStatus')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                code: true,
                            },
                            {
                                filter: {
                                    code: { _or: selectedStatus },
                                },
                            },
                        ),
                    )
                    .execute(),
            ).map(status => status.code);
        } catch (e) {
            await dialogMessage(
                page,
                'error',
                ui.localize('@sage/x3-purchasing/error-loading-stock-statuses', 'Error loading stock statuses'),
                String(e),
            ); // (X3-218179) TODO Issue: Better error handling: String(e) calculates to "Error: testing"
            return false;
        }
        //default status
        const defaultStatus = page.purchaseSession.purchaseStockManagementRules.defaultStatus;
        const lotManagementRule = page.purchaseSession.purchaseStockManagementRules.lotEntry;
        const lotByDefault = page.purchaseSession.purchaseStockManagementRules.lotByDefault;
        const lpnManagementRule = page.purchaseSession.purchaseStockManagementRules?.licensePlateNumberEntry;

        //PurchaseReceiptDetailsInitializer.initReceiptLines(page, !!productSite); // TODO Implement or removed
        PurchaseReceiptDetailsInitializer.initLotFields(
            page,
            product,
            transactionEntry,
            lotManagementRule,
            lotByDefault,
        );
        PurchaseReceiptDetailsInitializer.initReceiptUnitFields(page, product as any, supplierProductUnits); // TODO remove 'as any' - published ProductSite.product doesn't have packingUnits at the moment.
        PurchaseReceiptDetailsInitializer.initMiscellaneousFields(
            page,
            product,
            productSite,
            statuses,
            defaultStatus,
            transactionEntry,
            lpnManagementRule,
        );
        //add suggested location
        if (productSite && !page.suggestedLocation.isHidden) {
            const suggestedLocation = await findDefaultLocation(
                productSite,
                page.purchaseSession.purchaseStockManagementRules,
                page,
            );
            if (!suggestedLocation) page.suggestedLocation.isHidden = true;
            else page.suggestedLocation.value = suggestedLocation;
            if (page.isLocationPreloaded === true && suggestedLocation) {
                const _location = await getLocationPreloaded(page.site.value, suggestedLocation, page);
                if (_location) {
                    page.location.value = _location;
                }
            }
        }
        PurchaseReceiptDetailsInitializer.initPoCloseLineSelect(page, closePoLineOptions);

        // Set receiptUnit select field to default unit based on selected PO Line's orderUnit or selected Supplier->Product's purchaseUnit
        let defaultReceiptUnit: string;
        let defaultReceiptUnitCode: String;
        let orderUnitToStockUnitConversionFactor: string;
        if (purchaseOrderLine) {
            page.purchaseSession.orderUnitToPurchaseUnitConversionFactor = Number(
                purchaseOrderLine.orderUnitToPurchaseUnitConversionFactor,
            );
            orderUnitToStockUnitConversionFactor = Number.parseFloat(
                (
                    Number(purchaseOrderLine.quantityInStockUnitOrdered) /
                    Number(purchaseOrderLine.quantityInOrderUnitOrdered)
                ).toFixed(PurchaseReceiptDetailsInitializer.CONVERSION_RATE_DECIMAL_PLACES),
            ).toString();
            if (
                purchaseOrderLine.orderUnit.code === product.stockUnit.code &&
                orderUnitToStockUnitConversionFactor === '1'
            ) {
                defaultReceiptUnit = purchaseOrderLine.orderUnit.code;
                defaultReceiptUnitCode = purchaseOrderLine.orderUnit.code;
            } else {
                defaultReceiptUnitCode = purchaseOrderLine.orderUnit.code;
                defaultReceiptUnit = formatUnitForDisplay(
                    purchaseOrderLine.orderUnit.code,
                    orderUnitToStockUnitConversionFactor,
                    product.stockUnit.code,
                );
            }
            // Add order unit calculated from supplier purchase unit when used on PO
            const findUnit = page.receiptUnits.find(
                packingUnit => packingUnit.node.packingUnit.code === defaultReceiptUnitCode,
            ); // TODO Issue: Rounding issue on orderUnitToStockUnitConversionFactor can cause an existing match to not be found and thus, create a 'duplicate' option in receiptUnit
            if (!findUnit) {
                page.receiptUnits.push({
                    node: {
                        packingUnit: {
                            code: purchaseOrderLine.orderUnit.code,
                            numberOfDecimals:
                                page.receiptUnits.find(
                                    packingUnit =>
                                        packingUnit.node.packingUnit.code === purchaseOrderLine?.orderUnit.code,
                                )?.node.packingUnit.numberOfDecimals ?? 0,
                            name: defaultReceiptUnit,
                        },
                        packingUnitToStockUnitConversionFactor: orderUnitToStockUnitConversionFactor.toString(),
                        isPackingFactorEntryAllowed: false,
                        isPackingUnit: true,
                    },
                });
                //Update available units
                page.receiptUnit.options?.push(defaultReceiptUnit);
            }
        } else {
            //If supplier product purchase unit exists, use it as default.  Otherwise use the product's purchase unit.
            let defaultOrderUnit: string;
            if (supplierProductUnits?.purchaseUnit?.code) {
                defaultOrderUnit = supplierProductUnits.purchaseUnit.code;
                orderUnitToStockUnitConversionFactor = Number.parseFloat(
                    supplierProductUnits.purchaseUnitToStockUnitConversionFactor,
                ).toString();
            } else {
                defaultOrderUnit = product.purchaseUnit.code;
                orderUnitToStockUnitConversionFactor = Number.parseFloat(
                    product.purchaseUnitToStockUnitConversionFactor,
                ).toString();
            }
            if (
                defaultOrderUnit === product.stockUnit.code &&
                orderUnitToStockUnitConversionFactor === '1'
            ) {
                defaultReceiptUnit = defaultOrderUnit;
                defaultReceiptUnitCode = defaultOrderUnit;
            } else {
                defaultReceiptUnitCode = defaultOrderUnit;
                defaultReceiptUnit = formatUnitForDisplay(
                    defaultOrderUnit,
                    orderUnitToStockUnitConversionFactor,
                    product.stockUnit.code,
                );
            }
        }

        const selectedUnit = page.receiptUnits.find(
            packingUnit => packingUnit.node.packingUnit.code === defaultReceiptUnitCode,
        );
        if (!selectedUnit) return false;
        page.packingUnitToStockUnitConversionFactor.value = Number(orderUnitToStockUnitConversionFactor);
        page.isPackingUnit = selectedUnit.node.isPackingUnit;
        page.quantityToReceive.scale = selectedUnit.node.packingUnit.numberOfDecimals; // set # of decimal places restriction based on selected unit of measure
        if (
            selectedUnit.node.packingUnit.code !== product.stockUnit.code &&
            orderUnitToStockUnitConversionFactor !== '1'
        ) {
            page.receiptUnit.value = formatUnitForDisplay(
                selectedUnit.node.packingUnit.code,
                selectedUnit.node.packingUnitToStockUnitConversionFactor,
                product.stockUnit.code,
            );
        } else page.receiptUnit.value = defaultReceiptUnit;
        page.receiptUnitCode.value = selectedUnit.node.packingUnit.code;
        page.previousReceiptUnit.value = page.receiptUnit.value;
        page.packingUnitToStockUnitConversionFactor.isDisabled = !selectedUnit.node.isPackingFactorEntryAllowed;

        // Calculating remaining quantity only applicable for PO Line workflow
        if (purchaseOrderLine) {
            PurchaseReceiptDetailsInitializer.initVersionFields(page, product, purchaseOrderLine);

            const remainingQuantity: number[] = [0, 0, 0];
            PurchaseReceiptDetailsInitializer.calculateRemainingQuantity(
                page,
                purchaseOrderLine,
                currReceipt.lines.slice(0, currReceipt.lines.length - 1), // do not include most recently added line as part of the equation, because that is the current line being edited
                shipmentLine,
                remainingQuantity,
            );
            page.remainingQuantityInStockUnit.value = remainingQuantity[0];
            //the packing unit is the order unit, so I can put remainingQuantityInOrderUnit in the remainingQuantityInPackingUnit field
            page.remainingQuantityInPackingUnit.value = remainingQuantity[1];
            //if (selectedUnit?.node.unit.numberOfDecimals && selectedUnit?.node.packingUnit.numberOfDecimals !== 0) {
            const tmp = Math.pow(10, selectedUnit?.node.packingUnit.numberOfDecimals);
            if (tmp !== 0) {
                page.remainingQuantityInStockUnit.value =
                    Math.round(page.remainingQuantityInStockUnit.value * tmp) / tmp;
            }
            //}
            page.quantityToReceive.value = page.remainingQuantityInPackingUnit.value; // should auto-round value appropriately based on scale property that was set for this field
            page.quantityToReceiveNoRounded = page.quantityToReceive.value;
            page.quantityToReceiveRounded = page.quantityToReceive.value;
            page.remainingQuantityNoRounded = page.remainingQuantityInPackingUnit.value;
            page.remainingQuantityRounded = page.remainingQuantityInPackingUnit.value;
            page.closePoLine.value = closePoLineOptions[0]; //Yes
            page.remainingQuantityInitialInStockUnit = shipmentLine
                ? Number(shipmentLine.quantityInStockUnitShipped)
                : Number(purchaseOrderLine.quantityInStockUnitOrdered);
            page.remainingQuantityInitialInOrderUnit = shipmentLine
                ? Number(shipmentLine.quantityInOrderUnitShipped)
                : Number(purchaseOrderLine.quantityInOrderUnitOrdered);
            page.orderUnit = shipmentLine ? shipmentLine.orderUnit : purchaseOrderLine.orderUnit;
        } else {
            PurchaseReceiptDetailsInitializer.initVersionFields(page, product);
            //Quantity values not used for direct receipts
            page.remainingQuantityInPackingUnit.value = 0;
            page.remainingQuantityInPackingUnit.isHidden = true;
            page.closePoLine.isHidden = true;
            page.remainingQuantityInitialInStockUnit = 0;
            page.remainingQuantityInitialInOrderUnit = 0;
        }

        page.productSite = productSite;
        page.productNode = product;

        PurchaseReceiptDetailsInitializer.initHeader(
            page,
            product,
            purchaseOrderLine,
            currReceipt.lines.slice(0, currReceipt.lines.length - 1),
            shipmentLine,
        );

        return true;
    }

    static calculateRemainingQuantity(
        page: PageWithHeaderFields,
        purchaseOrderLine: PurchaseOrderNodeResponse,
        lines: PurchaseReceiptLineInput[] = [],
        shipmentLine: shipmentLineNodeResponse | null,
        remainingQuantity: number[],
    ) {
        // calculate the conversion rate from stock unit to order unit
        // TODO: Unfortunately there appears to be cases in which toFixed() doesn't round correctly
        // TODO: Verify if there is a better way of calculating conversion rate. This is based on 4GL ENVADC.src
        let conversionRatePurchaseUnitToStockUnit = 0;
        let conversionRateOrderUnitToStockUnit = 0;
        let pendingQuantityInStockUnitReceived = 0;
        let quantityInStockUnitExpected = 0;
        let remainingQuantityInStockUnit = 0;
        let remainingQuantityInOrderUnit = 0;
        let remainingQuantityInPurchaseUnit = 0;

        if (shipmentLine) {
            conversionRatePurchaseUnitToStockUnit = Number.parseFloat(
                (
                    Number(shipmentLine.quantityInStockUnitShipped) / Number(shipmentLine.quantityInPurchaseUnitShipped)
                ).toFixed(PurchaseReceiptDetailsInitializer.QTY_DECIMAL_PLACES),
            );
            conversionRateOrderUnitToStockUnit = Number.parseFloat(
                (
                    Number(purchaseOrderLine.quantityInStockUnitOrdered) /
                    Number(purchaseOrderLine.quantityInOrderUnitOrdered)
                ).toFixed(PurchaseReceiptDetailsInitializer.QTY_DECIMAL_PLACES),
            ); // using Number.parseFloat() to remove any trailing decimal zeros due to toFixed()
        } else {
            conversionRateOrderUnitToStockUnit = Number.parseFloat(
                (
                    Number(purchaseOrderLine.quantityInStockUnitOrdered) /
                    Number(purchaseOrderLine.quantityInOrderUnitOrdered)
                ).toFixed(PurchaseReceiptDetailsInitializer.QTY_DECIMAL_PLACES),
            ); // using Number.parseFloat() to remove any trailing decimal zeros due to toFixed()
            conversionRatePurchaseUnitToStockUnit = Number.parseFloat(
                (
                    Number(purchaseOrderLine.quantityInStockUnitOrdered) /
                    Number(purchaseOrderLine.quantityInPurchaseUnitOrdered)
                ).toFixed(PurchaseReceiptDetailsInitializer.QTY_DECIMAL_PLACES),
            ); // using Number.parseFloat() to remove any trailing decimal zeros due to toFixed()
        }

        try {
            if (lines?.length > 0) {
                // iterate through receipt lines user has added so far for the same PO Line & product as this selected PO line
                if (shipmentLine?.purchaseShipment) {
                    pendingQuantityInStockUnitReceived = calculCurrentQuantityInStockUnitReceived(lines, shipmentLine);
                } else {
                    lines
                        .filter(
                            (line: PurchaseReceiptLineInput) =>
                                line.purchaseOrder === page.purchaseOrderNumber &&
                                line.purchaseOrderLineNumber === page.purchaseOrderLineNumber &&
                                line.product === page.product.value,
                        )
                        .map(
                            (line: PurchaseReceiptLineInput) =>
                                (pendingQuantityInStockUnitReceived +=
                                    Number(line.quantityInReceiptUnitReceived) *
                                    Number(line.receiptUnitToStockUnitConversionFactor)),
                        );
                }
            }
        } catch (error) {
            ui.console.warn(`Error loading saved purchase receipt session data: ${error}`);
        }

        // quantity is initially calculated based on remaining stock quantity and then converted to order unit
        if (shipmentLine) {
            quantityInStockUnitExpected =
                Number(shipmentLine?.quantityInPurchaseUnitExpected) * conversionRatePurchaseUnitToStockUnit;
            remainingQuantityInStockUnit = Math.max(
                0,
                Number(quantityInStockUnitExpected) - pendingQuantityInStockUnitReceived,
            );
            if (conversionRateOrderUnitToStockUnit !== 0) {
                remainingQuantityInOrderUnit = remainingQuantityInStockUnit / conversionRateOrderUnitToStockUnit;
            }
            if (conversionRatePurchaseUnitToStockUnit !== 0) {
                const tmp = Math.pow(10, purchaseOrderLine.purchaseUnit.numberOfDecimals);
                if (tmp !== 0) {
                    remainingQuantityInPurchaseUnit =
                        Math.round((remainingQuantityInStockUnit / conversionRatePurchaseUnitToStockUnit) * tmp) / tmp;
                } else {
                    remainingQuantityInPurchaseUnit =
                        remainingQuantityInStockUnit / conversionRatePurchaseUnitToStockUnit;
                }
            }
        } else {
            remainingQuantityInStockUnit = Math.max(
                0,
                Number(purchaseOrderLine.quantityInStockUnitOrdered) -
                    Number(purchaseOrderLine.quantityInStockUnitReceived) -
                    pendingQuantityInStockUnitReceived,
            );
            if (conversionRateOrderUnitToStockUnit !== 0) {
                remainingQuantityInOrderUnit = remainingQuantityInStockUnit / conversionRateOrderUnitToStockUnit;
            }

            if (conversionRatePurchaseUnitToStockUnit !== 0) {
                const tmp = Math.pow(10, purchaseOrderLine.purchaseUnit.numberOfDecimals);
                if (tmp !== 0) {
                    remainingQuantityInPurchaseUnit =
                        Math.round((remainingQuantityInStockUnit / conversionRatePurchaseUnitToStockUnit) * tmp) / tmp;
                } else {
                    remainingQuantityInPurchaseUnit =
                        remainingQuantityInStockUnit / conversionRatePurchaseUnitToStockUnit;
                }
            }
        }
        remainingQuantity[0] = remainingQuantityInStockUnit;
        remainingQuantity[1] = remainingQuantityInOrderUnit;
        remainingQuantity[2] = remainingQuantityInPurchaseUnit;
    }

    static initReceiptLines(page: PurchaseReceiptsDetailPage, newDetail = false) {
        // TODO: fill in line management later
    }

    static initHeader(
        page: PageWithHeaderFields,
        productNode: { code: string; localizedDescription1: string },
        purchaseOrderLine: PurchaseOrderNodeResponse | null,
        lines: PurchaseReceiptLineInput[] | undefined,
        shipmentLine: shipmentLineNodeResponse | null,
    ) {
        page.product.value = productNode.code;
        page.description.value = productNode.localizedDescription1;

        if (purchaseOrderLine && page.purchaseOrderNumber && page.purchaseOrderLineNumber) {
            page.headerTitleLeft.value = page.purchaseOrderNumber;
            page.headerTitleRight.value = page.purchaseOrderLineNumber.toString();
            page.headerTitleRight.prefix = ui.localize(
                '@sage/x3-purchasing/pages__adc_purchase_receipt_header_detail__line_no',
                'Line No.',
            );
            page.headerLine2Left.value = page.product.value;
            page.headerLine2Right.value = page.description.value;
            page.headerLine3Left.value = ui.localize(
                '@sage/x3-purchasing/pages__adc_purchase_receipt_header_detail__expected_quantity',
                'Expected quantity:',
            );
            const remainingQuantity: number[] = [0, 0, 0];
            PurchaseReceiptDetailsInitializer.calculateRemainingQuantity(
                page,
                purchaseOrderLine,
                lines,
                shipmentLine,
                remainingQuantity,
            );

            page.headerLine3Right.value = `${Number(remainingQuantity[2]).toFixed(purchaseOrderLine.purchaseUnit.numberOfDecimals)} ${purchaseOrderLine.purchaseUnit.code}`;
        } else {
            page.headerTitleLeft.value = page.product.value;
            page.headerTitleRight.value = '';
            page.headerLine2Left.value = page.description.value;
            page.headerLine2Right.value = '';
            page.headerLine3Left.value = page.supplier.value;
            page.headerLine3Right.value = '';
        }
    }

    static initVersionFields(
        page: PageWithVersionFields,
        productNode: { stockVersionMode: StockVersionMode },
        purchaseOrderLineNode?: { majorProductVersion: { code: string }; minorProductVersion: string },
    ) {
        const versionManagementMode = productNode.stockVersionMode;

        page.majorVersion.isHidden = !['major', 'majorAndMinor'].includes(versionManagementMode);
        page.minorVersion.isHidden = versionManagementMode !== 'majorAndMinor';
        page.minorVersion.isDisabled = true;

        // if workflow is PO Line, display version from the PO Line if provided
        if (
            purchaseOrderLineNode &&
            !page.majorVersion.isHidden &&
            purchaseOrderLineNode.majorProductVersion &&
            purchaseOrderLineNode.majorProductVersion.code
        ) {
            page.majorVersion.value = { code: purchaseOrderLineNode.majorProductVersion.code };

            // check for minor version
            if (!page.minorVersion.isHidden && purchaseOrderLineNode.minorProductVersion) {
                page.minorVersion.value = { minorVersion: purchaseOrderLineNode.minorProductVersion };
                page.minorVersion.isDisabled = false;
            }
        }
        // no need to check at this point if only minor version is hidden. Cannot have minor without major
    }

    static initLotFields(
        page: PageWithLotFields,
        productNode: {
            lotManagementMode: LotManagementMode;
            lotSequenceNumber: string;
        },
        transactionEntry: PurchaseEntryTransactionInput,
        lotManagementRule: string,
        lotByDefault: string,
    ) {
        const lotManagement = productNode.lotManagementMode;

        const lotNotManaged = lotManagement === 'notManaged';
        page.lot.isHidden = lotNotManaged;
        page.supplierLot.isHidden = lotNotManaged;
        page.sublot.isHidden = lotManagement !== 'lotAndSublot';
        page.sublot.isMandatory = !page.sublot.isHidden;
        if (!page.lot.isHidden) {
            if (lotManagementRule === 'no') {
                page.lot.isDisabled = true;
                if (page.sublot.isHidden === false) {
                    page.sublot.value = '00001';
                    page.sublot.isDisabled = true;
                }
            } else if (lotManagementRule === 'newLot') {
                if (page.sublot.isHidden === false) page.sublot.value = '00001';
            } else if (lotManagementRule === 'free' && lotByDefault === 'documentNumber') {
                if (page.sublot.isHidden === false) page.sublot.value = '00001';
            }
        }

        page.lotCustomField1.isHidden = !transactionEntry.isLotCustomField1Allowed || lotNotManaged;
        page.lotCustomField2.isHidden = !transactionEntry.isLotCustomField2Allowed || lotNotManaged;
        page.lotCustomField3.isHidden = !transactionEntry.isLotCustomField3Allowed || lotNotManaged;
        page.lotCustomField4.isHidden = !transactionEntry.isLotCustomField4Allowed || lotNotManaged;

        page.lot.isMandatory =
            !page.lot.isHidden &&
            !productNode.lotSequenceNumber &&
            lotManagement !== 'optionalLot' &&
            lotByDefault !== 'documentNumber';
    }

    static initReceiptUnitFields(
        page: PageWithPackUnitFields & {
            quantityToReceive: {
                scale?: number;
            };
        },
        productNode: NodeWithUnitData,
        supplierProduct: SupplierProduct,
    ) {
        const packingUnitList = productNode.packingUnits;
        page.receiptUnit.options = [];
        page.receiptUnits = [];

        const unitCodeOptions: string[] = [];
        const stockUnit: string = productNode.stockUnit.code;

        // Add stock unit
        page.receiptUnits.push({
            node: {
                packingUnit: {
                    code: stockUnit,
                    numberOfDecimals: productNode.stockUnit.numberOfDecimals,
                    name: stockUnit,
                },
                packingUnitToStockUnitConversionFactor: '1',
                isPackingFactorEntryAllowed: false,
                isPackingUnit: false,
            },
        });
        unitCodeOptions.push(productNode.stockUnit.code);
        let currentUnitDisplay: string;
        currentUnitDisplay = '';

        // Add purchase unit if different from stock unit
        if (productNode.stockUnit.code !== productNode.purchaseUnit.code) {
            currentUnitDisplay = formatUnitForDisplay(
                productNode.purchaseUnit.code,
                productNode.purchaseUnitToStockUnitConversionFactor,
                stockUnit,
            );
            page.receiptUnits.push({
                node: {
                    packingUnit: {
                        code: productNode.purchaseUnit.code,
                        name: currentUnitDisplay,
                        numberOfDecimals: productNode.purchaseUnit.numberOfDecimals,
                    },
                    packingUnitToStockUnitConversionFactor:
                        productNode.purchaseUnitToStockUnitConversionFactor.toString(),
                    isPackingFactorEntryAllowed: false,
                    isPackingUnit: false,
                },
            });
            unitCodeOptions.push(currentUnitDisplay);
        }

        // Add all units from packingUnitList.  Packing unit cannot be a stock unit.
        let duplicateUnit = false;
        if (packingUnitList.query?.edges?.length > 0) {
            for (const packUnit of packingUnitList.query.edges) {
                duplicateUnit = false;
                for (const receiptUnit of page.receiptUnits) {
                    currentUnitDisplay = formatUnitForDisplay(
                        // TODO Issue: This logic only needs to be done one per packing unit. Should be moved outside of this for loop
                        packUnit.node.packingUnit.code,
                        packUnit.node.packingUnitToStockUnitConversionFactor,
                        stockUnit,
                    );
                    if (
                        packUnit.node.packingUnit.code === receiptUnit.node.packingUnit.code &&
                        packUnit.node.packingUnitToStockUnitConversionFactor ===
                            receiptUnit.node.packingUnitToStockUnitConversionFactor
                    ) {
                        duplicateUnit = true;
                        //If a packing unit matches the product's purchase unit, convert is to a packing unit
                        receiptUnit.node.isPackingUnit = true;
                        break;
                    }
                }
                if (!duplicateUnit) {
                    page.receiptUnits.push({
                        node: {
                            packingUnit: {
                                code: packUnit.node.packingUnit.code,
                                name: currentUnitDisplay,
                                numberOfDecimals: packUnit.node.packingUnit.numberOfDecimals,
                            },
                            packingUnitToStockUnitConversionFactor:
                                packUnit.node.packingUnitToStockUnitConversionFactor,
                            isPackingFactorEntryAllowed: packUnit.node.isPackingFactorEntryAllowed,
                            isPackingUnit: true,
                        },
                    });
                    unitCodeOptions.push(currentUnitDisplay);
                }
            }
        }
        // Add supplier-product units
        duplicateUnit = false;
        if (supplierProduct?.purchaseUnit?.code) {
            for (const receiptUnit of page.receiptUnits) {
                currentUnitDisplay = formatUnitForDisplay(
                    // TODO Issue: This logic only needs to be done one per packing unit. Should be moved outside of this for loop
                    supplierProduct.purchaseUnit.code,
                    supplierProduct.purchaseUnitToStockUnitConversionFactor,
                    stockUnit,
                );
                if (
                    supplierProduct.purchaseUnit.code === receiptUnit.node.packingUnit.code &&
                    supplierProduct.purchaseUnitToStockUnitConversionFactor ===
                        receiptUnit.node.packingUnitToStockUnitConversionFactor
                ) {
                    duplicateUnit = true;
                    break;
                }
            }
            if (!duplicateUnit) {
                page.receiptUnits.push({
                    node: {
                        packingUnit: {
                            code: supplierProduct.purchaseUnit.code,
                            name: currentUnitDisplay,
                            numberOfDecimals: supplierProduct.purchaseUnit.numberOfDecimals,
                        },
                        packingUnitToStockUnitConversionFactor: supplierProduct.purchaseUnitToStockUnitConversionFactor,
                        isPackingFactorEntryAllowed: false,
                        isPackingUnit: false,
                    },
                });
                unitCodeOptions.push(currentUnitDisplay);
            }
        }
        duplicateUnit = false;
        if (supplierProduct?.packingUnit?.code) {
            const unitConversion: string = Number.parseFloat(
                (
                    Number(supplierProduct.packingUnitToPurchaseUnitConversionFactor) *
                    Number(supplierProduct.purchaseUnitToStockUnitConversionFactor)
                ).toFixed(PurchaseReceiptDetailsInitializer.CONVERSION_RATE_DECIMAL_PLACES),
            ).toString();

            for (const receiptUnit of page.receiptUnits) {
                currentUnitDisplay = formatUnitForDisplay(supplierProduct.packingUnit.code, unitConversion, stockUnit); // TODO Issue: This logic does not need to be repeated. Should be moved outside of this for loop

                //Comparison of conversion rates required converting both to string.  unitConversion is a number and
                //packingUnitToStockUnitConversionFactor is defined as a number.  However, packingUnitToStockUnitConversionFactor
                //is retrieved with quotes around it so the validation always returns false which can result in duplicate
                //units displayed.  If I convert just unitConversion to a string, I receive a compile error that I am
                //comparing a number to string.
                if (
                    supplierProduct.packingUnit.code === receiptUnit.node.packingUnit.code &&
                    unitConversion.toString() === receiptUnit.node.packingUnitToStockUnitConversionFactor.toString()
                ) {
                    duplicateUnit = true;
                    break;
                }
            }
            if (!duplicateUnit) {
                page.receiptUnits.push({
                    node: {
                        packingUnit: {
                            code: supplierProduct.packingUnit.code,
                            name: currentUnitDisplay,
                            numberOfDecimals: supplierProduct.packingUnit.numberOfDecimals,
                        },
                        packingUnitToStockUnitConversionFactor: unitConversion,
                        isPackingFactorEntryAllowed: false,
                        isPackingUnit: false,
                    },
                });
                unitCodeOptions.push(currentUnitDisplay);
            }
        }

        page.receiptUnit.options = unitCodeOptions;
        page.receiptUnit.isReadOnly = page.receiptUnit.options.length <= 1;
        page.receiptUnit.isDisabled = false;
    }

    static initMiscellaneousFields(
        page: PageWithPurchaseReceiptMiscProperties,
        productNode: NodeProductWithMiscellaneousData,
        productSiteNode: NodeWithMiscellaneousData | undefined,
        statuses: string[],
        defaultStatus: string,
        transactionEntry: PurchaseEntryTransactionInput,
        lpnManagementRule: string,
    ) {
        //TODO: Retrieve default status value from StockManagementRules node.  This node will not be published in
        //time for 2020 R3 release.
        page.status.options = statuses;
        //page.status.value = 'A'; // TODO Verify: hard-code status?
        page.status.value = defaultStatus;

        // Potency management
        const potencyManagement = productNode.stockManagementMode;
        page.potency.isHidden =
            potencyManagement !== 'potencyManaged' || !transactionEntry.isLotPotencyAllowed || page.lot.isHidden;
        if (!page.potency.isHidden) {
            page.potency.value = Number(productNode.defaultPotencyInPercentage);
        }

        const serialNumberManagement = productNode.serialNumberManagementMode;
        page.serialNumber.isHidden = serialNumberManagement === 'notManaged' || serialNumberManagement === 'issued';
        page.serialNumber.isMandatory = !page.serialNumber.isHidden && !productNode.serialSequenceNumber;

        page.licensePlateNumber.isHidden = !productSiteNode || !productSiteNode.isLicensePlateNumberManaged;

        page.location.isHidden = !productSiteNode || !productSiteNode.isLocationManaged;
        page.location.isMandatory = !page.location.isHidden;
        page.suggestedLocation.isHidden = page.location.isHidden;

        const expirationManagement = productNode.expirationManagementMode;
        page.isExpirationManaged = expirationManagement !== 'notManaged';
        page.expirationDate.isHidden = page.useByDate.isHidden = !page.isExpirationManaged;

        if (page.isExpirationManaged) {
            page.expirationDate.isDisabled = page.useByDate.isDisabled = false;
            if (transactionEntry.isLotExpirationDateAllowed) {
                page.useByDate.isDisabled =
                    expirationManagement === 'roundingBeginningMonth1' || expirationManagement === 'roundingMonthEnd';
            } else {
                page.expirationDate.isDisabled = page.useByDate.isDisabled =
                    expirationManagement === 'roundingBeginningMonth1' ||
                    expirationManagement === 'roundingMonthEnd' ||
                    expirationManagement === 'withoutRounding';
            }
        }
        const effectiveDate = DateValue.today().toString();
        if (!page.expirationDate.isHidden && expirationManagement !== 'manualEntry') {
            page.expirationDate.value =
                expirationDateDefaultValue(
                    productNode.expirationManagementMode,
                    productNode.expirationLeadTime,
                    productNode.expirationTimeUnit,
                    effectiveDate,
                ) ?? '';

            page.useByDate.value =
                useByDateDefaultValue(
                    page.expirationDate.value,
                    effectiveDate,
                    productNode.expirationManagementMode,
                    Number(productNode.useByDateCoefficient),
                ) ?? '';
            page.useByDate.maxDate = page.expirationDate.value;
        }

        switch (transactionEntry.identifier1Detail) {
            case 'entered': {
                page.identifier1.isHidden = false;
                page.identifier1.isDisabled = false;
                break;
            }
            case 'displayed': {
                page.identifier1.isHidden = false;
                page.identifier1.isDisabled = true;
                break;
            }
            default: {
                page.identifier1.isHidden = true;
            }
        }

        switch (transactionEntry.identifier2Detail) {
            case 'entered': {
                page.identifier2.isHidden = false;
                page.identifier2.isDisabled = false;
                break;
            }
            case 'displayed': {
                page.identifier2.isHidden = false;
                page.identifier2.isDisabled = true;
                break;
            }
            default: {
                page.identifier2.isHidden = true;
            }
        }

        page.container.isHidden = lpnManagementRule !== 'mandatory' || !productSiteNode || !productSiteNode.isLicensePlateNumberManaged;
        if (productSiteNode && !page.container.isHidden) page.container.value = productSiteNode.defaultInternalContainer;
    }

    static initPoCloseLineSelect(page: PageWithPurchaseReceiptMiscProperties, closePoLineOptions: string[]) {
        page.closePoLine.options = closePoLineOptions;
    }
}

function formatUnitForDisplay(code: string, conversionRate: string, stockUnit: string): string {
    return `${code} ${'='} ${conversionRate} ${stockUnit}`;
}

function calculCurrentQuantityInStockUnitReceived(
    lines: PurchaseReceiptLineInput[] = [],
    shipmentLine: shipmentLineNodeResponse,
): number {
    let currentQuantityInStockUnitReceived = 0;
    lines
        ?.filter(
            (line: PurchaseReceiptLineInput) =>
                line?.purchaseShipment === shipmentLine.purchaseShipment.id &&
                Number(line?.purchaseShipmentLine) === Number(shipmentLine.lineNumber),
        )
        .map(
            (line: PurchaseReceiptLineInput) =>
                (currentQuantityInStockUnitReceived +=
                    Number(line.quantityInReceiptUnitReceived) * Number(line.receiptUnitToStockUnitConversionFactor)),
        );
    return currentQuantityInStockUnitReceived;
}
