import {
    GraphApi,
    PurchaseEntryTransactionInput,
    PurchaseReceiptInput,
    PurchaseReceiptLineInput,
} from '@sage/x3-purchasing-api';
import { StockJournalInput, StockManagementRules } from '@sage/x3-stock-data-api';
import { extractEdges } from '@sage/xtrem-client';
import { decimal } from '@sage/xtrem-shared';
import * as ui from '@sage/xtrem-ui';
import { Page, localize, queryUtils } from '@sage/xtrem-ui';
interface receiptUnit {
    code: string;
    quantityInReceiptUnit: number;
    stockUnitConversionFactor: number;
}

export async function validate(page: Page): Promise<boolean> {
    // (X3-201046, X3-203885) TODO Issue: Have better validation error messages that specifically tell which fields have issues
    const errors = await page.$.page.validate();
    if (errors.length === 0) {
        return true;
    }

    page.$.showToast(`${localize('@sage/x3-purchasing/dialog-error-title', 'Error')}: ${errors[0]}`, {
        type: 'error',
        timeout: 30000,
    });

    return false;
}
export async function controlLotReceipt(
    pageInstance: Page,
    lot: string,
    product: string,
    entryType: string,
    site: string,
): Promise<boolean> {
    if (!lot || !product || !entryType) throw new Error('Invalid arguments');
    let stockJournalFilter = {
        stockSite: site,
        documentType: entryType,
        product: { code: product },
        isUpdated: true,
        lot: lot,
    };

    const response = extractEdges(
        await pageInstance.$.graph
            .node('@sage/x3-stock-data/StockJournal')
            .query(
                queryUtils.edgesSelector(
                    {
                        lot: true,
                        sublot: true,
                    },
                    {
                        filter: stockJournalFilter,
                    },
                ),
            )
            .execute(),
    );

    if (response.length > 0) {
        pageInstance.$.showToast(
            localize(
                '@sage/x3-purchasing/notification-error-receipt-lot',
                'the lot number {{ lot }} already exists for this product',
                { lot: lot },
            ),
            { type: 'error' },
        );
        return false;
    }
    return true;
}

enum YesNoEnum {
    no = 1,
    yes = 2,
}

export type PurchaseReceiptSession = {
    purchaseReceipt: PurchaseReceiptInput;
    purchaseEntryTransaction: PurchaseEntryTransactionInput;
    purchaseStockManagementRules?: StockManagementRules;
    orderUnitToPurchaseUnitConversionFactor: number;
} | null;

export class PurchaseReceiptDetailsRecordManager {
    constructor(
        private page: Page<GraphApi>,
        cleanupIncompleteSessionLines = false,
    ) {
        // An incomplete purchase receipt line can be added to the session when user is transitioning from main page to enter detail page
        if (cleanupIncompleteSessionLines) this._cleanupSession();
    }

    private static readonly PURCHASE_RECEIPT_KEY: string = 'purchaseReceipt';

    private _purchaseSession: PurchaseReceiptSession;

    private _stockDetails: Partial<StockJournalInput>[] = [];
    private _receiptUnit: receiptUnit[] = [];
    private _unitIsSame: boolean = true;

    public get purchaseSession(): PurchaseReceiptSession {
        if (this._purchaseSession) {
            return this._purchaseSession;
        }

        const storedSession = this.page.$.storage.get(
            PurchaseReceiptDetailsRecordManager.PURCHASE_RECEIPT_KEY,
        ) as string;
        if (storedSession) {
            const parsedSession = JSON.parse(storedSession) as PurchaseReceiptSession;
            if (this._checkStorage(parsedSession)) {
                this._purchaseSession = parsedSession;
            } else {
                // purchase receipt in the session storage is corrupted/malformed
                this.clearSession();
            }
        }
        return this._purchaseSession;
    }

    public set purchaseSession(purchaseSession: PurchaseReceiptSession) {
        if (!purchaseSession) {
            return;
        }

        if (!this._checkStorage(purchaseSession)) {
            throw new Error('Invalid Purchase Receipt session');
        }

        this.page.$.storage.set(
            PurchaseReceiptDetailsRecordManager.PURCHASE_RECEIPT_KEY,
            JSON.stringify(purchaseSession),
        );
        this._purchaseSession = purchaseSession;
    }

    private _checkStorage(sessionData: PurchaseReceiptSession): boolean {
        // check if sessionData contains at least one purchase receipt line that has at least one stock detail line
        if (
            !sessionData ||
            !sessionData.purchaseEntryTransaction.code ||
            !sessionData.purchaseReceipt.receiptDate ||
            !sessionData.purchaseReceipt.lines ||
            sessionData.purchaseReceipt.lines.length === 0 ||
            !sessionData.purchaseReceipt.receiptSite ||
            !sessionData.purchaseReceipt.supplier
        ) {
            return false;
        }

        return !sessionData.purchaseReceipt.lines.some((line: PurchaseReceiptLineInput) => {
            // skip most recent line that is pending to be completed in detail page
            if (!line.stockDetails) {
                return false;
            }

            // check for each line, the receipt unit & quantity are defined
            if (
                !line.receiptUnit ||
                line.quantityInReceiptUnitReceived === null ||
                line.quantityInReceiptUnitReceived === undefined
            ) {
                return true;
            }

            return line.stockDetails.some((detail: StockJournalInput) => {
                // check for each stock detail on this line, the packing unit & quantity are defined
                return (
                    !detail.packingUnit ||
                    detail.quantityInPackingUnit === null ||
                    detail.quantityInPackingUnit === undefined
                );
            });
        });
    }

    private _cleanupSession() {
        this._purchaseSession = this.purchaseSession; // get the latest from session
        if (!this._purchaseSession || !this._purchaseSession.purchaseReceipt.lines) return;

        // remove any incomplete purchase receipt lines (ex. a line without stockDetails)
        const prevLineCount = this._purchaseSession.purchaseReceipt.lines?.length;
        this._purchaseSession.purchaseReceipt.lines = this._purchaseSession.purchaseReceipt.lines.filter(
            (line: PurchaseReceiptLineInput) => {
                if (
                    !line.receiptUnit ||
                    line.quantityInReceiptUnitReceived === null ||
                    line.quantityInReceiptUnitReceived === undefined ||
                    !line.stockDetails
                ) {
                    return false;
                }

                return !line.stockDetails.some((detail: StockJournalInput) => {
                    // check for each stock detail on this line, the packing unit & quantity are defined
                    return (
                        !detail.packingUnit ||
                        detail.quantityInPackingUnit === null ||
                        detail.quantityInPackingUnit === undefined
                    );
                });
            },
        );

        // if all lines have been removed, clear the session
        if (this._purchaseSession.purchaseReceipt.lines?.length === 0) {
            this.clearSession();
        } else if (prevLineCount !== this._purchaseSession.purchaseReceipt.lines.length) {
            // otherwise if any line got removed, overwrite the session
            this.purchaseSession = this._purchaseSession;
        }
    }

    public clearSession(): void {
        this.page.$.storage.remove(PurchaseReceiptDetailsRecordManager.PURCHASE_RECEIPT_KEY);
        this._purchaseSession = null;
    }

    public createLine(
        page: {
            closePoLine: { value: string | null; options?: string[] };
            description: { value: string | null };
            site: { value: string | null };
            supplier: { value: string | null };
            purchaseOrderNumber: string;
            purchaseOrderLineNumber: string | number | boolean;
            warehouse: { value: string | null };
            container: { value: { code?: string } | null };
            packingUnitToStockUnitConversionFactor: { value: number | null };
        },
        product: {
            stockUnit: {
                code: string;
            };
            code: string;           
        }
    ) {
        if (!this._purchaseSession || !this._purchaseSession.purchaseReceipt.lines) return;
        const currentLineNumber = this._purchaseSession.purchaseReceipt.lines.length - 1;
        const currentLine = this._purchaseSession.purchaseReceipt.lines[currentLineNumber];
        currentLine.stockDetails = this._stockDetails;

        // If all receipt units are the same, use that unit for the line. If there are multiple receipt units,
        // use stock unit for the line.
        if (this._unitIsSame) {
            currentLine.quantityInReceiptUnitReceived = this._receiptUnit.reduce<decimal>((totalQty, line) => {
                return totalQty + line.quantityInReceiptUnit;
            }, 0);
            currentLine.receiptUnit = this._receiptUnit[0].code;
            currentLine.receiptUnitToStockUnitConversionFactor = Number(
                page.packingUnitToStockUnitConversionFactor.value,
            );
        } else {
            currentLine.quantityInReceiptUnitReceived = this._receiptUnit.reduce<decimal>((totalQty, line) => {
                return totalQty + Number(line.quantityInReceiptUnit) * Number(line.stockUnitConversionFactor);
            }, 0);
            currentLine.receiptUnit = product.stockUnit.code;
            currentLine.receiptUnitToStockUnitConversionFactor = Number(
                page.packingUnitToStockUnitConversionFactor.value,
            );
        }

        //Set receipt line number
        currentLine.lineNumber = (currentLineNumber + 1) * 1000;

        /*  if (
            currentLine.purchaseOrderNumber &&
            currentLine.purchaseOrderLineNumber &&
            currentLine.purchaseOrderLine
        ) {
            currentLine._forMutationOnlyDoClosePurchaseOrderLine =
                page.closePoLine.options.indexOf(page.closePoLine.value) === 0 ? YesNoEnum.yes : YesNoEnum.no;
        } _forMutationOnlyDoClosePurchaseOrderLine missing */
        currentLine.balance = (
            page.closePoLine.value === ui.localize('@sage/x3-purchasing/Yes', 'Yes') ? YesNoEnum.yes : YesNoEnum.no
        ).toString();
        currentLine.warehouse = page.warehouse.value ?? undefined;
        currentLine.container = page.container?.value?.code;
        this.purchaseSession = {
            purchaseEntryTransaction: this._purchaseSession.purchaseEntryTransaction,
            purchaseReceipt: this._purchaseSession.purchaseReceipt,
            orderUnitToPurchaseUnitConversionFactor: page.packingUnitToStockUnitConversionFactor.value ?? 0,
        };
    }

    public loadStockDetails(
        page: {
            receiptUnit: { value: string | null };
            receiptUnitCode: { value: string | null };
            quantityToReceive: { value: number | null };
            quantityToReceiveNoRounded: number;
            isPackingUnit: boolean;
            remainingQuantityInPackingUnit: { value: number | null };
            remainingQuantityInStockUnit: { value: number | null };
            packingUnitToStockUnitConversionFactor: { value: number | null };
            warehouse: { value: string | null };
            status: { value: string | null };
            licensePlateNumber: { value: { code?: string } | null };
            location: { value: { code?: string } | null; isDisabled?: boolean };
            supplierLot: { value: string | null };
            lot: { value: string | null };
            sublot: { value: string | null };
            potency: { value: number | null };
            expirationDate: { value: string | null; isDisabled?: boolean };
            calculatedExpirationDate: { value: string | null };
            useByDate: { value: string | null };
            serialNumber: { value: string | null };
            majorVersion: { value: { code?: string } | null };
            minorVersion: { value: { minorVersion?: string } | null };
            lotCustomField1: { value: string | null };
            lotCustomField2: { value: string | null };
            lotCustomField3: { value: number | null };
            lotCustomField4: { value: string | null };
            identifier1: { value: string | null };
            identifier2: { value: string | null };
        },
        product: {
            stockUnit: {
                code: string;
            };
            code: string;
        }
    ) {
        const quantityReceivedInStockUnit =
            (page.quantityToReceive.value ?? 0) * (page.packingUnitToStockUnitConversionFactor.value ?? 1);

        // Saving receipt units is separate array because only stock and packing units are saved at stock details level.
        this._receiptUnit.push({
            code: page.receiptUnitCode.value ?? '',
            quantityInReceiptUnit: page.quantityToReceive.value ?? 0,
            stockUnitConversionFactor: page.packingUnitToStockUnitConversionFactor.value ?? 1,
        });

        // Check if unit is different from previous line
        if (this._receiptUnit.length > 1 && this._unitIsSame) {
            if (page.receiptUnitCode.value !== this._receiptUnit[this._receiptUnit.length - 2].code) {
                this._unitIsSame = false;
            }
        }

        const detailLine: Partial<StockJournalInput> = {
            packingUnit: page.isPackingUnit ? (page.receiptUnitCode.value ?? '') : product.stockUnit.code,
            quantityInPackingUnit: page.isPackingUnit
                ? (page.quantityToReceive.value ?? 0)
                : quantityReceivedInStockUnit,
            packingUnitToStockUnitConversionFactor: page.isPackingUnit
                ? (page.packingUnitToStockUnitConversionFactor.value ?? 1)
                : 1,
            lot: page.lot.value ?? undefined, // using undefined, omits the property entirely from JSON when stringified
            sublot: page.sublot.value ?? undefined,
            supplierLot: page.supplierLot.value ?? undefined,
            potency: page.potency.value ?? undefined,
            location: page.location.value?.code,
            serialNumber: page.serialNumber.value ?? undefined,
            status: page.status.value ?? undefined, // this is always required
            majorVersion: page.majorVersion.value?.code,
            minorVersion: page.minorVersion.value?.minorVersion,
            licensePlateNumber: page.licensePlateNumber.value?.code,
            expirationDate:
                (page.expirationDate.isDisabled ? page.calculatedExpirationDate.value : page.expirationDate.value) ??
                undefined,
            useByDate: page.useByDate.value ?? undefined,
            lotCustomField1: page.lotCustomField1.value ?? undefined,
            lotCustomField2: page.lotCustomField2.value ?? undefined,
            lotCustomField3: page.lotCustomField3.value ?? undefined,
            lotCustomField4: page.lotCustomField4.value ?? undefined,
            identifier1: page.identifier1.value ?? undefined,
            identifier2: page.identifier2.value ?? undefined,
        };

        this._stockDetails.push(detailLine);

        // Initialize detail page for next detail line.  Most field values will default to previous detail line values.
        page.location.isDisabled = false;
        page.licensePlateNumber.value = { code: '' };
        page.sublot.value = '';
        page.serialNumber.value = '';

        if (!this._purchaseSession || !this._purchaseSession.purchaseReceipt.lines) return;

        const currLine =
            this._purchaseSession.purchaseReceipt.lines[this._purchaseSession.purchaseReceipt.lines.length - 1];

        if (!currLine.purchaseOrder) {
            // Set remaining quantity to 0 for direct receipts
            page.quantityToReceive.value = 0;
        } else {
            // Calculate remaining quantity for PO receipt lines (not direct receipt)
            // remainingQuantity and quantityToReceive will always be the same unit and that unit will be the default
            // for the next detail line.  Value can never be negative

            page.remainingQuantityInStockUnit.value =
                (page.remainingQuantityInStockUnit.value ?? 0) -
                (page.quantityToReceive.value ?? 0) * (page.packingUnitToStockUnitConversionFactor.value ?? 1);
            if (page.packingUnitToStockUnitConversionFactor.value !== 0)
                page.remainingQuantityInPackingUnit.value =
                    page.remainingQuantityInStockUnit.value / (page.packingUnitToStockUnitConversionFactor.value ?? 1);
            page.quantityToReceive.value = Math.max(0, page.remainingQuantityInPackingUnit.value ?? 0);
            page.quantityToReceiveNoRounded = page.quantityToReceive.value;
        }
    }
}
