"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PurchaseReceiptDetailsRecordManager = exports.controlLotReceipt = exports.validate = void 0;
const xtrem_decimal_1 = require("@sage/xtrem-decimal"), typesLib = xtrem_decimal_1;
const xtrem_client_1 = require("@sage/xtrem-client");
const ui = require("@sage/xtrem-ui");
const xtrem_ui_1 = require("@sage/xtrem-ui");
async function validate(page) {
    // (X3-201046, X3-203885) TODO Issue: Have better validation error messages that specifically tell which fields have issues
    const errors = await page.$.page.validate();
    if (typesLib.strictEq(errors.length, 0)) {
        return true;
    }
    page.$.showToast(`${(0, xtrem_ui_1.localize)('@sage/x3-purchasing/dialog-error-title', 'Error')}: ${errors[0]}`, {
        type: 'error',
        timeout: 30000,
    });
    return false;
}
exports.validate = validate;
async function controlLotReceipt(pageInstance, lot, product, entryType, site) {
    if (!lot || !product || !entryType)
        throw new Error('Invalid arguments');
    let stockJournalFilter = {
        stockSite: site,
        documentType: entryType,
        product: { code: product },
        isUpdated: true,
        lot: lot,
    };
    const response = (0, xtrem_client_1.extractEdges)(await pageInstance.$.graph
        .node('@sage/x3-stock-data/StockJournal')
        .query(xtrem_ui_1.queryUtils.edgesSelector({
        lot: true,
        sublot: true,
    }, {
        filter: stockJournalFilter,
    }))
        .execute());
    if (typesLib.gt(response.length, 0)) {
        pageInstance.$.showToast((0, xtrem_ui_1.localize)('@sage/x3-purchasing/notification-error-receipt-lot', 'the lot number {{ lot }} already exists for this product', { lot: lot }), { type: 'error' });
        return false;
    }
    return true;
}
exports.controlLotReceipt = controlLotReceipt;
var YesNoEnum;
(function (YesNoEnum) {
    YesNoEnum[YesNoEnum["no"] = 1] = "no";
    YesNoEnum[YesNoEnum["yes"] = 2] = "yes";
})(YesNoEnum || (YesNoEnum = {}));
class PurchaseReceiptDetailsRecordManager {
    constructor(page, cleanupIncompleteSessionLines = false) {
        this.page = page;
        this._stockDetails = [];
        this._receiptUnit = [];
        this._unitIsSame = true;
        // An incomplete purchase receipt line can be added to the session when user is transitioning from main page to enter detail page
        if (cleanupIncompleteSessionLines)
            this._cleanupSession();
    }
    static { this.PURCHASE_RECEIPT_KEY = 'purchaseReceipt'; }
    get purchaseSession() {
        if (this._purchaseSession) {
            return this._purchaseSession;
        }
        const storedSession = this.page.$.storage.get(PurchaseReceiptDetailsRecordManager.PURCHASE_RECEIPT_KEY);
        if (storedSession) {
            const parsedSession = JSON.parse(storedSession);
            if (this._checkStorage(parsedSession)) {
                this._purchaseSession = parsedSession;
            }
            else {
                // purchase receipt in the session storage is corrupted/malformed
                this.clearSession();
            }
        }
        return this._purchaseSession;
    }
    set purchaseSession(purchaseSession) {
        if (!purchaseSession) {
            return;
        }
        if (!this._checkStorage(purchaseSession)) {
            throw new Error('Invalid Purchase Receipt session');
        }
        this.page.$.storage.set(PurchaseReceiptDetailsRecordManager.PURCHASE_RECEIPT_KEY, JSON.stringify(purchaseSession));
        this._purchaseSession = purchaseSession;
    }
    _checkStorage(sessionData) {
        // check if sessionData contains at least one purchase receipt line that has at least one stock detail line
        if (!sessionData ||
            !sessionData.purchaseEntryTransaction.code ||
            !sessionData.purchaseReceipt.receiptDate ||
            !sessionData.purchaseReceipt.lines || typesLib.strictEq(sessionData.purchaseReceipt.lines.length, 0) ||
            !sessionData.purchaseReceipt.receiptSite ||
            !sessionData.purchaseReceipt.supplier) {
            return false;
        }
        return !sessionData.purchaseReceipt.lines.some((line) => {
            // skip most recent line that is pending to be completed in detail page
            if (!line.stockDetails) {
                return false;
            }
            // check for each line, the receipt unit & quantity are defined
            if (!line.receiptUnit || typesLib.strictEq(line.quantityInReceiptUnitReceived, null) || typesLib.strictEq(line.quantityInReceiptUnitReceived, undefined)) {
                return true;
            }
            return line.stockDetails.some((detail) => {
                // check for each stock detail on this line, the packing unit & quantity are defined
                return (!detail.packingUnit || typesLib.strictEq(detail.quantityInPackingUnit, null) || typesLib.strictEq(detail.quantityInPackingUnit, undefined));
            });
        });
    }
    _cleanupSession() {
        this._purchaseSession = this.purchaseSession; // get the latest from session
        if (!this._purchaseSession || !this._purchaseSession.purchaseReceipt.lines)
            return;
        // remove any incomplete purchase receipt lines (ex. a line without stockDetails)
        const prevLineCount = this._purchaseSession.purchaseReceipt.lines?.length;
        this._purchaseSession.purchaseReceipt.lines = this._purchaseSession.purchaseReceipt.lines.filter((line) => {
            if (!line.receiptUnit || typesLib.strictEq(line.quantityInReceiptUnitReceived, null) || typesLib.strictEq(line.quantityInReceiptUnitReceived, undefined) ||
                !line.stockDetails) {
                return false;
            }
            return !line.stockDetails.some((detail) => {
                // check for each stock detail on this line, the packing unit & quantity are defined
                return (!detail.packingUnit || typesLib.strictEq(detail.quantityInPackingUnit, null) || typesLib.strictEq(detail.quantityInPackingUnit, undefined));
            });
        });
        // if all lines have been removed, clear the session
        if (typesLib.strictEq(this._purchaseSession.purchaseReceipt.lines?.length, 0)) {
            this.clearSession();
        }
        else if (typesLib.strictNe(prevLineCount, this._purchaseSession.purchaseReceipt.lines.length)) {
            // otherwise if any line got removed, overwrite the session
            this.purchaseSession = this._purchaseSession;
        }
    }
    clearSession() {
        this.page.$.storage.remove(PurchaseReceiptDetailsRecordManager.PURCHASE_RECEIPT_KEY);
        this._purchaseSession = null;
    }
    createLine(page, product) {
        if (!this._purchaseSession || !this._purchaseSession.purchaseReceipt.lines)
            return;
        const currentLineNumber = typesLib.sub(this._purchaseSession.purchaseReceipt.lines.length, 1);
        const currentLine = this._purchaseSession.purchaseReceipt.lines[currentLineNumber];
        currentLine.stockDetails = this._stockDetails;
        // If all receipt units are the same, use that unit for the line. If there are multiple receipt units,
        // use stock unit for the line.
        if (this._unitIsSame) {
            currentLine.quantityInReceiptUnitReceived = this._receiptUnit.reduce((totalQty, line) => {
                return typesLib.add(totalQty, line.quantityInReceiptUnit);
            }, 0);
            currentLine.receiptUnit = this._receiptUnit[0].code;
            currentLine.receiptUnitToStockUnitConversionFactor = Number(page.packingUnitToStockUnitConversionFactor.value);
        }
        else {
            currentLine.quantityInReceiptUnitReceived = this._receiptUnit.reduce((totalQty, line) => {
                return typesLib.add(totalQty, typesLib.mul(Number(line.quantityInReceiptUnit), Number(line.stockUnitConversionFactor)));
            }, 0);
            currentLine.receiptUnit = product.stockUnit.code;
            currentLine.receiptUnitToStockUnitConversionFactor = Number(page.packingUnitToStockUnitConversionFactor.value);
        }
        //Set receipt line number
        currentLine.lineNumber = typesLib.mul((typesLib.add(currentLineNumber, 1)), 1000);
        /*  if (
            currentLine.purchaseOrderNumber &&
            currentLine.purchaseOrderLineNumber &&
            currentLine.purchaseOrderLine
        ) {
            currentLine._forMutationOnlyDoClosePurchaseOrderLine =
                page.closePoLine.options.indexOf(page.closePoLine.value) === 0 ? YesNoEnum.yes : YesNoEnum.no;
        } _forMutationOnlyDoClosePurchaseOrderLine missing */
        currentLine.balance = (typesLib.strictEq(page.closePoLine.value, ui.localize('@sage/x3-purchasing/Yes', 'Yes')) ? YesNoEnum.yes : YesNoEnum.no).toString();
        currentLine.warehouse = page.warehouse.value ?? undefined;
        currentLine.container = page.container?.value?.code;
        this.purchaseSession = {
            purchaseEntryTransaction: this._purchaseSession.purchaseEntryTransaction,
            purchaseReceipt: this._purchaseSession.purchaseReceipt,
            orderUnitToPurchaseUnitConversionFactor: page.packingUnitToStockUnitConversionFactor.value ?? 0,
        };
    }
    loadStockDetails(page, product) {
        const quantityReceivedInStockUnit = typesLib.mul((page.quantityToReceive.value ?? 0), (page.packingUnitToStockUnitConversionFactor.value ?? 1));
        // Saving receipt units is separate array because only stock and packing units are saved at stock details level.
        this._receiptUnit.push({
            code: page.receiptUnitCode.value ?? '',
            quantityInReceiptUnit: page.quantityToReceive.value ?? 0,
            stockUnitConversionFactor: page.packingUnitToStockUnitConversionFactor.value ?? 1,
        });
        // Check if unit is different from previous line
        if (typesLib.gt(this._receiptUnit.length, 1) && this._unitIsSame) {
            if (typesLib.strictNe(page.receiptUnitCode.value, this._receiptUnit[typesLib.sub(this._receiptUnit.length, 2)].code)) {
                this._unitIsSame = false;
            }
        }
        const detailLine = {
            packingUnit: page.isPackingUnit ? (page.receiptUnitCode.value ?? '') : product.stockUnit.code,
            quantityInPackingUnit: page.isPackingUnit
                ? (page.quantityToReceive.value ?? 0)
                : quantityReceivedInStockUnit,
            packingUnitToStockUnitConversionFactor: page.isPackingUnit
                ? (page.packingUnitToStockUnitConversionFactor.value ?? 1)
                : 1,
            lot: page.lot.value ?? undefined, // using undefined, omits the property entirely from JSON when stringified
            sublot: page.sublot.value ?? undefined,
            supplierLot: page.supplierLot.value ?? undefined,
            potency: page.potency.value ?? undefined,
            location: page.location.value?.code,
            serialNumber: page.serialNumber.value ?? undefined,
            status: page.status.value ?? undefined, // this is always required
            majorVersion: page.majorVersion.value?.code,
            minorVersion: page.minorVersion.value?.minorVersion,
            licensePlateNumber: page.licensePlateNumber.value?.code,
            expirationDate: (page.expirationDate.isDisabled ? page.calculatedExpirationDate.value : page.expirationDate.value) ??
                undefined,
            useByDate: page.useByDate.value ?? undefined,
            lotCustomField1: page.lotCustomField1.value ?? undefined,
            lotCustomField2: page.lotCustomField2.value ?? undefined,
            lotCustomField3: page.lotCustomField3.value ?? undefined,
            lotCustomField4: page.lotCustomField4.value ?? undefined,
            identifier1: page.identifier1.value ?? undefined,
            identifier2: page.identifier2.value ?? undefined,
        };
        this._stockDetails.push(detailLine);
        // Initialize detail page for next detail line.  Most field values will default to previous detail line values.
        page.location.isDisabled = false;
        page.licensePlateNumber.value = { code: '' };
        page.sublot.value = '';
        page.serialNumber.value = '';
        if (!this._purchaseSession || !this._purchaseSession.purchaseReceipt.lines)
            return;
        const currLine = this._purchaseSession.purchaseReceipt.lines[typesLib.sub(this._purchaseSession.purchaseReceipt.lines.length, 1)];
        if (!currLine.purchaseOrder) {
            // Set remaining quantity to 0 for direct receipts
            page.quantityToReceive.value = 0;
        }
        else {
            // Calculate remaining quantity for PO receipt lines (not direct receipt)
            // remainingQuantity and quantityToReceive will always be the same unit and that unit will be the default
            // for the next detail line.  Value can never be negative
            page.remainingQuantityInStockUnit.value = typesLib.sub((page.remainingQuantityInStockUnit.value ?? 0), typesLib.mul((page.quantityToReceive.value ?? 0), (page.packingUnitToStockUnitConversionFactor.value ?? 1)));
            if (typesLib.strictNe(page.packingUnitToStockUnitConversionFactor.value, 0))
                page.remainingQuantityInPackingUnit.value = typesLib.div(page.remainingQuantityInStockUnit.value, (page.packingUnitToStockUnitConversionFactor.value ?? 1));
            page.quantityToReceive.value = typesLib.max(0, page.remainingQuantityInPackingUnit.value ?? 0);
            page.quantityToReceiveNoRounded = page.quantityToReceive.value;
        }
    }
}
exports.PurchaseReceiptDetailsRecordManager = PurchaseReceiptDetailsRecordManager;
//# sourceMappingURL=purchase-receipt-details-control.js.map