import { Product } from '@sage/x3-master-data-api';
import { getSelectedStockSite } from '@sage/x3-master-data/lib/client-functions/get-selected-stock-site';
import { PurchaseOrderLine } from '@sage/x3-purchasing-api';
import { receipt } from '@sage/x3-stock-data/build/lib/menu-items/receipt';
import { DateValue } from '@sage/xtrem-date-time';
import * as ui from '@sage/xtrem-ui';

@ui.decorators.page<MobileViewPurchaseOrderToReceive, PurchaseOrderLine>({
    module: 'purchasing',
    title: 'View purchase orders to receive',
    isTitleHidden: true,
    menuItem: receipt,
    priority: 300,
    mode: 'default',
    access: { node: '@sage/x3-purchasing/PurchaseOrderLine' },
    authorizationCode: 'INQPOR',
    skipDirtyCheck: true,
    async onLoad() {
        this.siteField.value = await getSelectedStockSite(
            this,
            ui.localize('@sage/x3-purchasing-sales/dialog-error-title', 'Error'),
            ui.localize(
                '@sage/x3-purchasing-sales/dialog-error-location-inquiry-set-site',
                'Define a default stock site on the user function profile.',
            ),
        );
        if (!this.siteField.value) {
            this._disablePage();
        }
        this._initializeDate();
    },
    businessActions() {
        return [this.searchButton];
    },
})
export class MobileViewPurchaseOrderToReceive extends ui.Page {
    @ui.decorators.textField<MobileViewPurchaseOrderToReceive>({
        isHidden: true,
    })
    siteField: ui.fields.Text;

    @ui.decorators.section<MobileViewPurchaseOrderToReceive>({
        isTitleHidden: true,
    })
    mainSection: ui.containers.Section;

    @ui.decorators.block<MobileViewPurchaseOrderToReceive>({
        parent() {
            return this.mainSection;
        },
    })
    mainBlock: ui.containers.Block;

    @ui.decorators.separatorField<MobileViewPurchaseOrderToReceive>({
        parent() {
            return this.mainBlock;
        },
        isInvisible: true,
    })
    separatorField: ui.fields.Separator;

    @ui.decorators.dateField<MobileViewPurchaseOrderToReceive>({
        parent() {
            return this.mainBlock;
        },
        title: 'Expected receipt date',
        isFullWidth: true,
    })
    date: ui.fields.Date;

    @ui.decorators.referenceField<MobileViewPurchaseOrderToReceive, Product>({
        parent() {
            return this.mainBlock;
        },
        title: 'Product',
        placeholder: 'Scan or select...',
        node: '@sage/x3-master-data/Product',
        valueField: 'code',
        helperTextField: 'upc',
        canFilter: false,
        filter() {
            return {
                productSites: { _atLeast: 1, stockSite: { code: this.siteField.value } },
                productStatus: 'active',
                isReceived: { _eq: true },
                _or: [
                    { isPurchased: { _eq: true } },
                    { isSubcontracted: { _eq: true } },
                    { isService: { _eq: true } },
                    { isPhantom: { _eq: true } },
                    { isDeliverable: { _eq: true } },
                    { isSold: { _eq: true } },
                ],
            };
        },
        isTransient: true,
        isFullWidth: true,
        isAutoSelectEnabled: true,
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Product',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'upc',
                title: 'UPC',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'description1',
                title: 'Description',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/ProductCategory',
                bind: 'productCategory',
                valueField: 'code',
                title: 'Product',
                isReadOnly: true,
            }),
        ],
    })
    product: ui.fields.Reference;

    @ui.decorators.pageAction<MobileViewPurchaseOrderToReceive>({
        title: 'Search',
        buttonType: 'primary',
        shortcut: ['f3'],
        async onClick() {
            if (!this.date.value) {
                this.$.storage.set('date', '');
            } else {
                this.$.storage.set('date', this.date.value);
            }
            this.$.router.goTo('@sage/x3-purchasing-sales/MobileViewPurchaseOrderToReceiveSelectAProduct', {
                product: this.product.value ? this.product.value.code : '',
            });
        },
    })
    searchButton: ui.PageAction;

    private _disablePage(): void {
        this.searchButton.isDisabled = true;
        this.date.isDisabled = true;
        this.product.isDisabled = true;
    }
    private _initializeDate(): void {
        this.date.value = DateValue.today().toString();
    }
}
