import { GraphApi } from '@sage/x3-purchasing-sales-api';
import { SalesDelivery } from '@sage/x3-sales-api';
import { extractEdges } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import { IntersiteReceiptDetailsRecordManager } from '../client-functions/intersite-receipt-details-control';

@ui.decorators.page<MobileViewPurchaseOrderToReceiveSelectADelivery>({
    title: 'Select a delivery',
    module: 'x3-purchasing-sales',
    mode: 'default',
    isTitleHidden: true,
    isTransient: true,
    headerCard() {
        return {
            title: this.purchaseOrderId,
            titleRight: this.supplierCode,
            line2: this.shortCompanyName,
        };
    },
    async onLoad() {
        this.siteField.value = this.$.storage.get('mobile-selected-stock-site').toString();
        this.purchaseOrderId.value = this.$.queryParameters.purchaseOrderId as string;
        await this._getPurchaseOrder(this.purchaseOrderId.value);
        this.salesDeliveries.value = JSON.parse(this.$.queryParameters.salesDeliveries as string);
    },
})
export class MobileViewPurchaseOrderToReceiveSelectADelivery extends ui.Page<GraphApi> {
    salesDeliveriesId: string[] = [];

    @ui.decorators.section<MobileViewPurchaseOrderToReceiveSelectADelivery>({
        isTitleHidden: true,
        isHidden: false,
    })
    mainSection: ui.containers.Section;

    @ui.decorators.block<MobileViewPurchaseOrderToReceiveSelectADelivery>({
        parent() {
            return this.mainSection;
        },
    })
    mainBlock: ui.containers.Block;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceiveSelectADelivery>({
        isTransient: true,
        isReadOnly: true,
    })
    purchaseOrderId: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceiveSelectADelivery>({
        isTransient: true,
        isReadOnly: true,
    })
    supplierCode: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceiveSelectADelivery>({
        isTransient: true,
        isReadOnly: true,
    })
    shortCompanyName: ui.fields.Text;

    @ui.decorators.labelField<MobileViewPurchaseOrderToReceiveSelectADelivery>({
        parent() {
            return this.mainBlock;
        },
        isTransient: true,
        isHidden: true,
    })
    siteField: ui.fields.Label;

    @ui.decorators.tableField<MobileViewPurchaseOrderToReceiveSelectADelivery, SalesDelivery>({
        parent() {
            return this.mainBlock;
        },
        node: '@sage/x3-sales/SalesDelivery',
        isChangeIndicatorDisabled: true,
        canFilter: false,
        canSelect: false,
        canExport: false,
        canResizeColumns: false,
        canUserHideColumns: false,
        isTitleHidden: false,
        isTransient: true,
        isFullWidth: true,
        isDisabled: false,
        hasSearchBoxMobile: false,
        cardView: true,
        displayMode: ui.fields.TableDisplayMode.compact,
        mobileCard: undefined,
        columns: [
            ui.nestedFields.text({
                bind: 'id',
                isReadOnly: true,
                canFilter: true,
                title: 'Delivery no.',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-system/Site',
                bind: 'salesSite',
                valueField: 'code',
                title: 'Sales site',
            }),
        ],
        async onRowClick(_recordId: string, rowItem: SalesDelivery) {
            const intersiteReceiptRecordManager = new IntersiteReceiptDetailsRecordManager(this, true);
            if (intersiteReceiptRecordManager.intersiteSession) {
                this.$.removeToasts();
                setTimeout(
                    () =>
                        this.$.showToast(
                            ui.localize(
                                '@sage/x3-purchasing-sales/notification-preloading-verification-intersite-receipt',
                                'Intersite receipt already in progress.  Starting a new intersite receipt is not allowed.',
                            ),
                            { type: 'warning', timeout: 30000 },
                        ),
                    10,
                );
                return false;
            } else {
                this.$.router.goTo('@sage/x3-purchasing-sales/MobileIntersiteReceipt', {
                    salesDeliveryId: rowItem.id,
                    salesSite: rowItem.salesSite.code,
                });
                return true;
            }
        },
    })
    salesDeliveries: ui.fields.Table<SalesDelivery>;

    private async _getPurchaseOrder(purchaseOrderId: string) {
        const _purchaseOrder = await this.$.graph
            .node('@sage/x3-purchasing/PurchaseOrder')
            .read(
                {
                    _id: true,
                    id: true,
                    orderFromSupplier: {
                        code: {
                            code: true,
                            shortCompanyName: true,
                        },
                    },
                },
                purchaseOrderId,
            )
            .execute();
        if (_purchaseOrder?.orderFromSupplier) {
            this.supplierCode.value = _purchaseOrder.orderFromSupplier.code.code;
            this.shortCompanyName.value = _purchaseOrder.orderFromSupplier.code.shortCompanyName;
        }
    }
}
