import { GraphApi, PurchaseOrderLine } from '@sage/x3-purchasing-api';
import { PurchaseReceiptDetailsRecordManager } from '@sage/x3-purchasing/lib/client-functions/purchase-receipt-details-control';
import * as ui from '@sage/xtrem-ui';
import { goToPurchaseOrderToReceiveSelectADelivery } from '../client-functions/purchase-order-control';

@ui.decorators.page<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
    module: 'purchasing',
    title: 'Purchase orders to receive',
    subtitle: 'Line details',
    isTitleHidden: true,
    isTransient: true,
    skipDirtyCheck: true,
    headerCard() {
        return {
            title: this.productCode,
            titleRight: this.remainingQuantityToReceive,
            line2: this.localizedDescription1,
            line3: this.purchaseOrderId,
            line3Right: this.expectedReceiptDate,
        };
    },
    async onLoad() {
        this.productCode.value = this.$.queryParameters['productCode'] as string;
        this.purchaseOrderId.value = this.$.queryParameters['purchaseOrderId'] as string;
        this.upcCode.value = this.$.queryParameters['upcCode'] as string;
        this.lineNumber.value = this.$.queryParameters['lineNumber'] as string;
        this.receiptSite.value = this.$.queryParameters['receiptSite'] as string;
        this.isIntersite.value = this.$.queryParameters['isIntersite'] as string;

        const response: any = await this.$.graph
            .node('@sage/x3-purchasing/PurchaseOrderLine')
            .query(
                ui.queryUtils.edgesSelector<PurchaseOrderLine>(
                    {
                        _id: true,
                        purchaseOrder: { id: true },
                        lineNumber: true,
                        receiptSite: { code: true },
                        orderFromSupplier: { code: { code: true } },
                        expectedReceiptDate: true,
                        quantityInPurchaseUnitOrdered: true,
                        quantityInStockUnitOrdered: true,
                        quantityInPurchaseUnitReceived: true,
                        quantityInStockUnitReceived: true,
                        quantityInOrderUnitOrdered: true,
                        purchaseUnit: {
                            code: true,
                        },
                        stockUnit: {
                            code: true,
                        },
                        orderUnit: {
                            code: true,
                        },
                        product: {
                            code: true,
                            localizedDescription1: true,
                            upc: true,
                            isVersionManaged: true,
                        },
                        majorProductVersion: { code: true },
                        minorProductVersion: true,
                        supplierProduct: {
                            supplierProduct: true,
                        },
                    },
                    {
                        filter: {
                            product: {
                                code: this.productCode.value,
                            },
                            lineNumber: Number(this.lineNumber.value),
                            lineStatus: { _ne: 'closed' },
                            purchaseOrder: {
                                id: this.purchaseOrderId.value,
                            },
                            receiptSite: {
                                code: this.receiptSite.value,
                            },
                        },
                    },
                ),
            )
            .execute();

        this.localizedDescription1.value = response.edges[0].node.product.localizedDescription1;
        this.expectedReceiptDate.value = ui.formatDateToCurrentLocale(response.edges[0].node.expectedReceiptDate);
        this.supplierCode = response.edges[0].node.orderFromSupplier.code.code;
        this.remainingQuantityToReceive.value = `${
            response.edges[0].node.quantityInStockUnitOrdered - response.edges[0].node.quantityInStockUnitReceived
        } ${response.edges[0].node.stockUnit.code}`;
        this.purchaseOrderLineDetailsList.value = response.edges.map((currentRecord: any) => {
            let supplierProduct: string;
            supplierProduct = '';
            if (currentRecord.node.supplierProduct !== null) {
                supplierProduct = currentRecord.node.supplierProduct.supplierProduct;
            }
            return {
                _id: currentRecord.node.$id,
                lineNumber: currentRecord.node.lineNumber,
                productDescription: currentRecord.node.product.localizedDescription1,
                productUpc: currentRecord.node.product.upc,
                orderFromSupplier: currentRecord.node.orderFromSupplier.code.code,
                supplierProduct: supplierProduct,
                quantityInOrderUnitOrdered: `${currentRecord.node.quantityInOrderUnitOrdered} ${currentRecord.node.orderUnit.code}`,
                quantityInStockUnitReceived: `${currentRecord.node.quantityInStockUnitReceived} ${currentRecord.node.stockUnit.code}`,
                quantityInStockUnitOrdered: `${currentRecord.node.quantityInStockUnitOrdered} ${currentRecord.node.stockUnit.code}`,
                quantityInPurchaseUnitOrdered: `${currentRecord.node.quantityInPurchaseUnitOrdered} ${currentRecord.node.purchaseUnit.code}`,
                quantityInPurchaseUnitReceived: `${currentRecord.node.quantityInPurchaseUnitReceived} ${currentRecord.node.purchaseUnit.code}`,
                majorProductVersion: currentRecord.node.majorProductVersion?.code,
                minorProductVersion: currentRecord.node.minorProductVersion,
            };
        });

        if (response.edges[0].node.product.isVersionManaged) {
            this.isProductVersionManaged = true;
        } else {
            this.isProductVersionManaged = false;
        }
        this.purchaseReceiptButton.isHidden = !!this.isIntersite.value;
        this.intersiteReceiptButton.isHidden = !this.isIntersite.value;
    },
    businessActions() {
        return [this.purchaseReceiptButton, this.intersiteReceiptButton];
    },
})
export class MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail extends ui.Page<GraphApi> {
    private supplierCode: string | number | boolean;
    private isProductVersionManaged: boolean;

    @ui.decorators.section<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        isTitleHidden: true,
    })
    mainSection: ui.containers.Section;

    @ui.decorators.block<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        parent() {
            return this.mainSection;
        },
    })
    mainBlock: ui.containers.Block;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        isTransient: false,
        isReadOnly: true,
    })
    receiptSite: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        isTransient: false,
        isReadOnly: true,
    })
    productCode: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        isReadOnly: true,
    })
    localizedDescription1: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        isTransient: false,
        isReadOnly: true,
    })
    purchaseOrderId: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        isTransient: false,
        isReadOnly: true,
    })
    upcCode: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        isTransient: false,
        isReadOnly: true,
    })
    lineNumber: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        isTransient: false,
        isReadOnly: true,
    })
    remainingQuantityToReceive: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        isTransient: false,
        isReadOnly: true,
    })
    remainingQtyToReceiveWithStockUnit: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        isTransient: false,
        isReadOnly: true,
        prefix: 'Expected',
    })
    expectedReceiptDate: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        isTransient: false,
        isReadOnly: true,
    })
    purchaseUnit: ui.fields.Text;

    @ui.decorators.textField<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        isTransient: false,
        isReadOnly: true,
    })
    isIntersite: ui.fields.Text;

    @ui.decorators.detailListField<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        parent() {
            return this.mainBlock;
        },
        isTransient: true,
        fields: [
            ui.nestedFields.text({
                title: 'Line number ',
                bind: 'lineNumber',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'Description',
                bind: 'productDescription',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'Product UPC',
                bind: 'productUpc',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'Supplier',
                bind: 'orderFromSupplier',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'Supplier product',
                bind: 'supplierProduct',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'Order qty.',
                bind: 'quantityInOrderUnitOrdered',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'STK order qty.',
                bind: 'quantityInStockUnitOrdered',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'STK qty. rec.',
                bind: 'quantityInStockUnitReceived',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'PUR order qty.',
                bind: 'quantityInPurchaseUnitOrdered',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'PUR receipt qty.',
                bind: 'quantityInPurchaseUnitReceived',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                title: 'Major version',
                bind: 'majorProductVersion',
                isReadOnly: true,
                isHidden() {
                    return !this.isProductVersionManaged; // return true to hide the field and return false to not hide the field
                },
            }),
            ui.nestedFields.text({
                title: 'Minor version',
                bind: 'minorProductVersion',
                isReadOnly: true,
                isHidden() {
                    return !this.isProductVersionManaged;
                },
            }),
        ],
    })
    purchaseOrderLineDetailsList: ui.fields.DetailList;

    @ui.decorators.pageAction<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        title: 'Purchase receipt',
        buttonType: 'primary',
        onClick() {
            const purchaseReceiptRecordManager = new PurchaseReceiptDetailsRecordManager(this, true);
            if (purchaseReceiptRecordManager.purchaseSession) {
                this.$.removeToasts();
                setTimeout(
                    () =>
                        this.$.showToast(
                            ui.localize(
                                '@sage/x3-purchasing-sales/notification-preloading-verification-purchase-receipt',
                                'Purchase receipt already in progress.  Starting a new purchase receipt is not allowed.',
                            ),
                            { type: 'warning', timeout: 30000 },
                        ),
                    10,
                );
            } else {
                this.$.router.goTo('@sage/x3-purchasing/MobilePurchaseReceipt', {
                    purchaseOrderId: this.purchaseOrderId.value ?? '',
                    supplierCode: this.supplierCode,
                });
            }
        },
    })
    purchaseReceiptButton: ui.PageAction;

    @ui.decorators.pageAction<MobileViewPurchaseOrderToReceivePurchaseOrderLineDetail>({
        title: 'Intersite receipt',
        buttonType: 'primary',
        async onClick() {
            await goToPurchaseOrderToReceiveSelectADelivery(this);
        },
    })
    intersiteReceiptButton: ui.PageAction;
}
