import { UnitOfMeasure } from '@sage/x3-master-data-api-partial';
import { dialogConfirmation, dialogMessage } from '@sage/x3-master-data/lib/client-functions/dialogs';
import { getSelectedStockSite } from '@sage/x3-master-data/lib/client-functions/get-selected-stock-site';
import {
    PurchaseEntryTransaction,
    PurchaseEntryTransactionInput,
    PurchaseReceiptLineInput,
} from '@sage/x3-purchasing-api';
import { SalesDelivery } from '@sage/x3-sales-api';
import { receipt } from '@sage/x3-stock-data/build/lib/menu-items/receipt';
import { ExtractEdgesPartial, extractEdges } from '@sage/xtrem-client';
import { DateValue } from '@sage/xtrem-date-time';
import * as ui from '@sage/xtrem-ui';
import { getNumberOfDecimal, getUnitNumberOfDecimalList } from '../client-functions/get-unit-number-decimals';
import {
    IntersiteReceiptDetailsRecordManager,
    IntersiteReceiptSession,
} from '../client-functions/intersite-receipt-details-control';

@ui.decorators.page<MobileIntersiteReceipt>({
    title: 'Intersite receipt',
    module: 'x3-purchasing-sales',
    mode: 'default',
    menuItem: receipt,
    priority: 150,
    isTransient: true,
    isTitleHidden: true,
    authorizationCode: 'CWSPTH',
    access: { node: '@sage/x3-purchasing/PurchaseReceipt' },
    async onLoad() {
        if (!(await this._init())) {
            this._disablePage();
        }
    },
    businessActions() {
        return [this.createButton, this.nextButton];
    },
})
export class MobileIntersiteReceipt extends ui.Page {
    /*
     *
     *  Technical properties
     *
     */
    private _intersiteReceiptRecordManager: IntersiteReceiptDetailsRecordManager; // to store the receipt to create from session data
    private _intersiteSession: IntersiteReceiptSession;
    private _entryTransactionMap: Map<string, PurchaseEntryTransactionInput>;
    private _numberOfDecimalList: ExtractEdgesPartial<UnitOfMeasure>[];
    private _isLocationPreloaded: boolean;

    @ui.decorators.textField<MobileIntersiteReceipt>({
        isHidden: true,
    })
    stockSite: ui.fields.Text;

    /*
     *
     *  Page Actions
     *
     */
    @ui.decorators.pageAction<MobileIntersiteReceipt>({
        title: 'Next',
        buttonType: 'primary',
        shortcut: ['f3'],
        isDisabled: true,
        async onClick() {
            if (this.salesDelivery.value) {
                this.$.removeToasts();
                this.$.setPageClean();
                if (!this._intersiteSession) {
                    const selectedEntryTransaction = this._entryTransactionMap.get(this.transaction.value ?? '');
                    this._intersiteReceiptRecordManager.intersiteSession = {
                        purchaseReceipt: {
                            receiptSite: this.stockSite.value ?? undefined,
                            receiptDate: this.receiptDate.value ?? undefined,
                            supplier: this.supplier.value ?? undefined,
                            stockAutomaticJournal: selectedEntryTransaction?.stockAutomaticJournal,
                            stockMovementCode: selectedEntryTransaction?.stockMovementCode ?? '',
                            transaction: selectedEntryTransaction?.code ?? '',
                            destination: this.$.storage.get('mobile-label-destination')?.toString() ?? '',
                            document: this.$.storage.get('mobile-document-destination')?.toString() ?? '',
                            supplierPackingSlip: this._getSupplierPackingSlip() ?? undefined,
                            supplierPackingSlipDate: DateValue.today().toString(),
                            lines: [],
                        },
                        purchaseEntryTransaction: {
                            code: this.transaction.value ?? undefined,
                            isLotCustomField1Allowed: selectedEntryTransaction?.isLotCustomField1Allowed,
                            isLotCustomField2Allowed: selectedEntryTransaction?.isLotCustomField2Allowed,
                            isLotCustomField3Allowed: selectedEntryTransaction?.isLotCustomField3Allowed,
                            isLotCustomField4Allowed: selectedEntryTransaction?.isLotCustomField4Allowed,
                            identifier1Entry: selectedEntryTransaction?.identifier1Entry,
                            identifier1Detail: selectedEntryTransaction?.identifier1Detail,
                            identifier2Entry: selectedEntryTransaction?.identifier2Entry,
                            identifier2Detail: selectedEntryTransaction?.identifier2Detail,
                            isLotPotencyAllowed: selectedEntryTransaction?.isLotPotencyAllowed,
                            isLotExpirationDateAllowed: selectedEntryTransaction?.isLotExpirationDateAllowed,
                            stockMovementCode: selectedEntryTransaction?.stockMovementCode,
                        },
                        salesUnitToStockUnitConversionFactor: 1,
                    };
                    this._intersiteSession = this._intersiteReceiptRecordManager.intersiteSession;
                } else {
                    this._intersiteReceiptRecordManager.intersiteSession = {
                        purchaseReceipt: {
                            ...this._intersiteReceiptRecordManager?.intersiteSession?.purchaseReceipt,
                            supplier: this.supplier.value ?? undefined,
                            supplierPackingSlip: this._getSupplierPackingSlip() ?? undefined,
                        },
                        purchaseEntryTransaction:
                            this._intersiteReceiptRecordManager.intersiteSession.purchaseEntryTransaction,
                        salesUnitToStockUnitConversionFactor: 1,
                    };
                }
                await this._goToPageTwo();
            }
        },
    })
    nextButton: ui.PageAction;

    private async _goToPageTwo(): Promise<void> {
        const errors = await this.$.page.validate();
        if (errors.length === 0) {
            this.$.setPageClean();
            this.$.router.goTo('@sage/x3-purchasing-sales/MobileIntersiteReceiptSelectFromDelivery', {
                _id: this.salesDelivery?.value?.id,
                deliveryLines: JSON.stringify(await this._getSalesDeliveryLines()),
                isLocationPreloaded: `${this._isLocationPreloaded ? '1' : '0'}`,
            });
        } else {
            this.$.showToast(`${ui.localize('@sage/x3-purchasing-sales/dialog-error-title', 'Error')}: ${errors[0]}`, {
                type: 'error',
                timeout: 30000,
            });
        }
    }

    private async _getSalesDeliveryLines(): Promise<string[]> {
        const response = extractEdges(
            await this.$.graph
                .node('@sage/x3-sales/SalesDeliveryLine')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            lineNumber: true,
                        },
                        {
                            filter: {
                                salesDelivery: { id: this.salesDelivery?.value?.id },
                                receiptStatus: { _ne: 2 },
                            },
                        },
                    ),
                )
                .execute(),
        ) as any[];
        return response.map(line => line.lineNumber);
    }

    @ui.decorators.pageAction<MobileIntersiteReceipt>({
        title: 'Create',
        buttonType: 'primary',
        shortcut: ['f2'],
        isDisabled: true,
        async onClick() {
            // Check if the purchase receipt to create contains at least one valid line
            if (!this._intersiteSession) {
                this.$.showToast(
                    ui.localize(
                        '@sage/x3-purchasing-sales/notification-error-intersite-receipt-no-products',
                        `Enter at least one product.`,
                    ),
                    { type: 'error', timeout: 5000 },
                );
                return;
            }

            // Only validate receipt date is populated
            const validateError = (await this.receiptDate.validate()) || (await this.supplierPackingSlip.validate());
            if (validateError) {
                this.$.removeToasts();
                this.$.showToast(
                    `${ui.localize(
                        '@sage/x3-purchasing-sales/notification-error-intersite-receipt-error',
                        'Error',
                    )}: ${validateError}`,
                    { type: 'error', timeout: 30000 },
                );
                return;
            }
            this.createButton.isDisabled = true; // to prevent extreme case of duplicate receipts by rapidly clicking on create button multiple times
            this.$.loader.isHidden = false;
            const result = await this._callCreationAPI();
            this.$.loader.isHidden = true;

            // Special case unable to connect check type of error
            if (!result || result instanceof Error) {
                this.$.loader.isHidden = true;
                const options: ui.dialogs.DialogOptions = {
                    acceptButton: {
                        text: ui.localize('@sage/x3-purchasing-sales/button-goback', 'Go back'),
                    },
                    cancelButton: {
                        text: ui.localize('@sage/x3-purchasing-sales/button-cancel', 'Cancel'),
                    },
                    size: 'small',
                    mdContent: true,
                };
                let message = '';

                if (!result?.message) {
                    message = `${ui.localize(
                        '@sage/x3-purchasing-sales/pages_creation_error_connexion_webservice_contact_administrator',
                        'An error has occurred (connection or webservice error). Please contact your administrator.',
                    )}`;
                } else {
                    const _messages = <string[]>[];
                    const _results = <any>result;
                    let _diagnoses = _results?.diagnoses;
                    if (_diagnoses?.length > 1) {
                        _diagnoses = _diagnoses.splice(0, _diagnoses.length - 1);
                    }

                    (
                        (_results?.errors
                            ? _results.errors[0]?.extensions?.diagnoses
                            : (_results?.innerError?.errors[0]?.extensions?.diagnoses ??
                              _results.extensions?.diagnoses ??
                              _diagnoses)) ?? []
                    )
                        .filter((d: { severity: number; message: any }) => d.severity > 2 && d.message)
                        .forEach((d: { message: any }) => {
                            const _message = d.message.split(`\n`);
                            _messages.push(..._message);
                        });

                    const _result = _messages.length ? <string[]>_messages : <string[]>result.message.split(`\n`);

                    message = `**${ui.localize(
                        '@sage/x3-purchasing-sales/dialog-error-intersite-receipt-creation',
                        'An error occurred',
                    )}**\n\n`;

                    if (_result.length === 1) {
                        message += `${_result[0]}`;
                    } else {
                        message += _result.map(item => `* ${item}`).join('\n');
                    }
                }
                await this.$.sound.error();

                if (
                    await dialogConfirmation(
                        this,
                        'error',
                        ui.localize('@sage/x3-purchasing-sales/dialog-error-title', 'Error'),
                        `${message}`,
                        options,
                    )
                ) {
                    await this.$.router.refresh();
                } else {
                    this._intersiteReceiptRecordManager.clearSession();
                    await this.$.router.emptyPage();
                }
            } else {
                this._intersiteReceiptRecordManager.clearSession();
                await this.$.sound.success();
                await dialogMessage(
                    this,
                    'success',
                    ui.localize('@sage/x3-purchasing-sales/dialog-success-title', 'Success'),
                    ui.localize(
                        '@sage/x3-purchasing-sales/dialog-success-intersite-receipt-creation',
                        'Document no. {{receiptId}} created.',
                        { receiptId: result.x3Purchasing.purchaseReceipt.create.id },
                    ),
                    {
                        fullScreen: true,
                        acceptButton: {
                            text: ui.localize('@sage/x3-purchasing-sales/button-accept-ok', 'OK'),
                        },
                    },
                );
                await this.$.router.emptyPage();
            }
        },
    })
    createButton: ui.PageAction;

    /*
     *
     *  Sections
     *
     */

    @ui.decorators.section<MobileIntersiteReceipt>({
        isTitleHidden: true,
    })
    mainSection: ui.containers.Section;

    /*
     *
     *  Blocks
     *
     */

    @ui.decorators.block<MobileIntersiteReceipt>({
        parent() {
            return this.mainSection;
        },
        isTitleHidden: true,
    })
    productBlock: ui.containers.Block;

    @ui.decorators.block<MobileIntersiteReceipt>({
        parent() {
            return this.mainSection;
        },
        title: 'Receipt to create',
        isHidden: true,
    })
    receiptBlock: ui.containers.Block;

    /*
     *
     *  Fields
     *
     */
    @ui.decorators.dateField<MobileIntersiteReceipt>({
        parent() {
            return this.productBlock;
        },
        title: 'Receipt date',
        isTransient: true,
        isMandatory: true,
        maxDate: DateValue.today().toString(), // restricts to current date or earlier
        onChange() {
            if (this._intersiteSession && this.receiptDate.value) {
                this._intersiteSession.purchaseReceipt.receiptDate = this.receiptDate.value;
                this._intersiteReceiptRecordManager.intersiteSession = this._intersiteSession; // trigger session manager's setter to save receipt date into session
            }
        },
    })
    receiptDate: ui.fields.Date;

    @ui.decorators.dropdownListField<MobileIntersiteReceipt>({
        parent() {
            return this.productBlock;
        },
        title: 'Transaction',
        isMandatory: true,
        isTransient: true,
    })
    transaction: ui.fields.DropdownList;

    @ui.decorators.labelField<MobileIntersiteReceipt>({
        isTransient: true,
        isHidden: true,
    })
    legalCompany: ui.fields.Label;

    @ui.decorators.referenceField<MobileIntersiteReceipt, SalesDelivery>({
        parent() {
            return this.productBlock;
        },
        title: 'Delivery number',
        placeholder: 'Scan or select...',
        node: '@sage/x3-sales/SalesDelivery',
        valueField: 'id',
        helperTextField: { salesSite: { code: true } },
        isHelperTextHidden: true,
        canFilter: false,
        filter() {
            let deliveryFilter: any = {
                salesSite: { code: { _ne: this.stockSite.value } },
                isIntersite: true,
                isValidated: true,
                isLoanDelivery: false,
                shipmentDate: { _lte: this.receiptDate.value },
                receivingSite: { code: this.stockSite.value },
                salesDeliveryLines: {
                    _atLeast: 1,
                    receiptStatus: { _ne: 2 },
                },
            };
            if (this._intersiteSession) {
                if (this._intersiteSession.purchaseReceipt) {
                    deliveryFilter = {
                        ...deliveryFilter,
                        id: {
                            _nin: this._intersiteSession.purchaseReceipt?.lines?.map(
                                (line: { salesDelivery: any }) => line.salesDelivery,
                            ),
                        },
                        salesDeliveryLines: {
                            _atLeast: 1,
                            receiptStatus: { _ne: 2 },
                            salesOrderLine: {
                                purchaseOrder: {
                                    orderFromSupplier: {
                                        code: {
                                            code: this._intersiteSession.purchaseReceipt.supplier,
                                        },
                                    },
                                },
                            },
                        },
                    };
                }
            }
            return deliveryFilter;
        },
        async onChange() {
            if (this.salesDelivery.value) {
                this.supplier.value = await this._getSupplier(this.salesDelivery.value.id);
                if (!this.supplier.value) {
                    this.supplier.value = this.salesDelivery.value.salesSite.code;
                }
                if (this._intersiteSession?.purchaseReceipt) {
                    this.supplierPackingSlip.value = null;
                } else {
                    this.supplierPackingSlip.value = this.salesDelivery.value.id;
                }
                this.nextButton.isHidden = false;
                this.nextButton.isDisabled = false;
            } else {
                if (this._intersiteSession?.purchaseReceipt) {
                    this.supplier.value = this._intersiteSession.purchaseReceipt.supplier ?? null;
                    this.supplierPackingSlip.value = this._getSupplierPackingSlip();
                } else {
                    this.supplier.value = null;
                    this.supplierPackingSlip.value = null;
                }
                this.nextButton.isHidden = false;
                this.nextButton.isDisabled = true;
            }
        },
        isTransient: true,
        isFullWidth: true,
        isAutoSelectEnabled: true,
        columns: [
            ui.nestedFields.text({
                bind: 'id',
                title: 'Order no.',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-system/Site',
                bind: 'salesSite',
                valueField: 'code',
                title: 'Sales site',
                isReadOnly: true,
            }),
        ],
    })
    salesDelivery: ui.fields.Reference;

    @ui.decorators.textField<MobileIntersiteReceipt>({
        parent() {
            return this.productBlock;
        },
        title: 'Supplier',
        placeholder: 'Scan or select...',
        isTransient: true,
        isFullWidth: true,
        validation: /^$|^[^|]+$/,
        isReadOnly: true,
    })
    supplier: ui.fields.Text;

    @ui.decorators.textField<MobileIntersiteReceipt>({
        parent() {
            return this.productBlock;
        },
        title: 'Supplier packing slip',
        placeholder: 'Scan or enter...',
        isTransient: true,
        isMandatory: false,
        isFullWidth: true,
        validation: /^$|^[^|]+$/,
        maxLength: 20,
        isReadOnly: true,
    })
    supplierPackingSlip: ui.fields.Text;

    @ui.decorators.tableField<MobileIntersiteReceipt>({
        parent() {
            return this.receiptBlock;
        },
        isTitleHidden: true,
        isTransient: true,
        canSelect: false,
        canFilter: false,
        columns: [
            ui.nestedFields.text({
                bind: 'product',
                title: 'Product',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'quantityAndStockUnit',
                title: 'Quantity',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'salesDeliveryLineNumber',
                title: 'Line No.',
                isReadOnly: true,
            }),
        ],
        dropdownActions: [
            {
                icon: 'delete',
                title: 'Delete',
                onClick(rowId: any, _data: any) {
                    if (this._intersiteSession) {
                        this._intersiteSession?.purchaseReceipt?.lines?.splice(rowId - 1, 1);
                        if (this._intersiteSession?.purchaseReceipt?.lines?.length === 0) {
                            this._intersiteReceiptRecordManager.clearSession();
                            this.salesDelivery.value = null;
                            this.supplier.value = null;
                            this.supplierPackingSlip.value = null;
                            this.salesDelivery.isDirty = false;
                            this.supplier.isDirty = false;
                            this.supplierPackingSlip.isDirty = false;
                            this.createButton.isDisabled = true;
                            this.$.router.goTo(`@sage/x3-purchasing-sales/${this.$.page.id}`);
                        } else {
                            this.receiptToCreate.value = this._mapPurchaseReceipt(
                                this._intersiteSession?.purchaseReceipt?.lines ?? [],
                            );
                            this.supplierPackingSlip.value = this._getSupplierPackingSlip();
                            this._intersiteReceiptRecordManager.intersiteSession = {
                                purchaseEntryTransaction: this._intersiteSession.purchaseEntryTransaction,
                                purchaseReceipt: {
                                    ...this._intersiteSession?.purchaseReceipt,
                                    supplierPackingSlip: this.supplierPackingSlip.value ?? undefined,
                                },
                                salesUnitToStockUnitConversionFactor: 1,
                            };
                        }
                    }
                },
            },
        ],
    })
    receiptToCreate: ui.fields.Table<any>;

    private async _init(): Promise<boolean> {
        this._numberOfDecimalList = await getUnitNumberOfDecimalList(this);
        this._intersiteReceiptRecordManager = new IntersiteReceiptDetailsRecordManager(this);
        if (!this.$.queryParameters._id) {
            this._intersiteReceiptRecordManager.clearSession();
        }
        this._intersiteSession = this._intersiteReceiptRecordManager.intersiteSession;
        const salesDeliveryId = this.$.queryParameters.salesDeliveryId;

        if (!this._intersiteSession) {
            // Initialize Receipt Date to the current date
            this.receiptDate.value = DateValue.today().toString();

            try {
                // Populate Default Site & Legal company & Entry Transaction fields
                this.stockSite.value = await getSelectedStockSite(
                    this,
                    ui.localize('@sage/x3-purchasing-sales/dialog-error-title', 'Error'),
                    ui.localize(
                        '@sage/x3-purchasing-sales/dialog-error-location-inquiry-set-site',
                        'Define a default stock site on the user function profile.',
                    ),
                );
                if (!this.stockSite.value) {
                    return false;
                }
                await this._initTransaction();
            } catch (error) {
                this.$.showToast(error.message, { type: 'error' });
                return false;
            }
            this._getIsLocationPreloaded();

            if (salesDeliveryId) {
                this.salesDelivery.value = {
                    id: salesDeliveryId,
                };
                this.supplier.value = (await this._getSupplier(salesDeliveryId as string)) ?? null;
                this.supplierPackingSlip.value = this.salesDelivery.value.id;
                this.nextButton.isHidden = false;
                this.nextButton.isDisabled = false;
            } else {
                this.supplier.value = null;
                this.supplierPackingSlip.value = null;
                this.supplier.isDirty = false;
                this.supplierPackingSlip.isDirty = false;
                this.nextButton.isHidden = false;
                this.nextButton.isDisabled = !this.salesDelivery.value;
            }
            this.createButton.isHidden = true;
        } else {
            const purchaseReceipt = this._intersiteSession.purchaseReceipt;

            this.stockSite.value = purchaseReceipt.receiptSite ?? null;

            try {
                await this._initTransaction();
            } catch (e) {
                ui.console.error(e);
            }

            this.transaction.options = [this._intersiteSession?.purchaseEntryTransaction?.code ?? ''];
            this.transaction.value = this._intersiteSession.purchaseEntryTransaction.code ?? null;

            this.receiptDate.value = purchaseReceipt.receiptDate ?? null;
            this.supplierPackingSlip.value = this._getSupplierPackingSlip();

            this.receiptToCreate.value = this._mapPurchaseReceipt(purchaseReceipt.lines ?? []);
            this.receiptBlock.isHidden = this.receiptToCreate.value.length === 0;

            if (purchaseReceipt?.lines?.length === 0) {
                this.salesDelivery.value = {
                    id: this.$.queryParameters._id,
                };
                this.salesDelivery.isReadOnly = false;
                this.nextButton.isHidden = false;
                this.nextButton.isDisabled = !this.salesDelivery.value;
                this.createButton.isHidden = true;
            } else {
                this.salesDelivery.value = {
                    id: purchaseReceipt.lines[0].salesDelivery,
                };
                this.salesDelivery.value = null;
                this.salesDelivery.isReadOnly = false;
                this.createButton.isHidden = false;
                this.createButton.isDisabled = false;
                this.nextButton.isHidden = true;
            }

            this.supplier.value = purchaseReceipt.supplier ?? null;
            this.supplier.isReadOnly = true;
        }
        this.legalCompany.value = await this._getLegalCompany(this.stockSite.value ?? '');
        return true;
    }

    private async _getLegalCompany(defaultSite: string): Promise<string> {
        const response = await this.$.graph
            .node('@sage/x3-system/Site')
            .read(
                {
                    _id: true,
                    legalCompany: { code: true },
                },
                defaultSite,
            )
            .execute();

        if (!response) {
            // This should never occur as long as the user function profile has a valid site set up
            throw new Error(
                ui.localize(
                    '@sage/x3-purchasing-sales/error-intersite-receipt-no-legal-company',
                    'The invalid site {{site}} has no legal company.',
                    {
                        site: defaultSite,
                    },
                ),
            );
        }

        return response.legalCompany.code;
    }

    private _getSupplierPackingSlip(): string | null {
        const salesDeliverys: string[] = [];
        if (this._intersiteSession?.purchaseReceipt?.lines) {
            this._intersiteSession.purchaseReceipt.lines.forEach(line => {
                if (!salesDeliverys.find(id => id === line.salesDelivery)) {
                    salesDeliverys.push(line.salesDelivery);
                }
            });
        }
        if (this.salesDelivery.value && !salesDeliverys.find(id => id === this.salesDelivery?.value?.id)) {
            salesDeliverys.push(this.salesDelivery?.value?.id);
        }
        if (salesDeliverys.length === 1) {
            return salesDeliverys[0];
        } else if (this._intersiteSession?.purchaseReceipt?.supplierPackingSlip) {
            return this._intersiteSession?.purchaseReceipt?.supplierPackingSlip;
        } else {
            return null;
        }
    }

    private async _getEntryTransactionOptions(): Promise<string[]> {
        const response = extractEdges(
            await this.$.graph
                .node('@sage/x3-purchasing/PurchaseEntryTransaction')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            _id: true,
                            code: true,
                            stockMovementCode: {
                                code: true,
                            },
                            defaultStockMovementGroup: {
                                code: true,
                            },
                            stockAutomaticJournal: {
                                code: true,
                            },
                            isLotCustomField1Allowed: true,
                            isLotCustomField2Allowed: true,
                            isLotCustomField3Allowed: true,
                            isLotCustomField4Allowed: true,
                            identifier1Entry: true,
                            identifier1Detail: true,
                            identifier2Entry: true,
                            identifier2Detail: true,
                            isLotPotencyAllowed: true,
                            isLotExpirationDateAllowed: true,
                        },
                        {
                            filter: {
                                entryTransactionType: 'receipt',
                                isActive: true,
                            },
                        },
                    ),
                )
                .execute(),
        ) as PurchaseEntryTransaction[];

        this._entryTransactionMap = new Map<string, PurchaseEntryTransactionInput>();

        // transform Entry Transaction response into a string array
        return response.map((entryTransaction: PurchaseEntryTransaction) => {
            this._entryTransactionMap.set(entryTransaction.code, {
                code: entryTransaction.code,
                _id: entryTransaction._id,
                stockAutomaticJournal: entryTransaction.stockAutomaticJournal?.code,
                stockMovementCode: entryTransaction.stockMovementCode?.code,
                defaultStockMovementGroup: entryTransaction.defaultStockMovementGroup?.code,
                isLotCustomField1Allowed: entryTransaction.isLotCustomField1Allowed,
                isLotCustomField2Allowed: entryTransaction.isLotCustomField2Allowed,
                isLotCustomField3Allowed: entryTransaction.isLotCustomField3Allowed,
                isLotCustomField4Allowed: entryTransaction.isLotCustomField4Allowed,
                identifier1Entry: entryTransaction.identifier1Entry,
                identifier1Detail: entryTransaction.identifier1Detail,
                identifier2Entry: entryTransaction.identifier2Entry,
                identifier2Detail: entryTransaction.identifier2Detail,
                isLotPotencyAllowed: entryTransaction.isLotPotencyAllowed,
                isLotExpirationDateAllowed: entryTransaction.isLotExpirationDateAllowed,
            });
            return entryTransaction.code;
        });
    }

    private async _getIsLocationPreloaded() {
        const responseSite = await this.$.graph
            .node('@sage/x3-master-data/MobileAutomationSetup')
            .query(
                ui.queryUtils.edgesSelector(
                    {
                        isLocationPreloaded: true,
                    },
                    {
                        filter: {
                            site: { code: this.stockSite.value },
                        },
                    },
                ),
            )
            .execute();
        if (responseSite.edges.length !== 0) {
            responseSite.edges.some((edge: { node: { isLocationPreloaded: boolean } }) => {
                this._isLocationPreloaded = edge.node.isLocationPreloaded;
            });
        } else {
            const response = await this.$.graph
                .node('@sage/x3-master-data/MobileAutomationSetup')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            isLocationPreloaded: true,
                        },
                        {
                            filter: {
                                site: null,
                            },
                        },
                    ),
                )
                .execute();
            if (response.edges.length !== 0) {
                response.edges.some((edge: { node: { isLocationPreloaded: boolean } }) => {
                    this._isLocationPreloaded = edge.node.isLocationPreloaded;
                });
            } else {
                this._isLocationPreloaded = true;
            }
        }
    }

    private async _callCreationAPI(): Promise<any | Error> {
        let result: any;
        const payload = `
        mutation {
            x3Purchasing {
                purchaseReceipt {
                    create(data: ${ui.queryUtils.serializeToGraphQL(this._intersiteSession?.purchaseReceipt)}){
                        id
                    }
                }
            }
        }
    `;

        try {
            result = await this.$.graph.raw(payload, true);
            if (!result) {
                throw Error(
                    ui.localize(
                        '@sage/x3-purchasing-sales/dialog-error-intersite-receipt-no-create-results',
                        'No results for the created intersite receipt',
                    ),
                );
            }
        } catch (error) {
            return error;
        }

        return result;
    }

    private _mapPurchaseReceipt(_receipt: Partial<PurchaseReceiptLineInput>[]) {
        let rowCount = 0;
        return _receipt.map((line: PurchaseReceiptLineInput) => {
            rowCount++;
            return {
                _id: String(rowCount), // this defines the rowId parameter in dropdownActions onClick() event
                product: line.product,
                quantityAndStockUnit: `${Number(line.quantityInReceiptUnitReceived).toFixed(
                    getNumberOfDecimal(this._numberOfDecimalList, line.receiptUnit),
                )} ${line.receiptUnit}`,
                purchaseShipment:
                    line.purchaseShipment && line.purchaseShipmentLine
                        ? `${line.purchaseShipment} ${line.purchaseShipmentLine}`
                        : '',
            };
        });
    }

    private _disablePage(): void {
        this.salesDelivery.isDisabled = true;
        this.supplier.isDisabled = true;
        this.receiptDate.isDisabled = true;
        this.transaction.isDisabled = true;
        this.supplierPackingSlip.isDisabled = true;
    }

    private async _initTransaction() {
        this.transaction.options = await this._getEntryTransactionOptions();
        switch (this.transaction.options.length) {
            case 0:
                this._disablePage();
                throw new Error('No transaction, cannot continue');
            case 1:
                this.transaction.value = this.transaction.options[0];
                this.transaction.isHidden = true;
                break;
            default:
                this.transaction.value = this.transaction.options[0];
                if (this._intersiteSession) {
                    this.transaction.isDisabled = true;
                }
                break;
        }
    }

    private async _getSupplier(salesDeliveryId: string): Promise<string | null> {
        const response = extractEdges(
            await this.$.graph
                .node('@sage/x3-sales/SalesDeliveryLine')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            salesDelivery: {
                                id: true,
                            },
                            salesOrderLine: {
                                purchaseOrder: {
                                    orderFromSupplier: {
                                        code: {
                                            code: true,
                                        },
                                    },
                                },
                            },
                        },
                        {
                            filter: {
                                salesDelivery: {
                                    id: salesDeliveryId,
                                },
                            },
                        },
                    ),
                )
                .execute(),
        ) as any[];
        let supplier: string | null;
        supplier = null;
        response.forEach(salesDeliveryLine => {
            if (salesDeliveryLine.salesOrderLine?.purchaseOrder) {
                supplier = salesDeliveryLine.salesOrderLine.purchaseOrder.orderFromSupplier?.code?.code;
            }
        });
        return supplier;
    }
}
