import { ProductSite, SupplierProduct } from '@sage/x3-master-data-api';
import { dialogMessage } from '@sage/x3-master-data/lib/client-functions/dialogs';
import { GraphApi, PurchaseEntryTransactionInput } from '@sage/x3-purchasing-api';
import { getRegExp } from '@sage/x3-system/lib/shared-functions/pat-converter';
import { extractEdges } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import { IntersiteReceiptSession } from '../client-functions/intersite-receipt-details-control';
import { findDefaultLocation, findStockManagementRules } from '../client-functions/stock-management-rules';
/*
 * Page Interfaces
 * These interfaces allow the inputs to the helper functions to be faked for testing.
 */
interface PageWithStorageAndSite {
    site: { value: string };
    intersiteSession: IntersiteReceiptSession;
}

interface PageWithProductSite {
    productSite: ProductSite;
}
interface PageWithHeaderFields {
    product: { value: string };
    description: { value: string };
    salesDeliveryLineNumber: string | number | boolean;
    salesDeliveryId: string;
    supplier: { value: string };
    headerTitleLeft: { value: string };
    headerTitleRight: { value: string; prefix?: string };
    headerLine2Left: { value: string };
    headerLine2Right: { value: string; prefix?: string };
    headerLine3Left: { value: string };
    headerLine3Right: { value: string; prefix?: string };
}
interface PageWithVersionFields {
    majorVersion: {
        isHidden: boolean;
        value: string;
    };
    minorVersion: {
        isHidden: boolean;
        value: string;
    };
}
interface PageWithLotFields {
    lot: {
        isHidden: boolean;
        value: string;
    };
    supplierLot: {
        isHidden: boolean;
        value: string;
    };
    sublot: {
        isHidden: boolean;
        value: string;
    };
    lotCustomField1: {
        isHidden: boolean;
        value: string;
    };
    lotCustomField2: {
        isHidden: boolean;
        value: string;
    };
    lotCustomField3: {
        isHidden: boolean;
        value: string;
    };
    lotCustomField4: {
        isHidden: boolean;
        value: string;
    };
    lotSequenceNumber: {
        value: string;
    };
}
interface PageWithPackUnitFields {
    receiptUnit: {
        options?: string[];
        isReadOnly: boolean;
        isDisabled: boolean;
        value: string;
    };
    receiptUnitCode: { value: string };
    previousReceiptUnit: {
        value: string;
    };
    receiptUnits: Array<{
        node: {
            packingUnit: {
                code: string;
                numberOfDecimals: number;
            };
            packingUnitToStockUnitConversionFactor: string;
            isPackingFactorEntryAllowed: boolean;
        };
    }>;
    packingUnitToStockUnitConversionFactor: {
        value: number;
        isDisabled: boolean;
    };
    salesUnitToStockUnitConversionFactor: string;
    previousSalesUnitToStockUnitConversionFactor: string;
}
interface PageWithPurchaseReceiptMiscProperties {
    status: {
        value: string;
        options?: string[];
    };
    serialNumber: {
        isHidden: boolean;
        value: string;
    };
    licensePlateNumber: {
        isHidden: boolean;
    };
    location: {
        isHidden: boolean;
        isMandatory: boolean;
        value: {
            code: string;
            type: string;
            stockSite: { code: string };
            warehouse: { code: string };
        };
    };
    suggestedLocation: {
        isHidden: boolean;
        value: string;
    };
    expirationDate: {
        isHidden: boolean;
        value: string;
    };
    useByDate: {
        isHidden: boolean;
        value: string;
    };
    remainingQuantity: {
        value: number;
        isHidden: boolean;
    };
    quantityToReceive: {
        value: number;
        scale?: number;
    };
    lot: {
        isHidden: boolean;
        value: string;
    };
    isExpirationManaged: boolean;
    identifier1: {
        isHidden: boolean;
        isDisabled: boolean;
    };
    identifier2: {
        isHidden: boolean;
        isDisabled: boolean;
    };
    container: {
        value: { code?: string };
        isHidden: boolean;
    };
    quantityToReceiveNoRounded: number;
    quantityToReceiveRounded: number;
    remainingQuantityNoRounded: number;
    remainingQuantityRounded: number;
    remainingQuantityInitial: number;
    isLocationPreloaded: boolean;
}

interface PageWithStockJournals {
    stockJournalNodeResponses: StockJournalNodeResponse[];
    currStockJournal: number;
}

export type IntersiteReceiptsDetailPage = ui.Page<GraphApi> &
    PageWithStorageAndSite &
    PageWithHeaderFields &
    PageWithVersionFields &
    PageWithLotFields &
    PageWithPackUnitFields &
    PageWithPurchaseReceiptMiscProperties &
    PageWithProductSite &
    PageWithStockJournals;

type SalesDeliveryNodeResponse = {
    salesDelivery: {
        id: string;
        salesSite: {
            code: string;
        };
        receivingSite: {
            code: string;
        };
    };
    stockUnit: {
        code: string;
        numberOfDecimals: number;
    };
    salesUnit: {
        code: string;
        numberOfDecimals: number;
    };
    product: {
        code: string;
    };
    salesOrderLine: {
        purchaseOrder: {
            orderFromSupplier: {
                code: {
                    code: string;
                };
            };
        };
    };
    quantityInStockUnitReceived: string;
    quantityInSalesUnitDelivered: string;
    salesUnitToStockUnitConversionFactor: string;
    majorProductVersion: {
        code: string;
    };
    minorProductVersion: string;
};

export type StockJournalNodeResponse = {
    stockSite: {
        code: string;
    };
    documentId: string;
    documentLine: number;
    sequence: number;
    product: {
        code: string;
    };
    quantityInStockUnit: number;
    isUpdated: boolean;
    lot: string;
    sublot: string;
    supplierLot: string;
    lotCustomField1: string;
    lotCustomField2: string;
    lotCustomField3: string;
    lotCustomField4: string;
    majorVersion: string;
    minorVersion: string;
    identifier1: string;
    identifier2: string;
    expirationDate: string;
    useByDate: string;
    status: string;
    serialNumber: string;
    startingSerialNumber: string;
    endingSerialNumber: string;
};

/*
 * Initializer helper class for the Purchase Receipts Detail page.
 */
export class IntersiteReceiptDetailsInitializer {
    private static readonly QTY_DECIMAL_PLACES: number = 8; // this precision is based on how classic Sage X3 4GL calculates remaining quantity in ENVADC.src
    private static readonly CONVERSION_RATE_DECIMAL_PLACES: number = 6; //Display and precision length is defined by the X3 datatype which we do not have access to currently.  Setting to 6 decimal places for now as this is the precisions of the conversion rate data type.

    static async initializePage(page: IntersiteReceiptsDetailPage): Promise<boolean> {
        if (!page.intersiteSession) {
            return false;
        }

        const salesDeliveryLine = await this._getSalesDeveryLine(page);
        if (!salesDeliveryLine) {
            return false;
        }

        const productSite = await this._getProductSite(
            page,
            salesDeliveryLine.product.code,
            salesDeliveryLine.salesDelivery.receivingSite.code,
        );
        if (!productSite) {
            return false;
        }

        const productReceivingSite = await this._getProductSite(
            page,
            salesDeliveryLine.product.code,
            salesDeliveryLine.salesDelivery.receivingSite.code,
        );
        if (!productReceivingSite) {
            return false;
        }

        await this._getStockJournals(
            page,
            salesDeliveryLine.product.code,
            salesDeliveryLine.salesDelivery.salesSite.code,
        );

        if (page.stockJournalNodeResponses.length === 0) {
            return false;
        }

        const transactionEntry: PurchaseEntryTransactionInput = page.intersiteSession.purchaseEntryTransaction;

        page.site.value = salesDeliveryLine.salesDelivery.receivingSite.code;
        page.supplier.value =
            salesDeliveryLine.salesOrderLine?.purchaseOrder?.orderFromSupplier?.code?.code ??
            salesDeliveryLine.salesDelivery.salesSite.code;

        let supplierProductUnits: SupplierProduct;
        try {
            supplierProductUnits = (await page.$.graph
                .node('@sage/x3-master-data/SupplierProduct')
                .read(
                    {
                        purchaseUnit: {
                            code: true,
                            numberOfDecimals: true,
                        },
                        purchaseUnitToStockUnitConversionFactor: true,
                        packingUnit: {
                            code: true,
                            numberOfDecimals: true,
                        },
                        packingUnitToPurchaseUnitConversionFactor: true,
                    },
                    `${salesDeliveryLine.product.code}|${salesDeliveryLine.salesOrderLine?.purchaseOrder?.orderFromSupplier?.code?.code}`,
                )
                .execute()) as SupplierProduct;
            if (supplierProductUnits) {
                const receiptUnitOptions = page.receiptUnit.options;
                if (
                    supplierProductUnits.purchaseUnit &&
                    !page.receiptUnits.find(
                        item => item.node.packingUnit.code === supplierProductUnits.purchaseUnit.code,
                    )
                ) {
                    page.receiptUnits.push({
                        node: {
                            packingUnit: {
                                code: supplierProductUnits.purchaseUnit.code,
                                numberOfDecimals: supplierProductUnits.purchaseUnit.numberOfDecimals,
                            },
                            packingUnitToStockUnitConversionFactor: '1',
                            isPackingFactorEntryAllowed: false,
                        },
                    });
                    receiptUnitOptions?.push(supplierProductUnits.purchaseUnit.code);
                }
                if (
                    supplierProductUnits.packingUnit &&
                    !page.receiptUnits.find(
                        item => item.node.packingUnit.code === supplierProductUnits.packingUnit.code,
                    )
                ) {
                    page.receiptUnits.push({
                        node: {
                            packingUnit: {
                                code: supplierProductUnits.packingUnit.code,
                                numberOfDecimals: supplierProductUnits.packingUnit.numberOfDecimals,
                            },
                            packingUnitToStockUnitConversionFactor: '1',
                            isPackingFactorEntryAllowed: false,
                        },
                    });
                    receiptUnitOptions?.push(supplierProductUnits.packingUnit.code);
                }
                page.receiptUnit.options = receiptUnitOptions;
            }
        } catch (e) {
            await dialogMessage(
                page,
                'error',
                ui.localize(
                    '@sage/x3-purchasing-sales/error-loading-supplier-product',
                    'Error loading supplier product',
                ),
                String(e),
            );
        }

        page.receiptUnit.value = salesDeliveryLine.salesUnit.code;
        page.previousReceiptUnit.value = page.receiptUnit.value;

        page.product.value = salesDeliveryLine.product.code;
        page.intersiteSession.purchaseStockManagementRules = await findStockManagementRules(
            page.site.value,
            productSite.product.productCategory.code,
            '3',
            transactionEntry?.stockMovementCode,
            page,
        );

        page.status.options = await this._getStatuses(page);
        page.status.value = page.intersiteSession?.purchaseStockManagementRules?.defaultStatus ?? '';

        const lpnManagementRule = page.intersiteSession.purchaseStockManagementRules?.licensePlateNumberEntry;
        const lotManagement = productSite.product.lotManagementMode;
        const lotNotManaged = lotManagement === 'notManaged';
        page.lot.isHidden = lotNotManaged;
        page.supplierLot.isHidden = lotNotManaged;
        page.sublot.isHidden = lotManagement !== 'lotAndSublot';
        page.lotCustomField1.isHidden = !transactionEntry.isLotCustomField1Allowed || lotNotManaged;
        page.lotCustomField2.isHidden = !transactionEntry.isLotCustomField2Allowed || lotNotManaged;
        page.lotCustomField3.isHidden = !transactionEntry.isLotCustomField3Allowed || lotNotManaged;
        page.lotCustomField4.isHidden = !transactionEntry.isLotCustomField4Allowed || lotNotManaged;

        const serialNumberManagement = productSite.product.serialNumberManagementMode;
        page.serialNumber.isHidden = serialNumberManagement === 'notManaged' || serialNumberManagement === 'issued';

        page.licensePlateNumber.isHidden = !productSite.isLicensePlateNumberManaged;
        page.location.isHidden = !productSite.isLocationManaged;
        page.location.isMandatory = !page.location.isHidden;

        const expirationManagement = productSite.product.expirationManagementMode;
        page.isExpirationManaged = expirationManagement !== 'notManaged';
        page.expirationDate.isHidden = page.useByDate.isHidden = !page.isExpirationManaged;

        switch (transactionEntry.identifier1Detail) {
            case 'entered':
            case 'displayed': {
                page.identifier1.isHidden = false;
                page.identifier1.isDisabled = true;
                break;
            }
            default: {
                page.identifier1.isHidden = true;
            }
        }

        switch (transactionEntry.identifier2Detail) {
            case 'entered':
            case 'displayed': {
                page.identifier2.isHidden = false;
                page.identifier2.isDisabled = true;
                break;
            }
            default: {
                page.identifier2.isHidden = true;
            }
        }

        page.container.isHidden = lpnManagementRule !== 'mandatory' || !productSite.isLicensePlateNumberManaged;
        if (!page.container.isHidden) page.container.value = productSite.defaultInternalContainer;

        const versionManagementMode = productSite.product.stockVersionMode;

        page.majorVersion.isHidden = !['major', 'majorAndMinor'].includes(versionManagementMode);
        page.minorVersion.isHidden = versionManagementMode !== 'majorAndMinor';

        page.quantityToReceive.value = page.remainingQuantity.value; // should auto-round value appropriately based on scale property that was set for this field
        page.quantityToReceiveNoRounded = page.quantityToReceive.value;
        page.quantityToReceiveRounded = page.quantityToReceive.value;
        page.remainingQuantityNoRounded = page.remainingQuantity.value;
        page.remainingQuantityRounded = page.remainingQuantity.value;
        page.remainingQuantityInitial = Number(salesDeliveryLine.quantityInStockUnitReceived);

        page.salesUnitToStockUnitConversionFactor = salesDeliveryLine.salesUnitToStockUnitConversionFactor;
        page.previousSalesUnitToStockUnitConversionFactor = salesDeliveryLine.salesUnitToStockUnitConversionFactor;
        const selectedUnit = page.receiptUnits.find(
            item => item.node.packingUnit.code === salesDeliveryLine.salesUnit.code,
        );
        page.packingUnitToStockUnitConversionFactor.isDisabled = selectedUnit
            ? !selectedUnit.node.isPackingFactorEntryAllowed
            : true;
        page.productSite = productSite;

        if (!page.suggestedLocation.isHidden && page.intersiteSession.purchaseStockManagementRules) {
            const suggestedLocation = await findDefaultLocation(
                productReceivingSite,
                page.intersiteSession.purchaseStockManagementRules,
                page,
            );
            if (!suggestedLocation) page.suggestedLocation.isHidden = true;
            else page.suggestedLocation.value = suggestedLocation;
            if (page.isLocationPreloaded === true) {
                try {
                    page.location.value = await page.$.graph
                        .node('@sage/x3-stock-data/Location')
                        .read(
                            {
                                code: true,
                                type: true,
                                stockSite: {
                                    code: true,
                                },
                                warehouse: {
                                    code: true,
                                },
                            },
                            `${page.site.value}|${suggestedLocation}`,
                        )
                        .execute();
                } catch (e) {
                    await dialogMessage(
                        page,
                        'error',
                        ui.localize('@sage/x3-purchasing-sales/error-loading-location', 'Error loading location'),
                        String(e),
                    );
                    return false;
                }
            }
        }

        IntersiteReceiptDetailsInitializer.initHeader(page, productSite, salesDeliveryLine);

        return true;
    }

    static async _getSalesDeveryLine(page: ui.Page): Promise<SalesDeliveryNodeResponse | null> {
        try {
            return await page.$.graph
                .node('@sage/x3-sales/SalesDeliveryLine')
                .read(
                    {
                        salesDelivery: {
                            id: true,
                            salesSite: {
                                code: true,
                            },
                            receivingSite: {
                                code: true,
                            },
                        },
                        product: {
                            code: true,
                        },
                        stockUnit: {
                            code: true,
                            numberOfDecimals: true,
                        },
                        salesUnit: {
                            code: true,
                            numberOfDecimals: true,
                        },
                        salesOrderLine: {
                            purchaseOrder: {
                                orderFromSupplier: {
                                    code: {
                                        code: true,
                                    },
                                },
                            },
                        },
                        quantityInStockUnitReceived: true,
                        quantityInSalesUnitDelivered: true,
                        salesUnitToStockUnitConversionFactor: true,
                        majorProductVersion: { code: true },
                        minorProductVersion: true,
                    },
                    `${(page as IntersiteReceiptsDetailPage).salesDeliveryId}|${
                        (page as IntersiteReceiptsDetailPage).salesDeliveryLineNumber
                    }`,
                )
                .execute();
        } catch (e) {
            await dialogMessage(
                page,
                'error',
                ui.localize('@sage/x3-purchasing-sales/error-loading-sales-delivery', 'Error loading sales delivery'),
                String(e),
            );
            return null;
        }
    }

    static async _getProductSite(
        page: IntersiteReceiptsDetailPage,
        productCode: string,
        salesSite: string,
    ): Promise<ProductSite | undefined> {
        const receiptUnitOptions: string[] = [];
        try {
            const _productSites = await page.$.graph
                .node('@sage/x3-master-data/ProductSite')
                .query(
                    ui.queryUtils.edgesSelector<ProductSite>(
                        {
                            isLicensePlateNumberManaged: true,
                            isLocationManaged: true,
                            stockSite: {
                                code: true,
                            },
                            defaultLocations: {
                                query: {
                                    edges: {
                                        node: {
                                            defaultLocationType: true,
                                            locationNumber: true,
                                            defaultLocation: true,
                                        },
                                    },
                                },
                            },
                            product: {
                                code: true,
                                productCategory: { code: true },
                                serialNumberManagementMode: true,
                                serialSequenceNumber: true,
                                localizedDescription1: true,
                                lotManagementMode: true,
                                lotSequenceNumber: true,
                                stockVersionMode: true,
                                expirationManagementMode: true,
                                stockManagementMode: true,
                                purchaseUnitToStockUnitConversionFactor: true,
                                useByDateCoefficient: true,
                                expirationLeadTime: true,
                                expirationTimeUnit: true,
                                stockUnit: {
                                    code: true,
                                    numberOfDecimals: true,
                                },
                                purchaseUnit: {
                                    code: true,
                                    numberOfDecimals: true,
                                },
                                packingUnits: {
                                    query: {
                                        edges: {
                                            node: {
                                                packingUnit: {
                                                    code: true,
                                                    numberOfDecimals: true,
                                                },
                                                packingUnitToStockUnitConversionFactor: true,
                                                isPackingFactorEntryAllowed: true,
                                            },
                                        },
                                    },
                                },
                            },
                            internalContainers: {
                                query: {
                                    edges: {
                                        node: {
                                            internalContainer: {
                                                code: true,
                                            },
                                        },
                                    },
                                },
                            },
                            defaultInternalContainer: {
                                code: true,
                            },
                        },
                        {
                            filter: {
                                product: {
                                    code: productCode,
                                },
                                stockSite: {
                                    code: salesSite,
                                },
                            },
                        },
                    ),
                )
                .execute();

            const _productSite =
                _productSites.edges.length > 0 ? (_productSites.edges[0].node as ProductSite) : undefined;
            if (_productSite) {
                page.receiptUnits = [];
                page.receiptUnits.push({
                    node: {
                        packingUnit: {
                            code: _productSite.product.stockUnit.code,
                            numberOfDecimals: _productSite.product.stockUnit.numberOfDecimals,
                        },
                        packingUnitToStockUnitConversionFactor: '1',
                        isPackingFactorEntryAllowed: false,
                    },
                });
                receiptUnitOptions.push(_productSite.product.stockUnit.code);
                if (
                    !page.receiptUnits.find(
                        item => item.node.packingUnit.code === _productSite.product.purchaseUnit.code,
                    )
                ) {
                    page.receiptUnits.push({
                        node: {
                            packingUnit: {
                                code: _productSite.product.purchaseUnit.code,
                                numberOfDecimals: _productSite.product.purchaseUnit.numberOfDecimals,
                            },
                            packingUnitToStockUnitConversionFactor: '1',
                            isPackingFactorEntryAllowed: false,
                        },
                    });
                    receiptUnitOptions.push(_productSite.product.purchaseUnit.code);
                }
                _productSite.product.packingUnits.query.edges.forEach(line => {
                    if (!page.receiptUnits.find(item => item.node.packingUnit.code === line.node.packingUnit.code)) {
                        page.receiptUnits.push({
                            node: {
                                packingUnit: {
                                    code: line.node.packingUnit.code,
                                    numberOfDecimals: line.node.packingUnit.numberOfDecimals,
                                },
                                packingUnitToStockUnitConversionFactor:
                                    line.node.packingUnitToStockUnitConversionFactor,
                                isPackingFactorEntryAllowed: line.node.isPackingFactorEntryAllowed,
                            },
                        });
                        receiptUnitOptions.push(line.node.packingUnit.code);
                    }
                });
                page.receiptUnit.options = receiptUnitOptions;
            }
            return _productSite;
        } catch (e) {
            await dialogMessage(
                page,
                'error',
                ui.localize('@sage/x3-purchasing-sales/error-loading-product-site', 'Error loading product site'),
                String(e),
            );
            return undefined;
        }
    }

    static async _getStatuses(page: IntersiteReceiptsDetailPage): Promise<string[]> {
        const selectedStatus: { _regex: string }[] = [];
        page.intersiteSession?.purchaseStockManagementRules?.authorizedSubstatus.split(',').forEach(function (status) {
            selectedStatus.push({ _regex: getRegExp(status).source });
        });
        try {
            return extractEdges<{ code: string }>(
                await page.$.graph
                    .node('@sage/x3-stock-data/StockStatus')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                code: true,
                            },
                            {
                                filter: {
                                    code: { _or: selectedStatus },
                                },
                            },
                        ),
                    )
                    .execute(),
            ).map(status => status.code);
        } catch (e) {
            await dialogMessage(
                page,
                'error',
                ui.localize('@sage/x3-purchasing-sales/error-loading-stock-statuses', 'Error loading stock statuses'),
                String(e),
            );
            return [];
        }
    }

    static async _getStockJournals(page: IntersiteReceiptsDetailPage, product: string, salesSite: string) {
        try {
            const stockJournals = await page.$.graph
                .node('@sage/x3-stock-data/StockJournal')
                .query(
                    ui.queryUtils.edgesSelector(
                        {
                            stockSite: {
                                code: true,
                            },
                            documentId: true,
                            documentLine: true,
                            sequence: true,
                            product: {
                                code: true,
                            },
                            quantityInStockUnit: true,
                            isUpdated: true,
                            lot: true,
                            sublot: true,
                            supplierLot: true,
                            lotCustomField1: true,
                            lotCustomField2: true,
                            lotCustomField3: true,
                            lotCustomField4: true,
                            identifier1: true,
                            identifier2: true,
                            majorVersion: {
                                code: true,
                            },
                            minorVersion: true,
                            expirationDate: true,
                            useByDate: true,
                            status: {
                                code: true,
                            },
                            serialNumber: true,
                            startingSerialNumber: true,
                            endingSerialNumber: true,
                        },
                        {
                            filter: {
                                stockSite: {
                                    code: salesSite,
                                },
                                documentType: 'delivery',
                                documentId: page.salesDeliveryId,
                                documentLine: page.salesDeliveryLineNumber,
                                product: { code: product },
                                quantityInStockUnit: { _lt: 0 },
                                isUpdated: true,
                                isAdjustedMovement: false,
                            },
                            orderBy: {
                                documentId: 1,
                                documentLine: 1,
                                sequence: -1,
                            },
                        },
                    ),
                )
                .execute();
            page.stockJournalNodeResponses = [];
            stockJournals.edges.forEach(stockJournal => {
                page.stockJournalNodeResponses.push({
                    stockSite: {
                        code: stockJournal.node.stockSite.code,
                    },
                    documentId: stockJournal.node.documentId,
                    documentLine: stockJournal.node.documentLine,
                    sequence: stockJournal.node.sequence,
                    product: {
                        code: stockJournal.node.product.code,
                    },
                    quantityInStockUnit: Number(stockJournal.node.quantityInStockUnit),
                    isUpdated: stockJournal.node.isUpdated,
                    lot: stockJournal.node.lot,
                    sublot: stockJournal.node.sublot,
                    supplierLot: stockJournal.node.supplierLot,
                    lotCustomField1: stockJournal.node.lotCustomField1,
                    lotCustomField2: stockJournal.node.lotCustomField2,
                    lotCustomField3: stockJournal.node.lotCustomField3,
                    lotCustomField4: stockJournal.node.lotCustomField4,
                    identifier1: stockJournal.node.identifier1,
                    identifier2: stockJournal.node.identifier2,
                    majorVersion: stockJournal.node.majorVersion?.code,
                    minorVersion: stockJournal.node.minorVersion,
                    expirationDate: stockJournal.node.expirationDate,
                    useByDate: stockJournal.node.useByDate,
                    status: stockJournal.node.status?.code,
                    serialNumber: stockJournal.node.serialNumber,
                    startingSerialNumber: stockJournal.node.startingSerialNumber,
                    endingSerialNumber: stockJournal.node.endingSerialNumber,
                });
            });
            page.currStockJournal = 0;
        } catch (e) {
            await dialogMessage(
                page,
                'error',
                ui.localize('@sage/x3-purchasing-sales/error-loading-stock-journals', 'Error loading stock journals'),
                String(e),
            );
        }
    }

    static initHeader(
        page: PageWithHeaderFields,
        productSiteNode: ProductSite,
        salesDeliveryLine: SalesDeliveryNodeResponse,
    ) {
        page.product.value = productSiteNode.product.code;
        page.description.value = productSiteNode.product.localizedDescription1;

        if (page.salesDeliveryId && page.salesDeliveryLineNumber) {
            page.headerTitleLeft.value = page.salesDeliveryId;
            page.headerTitleRight.value = page.salesDeliveryLineNumber.toString();
            page.headerTitleRight.prefix = ui.localize(
                '@sage/x3-purchasing-sales/pages__mobile_intersite_receipt_header_detail__line_no',
                'Line No.',
            );
            page.headerLine2Left.value = page.product.value;
            page.headerLine2Right.value = page.description.value;
            page.headerLine3Left.value = ui.localize(
                '@sage/x3-purchasing-sales/pages__mobile_intersite_receipt_header_detail__expected_quantity',
                'Expected quantity:',
            );
            page.headerLine3Right.value = `${Number(salesDeliveryLine.quantityInSalesUnitDelivered)} ${
                salesDeliveryLine.salesUnit.code
            }`;
        } else {
            page.headerTitleLeft.value = page.product.value;
            page.headerTitleRight.value = '';
            page.headerLine2Left.value = page.description.value;
            page.headerLine2Right.value = '';
            page.headerLine3Left.value = page.supplier.value;
            page.headerLine3Right.value = '';
        }
    }
}
