"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntersiteReceiptDetailsInitializer = void 0;
const xtrem_decimal_1 = require("@sage/xtrem-decimal"), typesLib = xtrem_decimal_1;
const dialogs_1 = require("@sage/x3-master-data/lib/client-functions/dialogs");
const pat_converter_1 = require("@sage/x3-system/lib/shared-functions/pat-converter");
const xtrem_client_1 = require("@sage/xtrem-client");
const ui = require("@sage/xtrem-ui");
const stock_management_rules_1 = require("../client-functions/stock-management-rules");
/*
 * Initializer helper class for the Purchase Receipts Detail page.
 */
class IntersiteReceiptDetailsInitializer {
    static { this.QTY_DECIMAL_PLACES = 8; } // this precision is based on how classic Sage X3 4GL calculates remaining quantity in ENVADC.src
    static { this.CONVERSION_RATE_DECIMAL_PLACES = 6; } //Display and precision length is defined by the X3 datatype which we do not have access to currently.  Setting to 6 decimal places for now as this is the precisions of the conversion rate data type.
    static async initializePage(page) {
        if (!page.intersiteSession) {
            return false;
        }
        const salesDeliveryLine = await this._getSalesDeveryLine(page);
        if (!salesDeliveryLine) {
            return false;
        }
        const productSite = await this._getProductSite(page, salesDeliveryLine.product.code, salesDeliveryLine.salesDelivery.receivingSite.code);
        if (!productSite) {
            return false;
        }
        const productReceivingSite = await this._getProductSite(page, salesDeliveryLine.product.code, salesDeliveryLine.salesDelivery.receivingSite.code);
        if (!productReceivingSite) {
            return false;
        }
        await this._getStockJournals(page, salesDeliveryLine.product.code, salesDeliveryLine.salesDelivery.salesSite.code);
        if (typesLib.strictEq(page.stockJournalNodeResponses.length, 0)) {
            return false;
        }
        const transactionEntry = page.intersiteSession.purchaseEntryTransaction;
        page.site.value = salesDeliveryLine.salesDelivery.receivingSite.code;
        page.supplier.value =
            salesDeliveryLine.salesOrderLine?.purchaseOrder?.orderFromSupplier?.code?.code ??
                salesDeliveryLine.salesDelivery.salesSite.code;
        let supplierProductUnits;
        try {
            supplierProductUnits = (await page.$.graph
                .node('@sage/x3-master-data/SupplierProduct')
                .read({
                purchaseUnit: {
                    code: true,
                    numberOfDecimals: true,
                },
                purchaseUnitToStockUnitConversionFactor: true,
                packingUnit: {
                    code: true,
                    numberOfDecimals: true,
                },
                packingUnitToPurchaseUnitConversionFactor: true,
            }, `${salesDeliveryLine.product.code}|${salesDeliveryLine.salesOrderLine?.purchaseOrder?.orderFromSupplier?.code?.code}`)
                .execute());
            if (supplierProductUnits) {
                const receiptUnitOptions = page.receiptUnit.options;
                if (supplierProductUnits.purchaseUnit &&
                    !page.receiptUnits.find(item => typesLib.strictEq(item.node.packingUnit.code, supplierProductUnits.purchaseUnit.code))) {
                    page.receiptUnits.push({
                        node: {
                            packingUnit: {
                                code: supplierProductUnits.purchaseUnit.code,
                                numberOfDecimals: supplierProductUnits.purchaseUnit.numberOfDecimals,
                            },
                            packingUnitToStockUnitConversionFactor: '1',
                            isPackingFactorEntryAllowed: false,
                        },
                    });
                    receiptUnitOptions?.push(supplierProductUnits.purchaseUnit.code);
                }
                if (supplierProductUnits.packingUnit &&
                    !page.receiptUnits.find(item => typesLib.strictEq(item.node.packingUnit.code, supplierProductUnits.packingUnit.code))) {
                    page.receiptUnits.push({
                        node: {
                            packingUnit: {
                                code: supplierProductUnits.packingUnit.code,
                                numberOfDecimals: supplierProductUnits.packingUnit.numberOfDecimals,
                            },
                            packingUnitToStockUnitConversionFactor: '1',
                            isPackingFactorEntryAllowed: false,
                        },
                    });
                    receiptUnitOptions?.push(supplierProductUnits.packingUnit.code);
                }
                page.receiptUnit.options = receiptUnitOptions;
            }
        }
        catch (e) {
            await (0, dialogs_1.dialogMessage)(page, 'error', ui.localize('@sage/x3-purchasing-sales/error-loading-supplier-product', 'Error loading supplier product'), String(e));
        }
        page.receiptUnit.value = salesDeliveryLine.salesUnit.code;
        page.previousReceiptUnit.value = page.receiptUnit.value;
        page.product.value = salesDeliveryLine.product.code;
        page.intersiteSession.purchaseStockManagementRules = await (0, stock_management_rules_1.findStockManagementRules)(page.site.value, productSite.product.productCategory.code, '3', transactionEntry?.stockMovementCode, page);
        page.status.options = await this._getStatuses(page);
        page.status.value = page.intersiteSession?.purchaseStockManagementRules?.defaultStatus ?? '';
        const lpnManagementRule = page.intersiteSession.purchaseStockManagementRules?.licensePlateNumberEntry;
        const lotManagement = productSite.product.lotManagementMode;
        const lotNotManaged = typesLib.strictEq(lotManagement, 'notManaged');
        page.lot.isHidden = lotNotManaged;
        page.supplierLot.isHidden = lotNotManaged;
        page.sublot.isHidden = typesLib.strictNe(lotManagement, 'lotAndSublot');
        page.lotCustomField1.isHidden = !transactionEntry.isLotCustomField1Allowed || lotNotManaged;
        page.lotCustomField2.isHidden = !transactionEntry.isLotCustomField2Allowed || lotNotManaged;
        page.lotCustomField3.isHidden = !transactionEntry.isLotCustomField3Allowed || lotNotManaged;
        page.lotCustomField4.isHidden = !transactionEntry.isLotCustomField4Allowed || lotNotManaged;
        const serialNumberManagement = productSite.product.serialNumberManagementMode;
        page.serialNumber.isHidden = typesLib.strictEq(serialNumberManagement, 'notManaged') || typesLib.strictEq(serialNumberManagement, 'issued');
        page.licensePlateNumber.isHidden = !productSite.isLicensePlateNumberManaged;
        page.location.isHidden = !productSite.isLocationManaged;
        page.location.isMandatory = !page.location.isHidden;
        const expirationManagement = productSite.product.expirationManagementMode;
        page.isExpirationManaged = typesLib.strictNe(expirationManagement, 'notManaged');
        page.expirationDate.isHidden = page.useByDate.isHidden = !page.isExpirationManaged;
        switch (transactionEntry.identifier1Detail) {
            case 'entered':
            case 'displayed': {
                page.identifier1.isHidden = false;
                page.identifier1.isDisabled = true;
                break;
            }
            default: {
                page.identifier1.isHidden = true;
            }
        }
        switch (transactionEntry.identifier2Detail) {
            case 'entered':
            case 'displayed': {
                page.identifier2.isHidden = false;
                page.identifier2.isDisabled = true;
                break;
            }
            default: {
                page.identifier2.isHidden = true;
            }
        }
        page.container.isHidden = typesLib.strictNe(lpnManagementRule, 'mandatory') || !productSite.isLicensePlateNumberManaged;
        if (!page.container.isHidden)
            page.container.value = productSite.defaultInternalContainer;
        const versionManagementMode = productSite.product.stockVersionMode;
        page.majorVersion.isHidden = !['major', 'majorAndMinor'].includes(versionManagementMode);
        page.minorVersion.isHidden = typesLib.strictNe(versionManagementMode, 'majorAndMinor');
        page.quantityToReceive.value = page.remainingQuantity.value; // should auto-round value appropriately based on scale property that was set for this field
        page.quantityToReceiveNoRounded = page.quantityToReceive.value;
        page.quantityToReceiveRounded = page.quantityToReceive.value;
        page.remainingQuantityNoRounded = page.remainingQuantity.value;
        page.remainingQuantityRounded = page.remainingQuantity.value;
        page.remainingQuantityInitial = Number(salesDeliveryLine.quantityInStockUnitReceived);
        page.salesUnitToStockUnitConversionFactor = salesDeliveryLine.salesUnitToStockUnitConversionFactor;
        page.previousSalesUnitToStockUnitConversionFactor = salesDeliveryLine.salesUnitToStockUnitConversionFactor;
        const selectedUnit = page.receiptUnits.find(item => typesLib.strictEq(item.node.packingUnit.code, salesDeliveryLine.salesUnit.code));
        page.packingUnitToStockUnitConversionFactor.isDisabled = selectedUnit
            ? !selectedUnit.node.isPackingFactorEntryAllowed
            : true;
        page.productSite = productSite;
        if (!page.suggestedLocation.isHidden && page.intersiteSession.purchaseStockManagementRules) {
            const suggestedLocation = await (0, stock_management_rules_1.findDefaultLocation)(productReceivingSite, page.intersiteSession.purchaseStockManagementRules, page);
            if (!suggestedLocation)
                page.suggestedLocation.isHidden = true;
            else
                page.suggestedLocation.value = suggestedLocation;
            if (typesLib.strictEq(page.isLocationPreloaded, true)) {
                try {
                    page.location.value = await page.$.graph
                        .node('@sage/x3-stock-data/Location')
                        .read({
                        code: true,
                        type: true,
                        stockSite: {
                            code: true,
                        },
                        warehouse: {
                            code: true,
                        },
                    }, `${page.site.value}|${suggestedLocation}`)
                        .execute();
                }
                catch (e) {
                    await (0, dialogs_1.dialogMessage)(page, 'error', ui.localize('@sage/x3-purchasing-sales/error-loading-location', 'Error loading location'), String(e));
                    return false;
                }
            }
        }
        IntersiteReceiptDetailsInitializer.initHeader(page, productSite, salesDeliveryLine);
        return true;
    }
    static async _getSalesDeveryLine(page) {
        try {
            return await page.$.graph
                .node('@sage/x3-sales/SalesDeliveryLine')
                .read({
                salesDelivery: {
                    id: true,
                    salesSite: {
                        code: true,
                    },
                    receivingSite: {
                        code: true,
                    },
                },
                product: {
                    code: true,
                },
                stockUnit: {
                    code: true,
                    numberOfDecimals: true,
                },
                salesUnit: {
                    code: true,
                    numberOfDecimals: true,
                },
                salesOrderLine: {
                    purchaseOrder: {
                        orderFromSupplier: {
                            code: {
                                code: true,
                            },
                        },
                    },
                },
                quantityInStockUnitReceived: true,
                quantityInSalesUnitDelivered: true,
                salesUnitToStockUnitConversionFactor: true,
                majorProductVersion: { code: true },
                minorProductVersion: true,
            }, `${page.salesDeliveryId}|${page.salesDeliveryLineNumber}`)
                .execute();
        }
        catch (e) {
            await (0, dialogs_1.dialogMessage)(page, 'error', ui.localize('@sage/x3-purchasing-sales/error-loading-sales-delivery', 'Error loading sales delivery'), String(e));
            return null;
        }
    }
    static async _getProductSite(page, productCode, salesSite) {
        const receiptUnitOptions = [];
        try {
            const _productSites = await page.$.graph
                .node('@sage/x3-master-data/ProductSite')
                .query(ui.queryUtils.edgesSelector({
                isLicensePlateNumberManaged: true,
                isLocationManaged: true,
                stockSite: {
                    code: true,
                },
                defaultLocations: {
                    query: {
                        edges: {
                            node: {
                                defaultLocationType: true,
                                locationNumber: true,
                                defaultLocation: true,
                            },
                        },
                    },
                },
                product: {
                    code: true,
                    productCategory: { code: true },
                    serialNumberManagementMode: true,
                    serialSequenceNumber: true,
                    localizedDescription1: true,
                    lotManagementMode: true,
                    lotSequenceNumber: true,
                    stockVersionMode: true,
                    expirationManagementMode: true,
                    stockManagementMode: true,
                    purchaseUnitToStockUnitConversionFactor: true,
                    useByDateCoefficient: true,
                    expirationLeadTime: true,
                    expirationTimeUnit: true,
                    stockUnit: {
                        code: true,
                        numberOfDecimals: true,
                    },
                    purchaseUnit: {
                        code: true,
                        numberOfDecimals: true,
                    },
                    packingUnits: {
                        query: {
                            edges: {
                                node: {
                                    packingUnit: {
                                        code: true,
                                        numberOfDecimals: true,
                                    },
                                    packingUnitToStockUnitConversionFactor: true,
                                    isPackingFactorEntryAllowed: true,
                                },
                            },
                        },
                    },
                },
                internalContainers: {
                    query: {
                        edges: {
                            node: {
                                internalContainer: {
                                    code: true,
                                },
                            },
                        },
                    },
                },
                defaultInternalContainer: {
                    code: true,
                },
            }, {
                filter: {
                    product: {
                        code: productCode,
                    },
                    stockSite: {
                        code: salesSite,
                    },
                },
            }))
                .execute();
            const _productSite = typesLib.gt(_productSites.edges.length, 0) ? _productSites.edges[0].node : undefined;
            if (_productSite) {
                page.receiptUnits = [];
                page.receiptUnits.push({
                    node: {
                        packingUnit: {
                            code: _productSite.product.stockUnit.code,
                            numberOfDecimals: _productSite.product.stockUnit.numberOfDecimals,
                        },
                        packingUnitToStockUnitConversionFactor: '1',
                        isPackingFactorEntryAllowed: false,
                    },
                });
                receiptUnitOptions.push(_productSite.product.stockUnit.code);
                if (!page.receiptUnits.find(item => typesLib.strictEq(item.node.packingUnit.code, _productSite.product.purchaseUnit.code))) {
                    page.receiptUnits.push({
                        node: {
                            packingUnit: {
                                code: _productSite.product.purchaseUnit.code,
                                numberOfDecimals: _productSite.product.purchaseUnit.numberOfDecimals,
                            },
                            packingUnitToStockUnitConversionFactor: '1',
                            isPackingFactorEntryAllowed: false,
                        },
                    });
                    receiptUnitOptions.push(_productSite.product.purchaseUnit.code);
                }
                _productSite.product.packingUnits.query.edges.forEach(line => {
                    if (!page.receiptUnits.find(item => typesLib.strictEq(item.node.packingUnit.code, line.node.packingUnit.code))) {
                        page.receiptUnits.push({
                            node: {
                                packingUnit: {
                                    code: line.node.packingUnit.code,
                                    numberOfDecimals: line.node.packingUnit.numberOfDecimals,
                                },
                                packingUnitToStockUnitConversionFactor: line.node.packingUnitToStockUnitConversionFactor,
                                isPackingFactorEntryAllowed: line.node.isPackingFactorEntryAllowed,
                            },
                        });
                        receiptUnitOptions.push(line.node.packingUnit.code);
                    }
                });
                page.receiptUnit.options = receiptUnitOptions;
            }
            return _productSite;
        }
        catch (e) {
            await (0, dialogs_1.dialogMessage)(page, 'error', ui.localize('@sage/x3-purchasing-sales/error-loading-product-site', 'Error loading product site'), String(e));
            return undefined;
        }
    }
    static async _getStatuses(page) {
        const selectedStatus = [];
        page.intersiteSession?.purchaseStockManagementRules?.authorizedSubstatus.split(',').forEach(function (status) {
            selectedStatus.push({ _regex: (0, pat_converter_1.getRegExp)(status).source });
        });
        try {
            return (0, xtrem_client_1.extractEdges)(await page.$.graph
                .node('@sage/x3-stock-data/StockStatus')
                .query(ui.queryUtils.edgesSelector({
                code: true,
            }, {
                filter: {
                    code: { _or: selectedStatus },
                },
            }))
                .execute()).map(status => status.code);
        }
        catch (e) {
            await (0, dialogs_1.dialogMessage)(page, 'error', ui.localize('@sage/x3-purchasing-sales/error-loading-stock-statuses', 'Error loading stock statuses'), String(e));
            return [];
        }
    }
    static async _getStockJournals(page, product, salesSite) {
        try {
            const stockJournals = await page.$.graph
                .node('@sage/x3-stock-data/StockJournal')
                .query(ui.queryUtils.edgesSelector({
                stockSite: {
                    code: true,
                },
                documentId: true,
                documentLine: true,
                sequence: true,
                product: {
                    code: true,
                },
                quantityInStockUnit: true,
                isUpdated: true,
                lot: true,
                sublot: true,
                supplierLot: true,
                lotCustomField1: true,
                lotCustomField2: true,
                lotCustomField3: true,
                lotCustomField4: true,
                identifier1: true,
                identifier2: true,
                majorVersion: {
                    code: true,
                },
                minorVersion: true,
                expirationDate: true,
                useByDate: true,
                status: {
                    code: true,
                },
                serialNumber: true,
                startingSerialNumber: true,
                endingSerialNumber: true,
            }, {
                filter: {
                    stockSite: {
                        code: salesSite,
                    },
                    documentType: 'delivery',
                    documentId: page.salesDeliveryId,
                    documentLine: page.salesDeliveryLineNumber,
                    product: { code: product },
                    quantityInStockUnit: { _lt: 0 },
                    isUpdated: true,
                    isAdjustedMovement: false,
                },
                orderBy: {
                    documentId: 1,
                    documentLine: 1,
                    sequence: -1,
                },
            }))
                .execute();
            page.stockJournalNodeResponses = [];
            stockJournals.edges.forEach(stockJournal => {
                page.stockJournalNodeResponses.push({
                    stockSite: {
                        code: stockJournal.node.stockSite.code,
                    },
                    documentId: stockJournal.node.documentId,
                    documentLine: stockJournal.node.documentLine,
                    sequence: stockJournal.node.sequence,
                    product: {
                        code: stockJournal.node.product.code,
                    },
                    quantityInStockUnit: Number(stockJournal.node.quantityInStockUnit),
                    isUpdated: stockJournal.node.isUpdated,
                    lot: stockJournal.node.lot,
                    sublot: stockJournal.node.sublot,
                    supplierLot: stockJournal.node.supplierLot,
                    lotCustomField1: stockJournal.node.lotCustomField1,
                    lotCustomField2: stockJournal.node.lotCustomField2,
                    lotCustomField3: stockJournal.node.lotCustomField3,
                    lotCustomField4: stockJournal.node.lotCustomField4,
                    identifier1: stockJournal.node.identifier1,
                    identifier2: stockJournal.node.identifier2,
                    majorVersion: stockJournal.node.majorVersion?.code,
                    minorVersion: stockJournal.node.minorVersion,
                    expirationDate: stockJournal.node.expirationDate,
                    useByDate: stockJournal.node.useByDate,
                    status: stockJournal.node.status?.code,
                    serialNumber: stockJournal.node.serialNumber,
                    startingSerialNumber: stockJournal.node.startingSerialNumber,
                    endingSerialNumber: stockJournal.node.endingSerialNumber,
                });
            });
            page.currStockJournal = 0;
        }
        catch (e) {
            await (0, dialogs_1.dialogMessage)(page, 'error', ui.localize('@sage/x3-purchasing-sales/error-loading-stock-journals', 'Error loading stock journals'), String(e));
        }
    }
    static initHeader(page, productSiteNode, salesDeliveryLine) {
        page.product.value = productSiteNode.product.code;
        page.description.value = productSiteNode.product.localizedDescription1;
        if (page.salesDeliveryId && page.salesDeliveryLineNumber) {
            page.headerTitleLeft.value = page.salesDeliveryId;
            page.headerTitleRight.value = page.salesDeliveryLineNumber.toString();
            page.headerTitleRight.prefix = ui.localize('@sage/x3-purchasing-sales/pages__mobile_intersite_receipt_header_detail__line_no', 'Line No.');
            page.headerLine2Left.value = page.product.value;
            page.headerLine2Right.value = page.description.value;
            page.headerLine3Left.value = ui.localize('@sage/x3-purchasing-sales/pages__mobile_intersite_receipt_header_detail__expected_quantity', 'Expected quantity:');
            page.headerLine3Right.value = `${Number(salesDeliveryLine.quantityInSalesUnitDelivered)} ${salesDeliveryLine.salesUnit.code}`;
        }
        else {
            page.headerTitleLeft.value = page.product.value;
            page.headerTitleRight.value = '';
            page.headerLine2Left.value = page.description.value;
            page.headerLine2Right.value = '';
            page.headerLine3Left.value = page.supplier.value;
            page.headerLine3Right.value = '';
        }
    }
}
exports.IntersiteReceiptDetailsInitializer = IntersiteReceiptDetailsInitializer;
//# sourceMappingURL=intersite-receipt-details-initializer.js.map