import { Currency, UnitOfMeasure } from '@sage/x3-master-data-api';
import {
    Budget,
    GraphApi,
    Operation,
    OperationAssignment,
    Project,
    ProjectUser,
    Task,
    TimeCategory,
    TimeEntryLine,
} from '@sage/x3-project-management-api';
import { CostType } from '@sage/x3-project-management-data-api';
import { User } from '@sage/x3-system-api';
import { setApplicativePageCrudActions } from '@sage/x3-system/build/lib/client-functions/applicative-crud-actions';
import { Filter } from '@sage/xtrem-client';
import { DateValue } from '@sage/xtrem-date-time';
import * as ui from '@sage/xtrem-ui';
import { project } from '../menu-items/project';

@ui.decorators.page<MobileTimeEntryInquiryByDate, TimeEntryLine>({
    isTitleHidden: true,
    title: 'Time entry by date',
    subtitle: 'Select a line',
    node: '@sage/x3-project-management/TimeEntryLine',
    menuItem: project,
    priority: 100,
    authorizationCode: 'PJMTE',
    objectTypeSingular: 'Time entry',
    objectTypePlural: 'Time entries',
    idField() {
        return this.employee;
    },
    access: { node: '@sage/x3-project-management/TimeEntryLine' },

    createAction() {
        return this.$standardNewAction;
    },

    headerDropDownActions() {
        return [
            this.$standardSaveAction,
            this.$standardCancelAction,
            this.$standardOpenRecordHistoryAction,
            this.$standardOpenAttachmentsDialog,
            this.myAssignmentButton,
        ];
    },

    async onLoad() {
        setApplicativePageCrudActions({
            page: this,
            isDirty: false,
            save: this.$standardSaveAction,
            cancel: this.$standardCancelAction,
            create: this.$standardNewAction,
            remove: this.$standardDeleteAction,
        });
        this._manageDisplayApplicativePageActions(false);

        //------------------------------
        this.timeCategoryIsNonProjectCategory();

        if (this.isTimeEntryAdministrator.value) {
            if (this.assignmentLine.value) {
                this.plannedLoad.value = `${this.assignmentLine.value.plannedLoad} ${this.assignmentLine.value.unit.code}`;
                this.consumedLoad.value = `${this.assignmentLine.value.consumedLoad} ${this.assignmentLine.value.unit.code}`;
                this.remainingLoad.value = `${this.assignmentLine.value.remainingLoad} ${this.assignmentLine.value.unit.code}`;
            } else if (this.operation.value?.unit) {
                this.plannedLoad.value = `${this.operation.value.plannedLoad} ${this.operation.value.unit.code}`;
                this.consumedLoad.value = `${this.operation.value.consumedLoad} ${this.operation.value.unit.code}`;
                this.remainingLoad.value = `${this.operation.value.remainingLoad} ${this.operation.value.unit.code}`;
            } else {
                this.loadBlock.isHidden = true;
            }
        } else {
            this.loadBlock.isHidden = true;
            this.costBlock.isHidden = true;
        }

        //------------------------------
        if (this.$.recordId === undefined) {
            this._CrudAction.value = 'C';
            await this.setCreateMode();
        } else {
            if (this._CrudAction.value === 'C') {
                this._CrudAction.value = '';
            }
            this.setDisplayMode();
        }
    },

    async onDirtyStateUpdated(isDirty: boolean) {
        setApplicativePageCrudActions({
            page: this,
            isDirty,
            save: this.$standardSaveAction,
            cancel: this.$standardCancelAction,
            remove: this.$standardDeleteAction,
        });
    },

    businessActions() {
        return [
            this.$standardSaveAction,
            this.$standardCancelAction,
            this.$standardNewAction,
            this.$standardDeleteAction,
            this.myAssignmentButton,
        ];
    },

    navigationPanel: {
        menuType: 'toggle',
        canFilter: true,
        isFirstLetterSeparatorHidden: true,
        listItem: {
            title: ui.nestedFields.date({
                bind: 'date',
                canFilter: true,
            }),
            titleRight: ui.nestedFields.text({
                bind: 'localizedDescription',
                title: 'Description',
                canFilter: true,
            }),
            line2: ui.nestedFields.reference({
                node: '@sage/x3-project-management-data/ProjectLink',
                bind: 'project',
                valueField: 'id',
                canFilter: true,
            }),
            line2Right: ui.nestedFields.numeric({
                bind: 'timeSpent',
                canFilter: false,
                postfix(value: any, rowData?: any) {
                    return rowData?.unit.code;
                },
            }),
            line3: ui.nestedFields.reference({
                node: '@sage/x3-project-management-data/ProjectLink',
                bind: 'project',
                valueField: { localizedDescription: true },
                canFilter: true,
            }),
            line3Right: ui.nestedFields.reference({
                node: '@sage/x3-project-management/TimeCategory',
                bind: 'timeCategory',
                valueField: 'code',
                canFilter: true,
            }),
            // HIDDEN
            line4: ui.nestedFields.numeric({
                bind: 'lineNumber',
                canFilter: false,
                isHidden: true,
            }),
            line4Right: ui.nestedFields.reference({
                node: '@sage/x3-master-data/UnitOfMeasure',
                bind: 'unit',
                valueField: 'code',
                canFilter: false,
                isHidden: true,
            }),
            title5Right: ui.nestedFields.text({
                bind: '_id',
                canFilter: false,
                isHidden: true,
            }),
        },
        orderBy: {
            date: -1,
            project: 1,
            lineNumber: -1,
        },
        async optionsMenu(_graph, _storage, _queryParam, _username, userCode) {
            return [
                {
                    title: ui.localize(
                        '@sage/x3-project-management/pages__mobile_time_entry_inquiry_by_date__toggle__pending',
                        'Pending',
                    ),
                    graphQLFilter: {
                        employee: {
                            employee: {
                                code: userCode,
                            },
                        },
                        status: { _in: ['pendingValidation', 'rejected'] },
                    },
                },
                {
                    title: ui.localize(
                        '@sage/x3-project-management/pages__mobile_time_entry_inquiry_by_date__toggle__validated',
                        'Validated',
                    ),
                    graphQLFilter: {
                        employee: {
                            employee: {
                                code: userCode,
                            },
                        },
                        status: 'validated',
                    },
                },
            ];
        },
    },
})
export class MobileTimeEntryInquiryByDate extends ui.Page<GraphApi> {
    private _manageDisplayApplicativePageActions(isDirty = false) {
        this._manageDisplayAdditionalPageActions(isDirty);
    }

    private _manageDisplayAdditionalPageActions(isDirty = false) {
        this.manageDisplayButtonCRUDActions(isDirty);
    }

    private manageDisplayButtonCRUDActions(isDirty = false) {
        if (this.isValidated.value) {
            this.$standardDeleteAction.isHidden = true;
        } else {
            this.$standardDeleteAction.isHidden = false;
        }
    }

    /*
     *
     *  Technical properties
     *
     */
    @ui.decorators.textField<MobileTimeEntryInquiryByDate>({
        isReadOnly: true,
        isTransient: true,
    })
    _CrudAction: ui.fields.Text;

    // 0 all  1 = isNonProjectCategory=true  2 = isNonProjectCategory=false
    @ui.decorators.numericField<MobileTimeEntryInquiryByDate>({
        isReadOnly: true,
        isTransient: true,
    })
    isNonProjectCategory: ui.fields.Numeric;

    // -----------------------------------------------
    // Page button
    // -----------------------------------------------
    @ui.decorators.pageAction<MobileTimeEntryInquiryByDate>({
        title: 'My assignments',
        shortcut: ['f4'],
        buttonType: 'secondary',
        async onClick() {
            this.myAssignment.value = null;
            this.myAssignmentBlock.isHidden = false;
            this.myAssignment.openDialog();
        },
    })
    myAssignmentButton: ui.PageAction;

    /*
     *
     *  Page blocks
     *
     */
    @ui.decorators.section<MobileTimeEntryInquiryByDate>({
        isTitleHidden: true,
    })
    mainSection: ui.containers.Section;

    @ui.decorators.block<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.mainSection;
        },
        title: 'My assignments',
    })
    myAssignmentBlock: ui.containers.Block;

    @ui.decorators.block<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.mainSection;
        },
        title: 'Time summary',
    })
    timeSummaryBlock: ui.containers.Block;

    @ui.decorators.block<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.mainSection;
        },
        title: 'Status',
    })
    statusBlock: ui.containers.Block;

    @ui.decorators.block<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.mainSection;
        },
        title: 'Project billing',
    })
    projectBillingBlock: ui.containers.Block;

    @ui.decorators.block<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.mainSection;
        },
        title: 'Load',
        isDisabled: true,
    })
    loadBlock: ui.containers.Block;

    @ui.decorators.block<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.mainSection;
        },
        title: 'Cost',
    })
    costBlock: ui.containers.Block;

    @ui.decorators.block<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.mainSection;
        },
        title: 'Creation',
        isDisabled: true,
    })
    creationBlock: ui.containers.Block;

    /*
     *
     *  Page properties
     *
     */
    // -----------------------------
    // Node computedValue properties
    // -----------------------------
    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, ProjectUser>({
        node: '@sage/x3-project-management/ProjectUser',
        title: 'Employee',
        fetchesDefaults: true, // to force default values loading
        valueField: { employee: { code: true } },
        bind: 'connectedEmployee',
        isMandatory: true,
        isReadOnly: true,
    })
    connectedEmployee: ui.fields.Reference;

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, TimeCategory>({
        node: '@sage/x3-project-management/TimeCategory',
        valueField: 'code',
        isReadOnly: true,
        columns: [
            ui.nestedFields.technical({ bind: 'code' }),
            ui.nestedFields.technical({ bind: 'localizedDescription' }),
            ui.nestedFields.technical({ bind: 'isNonProjectCategory' }),
        ],
    })
    defaultTimeCategory: ui.fields.Reference<TimeCategory>;

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, UnitOfMeasure>({
        node: '@sage/x3-master-data/UnitOfMeasure',
        valueField: 'code',
        isReadOnly: true,
        filter() {
            const filter: Filter<UnitOfMeasure> = {
                unitType: { _eq: 'time' },
            };
            return filter;
        },
        columns: [
            ui.nestedFields.technical({ bind: 'code' }),
            ui.nestedFields.technical({ bind: 'localizedDescription' }),
            ui.nestedFields.technical({ bind: 'numberOfDecimals' }),
        ],
    })
    defaultUnit: ui.fields.Reference<UnitOfMeasure>;

    @ui.decorators.checkboxField<MobileTimeEntryInquiryByDate>({
        isReadOnly: true,
    })
    isTimeEntryAdministrator: ui.fields.Checkbox;

    @ui.decorators.checkboxField<MobileTimeEntryInquiryByDate>({
        isReadOnly: true,
    })
    isTimeEntryEmployee: ui.fields.Checkbox;

    @ui.decorators.checkboxField<MobileTimeEntryInquiryByDate>({
        isReadOnly: true,
    })
    isNegativeTimeSpentAllowed: ui.fields.Checkbox;

    @ui.decorators.checkboxField<MobileTimeEntryInquiryByDate>({
        isReadOnly: true,
    })
    isValidatedDefault: ui.fields.Checkbox;

    // -----------------------------------------------
    // No Block: non-transient properties not displayed
    // -----------------------------------------------
    @ui.decorators.textField<MobileTimeEntryInquiryByDate>({
        title: '_id',
        isReadOnly: true,
    })
    _id: ui.fields.Text;

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, ProjectUser>({
        node: '@sage/x3-project-management/ProjectUser',
        title: 'Employee',
        valueField: { employee: { code: true } },
        isMandatory: true,
        isReadOnly: true,
    })
    employee: ui.fields.Reference;

    @ui.decorators.numericField<MobileTimeEntryInquiryByDate>({
        title: 'Line number',
        isReadOnly: true,
    })
    lineNumber: ui.fields.Numeric;

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, OperationAssignment>({
        title: 'Assignment line',
        node: '@sage/x3-project-management/OperationAssignment',
        valueField: 'lineNumber',
        columns: [
            ui.nestedFields.technical({ bind: 'plannedLoad' }),
            ui.nestedFields.technical({ bind: 'consumedLoad' }),
            ui.nestedFields.technical({ bind: 'remainingLoad' }),
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/UnitOfMeasure',
                bind: 'unit',
                valueField: 'code',
            }),
        ],
        isReadOnly: true,
    })
    assignmentLine: ui.fields.Reference;

    @ui.decorators.labelField<MobileTimeEntryInquiryByDate>({
        optionType: '@sage/x3-master-data/ExchangeRateType',
        isDisabled: true,
    })
    rateType: ui.fields.Label;

    // -----------------------------------------------
    // Block: myAssignmentBlock
    // -----------------------------------------------
    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, OperationAssignment>({
        parent() {
            return this.myAssignmentBlock;
        },
        isTransient: true,
        canFilter: true,
        isAutoSelectEnabled: true,
        shouldSuggestionsIncludeColumns: true,
        isFullWidth: true,
        valueField: '_id',
        lookupDialogTitle: 'My assignments',
        node: '@sage/x3-project-management/OperationAssignment',
        columns: [
            ui.nestedFields.text({ bind: '_id', title: 'ID', isHidden: true, canFilter: false }),
            ui.nestedFields.reference({
                bind: 'project',
                node: '@sage/x3-project-management/Project',
                valueField: 'id',
                title: 'Project',
                canFilter: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/Project',
                bind: 'project',
                valueField: { localizedDescription: true },
                canFilter: true,
            }),
            ui.nestedFields.reference({
                bind: 'task',
                node: '@sage/x3-project-management/Task',
                valueField: 'code',
                title: 'Task',
                canFilter: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/Task',
                bind: 'task',
                valueField: { localizedDescription: true },
                canFilter: true,
            }),
            ui.nestedFields.reference({
                bind: 'operation',
                node: '@sage/x3-project-management/Operation',
                valueField: 'number',
                title: 'Operation',
                canFilter: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/Operation',
                bind: 'operation',
                valueField: { localizedDescription: true },
                canFilter: true,
            }),
            ui.nestedFields.reference({
                bind: 'timeCategory',
                node: '@sage/x3-project-management/TimeCategory',
                valueField: 'code',
                title: 'Time category',
                canFilter: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/TimeCategory',
                bind: 'timeCategory',
                valueField: { localizedDescription: true },
                canFilter: true,
            }),
        ],
        filter() {
            const employee = this.$.userCode ? this.$.userCode.toUpperCase() : '';
            const filter: Filter<OperationAssignment> = {
                employee: { employee: { code: employee } },
                project: { projectStatus: { _eq: 'launched' } },
                _and: [
                    { task: { taskType: { _eq: 'operational' } } },
                    { task: { category: { type: { _in: ['labor', 'mixed'] } } } },
                ],
                operation: { status: { _eq: 'inProcess' } },
            };
            return filter;
        },
        async onChange() {
            if (this.myAssignment.value?.project) {
                this.project.value = this.myAssignment.value.project;
                this.projectIsDisable();
            }
            if (this.myAssignment.value?.task) {
                this.task.value = this.myAssignment.value.task;
                this.taskOnChangeBudget();
                this.taskIsDisable();
            }
            if (this.myAssignment.value?.operation) {
                this.operation.value = this.myAssignment.value.operation;
                this.localizedDescription.value = this.myAssignment.value.operation.localizedDescription;
                this.myAssignmentBlock.isHidden = true;
            }
            await this.$.commitValueAndPropertyChanges();
            this.timeSpent.focus();
        },
    })
    myAssignment: ui.fields.Reference<OperationAssignment>;

    // -----------------------------------------------
    // Block: timeSummary
    // -----------------------------------------------
    @ui.decorators.dateField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.timeSummaryBlock;
        },
        title: 'Date',
        async onChange() {
            this.date.getNextField(true)?.focus();
        },
    })
    date: ui.fields.Date;

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, TimeCategory>({
        parent() {
            return this.timeSummaryBlock;
        },
        title: 'Time category',
        node: '@sage/x3-project-management/TimeCategory',
        valueField: 'code',
        helperTextField: 'localizedDescription',
        filter() {
            if (this.isNonProjectCategory.value !== 0) {
                const filter: Filter<TimeCategory> = {
                    isNonProjectCategory: { _eq: this.isNonProjectCategory.value === 1 ? true : false },
                };
                return filter;
            }
            const filter: Filter<TimeCategory> = {
                code: { _ne: null },
            };
            return filter;
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Time category',
            }),
            ui.nestedFields.text({
                bind: 'localizedDescription',
                title: 'Description',
            }),
            ui.nestedFields.label({
                bind: 'isNonProjectCategory' as any,
                title: 'Non-project category',
                map(value: any) {
                    switch (value) {
                        case true:
                            return ui.localize(
                                '@sage/x3-project-management/pages__mobile_time_entry_inquiry_by_date__non-project',
                                'Non-project',
                            );
                        default:
                            return ui.localize(
                                '@sage/x3-project-management/pages__mobile_time_entry_inquiry_by_date__time-on-project',
                                'Time on project',
                            );
                    }
                },
            }),
            ui.nestedFields.technical({ bind: 'isNonProjectCategory' }),
        ],
        async onChange() {
            this.LoadIsNonProjectCategoryValue();
            this.timeCategoryOnChange();
            await this._callComputeProjectEmployeeCostTypeAPI();
            await this.$.commitValueAndPropertyChanges();
            if (this.timeCategory.value) {
                this.timeCategory.getNextField(true)?.focus();
            }
        },
    })
    timeCategory: ui.fields.Reference<TimeCategory>;

    async timeCategoryOnChange() {
        this.timeCategoryIsNonProjectCategory();
        if (this.isNonProjectCategory.value === 1) {
            this.project.value = null;
            this.task.value = null;
            this.budget.value = null;
            this.operation.value = null;
        }
    }

    async timeCategoryIsNonProjectCategory() {
        this.LoadIsNonProjectCategoryValue();
        if (this.isNonProjectCategory.value === 1) {
            this.project.isMandatory = false;
            this.task.isMandatory = false;
            this.budget.isMandatory = false;
            this.operation.isMandatory = false;

            this.project.isHidden = true;
            this.task.isHidden = true;
            this.budget.isHidden = true;
            this.operation.isHidden = true;

            this.projectBillingBlock.isHidden = true;
        } else {
            this.project.isMandatory = true;
            this.task.isMandatory = false;
            this.budget.isMandatory = false;
            this.operation.isMandatory = false;

            this.project.isHidden = false;
            this.task.isHidden = false;
            this.budget.isHidden = false;
            this.operation.isHidden = false;

            this.projectBillingBlock.isHidden = false;
        }
    }

    async LoadIsNonProjectCategoryValue() {
        this.isNonProjectCategory.value = 0;
        if (this.timeCategory.value) {
            if (this.timeCategory.value.isNonProjectCategory) {
                this.isNonProjectCategory.value = 1;
            }
        }
    }

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, UnitOfMeasure>({
        parent() {
            return this.timeSummaryBlock;
        },
        title: 'Time unit',
        node: '@sage/x3-master-data/UnitOfMeasure',
        valueField: 'code',
        helperTextField: 'localizedDescription',
        isMandatory: true,
        filter() {
            const filter: Filter<UnitOfMeasure> = {
                unitType: { _eq: 'time' },
            };
            return filter;
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'code',
            }),
            ui.nestedFields.text({
                bind: 'localizedDescription',
                title: 'Description',
            }),
            ui.nestedFields.technical({ bind: 'numberOfDecimals' }),
        ],
        async onChange() {
            if (this.unit.value) {
                this._GetNumberOfDecimals();
            }
            await this.$.commitValueAndPropertyChanges();
            if (this.unit.value) {
                this.unit.getNextField(true)?.focus();
            }
        },
    })
    unit: ui.fields.Reference<UnitOfMeasure>;

    @ui.decorators.numericField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.timeSummaryBlock;
        },
        title: 'Time spent',
        validation: /[0-9]{1,}/,
        min() {
            return this.isNegativeTimeSpentAllowed.value ? -100000 : 0;
        },
        isNotZero: true,
        async onChange() {
            this._GetNumberOfDecimals();
            await this.$.commitValueAndPropertyChanges();
            this.timeSpent.getNextField(true)?.focus();
        },
    })
    timeSpent: ui.fields.Numeric;

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, Project>({
        parent() {
            return this.timeSummaryBlock;
        },
        title: 'Project',
        node: '@sage/x3-project-management/Project',
        valueField: 'id',
        helperTextField: 'localizedDescription',
        isAutoSelectEnabled: true,
        shouldSuggestionsIncludeColumns: true,
        filter() {
            const filter: Filter<Project> = {
                isActive: { _eq: true },
                _and: [
                    {
                        _or: [
                            {
                                type: 'project',
                            },
                            {
                                type: 'mixedProject',
                            },
                        ],
                    },
                ],
            };
            return filter;
        },
        columns: [
            ui.nestedFields.text({
                bind: 'id',
                title: 'Project',
            }),
            ui.nestedFields.text({
                bind: 'localizedDescription',
                title: 'Description',
            }),
            // TODO
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/ProjectUser',
                bind: 'responsiblePerson',
                valueField: { employee: { name: true } },
                title: 'Responsible Person',
                canFilter: false,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-system/Site',
                bind: 'site',
                valueField: 'code',
                title: 'Site',
            }),
            // HIDDEN
            ui.nestedFields.label({
                bind: 'type',
                title: 'Type',
                isHidden: true,
                optionType: '@sage/x3-project-management-data/ProjectAllocationType',
            }),
        ],
        async onChange() {
            this.projectOnChange();
            await this.$.commitValueAndPropertyChanges();
            this.project.getNextField(true)?.focus();
        },
    })
    project: ui.fields.Reference<Project>;

    private async projectOnChange() {
        if (this.project.value) {
            this.task.value = null;
            this.budget.value = null;
            this.operation.value = null;

            await this._callComputeProjectEmployeeCostTypeAPI();
        } else {
            this.task.value = null;
            this.budget.value = null;
            this.operation.value = null;
        }
        this.projectIsDisable();
    }

    private async projectIsDisable() {
        if (this.project.value) {
            this.task.isMandatory = false;
            this.budget.isMandatory = false;
            this.operation.isMandatory = false;

            this.task.isDisabled = false;
            this.budget.isDisabled = false;
            this.operation.isDisabled = true; // Will be activated by Task if necessary
        } else {
            this.task.isMandatory = false;
            this.budget.isMandatory = false;
            this.operation.isMandatory = false;

            this.task.isDisabled = true;
            this.budget.isDisabled = true;
            this.operation.isDisabled = true; // Will be activated by Task if necessary
        }
    }

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, Task>({
        parent() {
            return this.timeSummaryBlock;
        },
        title: 'Task',
        node: '@sage/x3-project-management/Task',
        valueField: 'code',
        helperTextField: 'localizedDescription',
        isAutoSelectEnabled: true,
        shouldSuggestionsIncludeColumns: true,
        filter() {
            const filter: Filter<Task> = {
                project: { id: { _eq: this.project.value ? this.project.value.id : '' } },
                isActive: true,
                _and: [
                    {
                        _or: [
                            {
                                type: { _eq: 'laborTask' },
                            },
                            {
                                type: { _eq: 'mixedTask' },
                            },
                        ],
                    },
                ],
            };
            return filter;
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Task',
            }),
            ui.nestedFields.text({
                bind: 'localizedDescription',
                title: 'Description',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/ProjectUser',
                bind: 'responsiblePerson',
                valueField: { employee: { name: true } },
                title: 'Responsible Person',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-system/Site',
                bind: 'site',
                valueField: 'code',
                title: 'Site',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/TimeCategory',
                bind: 'timeCategory',
                valueField: { localizedDescription: true },
                title: 'Time category',
            }),
            // HIDDEN
            ui.nestedFields.label({
                bind: 'type',
                title: 'Type',
                optionType: '@sage/x3-project-management-data/ProjectAllocationType',
                isHidden: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/Project',
                bind: 'project',
                valueField: 'id',
                title: 'Project',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isActive',
                title: 'Active',
                isHidden: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/Budget',
                bind: 'budget',
                valueField: 'code',
                title: 'Budget',
                isHidden: true,
            }),
        ],
        async onChange() {
            this.taskOnChange();
            await this.$.commitValueAndPropertyChanges();
            this.task.getNextField(true)?.focus();
        },
    })
    task: ui.fields.Reference<Task>;

    private async taskOnChange() {
        if (this.task.value) {
            this.operation.value = null;
            this.taskOnChangeBudget();
            await this._callComputeProjectEmployeeCostTypeAPI();
        } else {
            this.budget.value = null;
            this.operation.value = null;
        }
        this.taskIsDisable();
    }

    private async taskOnChangeBudget() {
        if (this.task.value) {
            const taskRead = await this.$.graph
                .node('@sage/x3-project-management/Task')
                .read(
                    {
                        _id: true,
                        budget: {
                            _id: true,
                            code: true,
                            localizedDescription: true,
                        },
                    },
                    `${this.project.value ? this.project.value.id : ''}|${this.task.value.code}`, // TODO: Better way to specify id for a Sage X3 record?
                )
                .execute();

            this.budget.value = taskRead.budget;
        }
    }

    private async taskIsDisable() {
        if (this.task.value) {
            this.budget.isMandatory = false;
            this.operation.isMandatory = false;

            this.budget.isDisabled = true;
            this.operation.isDisabled = false;
            this.operation.isReadOnly = false;
        } else {
            this.budget.isMandatory = true;
            this.operation.isMandatory = false;

            this.budget.isDisabled = false;
            this.operation.isDisabled = true;
        }
    }

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, Operation>({
        parent() {
            return this.timeSummaryBlock;
        },
        title: 'Operation',
        node: '@sage/x3-project-management/Operation',
        valueField: 'number',
        helperTextField: 'localizedDescription',
        isAutoSelectEnabled: true,
        shouldSuggestionsIncludeColumns: true,
        filter() {
            const filter: Filter<Operation> = {
                project: { id: { _eq: this.project.value ? this.project.value.id : '' } },
                task: { code: { _eq: this.task.value ? this.task.value.code : '' } },
            };
            return filter;
        },
        columns: [
            ui.nestedFields.text({
                bind: 'number',
                title: 'Operation',
            }),
            ui.nestedFields.text({
                bind: 'localizedDescription',
                title: 'Description',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/TimeCategory',
                bind: 'timeCategory',
                valueField: { localizedDescription: true },
                title: 'Time category',
            }),
            // HIDDEN
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/Project',
                bind: 'project',
                valueField: 'id',
                title: 'Project',
                isHidden: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/Task',
                bind: 'task',
                valueField: 'code',
                title: 'Task',
                isHidden: true,
            }),
            ui.nestedFields.technical({ bind: 'plannedLoad' }),
            ui.nestedFields.technical({ bind: 'consumedLoad' }),
            ui.nestedFields.technical({ bind: 'remainingLoad' }),
            ui.nestedFields.reference({
                node: '@sage/x3-master-data/UnitOfMeasure',
                bind: 'unit',
                valueField: 'code',
            }),
        ],
        async onChange() {
            if (this.operation.value) {
                await this._callComputeProjectEmployeeCostTypeAPI();
                await this.$.commitValueAndPropertyChanges();
                this.operation.getNextField(true)?.focus();
            }
        },
    })
    operation: ui.fields.Reference<Operation>;

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, Budget>({
        parent() {
            return this.timeSummaryBlock;
        },
        title: 'Budget',
        node: '@sage/x3-project-management/Budget',
        valueField: 'code',
        helperTextField: 'localizedDescription',
        isAutoSelectEnabled: true,
        shouldSuggestionsIncludeColumns: true,
        filter() {
            const filter: Filter<Budget> = {
                project: { id: { _eq: this.project.value ? this.project.value.id : '' } },
                isActive: true,
                type: { _eq: 'budget' },
            };
            return filter;
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Budget',
            }),
            ui.nestedFields.text({
                bind: 'localizedDescription',
                title: 'Description',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/ProjectUser',
                bind: 'responsiblePerson',
                valueField: { employee: { name: true } },
                title: 'Responsible Person',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-system/Site',
                bind: 'site',
                valueField: 'code',
                title: 'Site',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/TimeCategory',
                bind: 'timeCategory',
                valueField: { localizedDescription: true },
                title: 'Time category',
            }),
            // HIDDEN fields
            ui.nestedFields.label({
                bind: 'type',
                title: 'Type',
                optionType: '@sage/x3-project-management-data/ProjectAllocationType',
                isHidden: true,
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-project-management/Project',
                bind: 'project',
                valueField: 'id',
                title: 'Project',
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isActive',
                title: 'Active',
                isHidden: true,
            }),
        ],
        async onChange() {
            if (this.budget.value) {
                this.task.value = null;
                this.operation.value = null;

                await this._callComputeProjectEmployeeCostTypeAPI();
                await this.$.commitValueAndPropertyChanges();
                this.budget.getNextField(true)?.focus();
            }
        },
    })
    budget: ui.fields.Reference<Budget>;

    @ui.decorators.textField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.timeSummaryBlock;
        },
        title: 'Description',
        isFullWidth: true,
    })
    localizedDescription: ui.fields.Text;

    // -----------------------------------------------
    // Block: Status block
    // -----------------------------------------------
    @ui.decorators.selectField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.statusBlock;
        },
        title: 'Status',
        isReadOnly: true,
        optionType: '@sage/x3-project-management/TimeEntryStatus',
    })
    status: ui.fields.Select;

    @ui.decorators.checkboxField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.statusBlock;
        },
        title: 'Validated',
    })
    isValidated: ui.fields.Checkbox;

    // -----------------------------------------------
    // Block: Project billing
    // -----------------------------------------------
    @ui.decorators.checkboxField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.projectBillingBlock;
        },
        title: 'Billable',
    })
    isBillable: ui.fields.Checkbox;

    @ui.decorators.dateField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.projectBillingBlock;
        },
        title: 'Billable from',
    })
    billableFrom: ui.fields.Date;

    // Block: Load
    @ui.decorators.textField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.loadBlock;
        },
        title: 'Planned load',
        isTransient: true,
    })
    plannedLoad: ui.fields.Text;

    @ui.decorators.textField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.loadBlock;
        },
        title: 'Consumed load',
        isTransient: true,
    })
    consumedLoad: ui.fields.Text;

    @ui.decorators.textField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.loadBlock;
        },
        title: 'Remaining load',
        isTransient: true,
    })
    remainingLoad: ui.fields.Text;

    // -----------------------------------------------
    // Block: cost
    // -----------------------------------------------
    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, CostType>({
        parent() {
            return this.costBlock;
        },
        title: 'Project cost type',
        node: '@sage/x3-project-management-data/CostType',
        valueField: 'code',
        helperTextField: 'localizedDescription',
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Cost type',
            }),
            ui.nestedFields.text({
                bind: 'localizedDescription',
                title: 'Description',
            }),
        ],
        async onChange() {
            if (this.projectCostType.value) {
                await this._callComputeUnitOfMeasureAPI();
                await this._callComputeProjectEmployeeLaborRateAPI(1);
                await this.$.commitValueAndPropertyChanges();
                this.projectCostType.getNextField(true)?.focus();
            }
        },
    })
    projectCostType: ui.fields.Reference<CostType>;

    @ui.decorators.numericField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.costBlock;
        },
        title: 'Project labor rate',
    })
    projectLaborRate: ui.fields.Numeric;

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, CostType>({
        parent() {
            return this.costBlock;
        },
        title: 'Employee cost type',
        node: '@sage/x3-project-management-data/CostType',
        valueField: 'code',
        helperTextField: 'localizedDescription',
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Cost type',
            }),
            ui.nestedFields.text({
                bind: 'localizedDescription',
                title: 'Description',
            }),
        ],
        async onChange() {
            if (this.employeeCostType.value) {
                await this._callComputeProjectEmployeeLaborRateAPI(2);
                await this.$.commitValueAndPropertyChanges();
                this.employeeCostType.getNextField(true)?.focus();
            }
        },
    })
    employeeCostType: ui.fields.Reference<CostType>;

    @ui.decorators.numericField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.costBlock;
        },
        title: 'Employee labor rate',
    })
    employeeLaborRate: ui.fields.Numeric;

    @ui.decorators.numericField<MobileTimeEntryInquiryByDate>({
        parent() {
            return this.costBlock;
        },
        title: 'Rate multiplier',
    })
    rateMultiplier: ui.fields.Numeric;

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, Currency>({
        parent() {
            return this.costBlock;
        },
        title: 'Currency',
        node: '@sage/x3-master-data/Currency',
        valueField: 'code',
        helperTextField: 'localizedDescription',
        isReadOnly: true,
    })
    currency: ui.fields.Reference<Currency>;

    // -----------------------------------------------
    // Block: creation
    // -----------------------------------------------
    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, User>({
        parent() {
            return this.creationBlock;
        },
        title: 'Validated by',
        node: '@sage/x3-system/User',
        valueField: 'code',
        helperTextField: 'nameDescription',
    })
    validatedBy: ui.fields.Reference<User>;

    @ui.decorators.referenceField<MobileTimeEntryInquiryByDate, User>({
        parent() {
            return this.creationBlock;
        },
        title: 'Rejected by',
        node: '@sage/x3-system/User',
        valueField: 'code',
        helperTextField: 'nameDescription',
    })
    rejectedBy: ui.fields.Reference<User>;

    /*
     *
     *  Functions - S/P
     *
     */
    private async _GetNumberOfDecimals() {
        if (this.unit.value) {
            this.timeSpent.scale = this.unit.value.numberOfDecimals;
        }
    }

    async setCreateMode() {
        this.myAssignmentBlock.isHidden = true;
        this.myAssignmentButton.isHidden = false;
        if (this.connectedEmployee.value) {
            this.employee.value = this.connectedEmployee.value;
        }
        if (this.defaultUnit.value) {
            this.unit.value = this.defaultUnit.value;
        }
        if (this.defaultTimeCategory.value) {
            this.timeCategory.value = this.defaultTimeCategory.value;
        }
        this.timeCategoryIsNonProjectCategory();

        this.date.value = DateValue.today().toString();
        this.isValidated.value = this.isValidatedDefault.value;

        this.date.isReadOnly = false;
        this.timeCategory.isReadOnly = false;
        this.unit.isReadOnly = false;
        this.timeSpent.isReadOnly = false;
        this.project.isReadOnly = false;
        this.task.isReadOnly = false;
        this.budget.isReadOnly = false;
        this.localizedDescription.isReadOnly = false;
        this.isValidated.isReadOnly = false;

        this.status.isHidden = true;

        // Entire hidden blocks
        this.projectBillingBlock.isHidden = true;
        this.loadBlock.isHidden = true;
        this.costBlock.isHidden = true;
        this.creationBlock.isHidden = true;
    }

    async setDisplayMode() {
        this.myAssignmentBlock.isHidden = true;
        this.myAssignmentButton.isHidden = true;
        if (this.isValidated.value) {
            this.timeSummaryBlock.isDisabled = true;

            this.projectBillingBlock.isDisabled = true;
            this.costBlock.isDisabled = true;

            this.projectBillingBlock.isHidden = false;
        } else {
            this.timeCategoryIsNonProjectCategory();
            this.projectIsDisable();
            this.taskIsDisable();
        }
    }

    async _callComputeUnitOfMeasureAPI(): Promise<void> {
        if (this.isNonProjectCategory.value === 0 && !this.costBlock.isHidden) {
            const result = await this.$.graph
                .node('@sage/x3-project-management/TimeEntryLine')
                .mutations.computeUnitOfMeasure(
                    {
                        unit: true,
                    },
                    {
                        parameters: {
                            project: this.project.value?.id ?? '',
                            task: this.task.value?.code ?? '',
                            operation: this.operation.value?.number ?? 0,
                            employee: this.employee.value?.employee.code ?? '',
                            timeCategory: this.timeCategory.value?.code ?? '',
                            costType: this.projectCostType.value?.code ?? '',
                        },
                    },
                )
                .execute();

            this.unit.value = await this.$.graph
                .node('@sage/x3-master-data/UnitOfMeasure')
                .read(
                    {
                        code: true,
                        localizedDescription: true,
                        numberOfDecimals: true,
                    },
                    `${result.unit}`,
                )
                .execute();
        }
    }

    async _callComputeProjectEmployeeLaborRateAPI(iWitchOne: number): Promise<void> {
        if (this.isNonProjectCategory.value === 0 && !this.costBlock.isHidden) {
            const result = await this.$.graph
                .node('@sage/x3-project-management/TimeEntryLine')
                .mutations.computeProjectEmployeeLaborRate(
                    {
                        projectLaborRate: true,
                        employeeLaborRate: true,
                        message: true,
                    },
                    {
                        parameters: {
                            employee: this.employee.value?.employee.code ?? '',
                            date: this.date.value ?? '',
                            unit: this.unit.value?.code ?? '',
                            currency: this.currency.value?.code ?? '',
                            rateType: this.rateType.value ?? 'dailyRate',
                            projectCostType: this.projectCostType.value?.code ?? '',
                            employeeCostType: this.employeeCostType.value?.code ?? '',
                        },
                    },
                )
                .execute();
            if (iWitchOne === 0 || iWitchOne === 1) {
                this.projectLaborRate.value = Number(result.projectLaborRate);
            }
            if (iWitchOne === 0 || iWitchOne === 2) {
                this.employeeLaborRate.value = Number(result.employeeLaborRate);
            }
        }
    }

    private async _callComputeProjectEmployeeCostTypeAPI(): Promise<void> {
        if (this.isNonProjectCategory.value === 0 && !this.costBlock.isHidden) {
            const result = await this.$.graph
                .node('@sage/x3-project-management/TimeEntryLine')
                .mutations.computeProjectEmployeeCostType(
                    {
                        employeeCostType: true,
                        projectCostType: true,
                    },
                    {
                        parameters: {
                            project: this.project.value?.id ?? '',
                            task: this.task.value?.code ?? '',
                            operation: this.operation.value?.number ?? 0,
                            employee: this.employee.value?.employee.code ?? '',
                            timeCategory: this.timeCategory.value?.code ?? '',
                        },
                    },
                )
                .execute();

            const resultProject = await this.$.graph
                .node('@sage/x3-project-management-data/CostType')
                .read(
                    {
                        code: true,
                        localizedDescription: true,
                    },
                    `${result.projectCostType}`,
                )
                .execute();

            const resultEmployee = await this.$.graph
                .node('@sage/x3-project-management-data/CostType')
                .read(
                    {
                        code: true,
                        localizedDescription: true,
                    },
                    `${result.employeeCostType}`,
                )
                .execute();

            const svProjectCostType = this.projectCostType.value?.code;
            const svEmployeeCostType = this.employeeCostType.value?.code;
            this.projectCostType.value = resultProject;
            this.employeeCostType.value = resultEmployee;
            if (
                svProjectCostType !== this.projectCostType.value?.code ||
                svEmployeeCostType !== this.employeeCostType.value?.code
            ) {
                await this._callComputeUnitOfMeasureAPI();
                let caseManaged: number = 0;
                if (
                    svProjectCostType !== this.projectCostType.value?.code &&
                    svEmployeeCostType !== this.employeeCostType.value?.code
                ) {
                    caseManaged = 0;
                } else {
                    if (svProjectCostType !== this.projectCostType.value?.code) {
                        caseManaged = 1;
                    } else {
                        caseManaged = 2;
                    }
                }
                await this._callComputeProjectEmployeeLaborRateAPI(caseManaged);
            }
        }
    }
}
