import {
    StandardOperations,
    TransactionProductionReporting,
    WorkOrder,
    WorkOrderOperationLine,
} from '@sage/x3-manufacturing-api';
import { RoutingHeader, WorkOrderProductLine, WorkOrderScheduling } from '@sage/x3-manufacturing-api-partial';
import { EmployeeId } from '@sage/x3-manufacturing-data-api';
import { ExtractEdges, extractEdges } from '@sage/xtrem-client';
import { DateValue } from '@sage/xtrem-date-time';
import * as ui from '@sage/xtrem-ui';
import { formatImportMessagefile } from '../client-functions/format-import-message-file';
import { readParameterValue } from '../client-functions/read-parameter';
import { manufacturing } from '../menu-items/manufacturing';

import { validateWithDetails } from '../client-functions/validation';
import { getWorkOrderOperationsList } from '../client-functions/work-order-operations-list';
import {
    SelectedOperationMethod,
    TimeTrackingLine,
    TimeTrackingStorageObject,
    TrackingLineTable,
    X3Response,
} from './utils/types';

let CtlOpepre: string; // Parameter CTLOPEPRE - Expected operation tracking
let showMobileTimeTrackingDetails: boolean;
let operationList: ExtractEdges<WorkOrderOperationLine>[];
let workOrderRoutingList: ExtractEdges<RoutingHeader>[];

@ui.decorators.page<MobileTimeTracking>({
    title: 'Time tracking',
    module: 'x3-manufacturing',
    mode: 'default',
    isTitleHidden: true,
    menuItem: manufacturing,
    authorizationCode: 'CWSAOT',
    access: { node: '@sage/x3-manufacturing/WorkOrder' },
    priority: 100,
    businessActions() {
        return [this.createButton];
    },
    async onLoad() {
        CtlOpepre = await this.getParamValue(await this.getStockSite(), 'CTLOPEPRE');
        await this.initPage();
    },
})
export class MobileTimeTracking extends ui.Page {
    private transactionsList: any[];

    @ui.decorators.section<MobileTimeTracking>({
        isTitleHidden: true,
    })
    section: ui.containers.Section;

    @ui.decorators.block<MobileTimeTracking>({
        parent() {
            return this.section;
        },
        isTitleHidden: true,
    })
    block: ui.containers.Block;

    @ui.decorators.dateField<MobileTimeTracking>({
        parent() {
            return this.block;
        },
        title: 'Tracking date',
        isMandatory: true,
        maxDate: DateValue.today().toString(),
        onChange() {
            this.transaction.focus();
        },
    })
    trackingDate: ui.fields.Date;

    @ui.decorators.labelField<MobileTimeTracking>({
        parent() {
            return this.block;
        },
        title: 'Site',
        isHidden: true,
    })
    stockSite: ui.fields.Label;

    @ui.decorators.selectField<MobileTimeTracking>({
        parent() {
            return this.block;
        },
        title: 'Transaction',
        isTransient: true,
        isMandatory: true,
        async onChange() {
            if (this.transaction.value) {
                this.employeeId.isDisabled = false;
                this.workOrder.isDisabled = false;
                this.employeeId.isMandatory = await this.isEmployeeMandatory(
                    this.transactionsList,
                    this.transaction.value,
                );
                this.employeeId.isHidden = await this.isEmployeeHidden(this.transactionsList, this.transaction.value);
                this.unplannedOperations.isHidden = await this.isUnplannedOperationsHidden(
                    this.transactionsList,
                    this.transaction.value,
                );
                this.isUnplannedOperationsEnabled();
            } else {
                this.employeeId.isDisabled = true;
                this.workOrder.isDisabled = true;
                this.unplannedOperations.isHidden = true;
                this.transaction.focus();
            }
        },
    })
    transaction: ui.fields.Select;

    @ui.decorators.referenceField<MobileTimeTracking, EmployeeId>({
        parent() {
            return this.block;
        },
        title: 'Employee ID',
        valueField: 'teamId',
        node: '@sage/x3-manufacturing-data/EmployeeId',
        placeholder: 'Scan or select...',
        isAutoSelectEnabled: true,
        isDisabled: true,
        filter() {
            return {
                isActive: true,
            };
        },
        onChange() {
            if (this.employeeId.isMandatory) {
                if (this.employeeId.value) {
                    this.workOrder.isDisabled = false;
                    this.workOrder.focus();
                } else {
                    this.workOrder.isDisabled = true;
                }
            } else {
                this.workOrder.isDisabled = false;
            }
        },
        columns: [
            ui.nestedFields.numeric({
                bind: 'teamId',
            }),
            ui.nestedFields.text({
                bind: 'description',
                title: 'Description',
                isReadOnly: true,
            }),
        ],
    })
    employeeId: ui.fields.Reference;

    @ui.decorators.referenceField<MobileTimeTracking, WorkOrder>({
        parent() {
            return this.block;
        },
        title: 'Work order',
        valueField: 'number',
        node: '@sage/x3-manufacturing/WorkOrder',
        placeholder: 'Scan or select...',
        isAutoSelectEnabled: true,
        isDisabled: false,
        isFullWidth: true,

        filter() {
            return {
                productionSite: { code: this.stockSite.value },
                orderStatus: { _in: ['firm'] },
                modeType: 'complete',
                workOrderSuspendedFlag: false,
            };
        },
        async onChange() {
            this.setOperationFieldsDisabledStatus();
            //this.workOrder.value ? this.workOrder.value : '';
            if (this.workOrder.value) {
                workOrderRoutingList = await this.getWorkOrderRoutingList();
                this.isUnplannedOperationsEnabled();
                operationList = await getWorkOrderOperationsList(this.workOrder.value.number, this);
                this.releaseUOM.value = await this.getReleaseUOM();
                this.releaseTimeUnit.value = await this.getReleaseTimeUnit();
            }
        },
        columns: [
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkOrder',
                bind: 'productionSite',
                valueField: 'code',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'number',
            }),

            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkOrder',
                bind: 'routingCode',
                valueField: 'code',
            }),
            ui.nestedFields.text({
                bind: 'startDate',
                title: 'Start',
                isReadOnly: true,
                prefix: 'Start:',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkOrder',
                bind: 'releasedRouting',
                valueField: { routing: { code: true } },
            }),
        ],
    })
    workOrder: ui.fields.Reference;

    @ui.decorators.textField<MobileTimeTracking>({
        parent() {
            return this.block;
        },
        title: 'Release unit',
        maxLength: 30,
        isTransient: true,
        isDisabled: true,
        isHidden: true,
    })
    releaseUOM: ui.fields.Text;

    @ui.decorators.textField<MobileTimeTracking>({
        parent() {
            return this.block;
        },
        title: 'Release time unit',
        maxLength: 30,
        isTransient: true,
        isDisabled: true,
        isHidden: true,
    })
    releaseTimeUnit: ui.fields.Text;

    @ui.decorators.checkboxField<MobileTimeTracking>({
        parent() {
            return this.block;
        },
        title: 'Unplanned operation tracking',
        isFullWidth: true,
        isHidden: true,
        isDisabled: true,
        async onChange() {
            this.isUnplannedOperationsEnabled();
        },
    })
    unplannedOperations: ui.fields.Checkbox;

    @ui.decorators.referenceField<MobileTimeTracking, StandardOperations>({
        parent() {
            return this.block;
        },
        title: 'Standard operation',
        valueField: 'code',
        node: '@sage/x3-manufacturing/StandardOperations',
        placeholder: 'Scan or select...',
        isHidden: true,
        isDisabled: true,
        isAutoSelectEnabled: true,
        isFullWidth: false,
        isMandatory: false,
        async onChange() {
            if (this.stdOperation.value !== null) {
                this.stdOperationDescription.value = this.stdOperation.value.description;
            } else {
                this.stdOperationDescription.value = null;
            }
            this.stdOperationDescription.focus();
        },
        filter() {
            return {
                site: {
                    code: this.stockSite.value ? this.stockSite.value : '',
                },
            };
        },
        columns: [
            ui.nestedFields.numeric({
                bind: 'code',
            }),
            ui.nestedFields.text({
                bind: 'description',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/StandardOperations',
                bind: 'operationUom',
                valueField: 'code',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/StandardOperations',
                bind: 'site',
                valueField: 'code',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/StandardOperations',
                bind: 'mainWorkCenter',
                valueField: 'code',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'timeUnit',
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'setupTime',
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'runTime',
                isHidden: true,
            }),
        ],
    })
    stdOperation: ui.fields.Reference;

    @ui.decorators.referenceField<MobileTimeTracking, WorkOrderOperationLine>({
        parent() {
            return this.block;
        },
        title: 'Operation',
        valueField: 'operationNumber',
        node: '@sage/x3-manufacturing/WorkOrderOperationLine',
        placeholder: 'Scan or select...',
        isDisabled: true,
        isAutoSelectEnabled: true,
        isFullWidth: true,
        async onChange() {
            if (!this.operationLine.value) {
                return;
            }

            const operationMilestonetype: string | undefined = await this.getOperationMilestoneType(
                this.operationLine.value?.operationNumber,
                this.workOrder.value?.number,
            );

            if (operationMilestonetype === 'none') {
                this.$.showToast(
                    ui.localize(
                        '@sage/x3-manufacturing/pages__mobile_time_tracking___operation_milestone_is_none',
                        'This operation will be tracked with a milestone.',
                    ),
                    { type: 'error', timeout: 5000 },
                );
                return;
            }

            if (!(await validateWithDetails(this))) return;

            this.updatePageStorage(SelectedOperationMethod.RouterOperation);
            this.setSessionStorage();
            this.$.setPageClean();

            if (CtlOpepre === '1') {
                showMobileTimeTrackingDetails = true;
            } else {
                if (
                    (await this.isPreviousOperationTracked(
                        this.operationLine.value?.operationNumber,
                        this.workOrder.value?.number,
                        operationMilestonetype,
                    )) === false
                ) {
                    if (CtlOpepre === '2') {
                        // "Yes, without block" display warning message
                        this.$.showToast(
                            ui.localize(
                                '@sage/x3-manufacturing/pages__mobile_time_tracking___Previous_operation_is_not_tracked',
                                'The previous operation has not been tracked',
                            ),
                            { type: 'warning', timeout: 5000 },
                        );
                        showMobileTimeTrackingDetails = true;
                    }
                    if (CtlOpepre === '3') {
                        // "Yes with block" display blocking message
                        this.$.showToast(
                            ui.localize(
                                '@sage/x3-manufacturing/pages__mobile_time_tracking___Previous_operation_is_not_tracked',
                                'The previous operation has not been tracked',
                            ),
                            { type: 'error', timeout: 5000 },
                        );
                        showMobileTimeTrackingDetails = false;
                    }
                } else {
                    showMobileTimeTrackingDetails = true;
                }
            }
            if (showMobileTimeTrackingDetails === true) {
                this.$.router.goTo('@sage/x3-manufacturing/MobileTimeTrackingDetails');
            }
        },

        filter() {
            return {
                workOrder: {
                    number: this.workOrder.value?.number,
                },
                expectedWorkCenter: {
                    type: { _ne: 3 },
                },
                workOrderScheduling: {
                    _atLeast: 1,
                    milestone: {
                        _ne: 'none',
                    },
                },
                operationStatus: { _ne: 6 },
            };
        },
        columns: [
            ui.nestedFields.numeric({
                bind: 'operationNumber',
            }),
            ui.nestedFields.text({
                bind: 'operationDescription',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkOrderOperationLine',
                bind: 'expectedWorkCenter',
                valueField: 'code',
            }),
            ui.nestedFields.reference({
                node: '@sage/x3-manufacturing/WorkOrderOperationLine',
                bind: 'operationUnit',
                valueField: 'code',
            }),
            ui.nestedFields.numeric({
                bind: 'expectedQuantity',
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'completedQuantity',
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'timeUnit',
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'setupTime',
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'runTime',
                isHidden: true,
            }),
        ],
    })
    operationLine: ui.fields.Reference;

    @ui.decorators.textField<MobileTimeTracking>({
        parent() {
            return this.block;
        },
        title: 'Description',
        maxLength: 30,
        isTransient: true,
        isHidden: true,
        isDisabled: true,
        isFullWidth: true,
        placeholder: 'Enter...',
    })
    stdOperationDescription: ui.fields.Text;

    @ui.decorators.numericField<MobileTimeTracking>({
        parent() {
            return this.block;
        },
        title: 'Operation',
        isHidden: true,
        validation: /^\d*[0-9](|\d*[0-9]|,\d*[0-9])?$/,
        isNotZero: true,
        max: 9999,
        isDisabled: true,
        isMandatory: true,
        async onChange() {
            let operation: string;
            if (!(await validateWithDetails(this))) {
                this.unplannedOperationsLine.value = null;
                return;
            }
            if (operationList.find(ope => ope.operationNumber == this.unplannedOperationsLine.value)) {
                const filteredNumbers = operationList.filter(
                    ope => ope.operationNumber == this.unplannedOperationsLine.value,
                );
                if (filteredNumbers.find(ope => ope.operationStatus === 'excluded')) {
                    this.$.dialog.message(
                        'error',
                        'Field error "Operation"',
                        `${ui.localize(
                            '@sage/x3-manufacturing/pages__mobile_time_tracking___operation_excluded',
                            'Excluded line.',
                        )}`,
                    );
                } else {
                    this.$.dialog.message(
                        'error',
                        'Field error "Operation"',
                        `${ui.localize(
                            '@sage/x3-manufacturing/pages__mobile_time_tracking___operation_exists',
                            'This operation is already planned.',
                        )}`,
                    );
                }
            } else {
                if (this.stdOperation.value === null) {
                    this.updatePageStorage(SelectedOperationMethod.UserDefinedOperation);
                } else {
                    this.updatePageStorage(SelectedOperationMethod.StandardOperationWithData);
                }

                this.setSessionStorage();
                this.$.setPageClean();
                this.$.router.goTo('@sage/x3-manufacturing/MobileTimeTrackingDetails');
            }
        },
    })
    unplannedOperationsLine: ui.fields.Numeric;

    @ui.decorators.block<MobileTimeTracking>({
        parent() {
            return this.section;
        },
        isHidden: true,
    })
    trackingLinesBlock: ui.containers.Block;

    @ui.decorators.tableField<MobileTimeTracking>({
        parent() {
            return this.trackingLinesBlock;
        },
        title: 'Operations',
        isTransient: true,
        canSelect: false,
        canFilter: false,
        columns: [
            ui.nestedFields.text({
                bind: 'workOrder',
            }),
            ui.nestedFields.numeric({
                bind: 'operationNumber',
            }),
            ui.nestedFields.text({
                bind: 'operationDescription',
            }),
            ui.nestedFields.text({
                bind: 'quantityAndStockUnit',
            }),
        ],
        dropdownActions: [
            {
                icon: 'delete',
                title: 'Delete',
                onClick(rowId: any) {
                    const trackingLines: any = this.getLines();
                    trackingLines.splice(rowId, 1);
                    if (!trackingLines.length) {
                        this.workOrder.isDirty = !this.workOrder.isDirty;
                        this.$.router.goTo('@sage/x3-manufacturing/MobileTimeTracking');
                        this.toggleCreateButton(trackingLines.length);
                    } else {
                        this.setTableTitle(trackingLines, true);
                        this.setTableLines(trackingLines);
                        this.setSessionStorage();
                    }
                },
            },
        ],
    })
    trackingLinesTable: ui.fields.Table<TrackingLineTable>;

    @ui.decorators.pageAction<MobileTimeTracking>({
        title: 'Create',
        buttonType: 'primary',
        shortcut: ['f2'],
        isDisabled: true,
        async onClick() {
            this.$.loader.isHidden = false;
            this.prepareCreation();
            const result = await this.callCreationAPI();
            this.$.loader.isHidden = true;

            if (this.getCreationStatus(result)) {
                this.workOrder.isDirty = !this.workOrder.isDirty;
                this.successScreen(result);
            } else {
                this.errorScreen(result);
            }
        },
    })
    createButton: ui.PageAction;

    /** Fields Management */

    async initPage() {
        this.transactionsList = await this.getEntryTransaction();
        await this.setMandatoryFields(this.transactionsList);
        await this.setStorageObject();
    }

    async setMandatoryFields(transactions: any[]): Promise<boolean | undefined> {
        let transactionIsDisabled: boolean;

        this.stockSite.value = await this.getStockSite();
        if (this.stockSite.value) {
            this.trackingDate.value = await this.getDate();

            if (!transactions || transactions.length === 0) {
                this.block.isDisabled = true;
                this.$.showToast(
                    ui.localize(
                        '@sage/x3-manufacturing/pages__mobile_completed_quantity___no_transaction_error',
                        'No transactions were found',
                    ),
                    { type: 'error', timeout: 5000 },
                );
                return (this.block.isDisabled = true);
            }
            transactionIsDisabled = transactions.length < 1;

            if (transactionIsDisabled) {
                this.setEntryTransaction(transactions, transactionIsDisabled, true);
                return;
            } else {
                this.setEntryTransaction(transactions, transactionIsDisabled, false);
            }

            //this.transaction.value ? this.transaction.value : '';
            this.employeeId.isMandatory = await this.isEmployeeMandatory(transactions, this.transaction.value);
            this.employeeId.isHidden = await this.isEmployeeHidden(transactions, this.transaction.value);
            this.unplannedOperations.isHidden = await this.isUnplannedOperationsHidden(
                transactions,
                this.transaction.value,
            );
        } else {
            this.block.isDisabled = true;
            this.$.dialog.message(
                'error',
                'Error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__default_stock_site_error',
                    'Define a default stock site on the user function profile.',
                ),
            );
            return (this.block.isDisabled = true);
        }
    }

    setEntryTransaction(transaction: any[], transactionIsDisabled: boolean, allFieldsDisabled: boolean) {
        const transactionOptions: string[] = transaction.map(transactions => transactions.transaction);
        this.transaction.options = transactionOptions;
        this.transaction.value = transaction[0].transaction;
        this.transaction.isDisabled = transactionIsDisabled;
        this.employeeId.isDisabled = allFieldsDisabled;
    }

    /**
     * Is the employee id mandatory
     * @param transactions
     * @param currentTransaction
     * @returns boolean
     */
    async isEmployeeMandatory(transactions: any[], currentTransaction: string | undefined | null): Promise<boolean> {
        const transactionIndex = transactions.findIndex(transaction => transaction.transaction === currentTransaction);

        switch (transactions[transactionIndex].employeeEntry) {
            case 'mandatory':
                await this.setDefaultEmployee();
                this.workOrder.isDisabled = !this.employeeId.value;
                !this.employeeId.value ? (this.workOrder.isDisabled = true) : (this.workOrder.isDisabled = false);
                return true;

            case 'optional':
                await this.setDefaultEmployee();
                return false;

            case 'prohibited':
                return false;

            default:
                return false;
        }
    }

    /**
     * Is the employee id fiedls to be hidden
     * @param transactions
     * @param currentTransaction
     * @returns boolean
     */
    async isEmployeeHidden(transactions: any[], currentTransaction: string | undefined | null): Promise<boolean> {
        const transaction = transactions.find(transaction => transaction.transaction === currentTransaction);

        if (transaction.employeeEntry === 'prohibited') {
            this.employeeId.value = null;
            return true;
        }

        return false;
    }

    /** Function returns whether unplanned operations checkbox is hidden based on unplanned operation flag
        in the entry transaction - True | False */
    async isUnplannedOperationsHidden(
        transactions: any[],
        currentTransaction: string | undefined | null,
    ): Promise<boolean> {
        const transactionIndex: number = transactions.findIndex(
            transaction => transaction.transaction === currentTransaction,
        );
        let isUnplannedOperationHiddenFlag: boolean = true;

        if (transactions[transactionIndex].unplannedOperations === true) {
            isUnplannedOperationHiddenFlag = false;
        } else {
            this.unplannedOperations.value = false;
        }

        return isUnplannedOperationHiddenFlag;
    }

    getLines(): TimeTrackingLine[] | null | undefined {
        const tracklingLines = this.storageObject?.timeTracking.workOrderOperationTrackingLines;
        if ((tracklingLines?.length ?? 0) > 0) {
            return tracklingLines;
        }
    }

    getLinesLength(): number {
        const trackingLines = this.getLines();
        if (trackingLines) {
            return trackingLines.length;
        }
        return 0;
    }

    async setStorageObject() {
        this.initStorageObject();
        this.getSessionStorage();

        const linesLength = this.getLinesLength();
        const lines = this.getLines();

        if (linesLength) {
            this.transaction.isDisabled = true;
            this.transaction.value = this.storageObject.transaction;
            this.workOrder.isDisabled = false;
            this.trackingDate.value = this.storageObject.trackingDate;
            this.setCurrentEmployee();
            this.employeeId.value ? (this.employeeId.isDisabled = true) : (this.employeeId.isHidden = true);
            this.unplannedOperations.isHidden = await this.isUnplannedOperationsHidden(
                this.transactionsList,
                this.transaction.value,
            );
            this.setCurrentWorkOrder();
            operationList = await getWorkOrderOperationsList(this.workOrder.value?.number, this);
            this.releaseUOM.value = await this.getReleaseUOM();
            this.releaseTimeUnit.value = await this.getReleaseTimeUnit();
            this.setOperationFieldsDisabledStatus();
            this.trackingLinesBlock.isHidden = this.isTableHidden(lines);
            this.toggleCreateButton(linesLength);
        }
    }

    setCurrentWorkOrder(): Promise<WorkOrder> {
        this.workOrder.isDirty = !this.workOrder.isDirty;
        return (this.workOrder.value = this.storageObject.workOrderNumber);
    }

    setCurrentEmployee(): EmployeeId {
        return (this.employeeId.value = this.storageObject.employeeId);
    }

    setOperationFieldsDisabledStatus(): void {
        if (this.workOrder.value) {
            this.unplannedOperations.isDisabled = false;
            this.operationLine.isDisabled = false;
            this.unplannedOperationsLine.isDisabled = false;
            this.stdOperation.isDisabled = false;
            this.stdOperationDescription.isDisabled = false;
            this.unplannedOperations.isHidden === true ? this.operationLine.focus() : this.unplannedOperations.focus();
        } else {
            this.unplannedOperations.isDisabled = true;
            this.unplannedOperations.value = false;
            this.operationLine.isDisabled = true;
            this.operationLine.value = null;
            this.unplannedOperationsLine.isDisabled = true;
            this.unplannedOperationsLine.value = null;
            this.stdOperation.isDisabled = true;
            this.stdOperation.value = null;
            this.stdOperationDescription.isDisabled = true;
            this.stdOperationDescription.value = null;
        }
    }

    isUnplannedOperationsEnabled(): void {
        if (this.unplannedOperations.value === true) {
            this.operationLine.isHidden = true;
            this.operationLine.value = null;
            this.unplannedOperationsLine.isHidden = false;
            this.unplannedOperationsLine.isDisabled = false;
            this.stdOperation.isHidden = false;
            this.stdOperation.isDisabled = false;
            this.stdOperationDescription.isHidden = false;
            this.stdOperationDescription.isDisabled = false;
        } else if (this.unplannedOperations.value === false) {
            this.operationLine.isHidden = false;
            this.operationLine.isDisabled = false;
            this.unplannedOperationsLine.isHidden = true;
            this.unplannedOperationsLine.value = null;
            this.stdOperation.isHidden = true;
            this.stdOperation.value = null;
            this.stdOperationDescription.isHidden = true;
            this.stdOperationDescription.value = null;
        }
    }

    isTableHidden(lines: TimeTrackingLine[]): boolean {
        let tableIsHidden: boolean = true;

        if (lines) {
            this.setTableTitle(lines);
            this.setTableLines(lines);
            tableIsHidden = false;
        }

        return tableIsHidden;
    }

    setTableTitle(lines: TimeTrackingLine[], tableLineDelete?: boolean): string {
        const linesCounter: number = lines?.length;
        if (tableLineDelete) {
            this.trackingLinesTable.title = this.trackingLinesTable.title?.split(':')[0];
        }
        return (this.trackingLinesTable.title = `${this.trackingLinesTable.title}: ${linesCounter}`);
    }

    setTableLines(lines: TimeTrackingLine[]): Partial<TrackingLineTable>[] {
        return (this.trackingLinesTable.value = this.mapTableLines(lines));
    }

    mapTableLines(lines: TimeTrackingLine[]) {
        return lines.map((line: TimeTrackingLine, index) => {
            const {
                workOrderNumber,
                operationNumber,
                description,
                totalCompletedQuantity,
                operationUnit,
            }: TimeTrackingLine = line;
            return {
                _id: `${index}`,
                workOrder: workOrderNumber,
                operationNumber: operationNumber,
                operationDescription: description,
                quantityAndStockUnit: `${totalCompletedQuantity} ${operationUnit}`,
            };
        });
    }

    toggleCreateButton(linesLength: number): boolean {
        if (linesLength > 0) {
            return (this.createButton.isDisabled = false);
        } else {
            return (this.createButton.isDisabled = true);
        }
    }

    private storageObject: TimeTrackingStorageObject;

    initStorageObject() {
        this.storageObject = {
            userDefinedOperationNumber: 0,
            operationMethod: 0,
            releaseTimeUnit: '',
            releaseUOM: '',
            transaction: '',
            employeeId: '',
            trackingDate: '',
            workOrderNumber: '',
            operationLine: {},
            timeTracking: {
                workOrderOperationTrackingLines: new Array<TimeTrackingLine>(),
            },
        } as TimeTrackingStorageObject;
        const gotLines = this.$.queryParameters['gotLines'];
        if (gotLines != 'true') this.setSessionStorage(this.storageObject);
    }

    getSessionStorage(): TimeTrackingStorageObject {
        return (this.storageObject = this.$.storage?.get('time_tracking') as unknown as TimeTrackingStorageObject);
    }

    updatePageStorage(operationMethod: SelectedOperationMethod): void {
        this.storageObject.userDefinedOperationNumber = this.unplannedOperationsLine.value;
        this.storageObject.userDefinedOperationDescription = this.stdOperationDescription.value;
        this.storageObject.operationMethod = operationMethod;
        this.storageObject.releaseTimeUnit = this.releaseTimeUnit.value;
        this.storageObject.releaseUOM = this.releaseUOM.value;
        this.storageObject.transaction = this.transaction.value;
        this.storageObject.employeeId = this.employeeId.value;
        this.storageObject.trackingDate = this.trackingDate.value;
        this.storageObject.workOrderNumber = this.workOrder.value;
        this.storageObject.operationLine = this.operationLine.value;
        this.storageObject.standardOperations = this.stdOperation.value;
    }

    setSessionStorage(data = this.storageObject) {
        this.$.storage.set('time_tracking', { ...data } as any);
    }

    async getStockSite(): Promise<string> {
        try {
            const site = this.$.storage.get('mobile-selected-stock-site') as string;
            return site;
        } catch (error) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_completed_quantity___stock_site_error',
                    'The stock site is not defined',
                ),
                { type: 'error', timeout: 5000 },
            );
            return '';
        }
    }

    async getDate(): Promise<string> {
        return DateValue.today().toString();
    }

    async setDefaultEmployee(): Promise<void> {
        try {
            const employeeIDList = extractEdges<EmployeeId>(
                await this.$.graph
                    .node('@sage/x3-master-data/EmployeeId')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                teamId: true,
                            },
                            {
                                filter: {
                                    isActive: true,
                                    userCode: this.$.userCode,
                                },
                            },
                        ),
                    )
                    .execute(),
            );

            if (employeeIDList) {
                this.employeeId.value = employeeIDList[0];
            } else {
                this.employeeId.value = null;
            }
        } catch (err) {}
    }

    async getEntryTransaction(): Promise<any[]> {
        try {
            const transactions = extractEdges<TransactionProductionReporting>(
                await this.$.graph
                    .node('@sage/x3-manufacturing/TransactionProductionReporting')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                transaction: true,
                                employeeEntry: true,
                                unplannedOperations: true,
                            },
                            {
                                filter: {
                                    type: 60,
                                    isActive: true,
                                },
                            },
                        ),
                    )
                    .execute(),
            ) as TransactionProductionReporting[];

            transactions.length === 1 ? (this.transaction.isHidden = true) : (this.transaction.isHidden = false);

            return transactions;
        } catch (err) {
            return [];
        }
    }

    async getWorkOrderRoutingList(): Promise<ExtractEdges<RoutingHeader>[]> {
        try {
            const woRoutingList = extractEdges<RoutingHeader>(
                await this.$.graph
                    .node('@sage/x3-manufacturing/RoutingHeader')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                timeUnit: true,
                            },
                            {
                                filter: {
                                    routing: { code: this.workOrder.value?.releasedRouting.routing.code },
                                    routingCode: { code: this.workOrder.value?.routingCode.code },
                                    site: this.stockSite.value,
                                },
                                first: 1000,
                            },
                        ),
                    )
                    .execute(),
            );
            return woRoutingList;
        } catch (err) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_time_tracking___cannot_retrieve_work_order_routing',
                    'Error while loading the work order routing.',
                ),
                { type: 'error', timeout: 5000 },
            );
            return [];
        }
    }

    /** function to get value for paramter CTLOPEPRE */
    private async getGeneralParameters() {
        /* CTLOPEPRE - Expected values
           1 No. 2 Yes without block, 3 Yes, with block */

        CtlOpepre = await readParameterValue(
            'CTLOPEPRE',
            this.$.userCode ? this.$.userCode.toUpperCase() : '',
            '',
            this,
        );
    }

    /** function to test if operation can be tracked - tracking is only permitted if previous operation
     *  has already been tracked and exists X3 or has been added to the grid pending creation */
    async isPreviousOperationTracked(
        operationNumber: number,
        workOrder: any,
        operationMilestonetype: string | undefined,
    ): Promise<boolean> {
        // If this is the first operation allow user to track as no previous operation will exist
        if (operationMilestonetype === 'normalTracking') {
            if ((await this.getPreviousOperation(operationNumber, workOrder)) === undefined) {
                return true;
            }
        }

        if (operationMilestonetype === 'range') {
            // If this is the first operation allow user to track as no previous operation will exist
            if ((await this.getPreviousOperation(operationNumber, workOrder)) === undefined) {
                return true;
            }

            let startFromOperation: number | undefined = operationNumber;
            let previousOperation: number | undefined;
            let result: string | undefined;

            do {
                previousOperation = await this.getPreviousOperation(startFromOperation, workOrder);
                // If the first operation in the routing has been reached and no
                // previous operation exists that can be tracked
                if (previousOperation === undefined) {
                    previousOperation = startFromOperation;
                    result = await this.getOperationMilestoneType(previousOperation, workOrder);
                    break;
                }
                result = await this.getOperationMilestoneType(previousOperation, workOrder);
                startFromOperation = previousOperation;
            } while (result === 'none');

            operationNumber = previousOperation;
            // When the first operation in the routing has been reached and type is = none
            if (result === 'none') {
                return true;
            }
        }

        // Test if previous operation has already been already been tracked and exists in X3
        const allowTracking = await this.isPreviousOperationTrackedDB(operationNumber, operationList);
        if (allowTracking === true) {
            return true;
        }

        // if operatrion not in X3 then test if operation has been added to mobile trackuing grid waiting to be processed
        if (allowTracking === false) {
            if (operationMilestonetype === 'normalTracking') {
                return await this.isOperationOnTrackingGrid(
                    await this.getPreviousOperation(operationNumber, workOrder),
                    workOrder,
                );
            }
            if (operationMilestonetype === 'range') {
                return await this.isOperationOnTrackingGrid(operationNumber, workOrder);
            }
        }
        return false;
    }

    /** function to test if an operation number exists on the mobile page 1 tracking grid */
    async isOperationOnTrackingGrid(operationNumber: number | undefined, workOrderNumber: number): Promise<boolean> {
        const trackingLines: TimeTrackingLine[] | null | undefined = this.getLines();

        if (trackingLines === undefined || trackingLines === null) {
            return false;
        }

        if (trackingLines.find(ope => ope.operation === operationNumber && ope.workOrderNumber === workOrderNumber)) {
            return true;
        }

        return false;
    }

    /** function to test if the previous operation for the current operation has been tracking in X3 */
    async isPreviousOperationTrackedDB(
        opNumber: number,
        opList: ExtractEdges<WorkOrderOperationLine>[],
    ): Promise<boolean | undefined> {
        try {
            let result: boolean = false;
            for (let i = 0; i <= opList.length - 1; i++) {
                if (opList[i]?.operationNumber === opNumber) {
                    if (opList[i]?.operationStatus !== 'pending') {
                        result = true;
                        break;
                    }
                    if (opList[i]?.operationStatus === 'pending' && i !== 0) {
                        break;
                    }
                }
            }
            return result;
        } catch (err) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_time_tracking___operation_tracking_status_error',
                    'Unable to check operation tracking status',
                ),
                { type: 'error', timeout: 5000 },
            );
            return undefined;
        }
    }

    /** Function prepare creation - remove the description from the storage lines */
    prepareCreation(): void {
        const mutationObject = this.storageObject.timeTracking.workOrderOperationTrackingLines;
        mutationObject?.forEach(line => delete line.description);
    }

    /** Function to call the API for creation */
    async callCreationAPI(): Promise<X3Response> {
        let result: X3Response;
        const inputData = { productionSite: this.stockSite.value, ...this.storageObject.timeTracking };
        const payload = `
        mutation {
            x3Manufacturing {
                workOrderOperationTracking {
                    create(data: ${ui.queryUtils.serializeToGraphQL(inputData)}){
                        trackingNumber
                    }
                }
            }
        }
    `;
        try {
            result = await this.$.graph.raw(payload, true);
            if (!result) {
                throw Error(
                    ui.localize(
                        '@sage/x3-manufacturing/pages__mobile_time_tracking___no_results',
                        'No results received for the creation',
                    ),
                );
            }
        } catch (error) {
            return error;
        }
        return result;
    }

    /** Create Button Methods */

    /**
     * Checks the response from X3 for errors. Controls the display success error screen.
     * First it handles a special case when in X3response:
     * severities 3 exists (hard fail),
     * workOrderProductionReporting.create.trackingNumber = null,
     * It Looks for severity 3 and 4, when none found Counts as success.
     * The rest is just an error.
     * @param {*} result The response from X3.
     * @return {*}  {boolean} True for win. False for fail.
     */

    getCreationStatus(result: X3Response): boolean {
        if (!!this.getTrackingNumber(result)) {
            return true;
        }
        if (this.checkForErrorSeverities(result)) {
            return true;
        }
        return false;
    }

    checkForErrorSeverities(result: X3Response | any): boolean {
        if (
            result.diagnoses.filter((diagnoses: { severity: number; message: string }) => {
                diagnoses.severity > 2 && diagnoses.message;
            }).length !== 0 ||
            result.message
        ) {
            return false;
        } else {
            return true;
        }
    }

    async errorScreen(result: X3Response): Promise<void> {
        await this.$.dialog
            .confirmation(
                'error',
                'Error',
                `${ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_time_tracking__creation_error',
                    'An error occurred in {{documentId}}.',
                    {
                        documentId: await formatImportMessagefile(result),
                    },
                )}`,
                this.errorDialogOption(),
            )
            .catch(() => {
                this.resetSessionStorage();
            });
    }

    async successScreen(result: X3Response): Promise<void> {
        await this.$.dialog
            .message(
                'info',
                'Success',
                `${ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_time_tracking__creation_success',
                    'Time tracking {{documentId}} created.{{errorMessages}}',
                    {
                        documentId: this.getTrackingNumber(result),
                        errorMessages: await formatImportMessagefile(result),
                    },
                )}`,
                this.successDialogOption(),
            )
            .finally(() => this.resetSessionStorage());
    }

    getTrackingNumber(result: X3Response) {
        const trackingNumber = result.x3Manufacturing?.workOrderOperationTracking.create?.trackingNumber;
        return trackingNumber;
    }

    /** Function to set gotLines to false will cause initPageStorageObject to reset the page and storage object */
    resetSessionStorage(): void {
        this.$.router.goTo('@sage/x3-manufacturing/MobileTimeTracking', {
            gotLines: 'false',
        });
    }

    infoDialogOption(): ui.dialogs.DialogOptions {
        const options = {
            acceptButton: {
                text: 'Yes',
            },
            cancelButton: {
                text: 'No',
            },
        };
        return options;
    }

    successDialogOption(): ui.dialogs.DialogOptions {
        const options = {
            acceptButton: {
                text: 'OK',
            },
        };
        return options;
    }

    errorDialogOption(): ui.dialogs.DialogOptions {
        const options = {
            acceptButton: {
                text: 'Back',
            },
            cancelButton: {
                text: 'Reset',
            },
        };
        return options;
    }

    /** function to return the milestone type for an operation */
    async getOperationMilestoneType(operation: number, woNumber: string): Promise<string | undefined> {
        try {
            const milestoneType = extractEdges<WorkOrderScheduling>(
                await this.$.graph
                    .node('@sage/x3-manufacturing/WorkOrderScheduling')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                milestone: true,
                            },
                            {
                                filter: {
                                    operationNumber: operation,
                                    workOrder: woNumber,
                                },
                            },
                        ),
                    )
                    .execute(),
            );

            return milestoneType[0].milestone;
        } catch (err) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_time_tracking___operation_milestone_type_error',
                    'Error while checking the operation milestone type.',
                ),
                { type: 'error', timeout: 5000 },
            );
            return undefined;
        }
    }

    /** Function to get release time unit for selected work order **/
    async getReleaseTimeUnit(): Promise<string> {
        try {
            const result = extractEdges<RoutingHeader>(
                await this.$.graph
                    .node('@sage/x3-manufacturing/RoutingHeader')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                timeUnit: true,
                            },
                            {
                                filter: {
                                    routing: { code: this.workOrder.value?.releasedRouting.routing.code },
                                    routingCode: { code: this.workOrder.value?.routingCode.code },
                                    site: this.stockSite.value,
                                },
                                first: 1000,
                            },
                        ),
                    )
                    .execute(),
            );

            return result[0].timeUnit;
        } catch (err) {
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_time_tracking___loading_rel_time_unit_code_error',
                    'Error while loading the release time unit.',
                ),
                { type: 'error', timeout: 5000 },
            );
            return '';
        }
    }

    async getReleaseUOM(): Promise<string> {
        try {
            const result = extractEdges<WorkOrderProductLine>(
                await this.$.graph
                    .node('@sage/x3-manufacturing/WorkOrderProductLine')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                releaseUnit: { code: true },
                            },
                            {
                                filter: {
                                    workOrder: this.workOrder.value?.number,
                                },
                            },
                        ),
                    )

                    .execute(),
            );
            return result[0].releaseUnit.code.toString();
        } catch (err) {
            this.$.dialog.message(
                'error',
                ui.localize(
                    '@sage/x3-manufacturing/pages__mobile_completed_time_tracking___loading_rel_BOM_code_error',
                    'Error while loading the release unit.',
                ),
                String(err),
            );
            return '';
        }
    }

    /** Function to return the previous operation number */
    async getPreviousOperation(
        operation: number | undefined,
        woNumber: string | undefined,
    ): Promise<number | undefined> {
        try {
            const previousOperation = extractEdges<WorkOrderScheduling>(
                await this.$.graph
                    .node('@sage/x3-manufacturing/WorkOrderScheduling')
                    .query(
                        ui.queryUtils.edgesSelector(
                            {
                                operationNumber: true,
                            },
                            {
                                filter: {
                                    downstreamOperationNumber: operation,
                                    workOrder: woNumber,
                                },
                            },
                        ),
                    )
                    .execute(),
            );

            return previousOperation[0].operationNumber;
        } catch (err) {
            return undefined;
        }
    }

    // TODO NEED REFACTOR AND MOVING TO CLIENT FUNCTION - PARAMETER READ LOGIN NEED IMPROVING
    // TODO ADD ERROR MESSAGE WAITING UNTIL TRANSLATION UTIL HAS BEEN RUN
    /**
     * Read Parameter Value
     *
     * @param parameterName
     * @param site
     * @returns
     */
    async getParamValue(site: string | null, parameterName: string): Promise<string> {
        let payload = '';
        payload = `
          mutation {
              x3System {
               generalParametersData {
                readParameterValueString(
                parameters: {company: "", site: "${site}", code: "${parameterName}"})
                {
                value
                }
                }}}
           `;
        try {
            const response = await this.$.graph.raw(payload, true);
            return response.x3System.generalParametersData.readParameterValueString.value;
        } catch (error) {
            return '';
        }
    }
}
