import { GraphApi } from '@sage/x3-manufacturing-api';
import { SerialNumber, Stock, StockManagementRules } from '@sage/x3-stock-data-api';
import { getRegExp } from '@sage/x3-system/build/lib/shared-functions/pat-converter';
import { Filter, extractEdges, integer } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import { GetUOMNumberOfDecimalPlaces } from '../client-functions/get-unit-decimal-places';
import {
    MaterialsConsumptionDetailsRecordManager,
    MaterialsConsumptionSession,
} from '../client-functions/materials-consumption-details-control';
import { findStockManagementRules } from '../client-functions/stock-management-rules';
import { ProductSiteData } from './utils/product-site-data';
interface EntryTransaction {
    type: integer;
    transaction: string;
    isPreloading: boolean;
}

const EntryTransaction: EntryTransaction = {
    type: 61,
    transaction: '',
    isPreloading: false,
};
interface ProductSetting {
    displayLot: boolean;
    displaySublot: boolean;
    displaySerial: boolean;
    displayLicensePlate: boolean;
    displayLocation: boolean;
    optionalLot: boolean;
}

@ui.decorators.page<MobileMaterialsConsumptionDetails>({
    title: 'Material consumption details',
    subtitle: 'Enter details',
    module: 'x3-manufacturing',
    mode: 'default',
    isTransient: true,
    isTitleHidden: true,
    headerCard() {
        return {
            title: this.product,
            line2: this.description,
            titleRight: this.titleTextRight,
            image: this.image,
        };
    },
    businessActions() {
        return [this.addDetails, this.addProduct];
    },
    async onLoad() {
        this.$.removeToasts();
        this._materialsConsumptionRecordManager = new MaterialsConsumptionDetailsRecordManager(this);

        this._materialsConsumptionSession = this._materialsConsumptionRecordManager.materialsConsumptionSession;
        const workOrder = this._materialsConsumptionSession.workOrder;
        if (!this._materialsConsumptionSession) {
            this.addDetails.isHidden = true;
            this.addProduct.isHidden = true;
            this.blockProduct.isHidden = true;
            this.licensePlateNumber.isHidden = true;
            this.location.isHidden = true;
            return;
        }

        const productSetting: ProductSetting = await this.getProductSettings();
        if (!productSetting.displayLot) {
            this.lot.isHidden = true;
        }
        if (productSetting.optionalLot) {
            this.lot.isMandatory = false;
        }
        if (!productSetting.displaySublot) {
            this.sublot.isHidden = true;
        }
        if (!productSetting.displaySerial) {
            this.serialNumber.isHidden = true;
        }
        if (!productSetting.displayLicensePlate) {
            this.licensePlateNumber.isHidden = true;
        }
        if (!productSetting.displayLocation) {
            this.location.isHidden = true;
        }

        this.unit.isDisabled = true;

        this._initHeader();

        EntryTransaction.transaction = this._materialsConsumptionSession.transaction;
        const preloadQty = await this.getEntryTransactionSettings(EntryTransaction);
        if (preloadQty) {
            this._calQuantity();
        } else {
            this.quantity.value = 0;
            this.quantity.scale = await GetUOMNumberOfDecimalPlaces(this, this.unit.value);
        }
    },
})
export class MobileMaterialsConsumptionDetails extends ui.Page<GraphApi> {
    private _materialsConsumptionRecordManager: MaterialsConsumptionDetailsRecordManager; // to store the receipt to create from session data
    private _materialsConsumptionSession: MaterialsConsumptionSession;
    private _materialsConsumptionStockManagementRules: StockManagementRules;

    private async getProductSettings(): Promise<ProductSetting> {
        const response = await this.$.graph
            .node('@sage/x3-master-data/Product')
            .query(
                ui.queryUtils.edgesSelector(
                    {
                        lotManagementMode: true,
                        stockManagementMode: true,
                        serialNumberManagementMode: true,
                        stockUnit: {
                            _id: true,
                            code: true,
                        },
                    },
                    {
                        filter: {
                            code: this._materialsConsumptionSession.productCode,
                        },
                    },
                ),
            )
            .execute();

        let productSetting: ProductSetting = {
            displayLot: true,
            displaySublot: true,
            displaySerial: true,
            displayLicensePlate: true,
            displayLocation: true,
            optionalLot: false,
        };

        for (let item of response.edges) {
            if (item.node.lotManagementMode == 'notManaged') {
                productSetting.displayLot = false;
            }
            if (item.node.lotManagementMode != 'lotAndSublot') {
                productSetting.displaySublot = false;
            }
            if (item.node.lotManagementMode == 'optionalLot') {
                productSetting.optionalLot = true;
            }

            if (item.node.serialNumberManagementMode == 'notManaged') {
                productSetting.displaySerial = false;
            }
        }

        //read stock rules management
        let statuses: string[];
        const selectedStatus: { _regex: string }[] = [];
        const productSiteData = new ProductSiteData(
            this._materialsConsumptionSession.productCode,
            this._materialsConsumptionSession.materialsConsumptionSite,
            this,
        );
        const productSitePromise = productSiteData.productSite;
        const productSite = await productSitePromise;

        if (productSite && productSite.product) {
            productSetting.displayLicensePlate = productSite.isLicensePlateNumberManaged;
            productSetting.displayLocation = productSite.isLocationManaged;

            this._materialsConsumptionStockManagementRules = await findStockManagementRules(
                this._materialsConsumptionSession.materialsConsumptionSite,
                productSite.product.productCategory.code,
                '6', // workOrderIssue
                null,
                this,
            );

            this._materialsConsumptionStockManagementRules.authorizedSubstatus.split(',').forEach(function (status) {
                selectedStatus.push({ _regex: getRegExp(status).source });
            });
            try {
                statuses = extractEdges<{ code: string }>(
                    await this.$.graph
                        .node('@sage/x3-stock-data/StockStatus')
                        .query(
                            ui.queryUtils.edgesSelector(
                                {
                                    code: true,
                                },
                                {
                                    filter: {
                                        code: { _or: selectedStatus },
                                    },
                                },
                            ),
                        )
                        .execute(),
                ).map(status => status.code);
            } catch (e) {
                this.$.dialog.message('error', 'Error loading stock statuses', String(e));
            }
            this.status.options = statuses;
            this.unit.value = response.edges[0].node.stockUnit.code;
        }

        return productSetting;
    }

    @ui.decorators.textField<MobileMaterialsConsumptionDetails>({
        isTransient: true,
    })
    titleTextRight: ui.fields.Text;

    @ui.decorators.textField<MobileMaterialsConsumptionDetails>({
        title: 'Product',
        isReadOnly: true,
        size: 'large',
    })
    product: ui.fields.Text;

    @ui.decorators.textField<MobileMaterialsConsumptionDetails>({
        title: 'Description',
        isReadOnly: true,
        size: 'small',
    })
    description: ui.fields.Text;

    @ui.decorators.imageField<MobileMaterialsConsumptionDetails>({
        isTransient: true,
        isReadOnly: true,
    })
    image: ui.fields.Image;

    /*
     *  Page Actions
     */

    @ui.decorators.pageAction<MobileMaterialsConsumptionDetails>({
        title: 'Add quantity',
        buttonType: 'secondary',
        shortcut: ['f7'],
        async onClick() {
            if (await this._validate()) {
                this._materialsConsumptionRecordManager.loadMaterialsConsumptionDetails(this);
                this.$.showToast('Quantity added', { type: 'success', timeout: 3000 });
                this.$.setPageClean();
                if (this.quantity.value == 0) {
                    this.addDetails.isDisabled = true;
                    this.$.router.goTo('@sage/x3-manufacturing/MobileMaterialsConsumption', {
                        returnFromDetailsPage: 'true',
                    });
                }
                // Fields to be initialized
                this.location.value = null;
                this.lot.value = null;
                this.sublot.value = null;
                this.licensePlateNumber.value = null;
                this.serialNumber.value = null;
                this._calQuantity();
                //return;
            }
        },
    })
    addDetails: ui.PageAction;

    @ui.decorators.pageAction<MobileMaterialsConsumptionDetails>({
        title: 'Next',
        buttonType: 'primary',
        shortcut: ['f3'],
        async onClick() {
            this.addProduct.isDisabled = true;
            if (await this._validate()) {
                this._materialsConsumptionRecordManager.loadMaterialsConsumptionDetails(this);
                this.$.setPageClean();
                this.$.router.goTo('@sage/x3-manufacturing/MobileMaterialsConsumption', {
                    returnFromDetailsPage: 'true',
                });
                return;
            }
            this.addProduct.isDisabled = false;
        },
    })
    addProduct: ui.PageAction;

    @ui.decorators.section<MobileMaterialsConsumptionDetails>({
        isTitleHidden: true,
    })
    sectionHeader: ui.containers.Section;

    @ui.decorators.block<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.sectionHeader;
        },
        isTitleHidden: true,
    })
    blockProduct: ui.containers.Block;

    /*
     *  Fields
     */

    @ui.decorators.textField<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.blockProduct;
        },
        title: 'Unit',
        isMandatory: true,
    })
    unit: ui.fields.Text;

    @ui.decorators.numericField<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.blockProduct;
        },
        title: 'Quantity',
        isMandatory: true,
        validation: /^([1-9][0-9]*(\.[0-9]+)?|[0]+\.[0-9]*[1-9][0-9]*)$/, // reg ex for any positive numbers (integers or decimals) excluding 0
        min: 0,
        async onChange() {
            let totalQte = await this._getTotalConsumedQTY();
            totalQte += Number(this.quantity.value);
            let requiredQte = Number(this._materialsConsumptionSession.product.requiredQuantity || 0);
            if (!this.serialNumber.isHidden && Number(this.quantity.value || 0) > 1) {
                this.$.showToast(
                    ui.localize(
                        '@sage/x3-manufacturing/pages__material-consumption-details___quantity__larger__expected__warning',
                        'Quantity must be 1 for serial number.',
                    ),
                    { type: 'warning', timeout: 5000 },
                );
                totalQte = totalQte + 1 - Number(this.quantity.value);
                this.quantity.value = 1;
                this.quantity.getNextField(true).focus();
            } else if (this._materialsConsumptionSession && this._materialsConsumptionSession.product) {
                if (totalQte > requiredQte) {
                    this.$.showToast(
                        ui.localize(
                            '@sage/x3-manufacturing/pages__material-consumption-details___quantity__serial__warning',
                            'The quantity is larger than expected.',
                        ),
                        { type: 'warning', timeout: 5000 },
                    );
                }
            }
            if (totalQte == requiredQte) {
                this.closeWoLine.value = true; // Yes
            } else if (totalQte < requiredQte) {
                this.closeWoLine.value = false; // No
            }
        },
    })
    quantity: ui.fields.Numeric;

    @ui.decorators.selectField<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.blockProduct;
        },
        title: 'Status',
        isMandatory: true,
        async onChange() {
            await this.setLocations();
            await this.setLicensePlateNumbers();
            this.status.getNextField(true)?.focus();
        },
    })
    status: ui.fields.Select;

    @ui.decorators.referenceField<MobileMaterialsConsumptionDetails, SerialNumber>({
        parent() {
            return this.blockProduct;
        },
        title: 'Serial number',
        width: 'large',
        placeholder: 'Scan or select...',
        node: '@sage/x3-stock-data/SerialNumber',
        valueField: 'code',
        isMandatory: true,
        filter() {
            return {
                product: { _id: this._materialsConsumptionSession.productCode },
                stockSite: { code: this._materialsConsumptionSession.materialsConsumptionSite },
                issueDate: { _eq: null }, // ignore if already issued
            };
        },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                title: 'Serial number',
            }),
        ],
    })
    serialNumber: ui.fields.Reference;

    @ui.decorators.referenceField<MobileMaterialsConsumptionDetails, Stock>({
        parent() {
            return this.blockProduct;
        },
        title: 'Lot',
        isMandatory: true,
        isAutoSelectEnabled: true,
        placeholder: 'Scan or select…',
        node: '@sage/x3-stock-data/Stock',
        valueField: 'lot',

        filter() {
            const filter: Filter<Stock> = {
                stockSite: { code: this._materialsConsumptionSession.materialsConsumptionSite },
                product: { product: { code: this._materialsConsumptionSession.productCode } },
                isBeingCounted: false,
                availableQuantity: { _gt: '0' },
                status: { code: { _eq: this.status.value } },
            };
            return filter;
        },
        async onChange() {
            if (this.lot.value) {
                let totalLotQty = this.lot.value.availableQuantity;
                totalLotQty -= await this._getTotalLotConsumedQTY();
                if (Number(this.quantity.value) > totalLotQty) {
                    this.$.showToast(
                        ui.localize(
                            '@sage/x3-manufacturing/pages__material-consumption-details___quantity__greater__than__lot__quantity',
                            `Quantity ${this.quantity.value} is greater than Lot available quantity ${totalLotQty}. \n Please change Lot or quantity.`,
                        ),
                        { type: 'error', timeout: 5000 },
                    );
                    this.lot.value = null;
                } else {
                    this.sublot.options = await this._getSubLot();
                    this.sublot.value = this.lot.value.sublot;
                    await this.setLocations();
                    await this.setLicensePlateNumbers();
                    this.lot.getNextField(true)?.focus();
                }
            }
        },
        columns: [
            ui.nestedFields.text({
                bind: 'lot',
                title: 'Lot',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'sublot',
                title: 'Sublot',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'availableQuantity',
                title: 'Available quantity',
                isHidden: false,
            }),
        ],
    })
    lot: ui.fields.Reference;

    @ui.decorators.selectField<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.blockProduct;
        },
        title: 'Sublot',
        placeholder: 'Scan or select...',
        isMandatory: true,
        async onChange() {
            this.sublot.getNextField(true)?.focus();
        },
    })
    sublot: ui.fields.Select;

    /** Location */

    @ui.decorators.selectField<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.blockProduct;
        },
        title: 'Location',
        placeholder: 'Scan or select...',
        isMandatory: true,
        isFullWidth: true,
        async onChange() {
            await this.setLicensePlateNumbers();
            this.location.getNextField();
        },
    })
    location: ui.fields.Select;

    async getLocations(): Promise<string[]> {
        const response = await this.$.graph
            .node('@sage/x3-stock-data/Stock')
            .query(
                ui.queryUtils.edgesSelector(
                    {
                        location: { code: true },
                    },
                    {
                        filter: {
                            stockSite: { code: this._materialsConsumptionSession.materialsConsumptionSite },
                            product: { _id: this._materialsConsumptionSession.productCode },
                            availableQuantity: { _gt: '0' },
                            status: { code: { _eq: this.status.value } },
                            lot: { _eq: this.lot.value?.lot },
                            sublot: { _eq: this.sublot?.value },
                            location: {
                                category: {
                                    _in: ['internal', 'subcontract'],
                                },
                                isBeingCounted: {
                                    _eq: false,
                                },
                            },
                        },
                    },
                ),
            )
            .execute();

        const locations = response.edges.map(edge => edge.node.location.code);
        const uniqueLocations = [...new Set(locations)];

        return uniqueLocations as string[];
    }

    async setLocations(): Promise<void> {
        this.location.options = await this.getLocations();
        if (this.location.options.length === 1) {
            this.location.value = this.location.options[0];
        }
    }

    /** Licence Plate Number */

    @ui.decorators.selectField<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.blockProduct;
        },
        title: 'License Plate Number',
        placeholder: 'Scan or select...',
        isFullWidth: true,
        isDisabled: true,
        async onChange() {
            this.licensePlateNumber.getNextField();
        },
    })
    licensePlateNumber: ui.fields.Select;

    async getLicensePlateNumbers(_excludeBlank: boolean): Promise<string[]> {
        let lpnFilter: any;
        let locFilter: any;
        let lotFilter: any;
        let sublotFilter: any;

        const baseFilter = {
            stockSite: { code: this._materialsConsumptionSession.materialsConsumptionSite },
            product: { _id: this._materialsConsumptionSession.productCode },
            availableQuantity: { _gt: '0' },
            status: { _eq: this.status?.value },
        };

        // add lot filter if exists
        if (this.lot.value != null) {
            lotFilter = {
                ...baseFilter,
                lot: { _eq: this.lot.value.lot },
            };
        } else {
            lotFilter = baseFilter;
        }

        // add sublot filter if exists
        if (this.sublot.value != null) {
            sublotFilter = {
                ...lotFilter,
                sublot: { _eq: this.sublot.value },
            };
        } else {
            sublotFilter = lotFilter;
        }

        // add location filter if exists
        if (this.location.value != null) {
            locFilter = {
                ...sublotFilter,
                location: { _eq: this.location.value },
            };
        } else {
            locFilter = sublotFilter;
        }

        // add filter to exclude blank lpn
        if (_excludeBlank) {
            lpnFilter = {
                ...locFilter,
                licensePlateNumber: { _ne: '' },
            };
        } else {
            lpnFilter = locFilter;
        }

        const response = await this.$.graph
            .node('@sage/x3-stock-data/Stock')
            .query(
                ui.queryUtils.edgesSelector(
                    {
                        licensePlateNumber: { code: true },
                    },
                    {
                        filter: lpnFilter,
                    },
                ),
            )
            .execute();

        const lpns = response.edges.map(edge => edge.node.licensePlateNumber?.code);
        const uniqueLpns = [...new Set(lpns)];
        return uniqueLpns as unknown as string[];
    }

    async setLicensePlateNumbers(): Promise<void> {
        // get LPNs excluding blank lpns
        const lpns = await this.getLicensePlateNumbers(true);
        // get LPNs including blank lpns
        const lpnsIncludeBlank = await this.getLicensePlateNumbers(false);

        if (lpns.length > 0) {
            this.licensePlateNumber.options = lpns;
            this.licensePlateNumber.isDisabled = false;
        } else {
            this.licensePlateNumber.options = [];
            this.licensePlateNumber.isDisabled = true;
        }

        if (lpnsIncludeBlank.length === 1) {
            this.licensePlateNumber.value = lpnsIncludeBlank[0];
        }
    }

    @ui.decorators.checkboxField<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.blockProduct;
        },
        title: 'Close WO',
        isFullWidth: true,
    })
    closeWoLine: ui.fields.Checkbox;

    @ui.decorators.numericField<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.blockProduct;
        },
        title: 'Line number',
        isHidden: true,
    })
    lineNumber: ui.fields.Numeric;

    @ui.decorators.numericField<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.blockProduct;
        },
        title: 'BOM sequence',
        isHidden: true,
    })
    bomSequence: ui.fields.Numeric;

    @ui.decorators.textField<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.blockProduct;
        },
        title: 'BOM number',
        isHidden: true,
    })
    bomNumber: ui.fields.Text;

    @ui.decorators.numericField<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.blockProduct;
        },
        title: 'BOM code',
        isHidden: true,
    })
    bomAlternative: ui.fields.Numeric;

    @ui.decorators.textField<MobileMaterialsConsumptionDetails>({
        parent() {
            return this.blockProduct;
        },
        title: 'Tracking date',
        isHidden: true,
    })
    trackingDate: ui.fields.Text;

    private async _validate(): Promise<boolean> {
        const fieldErrors = new Array<string>();
        if (await this._isFieldInvalid(this.unit)) fieldErrors.push(this.unit.title);
        if (await this._isFieldInvalid(this.quantity)) fieldErrors.push(this.quantity.title);
        if (await this._isFieldInvalid(this.status)) fieldErrors.push(this.status.title);
        if (await this._isFieldInvalid(this.location)) fieldErrors.push(this.location.title);
        if (await this._isFieldInvalid(this.serialNumber)) fieldErrors.push(this.serialNumber.title);
        if (await this._isFieldInvalid(this.lot)) fieldErrors.push(this.lot.title);
        if (await this._isFieldInvalid(this.licensePlateNumber)) fieldErrors.push(this.licensePlateNumber.title);
        if (fieldErrors.length > 0) {
            this.$.removeToasts();
            this.$.showToast(
                ui.localize(
                    '@sage/x3-manufacturing/pages__utils__notification__invalid_inputs_error',
                    `Please check your inputs for {{#each fieldNames}}\n - {{this}}{{/each}}`,
                    { fieldNames: fieldErrors },
                ),
                { type: 'error', timeout: 5000 },
            );
            await this.$.page.validate();
        }

        return fieldErrors.length === 0;
    }

    private async _isFieldInvalid(
        field:
            | ui.fields.FilterSelect
            | ui.fields.Text
            | ui.fields.Reference
            | ui.fields.Numeric
            | ui.fields.Select
            | ui.fields.Date,
    ): Promise<boolean> {
        return !field.isHidden && ((field.isMandatory && !field.value) || !!(await field.validate()));
    }

    private async _initHeader() {
        this.product.value = this._materialsConsumptionSession.productCode;
        this.description.value = this._materialsConsumptionSession.productDescription;
        this.titleTextRight.value = this._materialsConsumptionSession.workOrder.number;

        // Display product picture
        const productPicture = await this.$.graph
            .node('@sage/x3-master-data/Product')
            .read({ picture: { value: true } }, `${this.product.value}| `)
            .execute();
        this.image.value = productPicture?.picture ?? undefined;
        this.$.setPageClean();
    }

    private async _calQuantity() {
        let TotalConsumedQTY = await this._getTotalConsumedQTY();
        let remainingQuantity =
            Number(this._materialsConsumptionSession.product.requiredQuantity) -
            Number(this._materialsConsumptionSession.product.consumedQuantity);
        let ConsumeQuantity = Number(remainingQuantity - TotalConsumedQTY);
        if (ConsumeQuantity > 0) {
            this.quantity.value = ConsumeQuantity;
        } else {
            this.quantity.value = 0;
        }

        this.closeWoLine.value = true; // Yes
        // for serial component
        if (!this.serialNumber.isHidden && this.quantity.value > 1) {
            this.quantity.value = 1;
            this.closeWoLine.value = false; // No
        }
        this.quantity.scale = await GetUOMNumberOfDecimalPlaces(this, this.unit.value);
    }

    private async _getTotalConsumedQTY(): Promise<number> {
        let TotalConsumedQTY: number = 0;
        if (this._materialsConsumptionSession.detail && this._materialsConsumptionSession.detail.length > 0) {
            this._materialsConsumptionSession.detail.forEach(line => {
                if (
                    line.product === this._materialsConsumptionSession.productCode &&
                    line.workOrder === this._materialsConsumptionSession.workOrder.number &&
                    line.lineNumber === this._materialsConsumptionSession.productsReleased.lineNumber &&
                    line.bomSequence === this._materialsConsumptionSession.product.bomSequence
                ) {
                    TotalConsumedQTY += Number(line.quantity);
                }
            });
        }
        return TotalConsumedQTY;
    }

    private async _getTotalLotConsumedQTY(): Promise<number> {
        if (this._materialsConsumptionSession.detail && this._materialsConsumptionSession.detail.length > 0) {
            return this._materialsConsumptionSession.detail.reduce((totalLotConsumedQTY, line) => {
                if (
                    line.product === this._materialsConsumptionSession.productCode &&
                    line.workOrder === this._materialsConsumptionSession.workOrder.number &&
                    line.lineNumber === this._materialsConsumptionSession.productsReleased.lineNumber &&
                    line.bomSequence === this._materialsConsumptionSession.product.bomSequence &&
                    line.lot.code === this.lot.value.lot
                ) {
                    return Number(totalLotConsumedQTY + line.quantity);
                }
                return Number(totalLotConsumedQTY);
            }, 0);
        }
        return 0;
    }

    private async _getSubLot(): Promise<string[]> {
        const response = await this.$.graph
            .node('@sage/x3-stock-data/Stock')
            .query(
                ui.queryUtils.edgesSelector(
                    {
                        sublot: true,
                    },
                    {
                        filter: {
                            product: { _id: this.product.value },
                            owner: this._materialsConsumptionSession.materialsConsumptionSite,
                            lot: this.lot.value?.lot,
                            status: { code: { _eq: this.status.value } },
                        },
                    },
                ),
            )
            .execute();

        if (!response.edges || response.edges.length === 0) {
            throw new Error(
                ui.localize(
                    '@sage/x3-manufacturing/pages__utils__notification__invalid_stock_subLot_error',
                    'No sublot',
                ),
            );
        }

        return response.edges.map((Sublot: any) => {
            if (Sublot.node.sublot !== null) {
                return Sublot.node.sublot;
            } else {
                return ' ';
            }
        });
    }

    /**
     * Gets setting of the chosen entry transaction
     * @param keys transaction code
     * @returns isPreloading
     */
    async getEntryTransactionSettings(keys: EntryTransaction): Promise<boolean> {
        try {
            const transactionSettings = await this.$.graph
                .node('@sage/x3-manufacturing/TransactionProductionReporting')
                .read({ type: true, transaction: true, isPreloading: true }, `${keys.type}|${keys.transaction}`)
                .execute();

            return transactionSettings.isPreloading;
        } catch (error) {
            return false;
        }
    }
}
