"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MaterialsConsumptionDetailsRecordManager = exports.validate = void 0;
const xtrem_decimal_1 = require("@sage/xtrem-decimal"), typesLib = xtrem_decimal_1;
const xtrem_ui_1 = require("@sage/xtrem-ui");
async function validate(page) {
    const errors = await page.$.page.validate();
    if (typesLib.strictEq(errors.length, 0)) {
        return true;
    }
    page.$.showToast(`${(0, xtrem_ui_1.localize)('@sage/x3-manufacturing/notification-error-materials-consumption-error', 'Error')}: ${errors[0]}`, { type: 'error', timeout: 30000 });
    return false;
}
exports.validate = validate;
class MaterialsConsumptionDetailsRecordManager {
    constructor(page, cleanupIncompleteSessionLines = false) {
        this.page = page;
        if (cleanupIncompleteSessionLines)
            this._cleanupSession();
    }
    static { this.WORK_ORDER_KEY = 'workOrder'; }
    get materialsConsumptionSession() {
        if (this._materialsConsumptionSession) {
            return this._materialsConsumptionSession;
        }
        const storedSession = this.page.$.storage.get(MaterialsConsumptionDetailsRecordManager.WORK_ORDER_KEY);
        if (storedSession) {
            const parsedSession = JSON.parse(storedSession);
            if (this._checkStorage(parsedSession)) {
                this._materialsConsumptionSession = parsedSession;
            }
            else {
                this.clearSession();
            }
        }
        return this._materialsConsumptionSession;
    }
    set materialsConsumptionSession(materialsConsumptionSession) {
        if (!materialsConsumptionSession) {
            return;
        }
        if (!this._checkStorage(materialsConsumptionSession)) {
            throw new Error('Invalid Materials Consumption session');
        }
        this.page.$.storage.set(MaterialsConsumptionDetailsRecordManager.WORK_ORDER_KEY, JSON.stringify(materialsConsumptionSession));
        this._materialsConsumptionSession = materialsConsumptionSession;
    }
    _checkStorage(sessionData) {
        if (!sessionData.transaction || !sessionData.workOrder.startDate || !sessionData.workOrder.productionSite) {
            return false;
        }
        else {
            return true;
        }
    }
    _cleanupSession() {
        let returnFromDetailsPage = this.page.$.queryParameters['returnFromDetailsPage'];
        this._materialsConsumptionSession = this.materialsConsumptionSession; // get the latest from session
        if (!this._materialsConsumptionSession)
            return;
        // if all lines have been removed, clear the session
        if (typesLib.strictEq(this._materialsConsumptionSession.detail.length, 0) || typesLib.ne(returnFromDetailsPage, 'true')) {
            this.clearSession();
        }
        else {
            this.materialsConsumptionSession = this._materialsConsumptionSession;
        }
    }
    clearSession() {
        this.page.$.storage.remove(MaterialsConsumptionDetailsRecordManager.WORK_ORDER_KEY);
        this._materialsConsumptionSession = null;
    }
    loadMaterialsConsumptionDetails(page, addNewDetail = false) {
        const detailLine = {
            workOrder: this._materialsConsumptionSession?.workOrder.number,
            product: this._materialsConsumptionSession?.productCode,
            description: this._materialsConsumptionSession?.productDescription,
            unit: page.unit.value ?? undefined,
            quantity: page.quantity.value ?? undefined,
            status: page.status.value ?? undefined, // this is always required
            lot: { code: page.lot.value?.lot ?? undefined }, // using undefined, omits the property entirely from JSON when stringified
            sublot: page.sublot.value ?? undefined,
            location: { code: page.location.value ?? undefined },
            serialNumber: { code: page.serialNumber.value?.code ?? undefined },
            licensePlateNumber: { code: page.licensePlateNumber.value ?? undefined },
            closeWoLine: page.closeWoLine.value ?? undefined,
            lineNumber: this._materialsConsumptionSession?.productsReleased.lineNumber,
            bomSequence: this._materialsConsumptionSession?.product.bomSequence,
            bomNumber: this._materialsConsumptionSession?.productsReleased.product,
            bomAlternative: this._materialsConsumptionSession?.productsReleased.bomCode,
            trackingDate: this._materialsConsumptionSession?.trackingDate,
        };
        // unplanned material has no bomSequence
        if (typesLib.strictEq(typeof detailLine.bomSequence, 'undefined')) {
            detailLine.bomSequence = 0;
        }
        this._materialsConsumptionSession?.detail.push(detailLine);
        this.materialsConsumptionSession = this._materialsConsumptionSession;
        if (addNewDetail) {
            let TotalConsumedQTY = 0;
            if (this.materialsConsumptionSession?.detail && typesLib.gt(this.materialsConsumptionSession.detail.length, 0)) {
                this.materialsConsumptionSession.detail.forEach(line => {
                    TotalConsumedQTY = typesLib.add(TotalConsumedQTY, Number(line.quantity));
                });
            }
            let remainingQuantity = typesLib.sub(Number(this.materialsConsumptionSession?.product.requiredQuantity), Number(this._materialsConsumptionSession?.product.consumedQuantity));
            if (typesLib.lte(Number(typesLib.sub(remainingQuantity, TotalConsumedQTY)), 0)) {
                page.quantity.value = 0;
            }
            else {
                page.quantity.value = Number(typesLib.sub(remainingQuantity, TotalConsumedQTY));
            }
            page.status.value = '';
            page.location.value = '';
            page.lot.value.lot = '';
            page.sublot.value = '';
            page.licensePlateNumber.value = '';
            page.serialNumber.value.code = '';
        }
        console.log('🚀 ~ MaterialsConsumptionDetailsRecordManager ~ detailLine:', detailLine);
    }
}
exports.MaterialsConsumptionDetailsRecordManager = MaterialsConsumptionDetailsRecordManager;
//# sourceMappingURL=materials-consumption-details-control.js.map