"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getErrorMsg = void 0;
let silentOutput = false;
let logToString = false;
const correlationHeader = "x-correlation";
const correlationIdPrefix = correlationHeader + "-id";
const getErrorMsgFromMaybeObject = (notError) => notError instanceof Object && "toString" in notError ? notError.toString() : "no msg";
const getErrorMsg = (error) => error instanceof Error ? error.message : getErrorMsgFromMaybeObject(error);
exports.getErrorMsg = getErrorMsg;
const ContextTool = {
    clearAll: () => global.CONTEXT = undefined,
    replaceAllWith: (ctx) => global.CONTEXT = ctx,
    set: (key, value) => {
        //This test is required because SDX is using this header
        if (key === correlationHeader) {
            key = correlationIdPrefix;
        }
        if (!key.startsWith(correlationIdPrefix + "-")) {
            key = `${correlationIdPrefix}-${key}`;
        }
        const globalTyped = global;
        if (!globalTyped.CONTEXT) {
            globalTyped.CONTEXT = {};
        }
        globalTyped.CONTEXT[key] = value;
    },
    get: () => global.CONTEXT || {},
    responseWithContextHeaders: (request, headersObj) => {
        if (!headersObj) {
            headersObj = {};
        }
        if (!request) {
            request = {};
        }
        const headers = Object.assign({}, ContextTool.get(), headersObj);
        const headerNames = Object.keys(headers);
        if (!request.headers) {
            request.headers = {};
        }
        for (const name of headerNames) {
            request.headers[name] = headers[name];
        }
        return request;
    },
    setRequestContext: (event, context) => {
        if (!context) {
            context = {};
        }
        if (!event.headers) {
            LogTool.warn(`Request ${context.awsRequestId} is missing headers`);
            return;
        }
        if (!context.awsRequestId) {
            LogTool.warn(`Request ${context.awsRequestId} is missing awsRequestId`);
            return;
        }
        const ctx = { awsRequestId: context.awsRequestId };
        for (const header in event.headers) {
            if (header.toLowerCase() === correlationHeader) {
                event.headers[header.toLowerCase()] = event.headers[header];
            }
            if (header.toLowerCase().startsWith("x-correlation-")) {
                ctx[header.toLowerCase()] = event.headers[header];
            }
        }
        if (!ctx[correlationIdPrefix]) {
            ctx[correlationIdPrefix] = ctx.awsRequestId;
        }
        //fallback due to SWD using another token for context
        if (event.headers[correlationHeader]) {
            ctx["x-correlation-id"] = event.headers[correlationHeader];
        }
        if (event.headers["User-Agent"]) {
            ctx["User-Agent"] = event.headers["User-Agent"];
        }
        if (event.headers["Debug-Log-Enabled"] === "true") {
            ctx["Debug-Log-Enabled"] = "true";
        }
        else {
            // enable debug logging on 5% of cases
            ctx["Debug-Log-Enabled"] = Math.random() < 0.05 ? "true" : "false";
        }
        if (event.headers["True-Client-IP"]) {
            ctx["True-Client-IP"] = event.headers["True-Client-IP"];
        }
        ContextTool.replaceAllWith(ctx);
    }
};
function getContext() {
    // note: this is a shallow copy
    return Object.assign({}, ContextTool.get());
}
function isDebugEnabled() {
    // disable debug logging by default, but allow override via env variables
    // or if enabled via forwarded request context
    return process.env.DEBUG_LOG === "true" || ContextTool.get()["Debug-Log-Enabled"] === "true";
}
function isSecretNotObfuscated() {
    // secret obfuscation is enabled by default by default, but allow override via env variables for debug
    return process.env.OBFUSCATE_SECRET_IN_LOGS === "false";
}
function autoConvertError(objOrErr) {
    if (objOrErr instanceof Error) {
        return {
            name: objOrErr.name,
            message: objOrErr.message,
            stack: isDebugEnabled() ? objOrErr.stack : ""
        };
    }
    else {
        return objOrErr;
    }
}
function log(level, msg, params) {
    if (level === "DEBUG" && !isDebugEnabled() || silentOutput) {
        return undefined;
    }
    const logMsg = getContext();
    logMsg.level = level;
    if (level === "SECURITY") {
        msg = "[SECURITY_ALERT] " + msg;
    }
    logMsg.message = msg;
    if (params) {
        logMsg.params = autoConvertError(params);
    }
    const res = JSON.stringify(logMsg);
    if (logToString) {
        return res;
    }
    else {
        console.log(res);
    }
    return undefined;
}
const LogTool = {
    isDebugLoggingEnabled: isDebugEnabled,
    debug: (msg, params) => log("DEBUG", msg, params),
    info: (msg, params) => log("INFO", msg, params),
    warn: (msg, params) => log("WARN", msg, params),
    error: (msg, params) => log("ERROR", msg, params),
    security: (msg, params) => log("SECURITY", msg, params),
    audit: (msg, params) => log("AUDIT", msg, params),
    _setConsoleSilent: (value) => {
        silentOutput = value;
    },
    _setLogToString: (value) => {
        logToString = value;
    }
};
const Tools = {
    Context: ContextTool,
    Log: LogTool,
    obfuscate: (txt, lastCharNbPercentToObfuscate) => {
        if (!txt) {
            return txt;
        }
        if (isSecretNotObfuscated()) {
            return txt;
        }
        if (lastCharNbPercentToObfuscate !== 0 && !lastCharNbPercentToObfuscate) {
            lastCharNbPercentToObfuscate = 50;
        }
        const nbChar = Math.ceil(txt.length / (100 / (100 - lastCharNbPercentToObfuscate)));
        let newTxt = "";
        for (let x = 0; x < txt.length; x++) {
            if (x <= (nbChar - 1)) {
                newTxt += txt[x];
            }
            else {
                newTxt += "-";
            }
        }
        if (txt !== "" && newTxt === txt) {
            newTxt = newTxt.substring(0, newTxt.length - 1) + "-";
        }
        return newTxt;
    },
    deepObfuscate: (obj, lastCharNbPercentToObfuscate) => {
        if (obj instanceof Array) {
            return obj.map((data) => {
                switch (true) {
                    case typeof (data) === "string":
                        return Tools.obfuscate(data, lastCharNbPercentToObfuscate);
                    case data instanceof Object:
                        return Tools.deepObfuscate(data, lastCharNbPercentToObfuscate);
                    default:
                        return data;
                }
            });
        }
        else if (obj instanceof Object) {
            const objType = obj;
            return Object.keys(obj).reduce(function (result, key) {
                switch (true) {
                    case typeof (objType[key]) === "string":
                        result[key] = Tools.obfuscate(objType[key], lastCharNbPercentToObfuscate);
                        break;
                    case objType[key] instanceof Object:
                        result[key] = Tools.deepObfuscate(objType[key], lastCharNbPercentToObfuscate);
                        break;
                    default:
                        result[key] = objType[key];
                }
                return result;
            }, {});
        }
        else if (typeof (obj) === "string") {
            return Tools.obfuscate(obj, lastCharNbPercentToObfuscate);
        }
        else {
            return obj;
        }
    }
};
exports.default = Tools;
