import { ColorChannel, ColorSpace, Color, ColorAreaProps, ColorSliderProps, ColorWheelProps, ColorFieldProps } from "@react-types/color";
import { SliderState } from "@react-stately/slider";
import { FormValidationState } from "@react-stately/form";
import { NumberFieldState } from "@react-stately/numberfield";
import { ValueBase } from "@react-types/shared";
/** Parses a color from a string value. Throws an error if the string could not be parsed. */
export function parseColor(value: string): Color;
/** Returns a list of color channels for a given color space. */
export function getColorChannels(colorSpace: ColorSpace): [ColorChannel, ColorChannel, ColorChannel];
export interface ColorAreaState {
    /** The current color value displayed by the color area. */
    readonly value: Color;
    /** Sets the current color value. If a string is passed, it will be parsed to a Color. */
    setValue(value: string | Color): void;
    /** The current value of the horizontal axis channel displayed by the color area. */
    xValue: number;
    /** Sets the value for the horizontal axis channel displayed by the color area, and triggers `onChange`. */
    setXValue(value: number): void;
    /** The current value of the vertical axis channel displayed by the color area. */
    yValue: number;
    /** Sets the value for the vertical axis channel displayed by the color area, and triggers `onChange`. */
    setYValue(value: number): void;
    /** Sets the x and y channels of the current color value based on a percentage of the width and height of the color area, and triggers `onChange`. */
    setColorFromPoint(x: number, y: number): void;
    /** Returns the coordinates of the thumb relative to the upper left corner of the color area as a percentage. */
    getThumbPosition(): {
        x: number;
        y: number;
    };
    /** Increments the value of the horizontal axis channel by the channel step or page amount. */
    incrementX(stepSize?: number): void;
    /** Decrements the value of the horizontal axis channel by the channel step or page amount. */
    decrementX(stepSize?: number): void;
    /** Increments the value of the vertical axis channel by the channel step or page amount. */
    incrementY(stepSize?: number): void;
    /** Decrements the value of the vertical axis channel by the channel step or page amount. */
    decrementY(stepSize?: number): void;
    /** Whether the color area is currently being dragged. */
    readonly isDragging: boolean;
    /** Sets whether the color area is being dragged. */
    setDragging(value: boolean): void;
    /** Returns the xChannel, yChannel and zChannel names based on the color value. */
    channels: {
        xChannel: ColorChannel;
        yChannel: ColorChannel;
        zChannel: ColorChannel;
    };
    /** The step value of the xChannel, used when incrementing and decrementing. */
    xChannelStep: number;
    /** The step value of the yChannel, used when incrementing and decrementing. */
    yChannelStep: number;
    /** The page step value of the xChannel, used when incrementing and decrementing. */
    xChannelPageStep: number;
    /** The page step value of the yChannel, used when incrementing and decrementing. */
    yChannelPageStep: number;
    /** Returns the color that should be displayed in the color area thumb instead of `value`. */
    getDisplayColor(): Color;
}
/**
 * Provides state management for a color area component.
 * Color area allows users to adjust two channels of an HSL, HSB or RGB color value against a two-dimensional gradient background.
 */
export function useColorAreaState(props: ColorAreaProps): ColorAreaState;
export interface ColorSliderState extends SliderState {
    /** The current color value represented by the color slider. */
    readonly value: Color;
    /** Sets the current color value. If a string is passed, it will be parsed to a Color. */
    setValue(value: string | Color): void;
    /** Returns the color that should be displayed in the slider instead of `value` or the optional parameter. */
    getDisplayColor(): Color;
    /** Whether the color slider is currently being dragged. */
    readonly isDragging: boolean;
}
export interface ColorSliderStateOptions extends ColorSliderProps {
    /** The locale to use for formatting the color channel value. */
    locale: string;
}
/**
 * Provides state management for a color slider component.
 * Color sliders allow users to adjust an individual channel of a color value.
 */
export function useColorSliderState(props: ColorSliderStateOptions): ColorSliderState;
export interface ColorWheelState {
    /** The current color value represented by the color wheel. */
    readonly value: Color;
    /** Sets the color value represented by the color wheel, and triggers `onChange`. */
    setValue(value: string | Color): void;
    /** The current value of the hue channel displayed by the color wheel. */
    readonly hue: number;
    /** Sets the hue channel of the current color value and triggers `onChange`. */
    setHue(value: number): void;
    /** Sets the hue channel of the current color value based on the given coordinates and radius of the color wheel, and triggers `onChange`. */
    setHueFromPoint(x: number, y: number, radius: number): void;
    /** Returns the coordinates of the thumb relative to the center point of the color wheel. */
    getThumbPosition(radius: number): {
        x: number;
        y: number;
    };
    /** Increments the hue by the given amount (defaults to 1). */
    increment(stepSize?: number): void;
    /** Decrements the hue by the given amount (defaults to 1). */
    decrement(stepSize?: number): void;
    /** Whether the color wheel is currently being dragged. */
    readonly isDragging: boolean;
    /** Sets whether the color wheel is being dragged. */
    setDragging(value: boolean): void;
    /** Returns the color that should be displayed in the color wheel instead of `value`. */
    getDisplayColor(): Color;
    /** The step value of the hue channel, used when incrementing and decrementing. */
    step: number;
    /** The page step value of the hue channel, used when incrementing and decrementing. */
    pageStep: number;
    /** Whether the color wheel is disabled. */
    readonly isDisabled: boolean;
}
/**
 * Provides state management for a color wheel component.
 * Color wheels allow users to adjust the hue of an HSL or HSB color value on a circular track.
 */
export function useColorWheelState(props: ColorWheelProps): ColorWheelState;
export interface ColorFieldState extends FormValidationState {
    /**
     * The current text value of the input. Updated as the user types,
     * and formatted according to `formatOptions` on blur.
     */
    readonly inputValue: string;
    /**
     * The currently parsed color value, or null if the field is empty.
     * Updated based on the `inputValue` as the user types.
     */
    readonly colorValue: Color | null;
    /** Sets the current text value of the input. */
    setInputValue(value: string): void;
    /**
     * Updates the input value based on the currently parsed color value.
     * Typically this is called when the field is blurred.
     */
    commit(): void;
    /** Increments the current input value to the next step boundary, and fires `onChange`. */
    increment(): void;
    /** Decrements the current input value to the next step boundary, and fires `onChange`. */
    decrement(): void;
    /** Sets the current value to the maximum color value, and fires `onChange`. */
    incrementToMax(): void;
    /** Sets the current value to the minimum color value, and fires `onChange`. */
    decrementToMin(): void;
    /**
     * Validates a user input string.
     * Values can be partially entered, and may be valid even if they cannot currently be parsed to a color.
     * Can be used to implement validation as a user types.
     */
    validate(value: string): boolean;
}
/**
 * Provides state management for a color field component. Color fields allow
 * users to enter and adjust a hex color value.
 */
export function useColorFieldState(props: ColorFieldProps): ColorFieldState;
export interface ColorChannelFieldProps extends ColorFieldProps {
    colorSpace?: ColorSpace;
    channel: ColorChannel;
}
export interface ColorChannelFieldStateOptions extends ColorChannelFieldProps {
    locale: string;
}
export interface ColorChannelFieldState extends NumberFieldState {
    colorValue: Color;
}
/**
 * Provides state management for a color channel field, allowing users to edit the
 * value of an individual color channel.
 */
export function useColorChannelFieldState(props: ColorChannelFieldStateOptions): ColorChannelFieldState;
export interface ColorPickerProps extends ValueBase<string | Color, Color> {
}
export interface ColorPickerState {
    /** The current color value of the color picker. */
    color: Color;
    /** Sets the current color value of the color picker. */
    setColor(color: Color | null): void;
}
export function useColorPickerState(props: ColorPickerProps): ColorPickerState;
export type { Color, ColorAreaProps, ColorFieldProps, ColorWheelProps } from '@react-types/color';

//# sourceMappingURL=types.d.ts.map
