/**
 * @license Copyright (c) 2003-2024, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md.
 */
/**
 * A hook that allows to execute an asynchronous function and provides the state of the execution.
 *
 * @param callback The asynchronous function to be executed.
 * @returns A tuple with the function that triggers the execution and the state of the execution.
 *
 * @example
 * ```tsx
 * const [ onFetchData, fetchDataStatus ] = useAsyncCallback( async () => {
 * 	const response = await fetch( 'https://api.example.com/data' );
 * 	const data = await response.json();
 * 	return data;
 * } );
 *
 * return (
 * 	<div>
 * 		<button onClick={ onFetchData }>Fetch data</button>
 * 		{ fetchDataStatus.status === 'loading' && <p>Loading...</p> }
 * 		{ fetchDataStatus.status === 'success' && <pre>{ JSON.stringify( fetchDataStatus.data, null, 2 ) }</pre> }
 * 		{ fetchDataStatus.status === 'error' && <p>Error: { fetchDataStatus.error.message }</p> }
 * 	</div>
 * );
 * ```
 */
export declare const useAsyncCallback: <A extends Array<unknown>, R>(callback: (...args: Array<A>) => Promise<R>) => AsyncCallbackHookResult<A, R>;
/**
 * Represents the result of the `useAsyncCallback` hook.
 */
export type AsyncCallbackHookResult<A extends Array<unknown>, R> = [
    (...args: Array<A>) => Promise<R | null>,
    AsyncCallbackState<R>
];
/**
 * Represents the state of an asynchronous callback.
 */
export type AsyncCallbackState<T> = {
    status: 'idle';
} | {
    status: 'loading';
} | {
    status: 'success';
    data: T;
} | {
    status: 'error';
    error: any;
};
