/**
 * @license Copyright (c) 2003-2024, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md.
 */
import React from 'react';
import PropTypes, { type InferProps, type Validator } from 'prop-types';
import type { EventInfo, Editor, EditorConfig, EditorWatchdog, ContextWatchdog, WatchdogConfig, EditorCreatorFunction } from 'ckeditor5';
import { type CKEditorConfigContextMetadata } from './context/setCKEditorReactContextMetadata.js';
export default class CKEditor<TEditor extends Editor> extends React.Component<Props<TEditor>> {
    /**
     * After mounting the editor, the variable will contain a reference to the created editor.
     * @see: https://ckeditor.com/docs/ckeditor5/latest/api/module_core_editor_editor-Editor.html
     */
    private domContainer;
    /**
     * Unlocks element in editor semaphore after destroy editor instance.
     */
    private editorSemaphore;
    constructor(props: Props<TEditor>);
    /**
     * Checks if the CKEditor version used in the application is compatible with the component.
     */
    private _checkVersion;
    private get _semaphoreValue();
    /**
     * An watchdog instance.
     */
    get watchdog(): EditorWatchdog<TEditor> | EditorWatchdogAdapter<TEditor> | null;
    /**
     * An editor instance.
     */
    get editor(): Editor | null;
    /**
     * The CKEditor component should not be updated by React itself.
     * However, if the component identifier changes, the whole structure should be created once again.
     */
    shouldComponentUpdate(nextProps: Readonly<Props<TEditor>>): boolean;
    /**
     * Initialize the editor when the component is mounted.
     */
    componentDidMount(): void;
    /**
     * Re-render the entire component once again. The old editor will be destroyed and the new one will be created.
     */
    componentDidUpdate(): void;
    /**
     * Destroy the editor before unmounting the component.
     */
    componentWillUnmount(): void;
    /**
     * Async destroy attached editor and unlock element semaphore.
     */
    private _unlockLifeCycleSemaphore;
    /**
     * Unlocks previous editor semaphore and creates new one..
     */
    private _initLifeCycleSemaphore;
    /**
     * Render a <div> element which will be replaced by CKEditor.
     */
    render(): React.ReactNode;
    /**
     * Initializes the editor by creating a proper watchdog and initializing it with the editor's configuration.
     */
    private _initializeEditor;
    /**
     * Creates an editor from the element and configuration.
     *
     * @param element The source element.
     * @param config CKEditor 5 editor configuration.
     */
    private _createEditor;
    /**
     * Destroys the editor by destroying the watchdog.
     */
    private _destroyEditor;
    /**
     * Returns true when the editor should be updated.
     *
     * @param prevProps Previous react's properties.
     * @param nextProps React's properties.
     * @param editor Current editor instance.
     */
    private _shouldUpdateEditorData;
    /**
     * Returns the editor configuration.
     */
    private _getConfig;
    static contextType: React.Context<import("./context/ckeditorcontext.js").ContextWatchdogValue<import("ckeditor5").Context> | null>;
    static propTypes: {
        editor: Validator<{
            create(...args: any): Promise<any>;
        }>;
        data: PropTypes.Requireable<string>;
        config: PropTypes.Requireable<object>;
        disableWatchdog: PropTypes.Requireable<boolean>;
        watchdogConfig: PropTypes.Requireable<object>;
        onChange: PropTypes.Requireable<(...args: any[]) => any>;
        onReady: PropTypes.Requireable<(...args: any[]) => any>;
        onFocus: PropTypes.Requireable<(...args: any[]) => any>;
        onBlur: PropTypes.Requireable<(...args: any[]) => any>;
        onError: PropTypes.Requireable<(...args: any[]) => any>;
        disabled: PropTypes.Requireable<boolean>;
        id: PropTypes.Requireable<any>;
    };
}
/**
 * TODO this is type space definition for props, the CKEditor.propTypes is a run-time props validation that should match.
 */
export interface Props<TEditor extends Editor> extends InferProps<typeof CKEditor.propTypes> {
    editor: {
        create(...args: any): Promise<TEditor>;
        EditorWatchdog: typeof EditorWatchdog;
        ContextWatchdog: typeof ContextWatchdog;
    };
    contextItemMetadata?: CKEditorConfigContextMetadata;
    config?: EditorConfig;
    watchdogConfig?: WatchdogConfig;
    disableWatchdog?: boolean;
    onReady?: (editor: TEditor) => void;
    onAfterDestroy?: (editor: TEditor) => void;
    onError?: (error: Error, details: ErrorDetails) => void;
    onChange?: (event: EventInfo, editor: TEditor) => void;
    onFocus?: (event: EventInfo, editor: TEditor) => void;
    onBlur?: (event: EventInfo, editor: TEditor) => void;
}
interface ErrorDetails {
    phase: 'initialization' | 'runtime';
    willEditorRestart?: boolean;
}
/**
 * An adapter aligning the context watchdog API to the editor watchdog API for easier usage.
 */
export declare class EditorWatchdogAdapter<TEditor extends Editor> {
    /**
     * The context watchdog instance that will be wrapped into editor watchdog API.
     */
    private readonly _contextWatchdog;
    /**
     * A unique id for the adapter to distinguish editor items when using the context watchdog API.
     */
    private readonly _id;
    /**
     * A watchdog's editor creator function.
     */
    private _creator?;
    /**
     * @param contextWatchdog The context watchdog instance that will be wrapped into editor watchdog API.
     */
    constructor(contextWatchdog: ContextWatchdog);
    /**
     *  @param creator A watchdog's editor creator function.
     */
    setCreator(creator: EditorCreatorFunction): void;
    /**
     * Adds an editor configuration to the context watchdog registry. Creates an instance of it.
     *
     * @param sourceElementOrData A source element or data for the new editor.
     * @param config CKEditor 5 editor config.
     */
    create(sourceElementOrData: HTMLElement | string, config: EditorConfig): Promise<unknown>;
    /**
     * Creates a listener that is attached to context watchdog's item and run when the context watchdog fires.
     * Currently works only for the `error` event.
     */
    on(_: string, callback: (_: null, data: {
        error: Error;
        causesRestart?: boolean;
    }) => void): void;
    destroy(): Promise<unknown>;
    /**
     * An editor instance.
     */
    get editor(): TEditor;
}
export {};
