"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.enumNameToStringKey = exports.artifactReferenceToTitleStringKey = exports.menuItemIdToStringKey = exports.getLocaleFromHeader = void 0;
exports.getPackageNameAndRoot = getPackageNameAndRoot;
const xtrem_log_1 = require("@sage/xtrem-log");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const parser = require("accept-language-parser");
const fs = require("fs");
const lodash_1 = require("lodash");
const path = require("path");
const logger = new xtrem_log_1.Logger(__filename, 'utils');
/**
 * Gets the name of the current package
 * @param dir the starting directory
 * @returns the name of the package (closest ancestor)
 */
function getPackageNameAndRoot(start) {
    const isExisting = fs.existsSync(start);
    if (!isExisting) {
        throw new Error(`'${start}' doesn't exist`);
    }
    const root = path.parse(start).root;
    let dir = start;
    if (fs.lstatSync(start).isFile()) {
        dir = path.dirname(start);
    }
    const packageJson = path.join(dir, 'package.json');
    if (!fs.existsSync(packageJson)) {
        const newDir = path.join(dir, '..');
        if (newDir === root) {
            throw new Error("Couldn't find the name of the package");
        }
        return getPackageNameAndRoot(newDir);
    }
    try {
        const name = JSON.parse(fs.readFileSync(packageJson, { encoding: 'utf8' })).name;
        return { name, root: dir };
    }
    catch (error) {
        throw new Error(`Couldn't read the 'name' property from '${packageJson}'`);
    }
}
const getLocaleFromHeader = (headers) => {
    if (headers) {
        const acceptLanguage = headers['accept-language'];
        if (acceptLanguage && typeof acceptLanguage === 'string') {
            const language = parser.pick(xtrem_shared_1.supportedLocales, acceptLanguage);
            return language || 'base';
        }
    }
    else {
        logger.debug(() => 'Request headers missing');
    }
    return 'base';
};
exports.getLocaleFromHeader = getLocaleFromHeader;
const menuItemIdToStringKey = (menuItemId) => {
    const parts = menuItemId.split('/');
    const packageName = `${parts[0]}/${parts[1]}`;
    return `${packageName}/menu_item__${menuItemId.replace(`${packageName}/`, '')}`;
};
exports.menuItemIdToStringKey = menuItemIdToStringKey;
const artifactReferenceToTitleStringKey = (artifactReference, artifactType) => {
    const parts = artifactReference.split('/');
    const packageName = `${parts[0]}/${parts[1]}`;
    return `${packageName}/${artifactType}__${(0, lodash_1.snakeCase)(parts[2])}____title`;
};
exports.artifactReferenceToTitleStringKey = artifactReferenceToTitleStringKey;
const enumNameToStringKey = (enumName, memberName) => {
    const parts = enumName.split('/');
    return `${parts[0]}/${parts[1]}/enums__${(0, lodash_1.snakeCase)(parts[2])}__${memberName}`;
};
exports.enumNameToStringKey = enumNameToStringKey;
//# sourceMappingURL=utils.js.map