"use strict";
/* eslint-disable no-console */
/**
Use http://mikemcl.github.io/decimal.js/
https://github.com/MikeMcl/decimal.js
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.toDecimal = toDecimal;
exports.toNumber = toNumber;
exports.toInteger = toInteger;
exports.newDecimal = newDecimal;
exports.add = add;
exports.sub = sub;
exports.mul = mul;
exports.div = div;
exports.compare = compare;
exports.eq = eq;
exports.strictEq = strictEq;
exports.ne = ne;
exports.strictNe = strictNe;
exports.lt = lt;
exports.lte = lte;
exports.gt = gt;
exports.gte = gte;
exports.negated = negated;
exports.plus = plus;
exports.abs = abs;
exports.ceil = ceil;
exports.floor = floor;
exports.pow = pow;
exports.max = max;
exports.min = min;
exports.round = round;
exports.sum = sum;
exports.uselessFunctionV3 = uselessFunctionV3;
const decimal_class_1 = require("./decimal-class");
const valueIsANumber = (value) => typeof value === 'number';
const valueIsNotANumber = (value) => typeof value !== 'number';
const quote = (str) => `'${str}'`;
function ensureValidDecimal(value, originalValue) {
    if (value.isNaN())
        throw new Error(`Invalid numeric value: ${originalValue != null ? quote(originalValue) : value}`);
    return value;
}
function ensureValidNumber(value, originalValue) {
    if (Number.isNaN(value))
        throw new Error(`Invalid numeric value: ${originalValue != null ? quote(originalValue) : value}`);
    return value;
}
/**
 * Converts a value to a decimal (an instance of Decimal).
 *
 * The value may be a string, a number or a Decimal.
 * Only finite values are allowed. Special values like NaN and Infinity will throw an error.
 */
function toDecimal(value) {
    if (value instanceof decimal_class_1.Decimal)
        return value;
    if (typeof value === 'string')
        return ensureValidDecimal(new decimal_class_1.Decimal(value), value);
    if (typeof value === 'number')
        return ensureValidDecimal(new decimal_class_1.Decimal(value));
    throw new Error(`Invalid numeric value: ${value}`);
}
/**
 * Converts a value to a number.
 *
 * The value may be a string, a number or a decimal (an instance of Decimal).
 * Only finite values are allowed. Special values like NaN and Infinity will throw an error.
 */
function toNumber(value) {
    if (decimal_class_1.Decimal.isDecimal(value))
        return value.toNumber();
    if (typeof value === 'number')
        return ensureValidNumber(value);
    if (typeof value === 'string')
        return ensureValidNumber(parseFloat(value), value);
    throw new Error(`Invalid numeric value: ${value}`);
}
/**
 * Converts a value to an integer.
 *
 * The value may be a string, a number or a decimal (an instance of Decimal).
 * Only integer values are allowed. Non-integer values like 1.5, NaN and Infinity will throw an error.
 */
function toInteger(value) {
    const result = toNumber(value);
    if (result !== Math.floor(result))
        throw new Error(`Invalid integer value: ${value}`);
    return result;
}
function newDecimal(s) {
    return new decimal_class_1.Decimal(s);
}
function add(a, b) {
    // Allow string promotion for concatenation
    return (Number.isInteger(a) && Number.isInteger(b)) || typeof a === 'string' || typeof b === 'string'
        ? a + b
        : decimal_class_1.Decimal.add(a, b);
}
function sub(a, b) {
    return Number.isInteger(a) && Number.isInteger(b) ? a - b : decimal_class_1.Decimal.sub(a, b);
}
function mul(a, b) {
    return Number.isInteger(a) && Number.isInteger(b) ? a * b : decimal_class_1.Decimal.mul(a, b);
}
function div(a, b) {
    return Number.isInteger(a) && Number.isInteger(b) ? a / b : decimal_class_1.Decimal.div(a, b);
}
function toDecimalOrUndefined(a) {
    if (decimal_class_1.Decimal.isDecimal(a))
        return a;
    if (valueIsANumber(a))
        return new decimal_class_1.Decimal(a);
    return undefined;
}
// const compare = operation('comparedTo');
function compare(a, b) {
    if (a === b)
        return 0;
    // LATER: we should review the comparision with null to be aligned with the SQL behavior
    // // comparing undefined is always false
    // if (a === undefined || b === undefined) return undefined;
    // // everything is greater than null but any other comparison is false
    // if (a === null) return -1;
    // if (b === null) return 1;
    // To prevent from prototype pollution (https://portswigger.net/web-security/prototype-pollution, https://learn.snyk.io/lesson/prototype-pollution/),
    // some objects may have been created with a null prototype using Object.create(null),
    // those objects cannot be converted to a primitive type and thus are not comparable to anything
    if (a != null && typeof a === 'object' && (Object.getPrototypeOf(a) == null || Array.isArray(a)))
        return undefined;
    if (b != null && typeof b === 'object' && (Object.getPrototypeOf(b) == null || Array.isArray(b)))
        return undefined;
    const decimalA = toDecimalOrUndefined(a);
    const decimalB = toDecimalOrUndefined(b);
    if (decimalA && decimalB)
        return decimalA.comparedTo(decimalB);
    if (a < b)
        return -1;
    if (a > b)
        return 1;
    return undefined;
}
function sameTypes(a, b) {
    return (typeof a === typeof b ||
        ((valueIsANumber(a) || decimal_class_1.Decimal.isDecimal(a)) && (valueIsANumber(b) || decimal_class_1.Decimal.isDecimal(b))));
}
function eq(a, b) {
    // eslint-disable-next-line eqeqeq
    return a == b || compare(a, b) === 0;
}
function strictEq(a, b) {
    return a === b || (sameTypes(a, b) && eq(a, b));
}
function ne(a, b) {
    return !eq(a, b);
}
function strictNe(a, b) {
    return !strictEq(a, b);
}
function lt(a, b) {
    try {
        return compare(a, b) === -1;
    }
    catch (e) {
        console.error(`lt(${a} (${typeof a}) ${b} (${typeof b})) => ${e.message}`);
    }
    return false;
}
function lte(a, b) {
    try {
        const cmp = compare(a, b);
        return cmp !== undefined && cmp <= 0;
    }
    catch (e) {
        console.error(`lte(${a} (${typeof a}) ${b} (${typeof b})) => ${e.message}`);
    }
    return false;
}
function gt(a, b) {
    try {
        return compare(a, b) === 1;
    }
    catch (e) {
        console.error(`gt(${a} (${typeof a}) ${b} (${typeof b})) => ${e.message}`);
    }
    return false;
}
function gte(a, b) {
    try {
        const cmp = compare(a, b);
        return cmp !== undefined && cmp >= 0;
    }
    catch (e) {
        console.error(`gte(${a} (${typeof a}) ${b} (${typeof b})) => ${e.message}`);
    }
    return false;
}
function negated(a) {
    if (valueIsANumber(a))
        return -a;
    return ((decimal_class_1.Decimal.isDecimal(a) && a) || new decimal_class_1.Decimal(a)).negated();
}
function plus(a) {
    return typeof a === 'string' ? new decimal_class_1.Decimal(a) : a;
}
function abs(value) {
    return valueIsANumber(value) ? Math.abs(value) : decimal_class_1.Decimal.abs(value);
}
function ceil(value) {
    return valueIsANumber(value) ? Math.ceil(value) : decimal_class_1.Decimal.ceil(value).toNumber();
}
function floor(value) {
    return valueIsANumber(value) ? Math.floor(value) : decimal_class_1.Decimal.floor(value).toNumber();
}
function pow(base, exponent) {
    return valueIsANumber(base) && valueIsANumber(exponent) ? base ** exponent : decimal_class_1.Decimal.pow(base, exponent);
}
function max(...values) {
    return values.find(valueIsNotANumber) ? decimal_class_1.Decimal.max(...values) : Math.max(...values);
}
function min(...values) {
    return values.find(valueIsNotANumber) ? decimal_class_1.Decimal.min(...values) : Math.min(...values);
}
function round(value) {
    return valueIsANumber(value) ? Math.round(value) : decimal_class_1.Decimal.round(value).toNumber();
}
function sum(...values) {
    return values.find(valueIsNotANumber)
        ? decimal_class_1.Decimal.sum(...values)
        : values.reduce((prevValue, currentValue) => prevValue + currentValue, 0);
}
function uselessFunctionV3() { }
//# sourceMappingURL=types-lib.js.map