import { _Object, Tagging } from "@aws-sdk/client-s3";
import { FileTimeToLive, TenantDataType } from "@sage/xtrem-deployment";
import { S3BodyGet, S3BodyPut } from "./utils";
export interface FileStorageWrapper {
    setUnitTestMode(value: boolean): void;
    createFile(tenantId: string, objectKey: string, description: string, body: S3BodyPut, expiration: FileTimeToLive): Promise<any>;
    updateExpiration(tenantId: string, objectKey: string, expiration: FileTimeToLive): Promise<any>;
    updateTagging(tenantId: string, objectKey: string, tagging: Tagging): Promise<any>;
    readFile(tenantId: string, objectKey: string): Promise<{
        tagData: any;
        body: S3BodyGet;
    } | void>;
    list(tenantDataType: TenantDataType, tenantId: string, continuationToken?: string): Promise<any>;
    listFiles(tenantId: string, continuationToken?: string): Promise<any>;
    listExports(tenantId: string, continuationToken?: string): Promise<any>;
    listImports(tenantId: string, continuationToken?: string): Promise<any>;
    listTenants(): Promise<{
        [key: string]: _Object[];
    }>;
    readAsyncContextResult(tenantId: string, contextId: string, contextKind: string): Promise<{
        tagData: any;
        body: S3BodyGet;
    } | void>;
    generateUploadPresignedUrl(tenantId: string, contextId: string, urlTimeToLiveInSec?: number): Promise<string>;
    generateDownloadPresignedUrl(tenantId: string, objectKey: string, urlTimeToLiveInSec?: number, filename?: string): Promise<string>;
    /***
     * Provide a S3 Upload url that will then be moved after antivirus scan to the attachment tenant subfolder
     * @param tenantId
     * @param contextId
     * @param urlTimeToLiveInSec
     */
    generateAttachmentUploadPresignedUrl(tenantId: string, contextId: string, urlTimeToLiveInSec?: number): Promise<string>;
    /***
     * Provide a S3 download presigned url to download the attachement, can be used for web client to get the file
     * @param tenantId related tenant nanoid
     * @param attachmentId related attachment nanoid
     * @param urlTimeToLiveInSec how long in sec the download link should be valid for
     */
    generateAttachmentDownloadPresignedUrl(tenantId: string, attachmentId: string, urlTimeToLiveInSec?: number, filename?: string): Promise<string>;
    /**
     * Copy a file to another folder. Note: this command will fail if the target file already exists
     * @param tenantId  related tenant nanoid
     * @param objectKey the object key of the file to copy (format=prefix/filename)
     * @param target the target object key of the file to copy (format=prefix/filename)
     * @param expiration the expiration for the new file (infinite if not set)
     * @returns the objectKey of the new file
     *
     * @example copyFileToFolder('xxxx', 'print-output/yyyy', 'uploads', FileTimeToLive.Expire30Days)
     * Copy the file 'print-output/yyyy' to the 'uploads' folder with a 30 days expiration
     */
    copyFileToFolder(tenantId: string, objectKey: string, target: string, expiration?: FileTimeToLive): Promise<string>;
}
