import { S3ClientConfig, Tagging } from "@aws-sdk/client-s3";
import { FileTimeToLive, TenantDataType } from "@sage/xtrem-deployment";
import { FileStorageWrapper } from "./FileStorageWrapper";
import { S3BodyGet, S3BodyPut } from "./utils";
export interface FileStorageConfig {
    awsConfigOverride?: S3ClientConfig;
    bucket?: string;
    app?: string;
    maxListFiles?: number;
    /**
     * Use this parameter to force the use/not use of the local wrapper (if not set, the 'XTREM_ENV' environment variable will be
     * used: use localWrapper if XTREM_ENV === 'local')
     */
    useLocal?: boolean;
    localBasePath?: string;
    /**
     * The logger to be used (default is console.debug)
     */
    logger?: (message: string) => void;
}
/**
 *  That handler can be run locally or by using S3
 *
 *  - To run locally: you can create a new instance of that handler without passing any params.
 *    The files will be created directly in the current directory
 *    ex: const handler = new FileStorageHandler()
 *
 *  - Using S3: you must pass your bucket name in the constructor.
 *    ex:
 *      const config = {
 *          bucket: "your-bucket-name"
 *      }
 *
 *      const handler = new FileStorageHandler(config)
 */
export declare class FileStorageHandler {
    fileStorageFunctions: FileStorageWrapper;
    UNIT_TEST: boolean;
    private readonly logger;
    constructor(config?: FileStorageConfig, s3Mock?: any);
    setUnitTestMode(value: boolean): void;
    private static checkParametersForNullOrBlankOrInvalid;
    createFile(tenantId: string, objectKey: string, description: string, byteStream: S3BodyPut, expiration: FileTimeToLive): Promise<any>;
    updateExpiration(tenantId: string, objectKey: string, expiration: FileTimeToLive): Promise<any>;
    updateTagging(tenantId: string, objectKey: string, tagging: Tagging): Promise<any>;
    /***
     * Return A stream you can use with .pipe(file) to store on a temp location.
     * @param tenantId
     * @param objectKey
     */
    readFile(tenantId: string, objectKey: string): Promise<void | {
        tagData: any;
        body: S3BodyGet;
    }>;
    list(tenantDataType: TenantDataType, tenantId: string, continuationToken?: string): Promise<any>;
    listFiles(tenantId: string, continuationToken?: string): Promise<any>;
    listExports(tenantId: string, continuationToken?: string): Promise<any>;
    listImports(tenantId: string, continuationToken?: string): Promise<any>;
    listTenants(): Promise<{
        [key: string]: import("@aws-sdk/client-s3")._Object[];
    }>;
    generateDownloadPresignedUrl(tenantId: string, objectKey: string, urlTimeToLiveInSec?: number, filename?: string): Promise<string>;
    generateUploadPresignedUrl(tenantId: string, contextId: string, urlTimeToLiveInSec?: number): Promise<string>;
    readAsyncContextResult(tenantId: string, contextId: string, contextKind: string): Promise<{
        tagData: any;
        body: S3BodyGet;
    } | void>;
    /***
     * Provide a presigned URL to be shared to the client to upload an attachment.
     * It will be uploaded in a dirty bucket and after scan is done it will end in the tenant's "attachments" subfolder
     * See https://confluence.sage.com/display/XTREEM/Tenant+S3+file+system+organization
     * @param tenantId related tenant id
     * @param contextId the async context id related to the upload make sure you pick attachment upload context kind
     * @param urlTimeToLiveInSec optional, how long in second the upload url should be valid for
     */
    generateAttachmentUploadPresignedUrl(tenantId: string, contextId: string, urlTimeToLiveInSec?: number): Promise<string>;
    /**
     * Provide a download presigned url to be shared with the client to download an existing attachment
     * It will link to file in the tenant's "attachments" subfolder
     * See https://confluence.sage.com/display/XTREEM/Tenant+S3+file+system+organization
     * @param tenantId Related tenant id
     * @param attachmentId the attachement id, make sure status was correctly uploadded otherwise the file may not exist
     * @param urlTimeToLiveInSec optional, how long in second the download url should be valid for
     */
    generateAttachmentDownloadPresignedUrl(tenantId: string, attachmentId: string, urlTimeToLiveInSec?: number, filename?: string): Promise<string>;
    /**
    * Copy a file to another folder. Note: this command will fail if the target file already exists
    * @param tenantId  related tenant nanoid
    * @param objectKey the object key of the file to copy (format=prefix/filename)
    * @param target the target folder to copy the file to
    * @param expiration the expiration for the new file (infinite if not set)
    * @returns the objectKey of the new file
    *
    * @example copyFileToFolder('xxxx', 'print-output/yyyy', 'uploads', FileTimeToLive.Expire30Days)
    * Copy the file 'print-output/yyyy' to the 'uploads' folder with a 30 days expiration
    */
    copyFileToFolder(tenantId: string, objectKey: string, target: string, expiration?: FileTimeToLive): Promise<string>;
}
