"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseObjectLiteralToObject = void 0;
exports.getDirName = getDirName;
exports.getFileName = getFileName;
exports.saveAsJson = saveAsJson;
exports.getCrudBulkMutationNames = getCrudBulkMutationNames;
exports.getDecoratorArguments = getDecoratorArguments;
exports.getNodeStorageType = getNodeStorageType;
exports.isSubNodeDecorator = isSubNodeDecorator;
exports.isNodeExtensionDecorator = isNodeExtensionDecorator;
exports.isAbstractNodeDecorator = isAbstractNodeDecorator;
exports.canExportNodeDecorator = canExportNodeDecorator;
exports.hashFile = hashFile;
exports.getMetadataFilePathForClientFile = getMetadataFilePathForClientFile;
const crypto = require("crypto");
const fs = require("fs");
const lodash_1 = require("lodash");
const path = require("path");
const ts = require("typescript");
/**
 * Gets the directory a file belongs to
 * @param file Typescript source file
 * @param relativeToLib Should we get directory relative to the lib folder in the path
 *                      or do we get that last directory in the path. Default to false
 * @returns name of th directory the given file belongs to
 */
function getDirName(file, relativeToLib = false) {
    const pathElements = path.normalize(path.dirname(file.fileName)).split(path.sep);
    const indexOfLib = pathElements.indexOf('lib');
    if (relativeToLib && indexOfLib >= 0 && indexOfLib < pathElements.length - 1) {
        return pathElements[indexOfLib + 1];
    }
    return pathElements.pop();
}
/**
 * Gets the base name of a file
 * @param file Typescript source file
 * @returns base name of the given file without any extension
 */
function getFileName(file) {
    return path.basename(file.fileName, '.ts');
}
const parseObjectLiteralToObject = (objectLiteral) => {
    const targetObject = {};
    objectLiteral.properties.forEach(p => {
        if (ts.isPropertyAssignment(p) && p.name && p.initializer) {
            const name = p.name.getText();
            if (ts.isStringLiteral(p.initializer)) {
                targetObject[name] = (0, lodash_1.trim)(p.initializer.getText(), '"\'');
            }
            else if (ts.isNumericLiteral(p.initializer)) {
                targetObject[name] = Number(p.initializer.getText());
            }
            else if (p.initializer.kind === ts.SyntaxKind.TrueKeyword) {
                targetObject[name] = true;
            }
            else if (p.initializer.kind === ts.SyntaxKind.FalseKeyword) {
                targetObject[name] = false;
            }
            else if (ts.isObjectLiteralExpression(p.initializer)) {
                targetObject[name] = (0, exports.parseObjectLiteralToObject)(p.initializer);
            }
        }
    });
    return targetObject;
};
exports.parseObjectLiteralToObject = parseObjectLiteralToObject;
function saveAsJson(filename, obj) {
    fs.writeFileSync(filename, `${JSON.stringify(obj, null, 4)}\n`, 'utf-8');
}
function getCrudBulkMutationNames(filename, node) {
    const names = [];
    getDecoratorArguments(filename, node).forEach(property => {
        if (ts.isPropertyAssignment(property) &&
            property.initializer &&
            property.initializer.kind === ts.SyntaxKind.TrueKeyword &&
            ts.isIdentifier(property.name)) {
            const propertyName = property.name;
            const m = /^canBulk(Delete|Update)$/.exec(propertyName.escapedText.toString());
            if (m) {
                names.push(`bulk${m[1]}`);
            }
        }
    });
    return names;
}
function getDecoratorArguments(filename, node) {
    if (!ts.isCallExpression(node.expression)) {
        throw new Error(`${filename}: expected CallExpression, got ${ts.SyntaxKind[node.expression.kind]}`);
    }
    const decoratorArgument = node.expression.arguments[0];
    if (!decoratorArgument || !ts.isObjectLiteralExpression(decoratorArgument)) {
        throw new Error(`${filename}: expected ObjectLiteralExpression, got ${decoratorArgument && ts.SyntaxKind[decoratorArgument.kind]}`);
    }
    return decoratorArgument.properties;
}
function getNodeStorageType(filename, node) {
    const storage = getDecoratorArguments(filename, node).find(property => ts.isPropertyAssignment(property) &&
        property.initializer &&
        property.initializer.kind === ts.SyntaxKind.StringLiteral &&
        ts.isIdentifier(property.name) &&
        property.name.escapedText.toString() === 'storage');
    if (storage &&
        ts.isPropertyAssignment(storage) &&
        storage.initializer &&
        storage.initializer.kind === ts.SyntaxKind.StringLiteral)
        return storage.initializer.getText();
    return undefined;
}
function isSubNodeDecorator(node) {
    return (ts.isDecorator(node) &&
        ts.isCallExpression(node.expression) &&
        ts.isPropertyAccessExpression(node.expression.expression) &&
        node.expression.expression.getText() === 'decorators.subNode');
}
function isNodeExtensionDecorator(node) {
    return (ts.isDecorator(node) &&
        ts.isCallExpression(node.expression) &&
        ts.isPropertyAccessExpression(node.expression.expression) &&
        node.expression.expression.getText() === 'decorators.nodeExtension');
}
function isAbstractNodeDecorator(filename, node) {
    if (!ts.isCallExpression(node.expression)) {
        throw new Error(`${filename}: expected CallExpression, got ${ts.SyntaxKind[node.expression.kind]}`);
    }
    const decoratorArgument = node.expression.arguments[0];
    if (!decoratorArgument || !ts.isObjectLiteralExpression(decoratorArgument)) {
        throw new Error(`${filename}: expected ObjectLiteralExpression, got ${decoratorArgument && ts.SyntaxKind[decoratorArgument.kind]}`);
    }
    return decoratorArgument.properties.some(property => {
        if (ts.isPropertyAssignment(property) &&
            property.initializer.kind === ts.SyntaxKind.TrueKeyword &&
            ts.isIdentifier(property.name)) {
            const propertyName = property.name;
            if (propertyName.escapedText.toString() === 'isAbstract') {
                return true;
            }
        }
        return false;
    });
}
function canExportNodeDecorator(filename, node) {
    if (!ts.isCallExpression(node.expression)) {
        throw new Error(`${filename}: expected CallExpression, got ${ts.SyntaxKind[node.expression.kind]}`);
    }
    const decoratorArgument = node.expression.arguments[0];
    if (!decoratorArgument || !ts.isObjectLiteralExpression(decoratorArgument)) {
        throw new Error(`${filename}: expected ObjectLiteralExpression, got ${decoratorArgument && ts.SyntaxKind[decoratorArgument.kind]}`);
    }
    return decoratorArgument.properties.every(property => {
        if (ts.isPropertyAssignment(property) &&
            property.initializer.kind === ts.SyntaxKind.FalseKeyword &&
            ts.isIdentifier(property.name)) {
            const propertyName = property.name;
            if (propertyName.escapedText.toString() === 'canExport') {
                return false;
            }
        }
        return true;
    });
}
function hashFile(filePath) {
    return crypto.createHash('md5').update(fs.readFileSync(filePath, 'utf-8')).digest('hex');
}
function getMetadataFilePathForClientFile(dir, filePath) {
    const artifactFolderName = path.basename(path.dirname(filePath));
    const buildLibArtifactDir = path.resolve(dir, `build/lib/${artifactFolderName}`);
    const metaFilename = path.basename(filePath).replace(/\.ts$/, '.meta.json');
    return path.resolve(buildLibArtifactDir, metaFilename);
}
//# sourceMappingURL=utils.js.map