"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.visitor = void 0;
exports.serviceOptionNameExtractor = serviceOptionNameExtractor;
const xtrem_i18n_1 = require("@sage/xtrem-i18n");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const lodash_1 = require("lodash");
const ts = require("typescript");
const utils_1 = require("../utils");
const transformer_utils_1 = require("./transformer-utils");
const excludedAstTypes = new Set([
    ts.SyntaxKind.ImportDeclaration,
    ts.SyntaxKind.TypePredicate,
    ts.SyntaxKind.TypeReference,
]);
/**
 * Top-level TS visitor
 * @param prefix prefix for translation keys
 */
const visitor = (dictionary, packageName, dirName, filename, ctx) => (node) => {
    if (excludedAstTypes.has(node.kind)) {
        return node;
    }
    if (ts.isVariableStatement(node) && dirName === 'service-options') {
        const declaration = node.declarationList.declarations.find(decl => decl.initializer &&
            ts.isNewExpression(decl.initializer) &&
            ts.isIdentifier(decl.initializer.expression) &&
            decl.initializer.expression.text === 'ServiceOption');
        if (declaration && ts.isIdentifier(declaration.name)) {
            const serviceOptionName = declaration.name.text;
            const key = `${packageName}/${(0, transformer_utils_1.createDictionaryKey)('service_options', (0, lodash_1.snakeCase)(serviceOptionName))}__name`;
            dictionary[key] = (0, xtrem_shared_1.titleCase)(serviceOptionName);
            return node;
        }
    }
    return ts.visitEachChild(node, (0, exports.visitor)(dictionary, packageName, dirName, filename, ctx), ctx);
};
exports.visitor = visitor;
/**
 * This transformer is meant to be run BEFORE the 'message-transformer' and its purpose is
 * to wrap some page/sticker decorator properties (see 'includedProperties') with a call to the 'ui.localize' function.
 *
 * @param ctx transformation context
 * @returns the transformed file
 */
function serviceOptionNameExtractor(ctx) {
    return (file) => {
        if (!(0, transformer_utils_1.isServiceOptionFile)(file) || (0, transformer_utils_1.isTestFile)(file)) {
            return file;
        }
        const filename = file.fileName;
        try {
            const dictionary = {};
            const dirName = (0, utils_1.getDirName)(file, true);
            const nameAndRoot = (0, xtrem_i18n_1.getPackageNameAndRoot)(filename);
            const result = ts.visitNode(file, (0, exports.visitor)(dictionary, nameAndRoot.name, dirName, filename, ctx));
            (0, transformer_utils_1.writeStringLiteralsToBase)(dictionary, nameAndRoot.root);
            return result;
        }
        catch (err) {
            throw new Error(`${filename}: decorator transformer failed: ${err.message}`);
        }
    };
}
//# sourceMappingURL=service-option-name-extractor.js.map