"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isNodeLikeArtifactFile = isNodeLikeArtifactFile;
exports.createQueueExtractor = createQueueExtractor;
const path = require("path");
const ts = require("typescript");
const __1 = require("..");
const logger = require("./queue-builder/logger");
function extractQueueInfo(node) {
    if (ts.isVariableDeclaration(node) && node.initializer && ts.isNewExpression(node.initializer)) {
        const queueType = /^Sqs(Receive|Send)Queue$/.exec(node.initializer.expression.getText())?.[1];
        if (queueType) {
            const args = node.initializer.arguments;
            const properties = args && ts.isObjectLiteralExpression(args[0]) ? args[0].properties : [];
            const property = properties.find(prop => ts.isPropertyAssignment(prop) && prop.name.getText() === 'name');
            if (property && ts.isPropertyAssignment(property) && ts.isStringLiteral(property.initializer)) {
                return {
                    name: property.initializer.text,
                    type: queueType.toLowerCase(),
                };
            }
            throw new Error(`Queue name not found or not a string literal in: ${node.getText()}`);
        }
    }
    return undefined;
}
const queueExtractor = (packageContext, context) => {
    const { routingBuilder, messageQueueBuilder } = packageContext;
    return (sourceFile) => {
        const filePath = sourceFile.fileName;
        const { isQueue, isDevOnly } = messageQueueBuilder.getQueueFileDetails(filePath);
        if (!isQueue && !isNodeLikeArtifactFile(sourceFile)) {
            return sourceFile;
        }
        routingBuilder.setVisitingFile(sourceFile.fileName);
        const visitor = (node) => {
            let visitingNodeDecorator = false;
            try {
                if (isListenerDecorator(node, 'notificationListener')) {
                    const listener = routingBuilder.createListener(node);
                    routingBuilder.updateRouting(listener);
                }
                else if (isListenerDecorator(node, 'asyncMutation') || isListenerDecorator(node, 'bulkMutation')) {
                    const listener = routingBuilder.createAsyncMutationListener(node);
                    routingBuilder.updateRouting(listener);
                }
                else if (isNodeDecorator(node) || (0, __1.isSubNodeDecorator)(node)) {
                    visitingNodeDecorator = true;
                    routingBuilder.setVisitingNodeDecorator(node);
                    routingBuilder.createListeners();
                }
                else if (isQueue) {
                    // Look for SqsReceiveQueue and SqsSendQueue instantiations to collect the queue names
                    const queueInfo = extractQueueInfo(node);
                    if (queueInfo) {
                        messageQueueBuilder.addQueue(queueInfo.name, {
                            type: queueInfo.type,
                            devOnly: isDevOnly,
                        });
                    }
                }
                return ts.visitEachChild(node, visitor, context);
            }
            catch (err) {
                const { line, character } = sourceFile.getLineAndCharacterOfPosition(node.getStart());
                logger.error(err);
                logger.error(`fileName: ${sourceFile.fileName}:${line + 1}:${character + 1}`);
                logger.error(`nodeText: ${node.getText()}`);
                throw err;
            }
            finally {
                if (visitingNodeDecorator) {
                    routingBuilder.setVisitingNodeDecorator(undefined);
                }
            }
        };
        const returnNode = ts.visitNode(sourceFile, visitor);
        return returnNode;
    };
};
function isNodeLikeArtifactFile(file) {
    const filePath = file.fileName;
    const fileName = path.basename(filePath, '.ts');
    return (fileName !== 'index' &&
        fileName !== '_index' &&
        !/\/(x3-services|wh-services|wms|test|fixtures)\//.test(filePath) &&
        /[/\\]lib[/\\](nodes|node-extensions)/.test(filePath));
}
function isListenerDecorator(node, name) {
    return (ts.isCallExpression(node) &&
        ts.isDecorator(node.parent) &&
        ts.isPropertyAccessExpression(node.expression) &&
        node.expression.name.escapedText === name);
}
function isNodeDecorator(node) {
    return (ts.isDecorator(node) &&
        ts.isClassDeclaration(node.parent) &&
        ts.isCallExpression(node.expression) &&
        ts.isPropertyAccessExpression(node.expression.expression) &&
        node.expression.expression.getText() === 'decorators.node');
}
function createQueueExtractor(packageContext) {
    return queueExtractor.bind(undefined, packageContext);
}
//# sourceMappingURL=queue-extractor.js.map