"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.defaultElasticMq = void 0;
exports.mergeRoutings = mergeRoutings;
exports.sortRoutesByTopic = sortRoutesByTopic;
exports.getRoutingData = getRoutingData;
exports.getRoutingFilePath = getRoutingFilePath;
exports.getMessageQueueFilePath = getMessageQueueFilePath;
exports.getMessageQueueData = getMessageQueueData;
exports.getAllElasticMqConfigs = getAllElasticMqConfigs;
exports.updateElasticMqConfigFile = updateElasticMqConfigFile;
exports.syncSharedElasticMqConfigFile = syncSharedElasticMqConfigFile;
exports.getElasticMqConfigs = getElasticMqConfigs;
const xtrem_shared_1 = require("@sage/xtrem-shared");
const fs = require("fs");
const _ = require("lodash");
const os = require("os");
const fsp = require("path");
const logger = require("./logger");
function mergeRoutings(routing1, routing2) {
    return sortRoutesByTopic(_.mergeWith(routing1, routing2, mergeRoutes));
}
function mergeRoutes(routes1, routes2) {
    return _.unionWith(routes1, routes2, _.isEqual);
}
function sortRoutesByTopic(routing) {
    // Sort each route array by topic
    Object.keys(routing).forEach(packageName => {
        routing[packageName].sort((a, b) => a.topic.localeCompare(b.topic));
    });
    return routing;
}
function getRoutingData(routingFile) {
    logger.info(`Getting data from file ${routingFile}`);
    const rawRoutingData = fs.readFileSync(routingFile, 'utf-8');
    return JSON.parse(rawRoutingData);
}
function getRoutingFilePath(packageDir) {
    return fsp.join(packageDir, 'routing.json');
}
function getMessageQueueFilePath(packageDir) {
    return fsp.join(packageDir, 'message-queue.json');
}
function getMessageQueueData(messageQueueFile) {
    logger.info(`Getting data from file ${messageQueueFile}`);
    const rawMessageQueueData = fs.readFileSync(messageQueueFile, 'utf-8');
    return JSON.parse(rawMessageQueueData);
}
function getQueues(routing) {
    const queueList = new Set();
    Object.values(routing).forEach(routes => {
        const queues = _.uniq(routes.map(route => route.queue));
        queues.forEach(queue => queueList.add(queue));
    });
    return [...queueList];
}
const defaultVisibilityTimeout = '30 seconds';
const delay = '0 seconds';
const receiveMessageWait = '0 seconds';
const fifo = 'true';
const deadLetterQueueName = 'dev-dead-letter-queue';
/**
 * Generates the Elastic Queue configuration for a given queue name.
 * @param queueName - The name of the queue.
 * @returns The Elastic Queue configuration.
 */
function elasticQueueConfig(queueName) {
    const deadLetterQueue = queueName === deadLetterQueueName ? undefined : { name: deadLetterQueueName, maxReceiveCount: 5 };
    return {
        [queueName]: {
            defaultVisibilityTimeout,
            delay,
            receiveMessageWait,
            fifo,
            deadLetterQueue,
        },
    };
}
/**
 * Retrieves all ElasticMQ configurations based on the provided application and routing.
 * Optionally, a test package name can be specified to generate unit test queues.
 *
 * @param application - The application object.
 * @param routing - The routing object.
 * @param testPackageName - The name of the optional test package.
 * @returns A dictionary of queue configurations.
 */
function getAllElasticMqConfigs(dir, routing, testPackageName, additionalQueues) {
    const queues = {
        ...getQueues(routing).reduce((r, k) => {
            // Prefix with test-${testPackageName} to generate the unit test queues
            // This name must match what we generate in application.sqsQueueName when application.applicationType is 'test'
            // we cannot use that method here because application.applicationType is 'dev-tool'.
            const queueConfig = elasticQueueConfig((0, xtrem_shared_1.limitSqsQueueName)(testPackageName ? `test-${testPackageName}-${k}` : k));
            const queueName = Object.keys(queueConfig)[0];
            if (r[queueName]) {
                throw new xtrem_shared_1.LogicError(`Duplicate queue name ${queueName} in routing file`);
            }
            if (additionalQueues && additionalQueues[queueName]) {
                throw new xtrem_shared_1.LogicError(`Duplicate queue name ${queueName} in additional queues`);
            }
            r[queueName] = queueConfig[queueName];
            return r;
        }, {}),
        ...additionalQueues,
    };
    if (!testPackageName) {
        // Add queue variants prefixed by the app name.
        // We have to generate all the queue names regardless of whether the 'app' key is present in the config file.
        let appName = dir.split('/').at(-3);
        // Special hacks to guess the app name from the directory structure.
        if (appName === 'services')
            appName = 'sdmo';
        else if (appName === 'tools')
            appName = dir.split('/').at(-2);
        Object.keys(queues).forEach(queueName => {
            if (!appName)
                throw new xtrem_shared_1.LogicError('Application directory does not match expected format');
            if (!queueName.startsWith(`${appName}--`)) {
                queues[(0, xtrem_shared_1.limitSqsQueueName)(`${appName}--${queueName}`)] = queues[queueName];
            }
        });
    }
    return queues;
}
/**
 * Sorts the ElasticMQ configurations based on their keys in ascending order.
 *
 * @param queues - The dictionary of queues to be sorted.
 * @returns The sorted dictionary of queues.
 */
function sortElasticMqConfigs(queues) {
    const sortedKeys = Object.keys(queues).sort((a, b) => a.localeCompare(b));
    return _.pick(queues, sortedKeys);
}
/**
 * Writes the ElasticMQ configuration to a file.
 *
 * @param filePath - The path of the file to write the configuration to.
 * @param queues - The dictionary of queue configurations.
 */
function writeElasticMqConfigFile(filePath, queues) {
    const include = 'include classpath("application.conf")';
    const queuesStr = JSON.stringify(sortElasticMqConfigs(queues), null, '\t')
        .replace(/: \{/g, ' {')
        .replace(/:/g, ' =')
        .replace(/[",]/g, '');
    const text = `${include}\n\nqueues ${queuesStr}\n`;
    fs.writeFileSync(filePath, text, 'utf-8');
}
/**
 * Reads the ElasticMQ configuration from the specified root path.
 * @param path The path of the configuration file.
 * @returns A dictionary of queue configurations.
 */
function readElasticMqConfigFile(path) {
    if (!fs.existsSync(path))
        return {};
    const text = fs.readFileSync(path, 'utf-8');
    let json = text
        .replace(/^include.*/, '')
        .replace(/\nqueues\s+{/, '{')
        .replace(/(\n\t+)((?:\w|-)+) \{/g, `$1"$2": {`)
        .replace(/(\n\t+)((?:\w|-)+) = (.*)/g, `$1"$2": "$3",`)
        .replace(/([}"])\n/g, `$1,\n`)
        .replace(/,(\n\t*)\}/g, '$1}')
        .trimEnd();
    if (json.slice(-1) === ',')
        json = json.slice(0, -1);
    return JSON.parse(json);
}
/**
 * Updates the ElasticMQ configuration with the provided queues.
 *
 * @param {string} projectConfigDir - The path to the project root dir where the local ElasticMq configuration file will be saved.
 * @param {Dict<QueueConfig>} queues - The queues to be added or updated in the configuration.
 * @returns {void}
 */
function updateElasticMqConfigFile(workspaceId, projectConfigDir, queues) {
    if (!fs.existsSync(projectConfigDir)) {
        throw new xtrem_shared_1.LogicError(`Local conf dir ${projectConfigDir} does not exist. It should be the root of the project.`);
    }
    // Merge the queues with the existing ones in the project config dir
    // Required for CI and local dev when we get builds from the cache to ensure we have all queues
    const projectConfigFile = fsp.join(projectConfigDir, 'elasticmq.conf');
    const oldProjectQueues = readElasticMqConfigFile(projectConfigFile);
    writeElasticMqConfigFile(projectConfigFile, _.merge(oldProjectQueues, queues));
    updateSharedElasticMqConfigFile(workspaceId, queues);
}
function updateSharedElasticMqConfigFile(workspaceId, queues) {
    // The shared conf dir is used to store the queues for all workspaces to use a single conf file
    // for multi apps/repos hosted on the same machine
    const sharedConfDir = fsp.join(os.homedir(), '.cache', 'elasticmq');
    if (!fs.existsSync(sharedConfDir)) {
        fs.mkdirSync(sharedConfDir, { recursive: true });
    }
    // create a new dict of queues prefixed with the workspaceId
    const newQueues = {};
    Object.keys(queues).forEach(sqsName => {
        const queueConfig = queues[sqsName];
        if (queueConfig) {
            const newSqsName = (0, xtrem_shared_1.limitSqsQueueName)(`${workspaceId}--${sqsName}`);
            newQueues[newSqsName] = queueConfig;
        }
    });
    const sharedConfFile = fsp.join(sharedConfDir, 'elasticmq.conf');
    const oldSharedQueues = readElasticMqConfigFile(sharedConfFile);
    writeElasticMqConfigFile(sharedConfFile, _.merge(oldSharedQueues, newQueues));
}
function syncSharedElasticMqConfigFile(workspaceId, projectConfigDir) {
    const projectQueues = readElasticMqConfigFile(fsp.join(projectConfigDir, 'elasticmq.conf'));
    updateSharedElasticMqConfigFile(workspaceId, projectQueues);
}
function getElasticMqConfigs(names) {
    return names.reduce((acc, name) => Object.assign(acc, elasticQueueConfig(name)), (0, xtrem_shared_1.createDictionary)());
}
exports.defaultElasticMq = {
    queues: {
        ...elasticQueueConfig(deadLetterQueueName),
    },
};
//# sourceMappingURL=routing.js.map