"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.messageTransformVisitor = void 0;
exports.messageTransformer = messageTransformer;
exports.tsPatchMessageTransformer = tsPatchMessageTransformer;
const xtrem_i18n_1 = require("@sage/xtrem-i18n");
const lodash_1 = require("lodash");
const ts = require("typescript");
const transformer_utils_1 = require("./transformer-utils");
const excludedAstTypes = new Set([
    ts.SyntaxKind.ImportDeclaration,
    ts.SyntaxKind.TypePredicate,
    ts.SyntaxKind.TypeReference,
]);
const serverLocalizableValidationErrorCalls = ['businessRuleError', 'authorizationError', 'dataInputError'];
const straightforwardLocalizationMethods = ['localize', 'addLocalized', 'withMessage'];
/**
 * Determines whether the given file is a source file
 * @param file the TS file to be checked
 * @returns true if file is a source file
 */
function isSourceFile(file) {
    return file.fileName.indexOf('node_modules') === -1 && Boolean(file.fileName.match(/(\/|\\)lib(\/|\\)/));
}
/**
 * Adds an entry to the dictionary
 * @param key the dictionary key
 * @param value the dictionary value
 */
function addDictionaryEntry(dictionary, key, value) {
    if (dictionary[key] !== undefined && dictionary[key] !== value) {
        throw new Error(`Duplicated key '${key}' with different values: '${dictionary[key]}' and '${value}'`);
    }
    dictionary[key] = value;
}
/**
 * Typescript visitor
 * @param ctx the transformation context
 */
const messageTransformVisitor = (dictionary, pkgName, fileName, ctx, skipPackageValidation = false) => (node) => {
    if (excludedAstTypes.has(node.kind)) {
        return node;
    }
    // Methods taking straightforward arguments: localize(key, template, data)
    if (ts.isCallExpression(node) &&
        ((node.expression.name &&
            straightforwardLocalizationMethods.includes(node.expression.name.text)) ||
            (node.expression.text &&
                straightforwardLocalizationMethods.includes(node.expression.text)))) {
        const key = node.arguments[0].text;
        if (key != null) {
            // Key is null if we parse something like `cx.localize(node.getLocalizedTitleKey(), ...)`
            if (!skipPackageValidation && key.indexOf(`${pkgName}/`) === -1) {
                throw new Error(`'${key}' in '${fileName}' should be prefixed with '${pkgName}/'`);
            }
            const value = node.arguments[1].text;
            addDictionaryEntry(dictionary, key, value);
        }
    }
    // Methods taking object arguments: localize({key, template, data})
    if (ts.isCallExpression(node) &&
        ((node.expression.name &&
            serverLocalizableValidationErrorCalls.includes(node.expression.name.text)) ||
            (node.expression.text &&
                serverLocalizableValidationErrorCalls.includes(node.expression.text) &&
                ts.isObjectLiteralExpression(node.arguments[0])))) {
        const objectLiteralExpression = node.arguments[0];
        const arg = objectLiteralExpression.properties.reduce((prevValue, element) => {
            if (ts.isPropertyAssignment(element) &&
                element.initializer &&
                ts.isStringLiteral(element.initializer)) {
                const name = element.name?.getText(node.getSourceFile());
                const value = (0, lodash_1.trim)(element.initializer.getText(node.getSourceFile()), '\'"');
                prevValue[name] = value;
            }
            return prevValue;
        }, {});
        if (arg.key && arg.message) {
            addDictionaryEntry(dictionary, arg.key, arg.message);
        }
    }
    return ts.visitEachChild(node, (0, exports.messageTransformVisitor)(dictionary, pkgName, fileName, ctx, skipPackageValidation), ctx);
};
exports.messageTransformVisitor = messageTransformVisitor;
/**
 * This transformer is meant to be run AFTER the 'decorator-transformer' for pages/stickers &
 * for every source file of an em-core package
 * @param ctx the transformation context
 * @returns the transformed file
 */
function messageTransformer(ctx) {
    return (file) => {
        if (!isSourceFile(file)) {
            return file;
        }
        try {
            const dictionary = {};
            const nameAndRoot = (0, xtrem_i18n_1.getPackageNameAndRoot)(file.fileName);
            const packageName = nameAndRoot.name;
            const result = ts.visitNode(file, (0, exports.messageTransformVisitor)(dictionary, packageName, file.fileName, ctx));
            (0, transformer_utils_1.writeStringLiteralsToBase)(dictionary, nameAndRoot.root);
            return result;
        }
        catch (err) {
            throw new Error(`Message transformer failed. ${err}`);
        }
    };
}
function tsPatchMessageTransformer() {
    return messageTransformer;
}
//# sourceMappingURL=message-transformer.js.map