"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.visitor = void 0;
exports.menuItemTransformer = menuItemTransformer;
const xtrem_i18n_1 = require("@sage/xtrem-i18n");
const lodash_1 = require("lodash");
const ts = require("typescript");
const transformer_utils_1 = require("./transformer-utils");
/**
 * Determines whether the given file is a menu item definition file
 * @param file the TS file to be checked
 * @returns true if file is a source file
 */
function isMenuItemFile(file) {
    return (file.fileName.indexOf('node_modules') === -1 &&
        !file.fileName.endsWith('index.ts') &&
        Boolean(file.fileName.match(/(\/|\\)lib(\/|\\)menu-items(\/|\\)/)));
}
/**
 * Adds an entry to the dictionary
 * @param key the dictionary key
 * @param value the dictionary value
 */
function addDictionaryEntry(dictionary, key, value) {
    if (dictionary[key] !== undefined && dictionary[key] !== value) {
        throw new Error(`Duplicated key '${key}' with different values: '${dictionary[key]}' and '${value}'`);
    }
    dictionary[key] = value;
}
/**
 * Typescript visitor
 * @param ctx the transformation context
 */
const visitor = (dictionary, pkgName, fileName, ctx) => (node) => {
    if (ts.isVariableStatement(node) &&
        node.declarationList.declarations[0] &&
        ts.isVariableDeclaration(node.declarationList.declarations[0]) &&
        node.declarationList.declarations[0].initializer &&
        ts.isObjectLiteralExpression(node.declarationList.declarations[0].initializer)) {
        const objectLiteral = node.declarationList.declarations[0].initializer;
        const id = objectLiteral.properties.find(p => ts.isPropertyAssignment(p) && p.name?.getText(node.getSourceFile()) === 'id');
        const title = objectLiteral.properties.find(p => ts.isPropertyAssignment(p) && p.name?.getText(node.getSourceFile()) === 'title');
        if (id && title) {
            const idContent = (0, lodash_1.trim)(id.initializer.getText(node.getSourceFile()), "'");
            const titleContent = (0, lodash_1.trim)(title.initializer.getText(node.getSourceFile()), "'");
            if (idContent && !idContent.startsWith(pkgName)) {
                throw new Error(`Menu item ID '${idContent}' must start with the package name '${pkgName}'`);
            }
            if (idContent && titleContent) {
                const key = (0, xtrem_i18n_1.menuItemIdToStringKey)(idContent);
                addDictionaryEntry(dictionary, key, titleContent);
            }
        }
    }
    return ts.visitEachChild(node, (0, exports.visitor)(dictionary, pkgName, fileName, ctx), ctx);
};
exports.visitor = visitor;
/**
 * This transformer extract menu item labels for translations from `menu-items` artifacts.
 * @param ctx the transformation context
 * @returns the transformed file
 */
function menuItemTransformer(ctx) {
    return (file) => {
        if (!isMenuItemFile(file)) {
            return file;
        }
        try {
            const dictionary = {};
            const nameAndRoot = (0, xtrem_i18n_1.getPackageNameAndRoot)(file.fileName);
            const packageName = nameAndRoot.name;
            ts.visitNode(file, (0, exports.visitor)(dictionary, packageName, file.fileName, ctx));
            (0, transformer_utils_1.writeStringLiteralsToBase)(dictionary, nameAndRoot.root);
            return file;
        }
        catch (err) {
            throw new Error(`Menu item transformer failed. ${err}`);
        }
    };
}
//# sourceMappingURL=menu-item-transformer.js.map