"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.enumTransformer = enumTransformer;
const xtrem_i18n_1 = require("@sage/xtrem-i18n");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const lodash_1 = require("lodash");
const ts = require("typescript");
const utils_1 = require("../utils");
const transformer_utils_1 = require("./transformer-utils");
/**
 * This transformer is meant to be run AFTER the 'message-transformer' for pages/stickers &
 * for every source file of a xtrem-services package
 * @param ctx the transformation context
 * @returns the transformed file
 */
function enumTransformer() {
    return (file) => {
        if ((0, transformer_utils_1.isEnumFile)(file)) {
            const enumDeclaration = file.statements.find(ts.isEnumDeclaration);
            const enumTypeDeclaration = file.statements.find(node => ts.isTypeAliasDeclaration(node) && node.modifiers?.[0].kind === ts.SyntaxKind.ExportKeyword);
            if (enumDeclaration) {
                const nameAndRoot = (0, xtrem_i18n_1.getPackageNameAndRoot)(file.fileName);
                const packageName = nameAndRoot.name;
                const fileName = (0, utils_1.getFileName)(file);
                const dirName = (0, utils_1.getDirName)(file);
                const fileKey = `${packageName}/${(0, transformer_utils_1.createDictionaryKey)(dirName, fileName)}`;
                const dictionary = {};
                const enumName = enumDeclaration.name.getText();
                if (!enumTypeDeclaration)
                    throw new Error(`Enum name ${enumName} must declare a type in file ${dirName}/${fileName}.ts`);
                const enumTypeName = enumTypeDeclaration.name.getText();
                const expectedFilename = (0, lodash_1.kebabCase)(enumTypeName);
                if (expectedFilename !== fileName)
                    throw new Error(`Invalid file name for enum type '${enumTypeName}'. Got ${fileName}.ts, expected ${expectedFilename}.ts`);
                enumDeclaration.members.forEach(m => {
                    const memberName = (0, lodash_1.trim)(m.name.getText(), "'");
                    dictionary[`${fileKey}__${memberName}`] = (0, xtrem_shared_1.titleCase)(memberName);
                });
                (0, transformer_utils_1.writeStringLiteralsToBase)(dictionary, nameAndRoot.root);
            }
        }
        return file;
    };
}
//# sourceMappingURL=enum-transformer.js.map