"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.visitor = void 0;
exports.dataTypesNameExtractor = dataTypesNameExtractor;
const xtrem_i18n_1 = require("@sage/xtrem-i18n");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const lodash_1 = require("lodash");
const ts = require("typescript");
const utils_1 = require("../utils");
const transformer_utils_1 = require("./transformer-utils");
const excludedAstTypes = new Set([
    ts.SyntaxKind.ImportDeclaration,
    ts.SyntaxKind.TypePredicate,
    ts.SyntaxKind.TypeReference,
]);
/**
 * Top-level TS visitor
 * @param prefix prefix for translation keys
 */
const visitor = (dictionary, packageName, dirName, file, ctx) => (node) => {
    if (excludedAstTypes.has(node.kind)) {
        return node;
    }
    if ((0, transformer_utils_1.isEnumFile)(file)) {
        const enumDeclaration = file.statements.find(ts.isEnumDeclaration);
        if (enumDeclaration) {
            const dataTypeName = enumDeclaration?.name?.getText();
            const key = `${packageName}/${(0, transformer_utils_1.createDictionaryKey)('data_types', (0, lodash_1.snakeCase)(dataTypeName))}__name`;
            dictionary[key] = (0, xtrem_shared_1.titleCase)(dataTypeName);
        }
    }
    else if (ts.isVariableDeclaration(node) &&
        node.initializer &&
        ts.isNewExpression(node.initializer) &&
        ts.isIdentifier(node.initializer.expression) &&
        node.name &&
        ts.isIdentifier(node.name) &&
        node.initializer.expression.text.endsWith('DataType')) {
        const dataTypeName = node.name.text;
        const key = `${packageName}/${(0, transformer_utils_1.createDictionaryKey)('data_types', (0, lodash_1.snakeCase)(dataTypeName))}__name`;
        dictionary[key] = (0, xtrem_shared_1.titleCase)(dataTypeName);
    }
    return ts.visitEachChild(node, (0, exports.visitor)(dictionary, packageName, dirName, file, ctx), ctx);
};
exports.visitor = visitor;
/**
 * This transformer extract data-types to create literals key/value for them
 *
 * @param ctx transformation context
 * @returns the transformed file
 */
function dataTypesNameExtractor(ctx) {
    return (file) => {
        if (!(0, transformer_utils_1.isDataTypeFile)(file)) {
            return file;
        }
        const filename = file.fileName;
        try {
            const dictionary = (0, xtrem_shared_1.createDictionary)();
            const dirName = (0, utils_1.getDirName)(file, true);
            const nameAndRoot = (0, xtrem_i18n_1.getPackageNameAndRoot)(filename);
            const packageName = nameAndRoot.name;
            const result = ts.visitNode(file, (0, exports.visitor)(dictionary, packageName, dirName, file, ctx));
            (0, transformer_utils_1.writeStringLiteralsToBase)(dictionary, nameAndRoot.root);
            return result;
        }
        catch (err) {
            throw new Error(`${filename}: dataType transformer failed: ${err.message}`);
        }
    };
}
//# sourceMappingURL=data-type-name-extractor.js.map