"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.visitor = void 0;
exports.clientListTransformer = clientListTransformer;
const xtrem_i18n_1 = require("@sage/xtrem-i18n");
const fs = require("fs");
const path = require("path");
const ts = require("typescript");
const utils_1 = require("../utils");
const excludedAstTypes = new Set([
    ts.SyntaxKind.ImportDeclaration,
    ts.SyntaxKind.TypePredicate,
    ts.SyntaxKind.TypeReference,
]);
/**
 * Determines whether the given file is a source file
 * @param file the TS file to be checked
 * @returns true if file is a source file
 */
function isSourceFile(file) {
    const fileName = path.parse(file.fileName).base;
    return (!['index.ts', '_index.ts'].includes(fileName) &&
        file.fileName.indexOf('node_modules') === -1 &&
        Boolean(file.fileName.match(/(\/|\\)lib(\/|\\)/)));
}
/**
 * Typescript visitor
 * @param ctx the transformation context
 */
const visitor = (usedStrings, usedEnums, file, ctx) => (node) => {
    if (excludedAstTypes.has(node.kind)) {
        return node;
    }
    // Check strings
    if (ts.isCallExpression(node) &&
        ((node.expression.name && node.expression.name.text === 'localize') ||
            (node.expression.text && node.expression.text === 'localize'))) {
        const key = node.arguments[0].text;
        usedStrings.push(key);
    }
    // Check enum localize calls
    if (ts.isCallExpression(node) &&
        ((node.expression.name && node.expression.name.text === 'localizeEnumMember') ||
            (node.expression.text && node.expression.text === 'localizeEnumMember'))) {
        const key = node.arguments[0].text;
        if (!usedEnums.includes(key))
            usedEnums.push(key);
    }
    // Check enums
    if (ts.isPropertyAssignment(node) && node.name.getText(file) === 'optionType') {
        const initializer = node.initializer;
        if (ts.isStringLiteral(initializer)) {
            usedEnums.push(initializer.text);
        }
    }
    return ts.visitEachChild(node, (0, exports.visitor)(usedStrings, usedEnums, file, ctx), ctx);
};
exports.visitor = visitor;
/**
 * This transformer is meant to be run AFTER the 'decorator-transformer' and 'message-transformer' for pages/stickers &
 * for every source file of an xtrem application package. For all client side artifacts it lists the strings that the
 * page uses so the server can send the right strings to the client.
 *
 * @param ctx the transformation context
 * @returns the transformed file
 */
function clientListTransformer(ctx) {
    return (file) => {
        if (isSourceFile(file)) {
            const strings = [];
            const enums = [];
            const nameAndRoot = (0, xtrem_i18n_1.getPackageNameAndRoot)(file.fileName);
            const artifactType = (0, utils_1.getDirName)(file);
            const artifactName = (0, utils_1.getFileName)(file);
            ts.visitNode(file, (0, exports.visitor)(strings, enums, file, ctx));
            const buildDir = path.resolve(nameAndRoot.root, 'build');
            safeMkdir(buildDir);
            const buildLibDir = path.resolve(buildDir, 'lib');
            safeMkdir(buildLibDir);
            const artifactDir = path.resolve(buildLibDir, artifactType);
            safeMkdir(artifactDir);
            const metadataFilePath = path.resolve(artifactDir, `${artifactName}.meta.json`);
            const artifactMetaData = {
                literals: { strings: Array.from(new Set(strings)), enums: Array.from(new Set(enums)) },
            };
            const dataToSave = fs.existsSync(metadataFilePath)
                ? { ...JSON.parse(fs.readFileSync(metadataFilePath, 'utf-8')), ...artifactMetaData }
                : artifactMetaData;
            (0, utils_1.saveAsJson)(metadataFilePath, dataToSave);
        }
        return file;
    };
}
// Normal and Binary build sometime reports EEXIST error. It might be due to a concurrency issue.
// It is easier to ignore EEXIST error and it avoids race conditions.
function safeMkdir(dir) {
    try {
        fs.mkdirSync(dir);
    }
    catch (e) {
        if (e.code !== 'EEXIST') {
            throw e;
        }
    }
}
//# sourceMappingURL=client-list-transformer.js.map