"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.visitor = void 0;
exports.activityNameExtractor = activityNameExtractor;
const xtrem_i18n_1 = require("@sage/xtrem-i18n");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const lodash_1 = require("lodash");
const ts = require("typescript");
const utils_1 = require("../utils");
const transformer_utils_1 = require("./transformer-utils");
const excludedAstTypes = new Set([
    ts.SyntaxKind.ImportDeclaration,
    ts.SyntaxKind.TypePredicate,
    ts.SyntaxKind.TypeReference,
]);
/**
 * Top-level TS visitor
 * @param prefix prefix for translation keys
 */
const visitor = (dictionary, packageName, dirName, filename, ctx) => (node) => {
    if (excludedAstTypes.has(node.kind)) {
        return node;
    }
    if (ts.isVariableStatement(node) && dirName === 'activities') {
        const declaration = node.declarationList.declarations.find(decl => decl.initializer &&
            ts.isNewExpression(decl.initializer) &&
            ts.isIdentifier(decl.initializer.expression) &&
            decl.initializer.expression.text === 'Activity');
        if (declaration && ts.isIdentifier(declaration.name)) {
            const activityName = declaration.name.text;
            const key = `${packageName}/${(0, transformer_utils_1.createDictionaryKey)('activity', (0, lodash_1.snakeCase)(activityName))}__name`;
            dictionary[key] = (0, xtrem_shared_1.titleCase)(activityName);
            if (declaration.initializer &&
                ts.isNewExpression(declaration.initializer) &&
                declaration.initializer.arguments?.[0] &&
                ts.isObjectLiteralExpression(declaration.initializer.arguments?.[0])) {
                const properties = declaration.initializer.arguments[0].properties;
                const permissionsProperty = properties.find(arg => ts.isPropertyAssignment(arg) &&
                    ts.isIdentifier(arg.name) &&
                    arg.name.text === 'permissions');
                if (ts.isArrayLiteralExpression(permissionsProperty.initializer)) {
                    const values = permissionsProperty.initializer.elements.map(e => e.getText());
                    values.forEach(permission => {
                        const permissionKey = `${packageName}/${(0, transformer_utils_1.createDictionaryKey)('permission', (0, lodash_1.snakeCase)(permission))}__name`;
                        dictionary[permissionKey] = (0, xtrem_shared_1.titleCase)(permission);
                    });
                }
            }
            return node;
        }
    }
    return ts.visitEachChild(node, (0, exports.visitor)(dictionary, packageName, dirName, filename, ctx), ctx);
};
exports.visitor = visitor;
/**
 * This transformer extract activities to create literals key/value for them
 *
 * @param ctx transformation context
 * @returns the transformed file
 */
function activityNameExtractor(ctx) {
    return (file) => {
        if (!(0, transformer_utils_1.isActivityFile)(file) || (0, transformer_utils_1.isTestFile)(file)) {
            return file;
        }
        const filename = file.fileName;
        try {
            const dictionary = (0, xtrem_shared_1.createDictionary)();
            const dirName = (0, utils_1.getDirName)(file, true);
            const nameAndRoot = (0, xtrem_i18n_1.getPackageNameAndRoot)(filename);
            const packageName = nameAndRoot.name;
            const result = ts.visitNode(file, (0, exports.visitor)(dictionary, packageName, dirName, filename, ctx));
            (0, transformer_utils_1.writeStringLiteralsToBase)(dictionary, nameAndRoot.root);
            return result;
        }
        catch (err) {
            throw new Error(`${filename}: activity transformer failed: ${err.message}`);
        }
    };
}
//# sourceMappingURL=activity-name-extractor.js.map