// From file: @sage/xtrem-cli-main/build/index.d.ts
declare module '@sage/xtrem-cli-main/index' {
	export * from '@sage/xtrem-cli-main/lib/index';
	//# sourceMappingURL=index.d.ts.map
}
// From file: @sage/xtrem-cli-main/build/lib/web-socket-hook.d.ts
declare module '@sage/xtrem-cli-main/lib/web-socket-hook' {
	/**
	 * Install the web socket hook into @sage/xtrem-service.
	 * This hook allows us to have a leaner packaging of X3 Services.
	 * It eliminates a direct dependency between @sage/xtrem-service and @sage/xtrem-infrastructure-adapter,
	 * which would bring a lot of heavy dependencies (@aws-sdk) into the service package.
	 */
	export function initializeWebSocketHook(): void;
	//# sourceMappingURL=web-socket-hook.d.ts.map
}
// From file: @sage/xtrem-cli-main/build/lib/index.d.ts
declare module '@sage/xtrem-cli-main/lib/index' {
	import type { Argv } from 'yargs';
	export { StartContext, StartOptions, startServices } from '@sage/xtrem-cli-main/lib/commands/handlers/start/index';
	export const plugin: (yargs: Argv) => Argv;
	export const builder: {};
	export const handler: {};
	//# sourceMappingURL=index.d.ts.map
}
// From file: @sage/xtrem-cli-main/build/lib/commands/start.d.ts
declare module '@sage/xtrem-cli-main/lib/commands/start' {
	import type { Argv } from 'yargs';
	import { StartOptions } from '@sage/xtrem-cli-main/lib/commands/handlers/start/index';
	/**
	 * The command name and its aliases
	 */
	export const command: string[];
	/**
	 * The command description
	 */
	export const desc = "Start an xtrem application";
	/**
	 * The command builder that defines the syntax of this command
	 */
	export const builder: (yargs: Argv) => Argv<{}>;
	/**
	 * The command handler called to execute the actions of this command
	 */
	export const handler: (argv: Argv<StartOptions>) => Promise<void>;
	//# sourceMappingURL=start.d.ts.map
}
// From file: @sage/xtrem-cli-main/build/lib/commands/handlers/start/worker.d.ts
declare module '@sage/xtrem-cli-main/lib/commands/handlers/start/worker' {
	import { Dict } from '@sage/xtrem-core';
	import { Cluster, Worker } from 'cluster';
	/**
	 * XtremWorker class manages individual worker processes in a Node.js cluster.
	 * It implements a least-active-requests load balancing strategy with auto-scaling capabilities.
	 *
	 * Key Features:
	 * - Dynamic worker pool management per request source
	 * - Least-active-requests load balancing algorithm
	 * - Auto-scaling based on request load and worker capacity
	 * - Request tracking and completion monitoring
	 * - Worker health monitoring and periodic logging
	 * - Graceful worker restart and cleanup
	 *
	 * Load Balancing Strategy:
	 * - Selects worker with fewest active requests
	 * - Uses last request time as tie-breaker when request counts are equal
	 * - Automatically scales up when workers exceed capacity thresholds
	 * - Scales down idle workers during low-load periods
	 */
	export class XtremWorker {
	    readonly initWorkerId: string;
	    readonly cluster: Cluster;
	    /** Promise that resolves when the worker is fully initialized */
	    promise: Promise<void>;
	    /**
	     * Creates a new XtremWorker instance.
	     *
	     * @param workerId the identifier of the worker (usually matches the request source)
	     * @param cluster the cluster instance used to manage (fork/spawn) worker processes
	     */
	    constructor(initWorkerId: string, cluster: Cluster);
	    /**
	     * Gets the unique identifier for this worker.
	     * @returns the worker ID string
	     */
	    get workerId(): string;
	    /**
	     * Initializes the worker by forking a new process.
	     * @returns this instance for method chaining
	     */
	    init(): this;
	    /**
	     * Forks a new worker process and sets up event handlers.
	     * This method:
	     * 1. Resets worker state (ready flag, port, active requests)
	     * 2. Forks a new Node.js cluster worker
	     * 3. Sets up IPC message handling for worker lifecycle events
	     * 4. Configures timeout handling for worker startup failures
	     */
	    fork(): void;
	    /**
	     * Gets the current Worker instance, creating a new one if necessary.
	     * If the worker is dead or doesn't exist, it automatically forks a new one.
	     * @returns the active Worker instance
	     */
	    get worker(): Worker;
	    /**
	     * Gets the index/ID of the worker in the cluster's worker dictionary.
	     * @returns the worker index, or -1 if no worker exists
	     */
	    get index(): number;
	    /**
	     * Checks if the worker is ready to receive requests.
	     * A worker is ready when it has completed initialization and sent the 'listening' message.
	     * @returns true if the worker is ready to handle requests
	     */
	    get isReady(): boolean;
	    /**
	     * Checks if the worker process exists and is alive.
	     * @returns true if the worker exists and is not dead
	     */
	    get workerExists(): boolean;
	    /**
	     * Gets the port number that this worker is listening on.
	     * @throws Error if the port has not been set yet
	     * @returns the port number
	     */
	    get port(): number;
	    /**
	     * Gets the current number of active requests being processed by this worker.
	     * Used by the load balancing algorithm to select the least busy worker.
	     * @returns the active request count
	     */
	    get activeRequestCount(): number;
	    /**
	     * Gets the timestamp of the last request sent to this worker.
	     * Used as a tie-breaker when multiple workers have the same active request count.
	     * @returns the last request timestamp in milliseconds
	     */
	    get lastRequestTime(): number;
	    /**
	     * Increments the active request count and updates the last request time.
	     * Called when a new request is routed to this worker.
	     */
	    incrementActiveRequests(): void;
	    /**
	     * Decrements the active request count, ensuring it doesn't go below zero.
	     * Called when a request completes on this worker.
	     */
	    decrementActiveRequests(): void;
	    /**
	     * Resets the active request count and last request time to zero.
	     * Used when a worker is restarted or killed to clear stale state.
	     */
	    resetActiveRequests(): void;
	    /** Global worker pool organized by request source (e.g., 'main', 'x3-main') */
	    static workers: Dict<XtremWorker[]>;
	    /** Tracks active request IDs by request source for cleanup and monitoring */
	    static activeRequests: Dict<Set<string>>;
	    /** Timer for periodic worker health logging (runs every hour) */
	    static loggingInterval: NodeJS.Timeout | undefined;
	    /**
	     * Starts periodic logging of worker health metrics.
	     * Logs worker status every hour to help with monitoring and debugging.
	     * Only starts if not already running to prevent multiple timers.
	     */
	    static startLoggingTimer(): void;
	    /**
	     * Stops the periodic worker health logging timer.
	     * Used during application shutdown or when monitoring is no longer needed.
	     */
	    static stopLoggingTimer(): void;
	    /**
	     * Logs comprehensive worker health information for monitoring and debugging.
	     * Includes worker count, active requests, and detailed worker status.
	     *
	     * @param requestSource - Optional specific request source to log, logs all if not provided
	     */
	    static logWorkerHealth(requestSource?: string): void;
	    /**
	     * Marks a request as completed and updates worker state.
	     * Decrements the worker's active request count and cleans up request tracking.
	     *
	     * @param requestSource - The source that originated the request
	     * @param workerId - The ID of the worker that processed the request
	     * @param requestId - The unique identifier of the completed request
	     */
	    static completeRequest(requestSource: string, workerId: string, requestId: string): void;
	    /**
	     * Generates a unique request ID for tracking request lifecycle.
	     *
	     * @returns a unique request identifier
	     */
	    static generateRequestId(): string;
	    /**
	     * Adds multiple workers to the worker pool for a specific request source.
	     * Creates new XtremWorker instances and initializes them.
	     *
	     * @param cluster - The Node.js cluster instance for forking processes
	     * @param requestSource - The request source identifier (e.g., 'main', 'x3-main')
	     * @param startIndex - Starting index for worker creation (usually 0)
	     * @param stopIndex - Ending index for worker creation (exclusive)
	     */
	    static addWorkers(cluster: Cluster, requestSource: string, startIndex: number, stopIndex: number): void;
	    /**
	     * Seeds the worker pool with initial workers if the pool is below the target size.
	     * Ensures there are enough workers available to handle incoming requests.
	     *
	     * @param cluster - The Node.js cluster instance for forking processes
	     * @param requestSource - The request source identifier
	     * @param maxWorkers - Maximum number of workers to maintain (default: 2)
	     */
	    static seedWorker(cluster: Cluster, requestSource: string, maxWorkers?: number): void;
	    /** Timer for default worker seeding operations */
	    static defaultSeedInterval: NodeJS.Timeout;
	    /** Tracks deferred worker creation timeouts by request source */
	    static deferredWorkerTimeout: Dict<NodeJS.Timeout>;
	    /** Flag indicating if initial worker seeding has completed */
	    static seeded: boolean;
	    /**
	     * Determines the number of workers to maintain for a specific request source.
	     * Checks configuration in order of precedence:
	     * 1. Source-specific configuration from GraphqlEndpointHooks
	     * 2. Global server configuration
	     * 3. Default value of 2
	     *
	     * @param requestSource - The request source identifier
	     * @returns the number of workers to maintain for this source
	     */
	    static workersPerSource(requestSource: string): number;
	    /**
	     * Seeds workers for all known request sources during application startup.
	     * Only creates workers for sources that don't already have workers.
	     * Sets up periodic health monitoring after seeding is complete.
	     *
	     * @param cluster - The Node.js cluster instance for forking processes
	     */
	    static seedWorkers(cluster: Cluster): void;
	    /** Timer for periodic worker downscaling operations */
	    static downscaleWorkerInterval: NodeJS.Timeout;
	    /** Tracks downscaling promises by request source to prevent concurrent scaling */
	    static downscaleWorkerPromises: Dict<Promise<void>>;
	    /**
	     * Downscales worker pools by removing idle workers that exceed the target pool size.
	     * Only removes workers with zero active requests to avoid interrupting ongoing work.
	     * Runs periodically to optimize resource usage during low-load periods.
	     */
	    static downscaleWorkers(): void;
	    /**
	     * Initializes the worker pool for a request source and sets up periodic downscaling.
	     * Creates the initial set of workers and configures auto-scaling timers.
	     *
	     * @param cluster - The Node.js cluster instance for forking processes
	     * @param requestSource - The request source identifier
	     * @param maxWorkers - Number of workers to create initially
	     */
	    static initializeWorkers(cluster: Cluster, requestSource: string, maxWorkers: number): void;
	    /**
	     * Ensures adequate worker pool size for a request source with optional upscaling.
	     * This method handles both initial worker seeding and dynamic scaling based on demand.
	     *
	     * Auto-scaling Logic:
	     * 1. Waits for initial seeding to complete
	     * 2. Creates minimum workers if pool is empty
	     * 3. Defers additional worker creation to avoid blocking requests
	     * 4. Respects maximum upscaling limits from configuration
	     *
	     * @param cluster - The Node.js cluster instance for forking processes
	     * @param requestSource - The request source identifier
	     * @param scaleUp - Whether to scale beyond normal limits (default: false)
	     * @returns Promise<boolean> - true if workers are available, false if scaling failed
	     */
	    private static fillRequestSource;
	    /** Tracks downscaling promises by request source to prevent concurrent scaling */
	    static upscaleWorkerPromises: Dict<Promise<void>>;
	    /**
	     * Selects the optimal worker using the least-active-requests algorithm.
	     *
	     * Load Balancing Strategy:
	     * 1. Finds worker with minimum active requests
	     * 2. Uses last request time as tie-breaker (prefer less recently used)
	     * 3. Triggers auto-scaling if selected worker is overloaded
	     * 4. Tracks request assignment for proper lifecycle management
	     *
	     * @param cluster - The Node.js cluster instance
	     * @param requestSource - The request source identifier
	     * @param afterUpscale - Whether this is called after auto-scaling (prevents infinite recursion)
	     * @returns Promise<XtremWorker | undefined> - The selected worker or undefined if none available
	     */
	    private static getNextAvailableWorker;
	    /**
	     * Main entry point for request routing and worker selection.
	     *
	     * This method orchestrates the complete request routing process:
	     * 1. Waits for any pending worker management operations
	     * 2. Ensures adequate worker pool size
	     * 3. Selects optimal worker using load balancing algorithm
	     * 4. Handles worker recovery and request tracking
	     * 5. Returns worker and request ID for lifecycle management
	     *
	     * @param cluster - The Node.js cluster instance for worker management
	     * @param requestSource - The request source identifier
	     * @returns Promise with worker and request ID, or undefined if no worker available
	     */
	    static getNextWorker(cluster: Cluster, requestSource: string): Promise<{
	        worker: XtremWorker;
	        requestId: string;
	    } | undefined>;
	    /** Tracks promises for worker kill operations by request source */
	    static killWorkersPromises: Dict<Promise<void>>;
	    /** Funnel to serialize worker kill operations (prevents race conditions) */
	    static workerKillFunnel: import("@sage/xtrem-core").Funnel;
	    /**
	     * Terminates this worker process gracefully.
	     * Resets active request count and attempts to kill the process using SIGTERM.
	     * Logs the termination process for monitoring and debugging.
	     */
	    kill(): void;
	    /**
	     * Terminates all workers for a specific request source.
	     *
	     * This method:
	     * 1. Uses a funnel to serialize kill operations
	     * 2. Clears the worker pool for the request source
	     * 3. Clears active request tracking
	     * 4. Terminates all worker processes
	     * 5. Prepares for new workers to be created on next request
	     *
	     * @param requestSource - The request source whose workers should be terminated
	     * @returns Promise that resolves when all workers are killed
	     */
	    static killWorkers(requestSource: string): Promise<void>;
	}
	//# sourceMappingURL=worker.d.ts.map
}
// From file: @sage/xtrem-cli-main/build/lib/commands/handlers/start/start-server.d.ts
declare module '@sage/xtrem-cli-main/lib/commands/handlers/start/start-server' {
	import { Application, ApplicationStartServicesOptions } from '@sage/xtrem-core';
	/**
	 * Start a service for the current process
	 * @param dir
	 * @param options
	 * @returns
	 */
	export const startWorkerServices: (application: Application) => Promise<void>;
	/**
	 * Manage the start command, if multi-worker service are enabled then start the proxy service, otherwise start a single process as normal
	 * @param dir
	 * @param options
	 * @returns
	 */
	export const startServices: (dir: string, options: ApplicationStartServicesOptions) => Promise<void>;
	//# sourceMappingURL=start-server.d.ts.map
}
// From file: @sage/xtrem-cli-main/build/lib/commands/handlers/start/start-cluster.d.ts
declare module '@sage/xtrem-cli-main/lib/commands/handlers/start/start-cluster' {
	import { ApplicationStartServicesOptions } from '@sage/xtrem-core';
	export const startCluster: (dir: string, options: ApplicationStartServicesOptions) => Promise<void>;
	//# sourceMappingURL=start-cluster.d.ts.map
}
// From file: @sage/xtrem-cli-main/build/lib/commands/handlers/start/index.d.ts
declare module '@sage/xtrem-cli-main/lib/commands/handlers/start/index' {
	import { ApplicationStartServicesOptions, StartChannel } from '@sage/xtrem-core';
	import { startServices } from '@sage/xtrem-cli-main/lib/commands/handlers/start/start-server';
	export { startServices };
	export interface StartOptions {
	    isUsingReferences?: boolean;
	    cluster?: boolean;
	    channels?: StartChannel[];
	    services?: string;
	    webSockets?: string;
	    configExtensionLocation?: string;
	    /**
	     * Names of queues to explicitly start
	     * If not set, all the queues will be started
	     */
	    queues?: string[];
	}
	export interface StartContext {
	    dir: string;
	    deployedApp: any;
	    options: StartOptions;
	    startServerOptions: ApplicationStartServicesOptions;
	}
	export const scriptName = ".cli-start.js";
	export function createStartContext(options: StartOptions): StartContext;
	export const start: (startContext: StartContext) => Promise<void>;
	//# sourceMappingURL=index.d.ts.map
}
declare module '@sage/xtrem-cli-main' {
	export * from '@sage/xtrem-cli-main/index';
}
