import { integration } from '@sage/xtrem-ui';
import * as React from 'react';
import { createRoot } from 'react-dom/client';
import { connect, Provider } from 'react-redux';
import { ConnectedXtremHeader } from './component/header';
import { ConnectedXtremNavigation } from './component/navigation';
import * as actions from './standalone-redux/actions';
import { getStore } from './standalone-redux/store';
import { lockRenewal, shouldRenewToken, unlockRenewal } from './service/auth-service';
import { ConnectedMobileNavigation } from './component/mobile-navigation';
import sageTheme from 'carbon-react/esm/style/themes/sage';
import Loader from 'carbon-react/esm/components/loader';
import { localize } from './service/standalone-i18n-service';
import I18nProvider from 'carbon-react/esm/components/i18n-provider';
import CarbonProvider from 'carbon-react/esm/components/carbon-provider';
import { clone, set } from 'lodash';
import { getNewRelicApi, getPendoApi } from './service/telemetry-service';
import { ErrorBoundary } from './error-boundary';
import { Chatbot } from './component/chatbot';
import { initiateSocketConnection } from './service/socket-service';
import './style.scss';
const LAST_USER_INTERACTION = 'LAST_USER_INTERACTION';
const IDLE_LIMIT = 60 * 60000;
const customizedTheme = set(clone(sageTheme), 'zIndex.popover', 5000);
const GMS_CHAT_INSIGHTS_FETCHED_EVENT = 'gms-chat-insights-fetched';
export class XtremStandalone extends React.Component {
    constructor(props) {
        super(props);
        this.intervalRef = null;
        this.iframeRef = React.createRef();
        this.xtremUiRef = React.createRef();
        this.appContext = {
            handleNavigation: (path, _queryParameters, doDirtyCheck = false) => this.props.onInternalNavigationChange(path, doDirtyCheck),
            updateMenu: this.props.setMenuItems,
            onPageTitleChange: this.props.setPageTitle,
            onPageContextChange: this.props.setPageContext,
        };
        this.hasInitializedWebsocket = false;
        this.onInsightsFetched = (event) => {
            if (!this.props.pageContext?.screenId) {
                return;
            }
            const { insights } = event.detail;
            this.xtremUiRef.current?.onInsightsFetched({
                screenId: this.props.pageContext.screenId,
                count: insights.length,
            });
        };
        this.checkAuthTokenState = async (locale, forceRenewToken) => new Promise((resolve, reject) => {
            const lastInteraction = window.sessionStorage.getItem(LAST_USER_INTERACTION);
            const idleTimeLimit = new Date().getTime() - IDLE_LIMIT;
            if (lastInteraction && Number(lastInteraction) < idleTimeLimit) {
                getPendoApi()?.track('userIdleTimeout');
                // redirects user to login page if LAST_USER_INTERACTION was more than 5 min ago
                this.onUserLogout();
            }
            if ((shouldRenewToken() || forceRenewToken) && this.props.loginService && this.iframeRef.current) {
                // It locks the renewal process so other tabs will not try to renew the session at the same time.
                lockRenewal();
                this.iframeRef.current.onerror = () => {
                    // Check if still need to renew the token because due to race condition could be updated by other tab
                    if (shouldRenewToken()) {
                        getPendoApi()?.track('renewTokenError');
                        // If the renewal fails, it retries after 1 second just in case the new token is not placed yet by other tab
                        setTimeout(() => {
                            if (shouldRenewToken()) {
                                getPendoApi()?.track('renewTokenErrorRetry');
                                this.onUserLogout();
                                reject();
                            }
                            else {
                                resolve();
                            }
                        }, 2000);
                    }
                    else {
                        resolve();
                    }
                };
                // Once the page is successfully loaded, it removed the lock
                this.iframeRef.current.onload = () => {
                    unlockRenewal();
                    resolve();
                };
                this.iframeRef.current.src = `${this.props.loginService}/renew?locale=${locale}`;
            }
        });
        this.onTelemetryEvent = (event, data) => {
            if (event === 'unhandledError') {
                const error = data;
                getPendoApi()?.track(event, {
                    message: error.message,
                    stack: error.stack,
                });
                getNewRelicApi()?.noticeError(error);
            }
            else {
                getPendoApi()?.track(event, data);
            }
        };
        this.onUserLogout = () => {
            window.location.replace(`${this.props.loginService}/logout?fromUrl=${document.location.href}`);
        };
        this.onUserUpdate = () => {
            window.sessionStorage.setItem(LAST_USER_INTERACTION, String(new Date().getTime()));
        };
        this.onUserChangeLocale = async (locale) => {
            try {
                await this.props.onInternalNavigationChange(this.props.path, true);
                await this.checkAuthTokenState(locale, true);
                this.setState({ removeXtremUi: true }, () => document.location.reload());
            }
            catch (e) {
                // ignore
            }
        };
        this.onUserUpdate();
        this.checkAuthTokenState(props.user?.locale, false);
        this.state = {
            removeXtremUi: false,
        };
    }
    componentDidMount() {
        // Start listening on browser navigation changes
        window.addEventListener('popstate', this.props.updateNavStateBasedOnLocation);
        // Subscribe to Sage Insights events
        document.addEventListener(GMS_CHAT_INSIGHTS_FETCHED_EVENT, this.onInsightsFetched);
        this.props.init();
        // Checks every 20 seconds if the session needs to be renewed
        this.intervalRef = setInterval(this.checkAuthTokenState, 20000);
    }
    componentDidUpdate() {
        if (this.props.isNavigationOpen && this.xtremUiRef.current) {
            this.xtremUiRef.current.cleanToasts();
        }
        if (!this.hasInitializedWebsocket && this.xtremUiRef.current && this.props.config) {
            initiateSocketConnection({
                protocols: this.props.config.app,
                onWebSocketMessage: this.xtremUiRef.current.onWebSocketMessage,
            });
            this.hasInitializedWebsocket = true;
        }
    }
    componentWillUnmount() {
        document.removeEventListener(GMS_CHAT_INSIGHTS_FETCHED_EVENT, this.onInsightsFetched);
        if (this.intervalRef) {
            clearInterval(this.intervalRef);
            this.intervalRef = null;
        }
    }
    render() {
        if (this.props.user && this.props.config) {
            const locale = this.props.user.locale || 'en-US';
            this.appContext.locale = locale;
            this.appContext.login = this.props.user.email;
            this.appContext.userCode = this.props.user.userCode || this.appContext.login;
            this.appContext.profilePictureUrl = this.props.user.photo
                ? `data:image;base64,${this.props.user.photo}`
                : undefined;
            this.appContext.displayName = `${this.props.user.firstName || ''} ${this.props.user.lastName || ''}`.trim();
            this.appContext.cacheEncryptionKey = this.props.user.clientEncryptionKey;
            this.appContext.agGridLicenceKey = this.props.config?.agGridLicenceKey;
            this.appContext.onDirtyStatusChange = this.props.onDirtyStatusChange;
            this.appContext.onTelemetryEvent = this.onTelemetryEvent;
            this.appContext.onApiRequestError = ({ status }) => {
                if (status === 401) {
                    this.onUserLogout();
                }
            };
            const carbonLocale = {
                link: {
                    skipLinkLabel: () => localize('@sage/xtrem-standalone/skip-to-main-content', 'Skip to main content', {}, this.props.translations, locale),
                },
            };
            const hasPage = !!this.props.path;
            return (React.createElement(ErrorBoundary, null,
                React.createElement(I18nProvider, { locale: carbonLocale },
                    React.createElement(ConnectedXtremHeader, { onUserChangeLocale: this.onUserChangeLocale }),
                    React.createElement(ConnectedMobileNavigation, null),
                    React.createElement(Chatbot, null),
                    this.props.isXtremChatbotOpen && (React.createElement(integration.CopilotComponent, { applicationContext: this.appContext, onClose: this.props.closeXtremChatbotOpen, pageContext: this.props.pageContext })),
                    React.createElement("div", { className: "xe-body", onKeyDown: this.onUserUpdate, onMouseMove: this.onUserUpdate },
                        !this.props.isExtraSmall && React.createElement(ConnectedXtremNavigation, null),
                        React.createElement("div", { className: "xe-app-container", style: {
                                height: hasPage ? '100vh' : 0,
                                paddingLeft: this.props.isExtraSmall ? 0 : '64px',
                            } },
                            React.createElement(ErrorBoundary, null, !this.state?.removeXtremUi && (React.createElement(integration.XtremUiIndex, { path: this.props.path, applicationContext: this.appContext, ref: this.xtremUiRef })))),
                        !hasPage && (React.createElement(integration.DashboardRootComponent, { applicationContext: this.appContext, group: "home" }))),
                    React.createElement("iframe", { ref: this.iframeRef, style: { display: 'none' }, title: "auth iframe" }))));
        }
        return (React.createElement("div", { className: "initial-loader-wrapper" },
            React.createElement("div", { className: "initial-loader" },
                React.createElement(Loader, { size: "large" }))));
    }
}
const mapStateToProps = (state) => ({
    pageContext: state.pageContext,
    config: state.config,
    isExtraSmall: state.browser.is.xs,
    isNavigationOpen: state.isNavigationOpen,
    isXtremChatbotOpen: state.isXtremChatbotOpen,
    loginService: state.loginService,
    path: state.path,
    translations: state.translations,
    user: state.user,
    init: actions.actionStub,
    onDirtyStatusChange: actions.actionStub,
    onInternalNavigationChange: actions.actionStub,
    setPageTitle: actions.actionStub,
    closeXtremChatbotOpen: actions.actionStub,
    setMenuItems: actions.actionStub,
    setPageContext: actions.actionStub,
    updateNavStateBasedOnLocation: actions.actionStub,
});
const mapDispatchToProps = (dispatch) => ({
    updateNavStateBasedOnLocation: () => dispatch(actions.updateNavStateBasedOnLocation()),
    onDirtyStatusChange: (isApplicationDirty, preNavigationConfirmation) => dispatch(actions.onDirtyStatusChange(isApplicationDirty, preNavigationConfirmation)),
    init: () => dispatch(actions.init()),
    onInternalNavigationChange: (path, doDirtyCheck = false) => dispatch(actions.onInternalNavigationChange(path, doDirtyCheck)),
    setPageTitle: (title) => {
        dispatch(actions.setPageTitle(title));
    },
    setPageContext: (pageContext) => {
        dispatch(actions.setPageContext(pageContext));
    },
    closeXtremChatbotOpen: () => {
        dispatch(actions.setXtremChatbotOpen(false));
    },
    setMenuItems: (menuItems) => dispatch(actions.setMenuItems(menuItems)),
});
// tslint:disable-next-line:variable-name
export const ConnectedXtremStandaloneComponent = connect(mapStateToProps, mapDispatchToProps)(XtremStandalone);
window.start = () => {
    createRoot(window.document.getElementById('root')).render(React.createElement(Provider, { store: getStore() },
        React.createElement(CarbonProvider, { theme: customizedTheme },
            React.createElement(ConnectedXtremStandaloneComponent, null))));
};
//# sourceMappingURL=index.js.map