import * as tokens from '@sage/design-tokens/js/base/common';
import Button from 'carbon-react/esm/components/button';
import Icon from 'carbon-react/esm/components/icon';
import Link from 'carbon-react/esm/components/link';
import Tooltip from 'carbon-react/esm/components/tooltip';
import Pill from 'carbon-react/esm/components/pill';
import * as React from 'react';
import { connect } from 'react-redux';
import { localize } from '../service/standalone-i18n-service';
import * as actions from '../standalone-redux/actions';
import './header.scss';
import { ConnectedNotificationCenter } from './notification/notification-center';
import { ConnectedNotificationPreviews } from './notification/notification-previews';
import { ProfileSettingsPanel } from './profile-settings';
import { CHATBOT_SUPPORTED_PAGES } from '../service/chatbot-service';
import { getPendoApi } from '../service/telemetry-service';
import { ChatbotIcon } from './chatbot-icon';
export function XtremHeader(props) {
    const profileStyles = {};
    const hasPhoto = props.user && props.user.photo;
    const hasUnreadNotifications = props.notifications.some(n => !n.isRead);
    const selectableTenants = props.tenantItem.tenantList.reduce((value, tenant) => {
        if (!tenant.current) {
            value.push(tenant.tenantId);
        }
        return value;
    }, []);
    const [isTenantsListContainerOpen, setIsTenantsListContainerOpen] = React.useState(false);
    const [focusedTenant, setFocusedTenant] = React.useState(-1);
    const [isProfileOpenedByKeyboard, setProfileOpenedByKeyboard] = React.useState(false);
    const toggleProfilePanelOpenByKeyboard = () => setProfileOpenedByKeyboard(!isProfileOpenedByKeyboard);
    const tenantsActionsRef = React.useRef(null);
    const tenantsContentRef = React.useRef(null);
    const getTenantsAnchor = (tenantId) => tenantsContentRef.current?.querySelector(`a[id="${tenantId}"]`);
    const onTenantsActionEvent = () => {
        setIsTenantsListContainerOpen(!isTenantsListContainerOpen);
        if (!isTenantsListContainerOpen) {
            setFocusedTenant(-1);
        }
    };
    const onTenantsActionKeyDown = (e) => {
        if (e.key === 'Enter' || e.key === ' ') {
            e.preventDefault();
            onTenantsActionEvent();
        }
        if (e.key === 'ArrowDown') {
            getTenantsAnchor(selectableTenants[0])?.focus();
            setFocusedTenant(0);
        }
    };
    const onTenantsContentKeyDown = (e) => {
        e.stopPropagation();
        if (e.key === ' ') {
            getTenantsAnchor(selectableTenants[focusedTenant]).click();
        }
        if (e.key === 'Escape') {
            onTenantsActionEvent();
            tenantsActionsRef.current?.focus();
        }
        if (e.key === 'ArrowDown' && focusedTenant < selectableTenants.length - 1) {
            getTenantsAnchor(selectableTenants[focusedTenant + 1]).focus();
            setFocusedTenant(focusedTenant + 1);
        }
        if (e.key === 'ArrowUp' && focusedTenant > 0) {
            getTenantsAnchor(selectableTenants[focusedTenant - 1]).focus();
            setFocusedTenant(focusedTenant - 1);
        }
    };
    if (hasPhoto) {
        profileStyles.backgroundImage = `url(data:image;base64,${props.user.photo})`;
    }
    const goHome = (event) => {
        event.preventDefault();
        props.setNavigationOpen(false);
        setProfileOpenedByKeyboard(false);
        props.goHome();
        getPendoApi()?.track('goHomeButtonClicked');
    };
    const currentTenant = props.tenantItem.tenantList.find(tenant => tenant.current);
    const hasTenants = props.tenantItem.tenantList?.length > 0 && !props.user?.isOperator;
    const canSwitchTenant = props.tenantItem.tenantList?.length > 1;
    const logoContainerClasses = ['xe-logo-container'];
    if (props.isExtraSmall) {
        logoContainerClasses.push('xe-logo-container-no-margin');
    }
    const profileClassNames = ['xe-profile-action', 'xe-profile-profile'];
    if (isProfileOpenedByKeyboard) {
        profileClassNames.push('xe-opened');
    }
    const onSkipLink = (event) => {
        event.preventDefault();
        const pageBodyElements = Array.from(document.querySelectorAll('main'));
        if (pageBodyElements.length > 1) {
            const focusableElements = Array.from(pageBodyElements[pageBodyElements.length - 1].querySelectorAll('a[href], button, input, textarea, select, details, [tabindex]:not([tabindex="-1"])')).filter(el => !el.hasAttribute('disabled') &&
                !el.getAttribute('aria-hidden') &&
                !el.classList.contains('e-navigation-panel-toggle'));
            const firstFocusableElement = focusableElements[0];
            firstFocusableElement.focus();
        }
        else {
            const focusableElements = Array.from(pageBodyElements[0].querySelectorAll('a[href], button, input, textarea, select, details, [tabindex]:not([tabindex="-1"])')).filter(el => !el.hasAttribute('disabled') &&
                !el.getAttribute('aria-hidden') &&
                !el.classList.contains('e-navigation-panel-toggle'));
            const firstFocusableElement = focusableElements[0];
            firstFocusableElement.focus();
        }
    };
    const openChatbot = React.useCallback(() => {
        if (!props.isXtremChatbotAvailable) {
            window.GmsChatUi?.setDisplayState('popup-right');
        }
        else {
            props.openXtremChatbotOpen();
        }
    }, [props.isXtremChatbotAvailable, props.openXtremChatbotOpen]);
    const getTenantHref = (tenant) => {
        return tenant.current
            ? ''
            : `${props.loginService}${tenant.directLoginUrl.substring(tenant.directLoginUrl.indexOf('/'))}`;
    };
    const profileAriaLabelText = isProfileOpenedByKeyboard
        ? localize('@sage/xtrem-standalone/profile-settings-close', 'Close profile settings', {}, props.translations, props.user?.locale ?? undefined)
        : localize('@sage/xtrem-standalone/profile-settings-open', 'Open profile settings', {}, props.translations, props.user?.locale ?? undefined);
    React.useEffect(() => {
        const handleKeyDown = (event) => {
            if (event.key === 'Escape') {
                toggleProfilePanelOpenByKeyboard();
            }
        };
        if (isProfileOpenedByKeyboard) {
            document.addEventListener('keydown', handleKeyDown);
        }
        return () => {
            if (isProfileOpenedByKeyboard) {
                document.removeEventListener('keydown', handleKeyDown);
            }
        };
    }, [isProfileOpenedByKeyboard, toggleProfilePanelOpenByKeyboard]);
    const isChatbotButtonHidden = !props.isXtremChatbotAvailable &&
        !CHATBOT_SUPPORTED_PAGES.find(p => props.path === p || props.path.startsWith(`${p}/`));
    const getTenantPill = (kind) => {
        if (!kind)
            return null;
        switch (kind) {
            case 'production':
                return (React.createElement(Pill, { isDarkBackground: true, borderColor: tokens.colorsSemanticPositive500, size: "S" }, kind));
            case 'demo':
                return (React.createElement(Pill, { isDarkBackground: true, borderColor: tokens.colorsSemanticInfo500, size: "S" }, kind));
            default:
                return null;
        }
    };
    return (React.createElement("header", { className: "xe-header" },
        React.createElement("div", { className: "xe-skip-link", "data-testid": "xe-skip-link" },
            React.createElement(Link, { isSkipLink: true, onClick: onSkipLink, href: "#" })),
        props.isExtraSmall && (React.createElement(Button, { buttonType: "tertiary", ml: 2, iconType: props.isNavigationOpen ? 'close' : 'list_view', onClick: () => props.setNavigationOpen(!props.isNavigationOpen) })),
        React.createElement("a", { href: "/", className: logoContainerClasses.join(' '), onClick: goHome },
            React.createElement("img", { className: "xe-text-sage-logo", src: "/images/sage-logo.svg", alt: "Sage" }),
            !props.isExtraSmall && React.createElement("span", { className: "xe-text-logo-main-name" }, props.productName)),
        React.createElement("ul", { className: "xe-actions" },
            hasTenants && (React.createElement("li", { className: `${canSwitchTenant ? 'xe-tenants-action' : 'xe-tenants-no-action'}`, "data-testid": `${canSwitchTenant ? 'xe-tenants-action' : 'xe-tenants-no-action'}` },
                React.createElement("button", { className: "xe-tenants-action-button", disabled: !canSwitchTenant, onClick: canSwitchTenant ? onTenantsActionEvent : undefined, onKeyDown: canSwitchTenant ? onTenantsActionKeyDown : undefined, ref: tenantsActionsRef, type: "button" },
                    React.createElement("div", { className: "xe-tenants-current-tenant-wrapper" },
                        !props.isExtraSmall && (React.createElement(React.Fragment, null,
                            React.createElement("span", { className: "xe-tenants-current-tenant", "data-testid": "xe-tenants-current-tenant", title: currentTenant?.tenantName }, currentTenant?.tenantName),
                            React.createElement("span", { className: "xe-tenants-current-pill" }, getTenantPill(currentTenant?.kind)))),
                        canSwitchTenant && (React.createElement(Icon, { color: tokens.colorsYang100, "data-testid": "xe-tenants-icon-dropdown", tabIndex: -1, type: "dropdown" }))),
                    isTenantsListContainerOpen && (React.createElement("div", { className: "xe-tenants-list-wrapper", "data-testid": "xe-tenants-list-wrapper" },
                        React.createElement("div", { className: "xe-tenants-list-container", "data-testid": "xe-tenants-list-container" },
                            React.createElement("div", { className: "xe-tenants-list-content", "data-testid": "xe-tenants-list-content", ref: tenantsContentRef, onKeyDown: onTenantsContentKeyDown },
                                React.createElement("div", null, (props.tenantItem.tenantList || []).map(tenant => (React.createElement("a", { className: "xe-tenant", id: tenant.tenantId, key: tenant.tenantId, href: getTenantHref(tenant) },
                                    React.createElement("span", { className: "xe-tenant-name" }, tenant.tenantName),
                                    React.createElement("span", { className: "xe-tenant-kind" }, getTenantPill(tenant.kind)),
                                    tenant.current && (React.createElement(Icon, { color: tokens.colorsYang100, "data-testid": "xe-tenant-icon", className: "xe-tenant-icon", tabIndex: -1, type: "tick" }))))))))))))),
            props.menuItems?.map(m => {
                return (React.createElement("li", { key: m.id, className: "xe-profile-action xe-profile-action-icon-button" },
                    React.createElement("button", { "aria-label": m.icon, className: "xe-profile-action-button xe-icon-button xe-icon-button-link", "data-testid": m.id, onClick: e => {
                            e.preventDefault();
                            m.onClick();
                        }, type: "button" },
                        React.createElement(Icon, { type: m.icon, tooltipMessage: m.title, color: tokens.colorsYang100, tabIndex: -1 })),
                    m.badgeContent && (React.createElement("span", { className: "xe-icon-label" },
                        React.createElement(Pill, { colorVariant: "neutral", fill: true, pillRole: "status", size: "S" }, String(m.badgeContent))))));
            }),
            props.chatbotEnabled && props.user?.locale && !isChatbotButtonHidden && (React.createElement("li", { className: "xe-profile-action xe-ai-assistant-action", "data-testid": "xe-ai-assistant-action" },
                React.createElement("button", { disabled: isChatbotButtonHidden, "aria-label": localize('@sage/xtrem-standalone/sage-copilot', 'Sage Copilot', {}, props.translations, props.user.locale), className: "xe-profile-action-button xe-icon-button xe-profile-action-icon-button", onClick: openChatbot, type: "button", "data-testid": "xe-ai-assistant-action-button" },
                    React.createElement(Tooltip, { message: localize('@sage/xtrem-standalone/sage-copilot', 'Sage Copilot', {}, props.translations, props.user.locale) },
                        React.createElement(ChatbotIcon, null))))),
            props.user?.locale ? (React.createElement(React.Fragment, null,
                React.createElement("li", { className: "xe-profile-action xe-notification-action", "data-testid": "xe-notification-action" },
                    React.createElement("button", { "aria-label": localize('@sage/xtrem-standalone/notification-menu', 'Notifications', {}, props.translations, props.user.locale), className: "xe-profile-action-button xe-icon-button xe-profile-action-icon-button xe-icon-button-link-notification", "data-testid": hasUnreadNotifications ? 'xe-notification-unread' : 'xe-notification-normal', onClick: ev => {
                            ev.preventDefault();
                            props.setNotificationCenterOpen(!props.isNotificationCenterOpen);
                        }, type: "button" },
                        React.createElement(Icon, { type: "alert", color: tokens.colorsYang100, tabIndex: -1, tooltipMessage: localize('@sage/xtrem-standalone/notification-menu', 'Notifications', {}, props.translations, props.user.locale) }),
                        hasUnreadNotifications && React.createElement("span", { className: "xe-icon-button-link-notification-badge" })),
                    React.createElement(ConnectedNotificationPreviews, null),
                    React.createElement(ConnectedNotificationCenter, { open: props.isNotificationCenterOpen })),
                React.createElement("li", { className: "xe-profile-action xe-profile-action-icon-button" },
                    React.createElement("a", { 
                        // Don't reuse this className as it uses by Pendo to show the help dialog
                        className: "xe-icon-button xe-icon-button-link", id: "pendo-help", href: "#", target: "_blank", onClick: ev => {
                            ev.preventDefault();
                            getPendoApi()?.track('helpButtonClicked');
                        }, "aria-label": localize('@sage/xtrem-standalone/help-menu', 'Help', {}, props.translations, props.user.locale) },
                        React.createElement(Icon, { color: tokens.colorsYang100, tabIndex: -1, tooltipMessage: localize('@sage/xtrem-standalone/help-menu', 'Help', {}, props.translations, props.user.locale), type: "help" }))))) : null,
            React.createElement("li", { className: profileClassNames.join(' ') },
                React.createElement("div", { className: "xe-profile-wrapper" },
                    hasPhoto ? (React.createElement("button", { type: "button", className: "xe-profile-picture", style: profileStyles, onClick: toggleProfilePanelOpenByKeyboard, "aria-label": profileAriaLabelText })) : (React.createElement("button", { type: "button", className: "xe-profile-picture xe-profile-picture-empty", onClick: toggleProfilePanelOpenByKeyboard, "aria-label": profileAriaLabelText })),
                    React.createElement(ProfileSettingsPanel, { user: props.user, translations: props.translations, loginService: props.loginService, localeOnChange: locale => props.onUserChangeLocale(locale), tenantItem: props.tenantItem }))))));
}
XtremHeader.displayName = 'XtremHeader';
const mapStateToProps = (state, ownProps) => ({
    chatbotEnabled: state.isXtremChatbotAvailable || !!state.config?.chatbotBackendUrl,
    goHome: actions.actionStub,
    isExtraSmall: state.browser.is.xs,
    isNavigationOpen: state.isNavigationOpen,
    isNotificationCenterOpen: state.isNotificationCenterOpen,
    isXtremChatbotAvailable: state.isXtremChatbotAvailable,
    loginService: state.loginService,
    menuItems: state.menuItems.filter(m => m.category === 'sticker'),
    notifications: state.notifications,
    onUserChangeLocale: ownProps.onUserChangeLocale,
    openXtremChatbotOpen: actions.actionStub,
    path: state.path,
    productName: state.config?.productName || null,
    setNavigationOpen: actions.actionStub,
    setNotificationCenterOpen: actions.actionStub,
    tenantItem: state.tenantsList,
    translations: state.translations,
    user: state.user,
});
const mapDispatchToProps = (dispatch) => ({
    goHome: () => dispatch(actions.goHome()),
    setNavigationOpen: (isOpen) => dispatch(actions.setNavigationOpen(isOpen)),
    setNotificationCenterOpen: (isOpen) => dispatch(actions.setNotificationCenterOpen(isOpen)),
    openXtremChatbotOpen: () => {
        dispatch(actions.setXtremChatbotOpen(true));
    },
});
export const ConnectedXtremHeader = connect(mapStateToProps, mapDispatchToProps)(XtremHeader);
//# sourceMappingURL=header.js.map