"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isStringDate = void 0;
exports.pad2 = pad2;
exports.pad3 = pad3;
exports.localizedGroup = localizedGroup;
exports.datePropertyValueToDateString = datePropertyValueToDateString;
exports.isValidDatePropertyValue = isValidDatePropertyValue;
const date_1 = require("./date");
const datetime_1 = require("./datetime");
const localize_1 = require("./localize");
function pad2(val) {
    const s = val.toString();
    return s.length === 1 ? `0${s}` : s;
}
function pad3(val) {
    const s = val.toString();
    switch (s.length) {
        case 1:
            return `00${s}`;
        case 2:
            return `0${s}`;
        default:
            return s;
    }
}
/**
 * All i18n keys are individual records. Some of these records needs to function in a group i.e. days of the week and months.
 * @param baseResource string[] - see datetime/baseResource.ts
 * @param data object | any[] - default is any. Data for i18n value handlebars
 * @param i18nKey string - example: @sage/xtrem-date-time/datetime__abbreviatedDayName - NB! without the index. see datetime/i18n/en-US.json
 * @param locale LocalizeLocale - default is base
 * @returns string[] of localized semantic values
 */
function localizedGroup(baseResource, i18nKey, data = {}, locale = 'base') {
    const result = [];
    for (let i = 0; i < baseResource.length; i += 1) {
        result.push((0, localize_1.localizedText)(`${i18nKey}${i}`, baseResource[i], data, locale));
    }
    return result;
}
// eslint-disable-next-line no-useless-escape
const dateRegexp = /^\d{4}\-(0?[1-9]|1[012])-(0?[1-9]|[12][0-9]|3[01])$/;
const isoDateTimeRegexp = 
// eslint-disable-next-line no-useless-escape
/^([\+-]?\d{4}(?!\d{2}\b))((-?)((0[1-9]|1[0-2])(\3([12]\d|0[1-9]|3[01]))?|W([0-4]\d|5[0-2])(-?[1-7])?|(00[1-9]|0[1-9]\d|[12]\d{2}|3([0-5]\d|6[1-6])))([T\s]((([01]\d|2[0-3])((:?)[0-5]\d)?|24\:?00)([\.,]\d+(?!:))?)?(\17[0-5]\d([\.,]\d+)?)?([zZ]|([\+-])([01]\d|2[0-3]):?([0-5]\d)?)?)?)?$/;
const isStringDate = (value) => typeof value === 'string' && (!!value.match(dateRegexp) || !!value.match(isoDateTimeRegexp));
exports.isStringDate = isStringDate;
function datePropertyValueToDateString(value, keepTime = false) {
    if (!value) {
        return null;
    }
    if (value instanceof Date && keepTime) {
        return datetime_1.Datetime.fromJsDate(value).toString();
    }
    if (value instanceof Date) {
        return date_1.DateValue.fromJsDate(value).toString();
    }
    if (typeof value === 'string' && (0, exports.isStringDate)(value) && keepTime) {
        if (value.length === 10) {
            return `${value}T00:00:00.000Z`;
        }
        return value;
    }
    if (typeof value === 'string' && (0, exports.isStringDate)(value)) {
        return value.substring(0, 10);
    }
    if (value instanceof date_1.DateValue) {
        return value.toString();
    }
    if (value instanceof datetime_1.Datetime) {
        if (!keepTime) {
            return value.date.toString();
        }
        return value.toString();
    }
    throw new Error(`Invalid date provided: ${value}`);
}
function isValidDatePropertyValue(value) {
    if (!value) {
        return false;
    }
    try {
        datePropertyValueToDateString(value);
        return true;
    }
    catch {
        return false;
    }
}
//# sourceMappingURL=utils.js.map