"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Range = void 0;
class Range {
    constructor(start, end, excludesStart = false, excludesEnd = true) {
        this.start = start;
        this.end = end;
        this.excludesStart = excludesStart;
        this.excludesEnd = excludesEnd;
        if (start) {
            if (this.excludesStart) {
                this.includedStart = this.incrementsByRangeUnit(this.start);
                this.excludedStart = this.start;
            }
            else {
                this.includedStart = this.start;
                this.excludedStart = this.incrementsByRangeUnit(this.start, -1);
            }
        }
        else {
            this.includedStart = null;
            this.excludedStart = null;
        }
        if (end) {
            if (this.excludesEnd) {
                this.includedEnd = this.incrementsByRangeUnit(this.end, -1);
                this.excludedEnd = this.end;
            }
            else {
                this.includedEnd = this.end;
                this.excludedEnd = this.incrementsByRangeUnit(this.end, 1);
            }
        }
        else {
            this.excludedEnd = null;
        }
        this.validateRange();
    }
    // should check if included start is greater than excluded end
    validateRange() {
        if (this.includedStart && this.excludedEnd) {
            if (this.getBoundValue(this.includedStart) >= this.getBoundValue(this.excludedEnd)) {
                throw new Error(`Invalid range: included start ${this.getStringBoundValue(this.includedStart)} should not be greater than excluded end ${this.getStringBoundValue(this.excludedEnd)}`);
            }
        }
    }
    includes(elmToCompareWith) {
        if (this.excludedEnd === null && this.includedStart === null)
            return true;
        if (this.includedStart) {
            if (this.excludedEnd) {
                return (this.getBoundValue(elmToCompareWith) >= this.getBoundValue(this.includedStart) &&
                    this.getBoundValue(elmToCompareWith) < this.getBoundValue(this.excludedEnd));
            }
            return this.getBoundValue(elmToCompareWith) >= this.getBoundValue(this.includedStart);
        }
        if (this.excludedEnd) {
            return this.getBoundValue(elmToCompareWith) < this.getBoundValue(this.excludedEnd);
        }
        return false;
    }
    equals(rangeToCompareWith) {
        return (this.getBoundValue(this.includedStart) === this.getBoundValue(rangeToCompareWith?.includedStart) &&
            this.getBoundValue(this.excludedEnd) === this.getBoundValue(rangeToCompareWith?.excludedEnd));
    }
    toString() {
        const rangeBeginInclusionMark = this.excludesStart ? '(' : '[';
        const rangeEndInclusionMark = this.excludesEnd ? ')' : ']';
        const rangeStartDateMark = this.start ? this.getStringBoundValue(this.start) : '';
        const rangeEndDateMark = this.end ? this.getStringBoundValue(this.end) : '';
        return `${rangeBeginInclusionMark}${rangeStartDateMark},${rangeEndDateMark}${rangeEndInclusionMark}`;
    }
    toJSON() {
        return this.toString();
    }
}
exports.Range = Range;
//# sourceMappingURL=range.js.map