"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isValidIsoDate = exports.isValidLocalizedDate = exports.parseIsoDate = exports.parseLocalizedDate = exports.formatTimeToLocale = exports.formatDateToCurrentLocale = void 0;
const _1 = require(".");
const date_1 = require("./date");
const localize_1 = require("./localize");
const localizeDatePresetFormat = (fmt, locale = 'en-US') => {
    switch (fmt) {
        case 'MonthDay': {
            return (0, localize_1.localizedText)('@sage/xtrem-date-time/date-format-month-day', 'M/d', {}, locale);
        }
        case 'MonthYear': {
            return (0, localize_1.localizedText)('@sage/xtrem-date-time/date-format-month-year', 'MM/YYYY', {}, locale);
        }
        case 'LongMonthDayYear': {
            return (0, localize_1.localizedText)('@sage/xtrem-date-time/date-format-long-month-day-year', 'MMMM DD YYYY', {}, locale);
        }
        case 'LongMonthYear': {
            return (0, localize_1.localizedText)('@sage/xtrem-date-time/date-format-long-month-year', 'MMMM YYYY', {}, locale);
        }
        case 'FullDate': {
            return (0, localize_1.localizedText)('@sage/xtrem-date-time/date-format-full-date', 'DD/MM/YYYY', {}, locale);
        }
        case 'LongMonth': {
            return (0, localize_1.localizedText)('@sage/xtrem-date-time/date-format-long-month', 'MMMM', {}, locale);
        }
        default:
            return (0, localize_1.localizedText)('@sage/xtrem-date-time/date-format', 'DD/MM/YYYY', {}, locale);
    }
};
const formatDateToCurrentLocale = (date, locale = 'en-US', fmt = 'FullDate') => {
    const localizedFmt = localizeDatePresetFormat(fmt, locale);
    let dateValue;
    if (date instanceof date_1.DateValue) {
        dateValue = date;
    }
    else if (date instanceof _1.Datetime) {
        dateValue = date.date;
    }
    else if (date instanceof Date) {
        dateValue = date_1.DateValue.fromJsDate(date);
    }
    else if (typeof date === 'string') {
        dateValue = date_1.DateValue.parse(date, locale, 'YYYY-MM-DD');
    }
    else {
        throw new Error(`Invalid date value: ${date}`);
    }
    return dateValue.format(localizedFmt, locale);
};
exports.formatDateToCurrentLocale = formatDateToCurrentLocale;
const formatTimeToLocale = (time, locale = 'en-US') => {
    const localizedFmt = (0, localize_1.localizedText)('@sage/xtrem-date-time/time-format-short', 'hh:mm A', {}, locale);
    let timeValue;
    if (time instanceof _1.Time) {
        timeValue = time;
    }
    else if (time instanceof _1.Datetime) {
        timeValue = time.time;
    }
    else if (time instanceof Date) {
        timeValue = _1.Time.fromJsDate(time);
    }
    else if (typeof time === 'string') {
        timeValue = _1.Time.parse(time, locale, 'HH:mm');
    }
    else {
        throw new Error(`Invalid time value: ${time}`);
    }
    return timeValue.format(locale, localizedFmt);
};
exports.formatTimeToLocale = formatTimeToLocale;
const parseLocalizedDate = (date, locale = 'en-US', fmt = 'FullDate') => {
    const localizedFormat = localizeDatePresetFormat(fmt, locale);
    return _1.date.parse(date, locale, localizedFormat);
};
exports.parseLocalizedDate = parseLocalizedDate;
const parseIsoDate = (date, locale = 'en-US') => {
    return _1.date.parse(date, locale, 'YYYY-MM-DD');
};
exports.parseIsoDate = parseIsoDate;
const isValidLocalizedDate = (date, locale = 'en-US', fmt = 'FullDate') => {
    try {
        (0, exports.parseLocalizedDate)(date, locale, fmt);
        return true;
    }
    catch {
        return false;
    }
};
exports.isValidLocalizedDate = isValidLocalizedDate;
const isValidIsoDate = (date, locale = 'en-US') => {
    try {
        (0, exports.parseIsoDate)(date, locale);
        return true;
    }
    catch {
        return false;
    }
};
exports.isValidIsoDate = isValidIsoDate;
//# sourceMappingURL=formatting-utils.js.map