import { LocalizeLocale } from '@sage/xtrem-shared';
import * as date from './date';
import { integer } from './date';
import * as time from './time';
export type DatetimeDiff = date.DateDiff & time.TimeDiff;
/**
 * A `Datetime` represents a specific point in time.
 *
 * The main differences with JavaScript `Date` are:
 *
 * - `Datetime` is immutable.
 * - The `month` property takes values between 1 and 12, not 0 and 11.
 */
export declare class Datetime {
    /** Internal value: GMT millis since epoch (Jan 1st, 1970) */
    private readonly _value;
    /** The time zone used to format the value */
    private readonly _timeZone;
    /**
     * Internal cache of the UTC values (year, month, day, hour, minute, second).
     * Packed int with year, month, day, hour, minute, second.
     * See Datetime.pack for details.
     */
    private _utcValues;
    /**
     * Internal cache of the time zone field values (year, month, day, hour, minute, second).
     * Packed int with year, month, day, hour, minute, second.
     * See Datetime.pack for details.
     */
    private _tzValues;
    private constructor();
    /**
     * Pack datetime components into an int.
     * JS int is 53 bits, so we can pack year, month, day, hour, minute, second on powers of 10 without overflow
     * The millisecond part is not included as it is independent from UTC / time zone and can be obtained from #value.
     */
    private static pack;
    private get utcValue();
    private get tzValue();
    get constructorName(): string;
    get timeZone(): string | undefined;
    /** The date (only year, month, day). */
    get date(): date.DateValue;
    /** The time (only hour, minute, second, millisecond). */
    get time(): time.Time;
    /** The year, between 0 and 9999. */
    get year(): integer;
    /** The month, between 1 and 12. */
    get month(): integer;
    /** The day, between 1 and 31. */
    get day(): integer;
    /** The week day, between 0 (Sunday) and 6 (Saturday). */
    get weekDay(): date.WeekDay;
    /** The day of the year, between 1 and 366. */
    get yearDay(): integer;
    /** The hour, between 0 and 23. */
    get hour(): integer;
    /** The minute, between 0 and 59. */
    get minute(): integer;
    /** The second, between 0 and 59. */
    get second(): integer;
    /** The millisecond, between 0 and 999. */
    get millisecond(): integer;
    get utcDate(): date.DateValue;
    get utcTime(): time.Time;
    get utcYear(): number;
    get utcMonth(): number;
    get utcDay(): number;
    get utcWeekDay(): date.WeekDay;
    get utcYearDay(): number;
    get utcHour(): number;
    get utcMinute(): number;
    get utcSecond(): number;
    get utcMillisecond(): number;
    get value(): number;
    compare(dt: Datetime): number;
    equals(dt: Datetime): boolean;
    isBetween(begin: Datetime, end: Datetime): boolean;
    addYears(years: number): Datetime;
    addMonths(months: number): Datetime;
    addWeeks(weeks: number): Datetime;
    addDays(days: number): Datetime;
    addHours(hours: number): Datetime;
    addMinutes(minutes: number): Datetime;
    addSeconds(seconds: number): Datetime;
    addMilliseconds(millis: number): Datetime;
    addDayFractions(fraction: number): Datetime;
    millisDiff(dt: Datetime): number;
    withoutMillis(): Datetime;
    inTimeZone(timeZone: string | undefined): Datetime;
    add(delta: DatetimeDiff): Datetime;
    toString(): string;
    static formatDirectiveError(format: string, letter: string, locale: LocalizeLocale, repeat: number): Error;
    format(locale: LocalizeLocale | undefined, format: string): string;
    toJsDate(): Date;
    toJSON(): string;
    withoutTimezoneOffset(): Datetime;
    isNull(): boolean;
    private static parseIso;
    static now(withMillis?: boolean, timeZone?: string): Datetime;
    static parse(str: string, locale?: LocalizeLocale, format?: string, timeZone?: string): Datetime;
    static fromJsDate(jsDate: Date, timeZone?: string): Datetime;
    static fromValue(value: number, timeZone?: string): Datetime;
    private static validateDateValues;
    static make(year: number, month: number, day: number, hour?: number, minute?: number, second?: number, millis?: number, timeZone?: string): Datetime;
    static makeUtc(year: number, month: number, day: number, hour?: number, minute?: number, second?: number, millis?: number): Datetime;
    /**
     * RegExp to test a string for an ISO 8601 Date spec
     *  YYYY
     *  YYYY-MM
     *  YYYY-MM-DD
     *  YYYY-MM-DD[T| ]HH:mmTZD
     *  YYYY-MM-DD[T| ]HH:mm:ssTZD
     *  YYYY-MM-DD[T| ]HH:mm:ss.sTZD
     * @see: https://www.w3.org/TR/NOTE-datetime
     */
    static isoDatetimeRegexISO8601: RegExp;
    static isDatetime(obj: any): obj is Datetime;
}
export declare const sunday = 0;
export declare const monday = 1;
export declare const tuesday = 2;
export declare const wednesday = 3;
export declare const thursday = 4;
export declare const friday = 5;
export declare const saturday = 6;
export declare function overrideNow(str: string | null): void;
export declare const now: typeof Datetime.now;
export declare const parse: typeof Datetime.parse;
export declare const fromJsDate: typeof Datetime.fromJsDate;
export declare const fromValue: typeof Datetime.fromValue;
export declare const make: typeof Datetime.make;
export declare const makeUtc: typeof Datetime.makeUtc;
export declare const isDatetime: typeof Datetime.isDatetime;
//# sourceMappingURL=datetime.d.ts.map