"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getWeekDayNumbers = exports.getWeekDayNames = exports.isDate = exports.makeInWeek = exports.make = exports.fromInternalValue = exports.fromJsDate = exports.today = exports.parse = exports.weekDayFromName = exports.weekDayName = exports.monthFromName = exports.monthName = exports.pmDesignator = exports.amDesignator = exports.DateValue = exports.dayNames = exports.abbreviatedDayNames = exports.WeekDay = exports.monthNames = exports.abbreviatedMonthNames = void 0;
exports.toOffset = toOffset;
exports.fromOffset = fromOffset;
exports.isLeap = isLeap;
exports.daysInMonth = daysInMonth;
const datetime = require("./datetime");
const datetime_1 = require("./datetime");
const localize_1 = require("./localize");
const parse_1 = require("./parse");
const utils_1 = require("./utils");
const walkformat_1 = require("./walkformat");
const monthLengths = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
const monthOffsets = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334];
exports.abbreviatedMonthNames = [
    'Jan',
    'Feb',
    'Mar',
    'Apr',
    'May',
    'Jun',
    'Jul',
    'Aug',
    'Sep',
    'Oct',
    'Nov',
    'Dec',
];
exports.monthNames = [
    'January',
    'February',
    'March',
    'April',
    'May',
    'June',
    'July',
    'August',
    'September',
    'October',
    'November',
    'December',
];
/** Day of the week */
var WeekDay;
(function (WeekDay) {
    WeekDay[WeekDay["sunday"] = 0] = "sunday";
    WeekDay[WeekDay["monday"] = 1] = "monday";
    WeekDay[WeekDay["tuesday"] = 2] = "tuesday";
    WeekDay[WeekDay["wednesday"] = 3] = "wednesday";
    WeekDay[WeekDay["thursday"] = 4] = "thursday";
    WeekDay[WeekDay["friday"] = 5] = "friday";
    WeekDay[WeekDay["saturday"] = 6] = "saturday";
})(WeekDay || (exports.WeekDay = WeekDay = {}));
exports.abbreviatedDayNames = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
exports.dayNames = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
// internal value is integer YYYYMMDD
// this is a very simple and compact representation that leads to
// very efficient component extraction and formatting.
// Also nice for debugging
function pad(val, len) {
    let s = val.toString();
    while (s.length < len)
        s = `0${s}`;
    return s;
}
/** @internal */
function toOffset(date) {
    const { year, month, day } = date;
    // Compute the number of leap days since Jan 1st, 1970.
    // The trick is to use the previous year if month is January or February,
    // the current year otherwise.
    // Then, we compute the number of multiples of 4, 100 and 400 since 1970.
    const y = month <= 2 ? year - 1 : year;
    const n4 = Math.floor(y / 4) - Math.floor(1970 / 4);
    const n100 = Math.floor(y / 100) - Math.floor(1970 / 100);
    const n400 = Math.floor(y / 400) - Math.floor(1970 / 400);
    // Years that are multiple of 400 (like 2000) contribute by 1 (1 -1 +1 in expression below)
    // Years that are multiple of 100 but not of 400 contribute by 0 (1 -1 +0 in expression below)
    // Years that are multiple of 4 but not 100 nor 400 contribute by 1 (1 -0 +0 in expression below).
    const nLeap = n4 - n100 + n400;
    // The offset is straightforward at this point.
    // The February/March transition on leap days will be handled by the fact that the
    // 'y' value above will change, and hence the 'nLeap' value.
    return (year - 1970) * 365 + nLeap + monthOffsets[month - 1] + day - 1;
}
/** @internal */
function fromOffset(offset) {
    const julian = 2440588 + offset;
    let l = julian + 68569;
    const n = Math.floor((4 * l) / 146097);
    l -= Math.floor((146097 * n + 3) / 4);
    const i = Math.floor((4000 * (l + 1)) / 1461001);
    l = l - Math.floor((1461 * i) / 4) + 31;
    const j = Math.floor((80 * l) / 2447);
    const day = l - Math.floor((2447 * j) / 80);
    l = Math.floor(j / 11);
    const month = j + 2 - 12 * l;
    const year = 100 * (n - 49) + i + l;
    return new DateValue(year * 10000 + month * 100 + day);
}
///
/// #### Misc. functions
/// * `dateApi.isLeap(year)`: returns true if `year` is leap.
function isLeap(year) {
    return year % 4 === 0 && (year % 100 !== 0 || year % 400 === 0);
}
/// * `dateApi.daysInMonth(year, month)`: returns the number of days in the specified month.
function daysInMonth(year, month) {
    return month === 2 ? (isLeap(year) ? 29 : 28) : monthLengths[month - 1];
}
/**
 * A `DateValue` represents a date.
 *
 * This class is called `DateValue` rather than Date to avoid confusions with JavaScript's built-in `Date` class.
 *
 * The main differences with JavaScript `Date` are:
 *
 * - `DateValue` is a pure date, without any time nor timezone information. Only year, month and day.
 * - `DateValue` is immutable.
 * - The `month` property takes values between 1 and 12, not 0 and 11.
 *
 * The API uses a simple chaining style. For example:
 *
 * ```ts
 * // the current date
 * const today = DateValue.today();
 * // the end of this month, as another date
 * const endOfThisMonth = today.endOfThisMonth();
 * // the week day of the first day of next month
 * const nextMonthWeekStart = today.begOfMonth().addMonth(1).weekDay;
 * ```
 */
class DateValue {
    constructor(value) {
        if (Number.isNaN(value))
            throw new Error(`Invalid date value ${value}`);
        this._value = value;
        const { year, month, day } = this;
        if (year < 1000 || year > 9999)
            throw new Error(`Invalid date value ${value}: invalid year ${year}`);
        if (month < 1 || month > 12)
            throw new Error(`Invalid date value ${value}: invalid month ${month}`);
        if (day < 1 || day > daysInMonth(year, month))
            throw new Error(`Invalid date value ${value}: invalid day ${day}`);
    }
    // eslint-disable-next-line class-methods-use-this
    get constructorName() {
        return 'DateValue';
    }
    /**
     * The date's internal value.
     *
     * Note: this is *not* the UNIX timestamp but the year, month and day packed into a single number,
     * for example 20200514 for 2020-05-14
     */
    get value() {
        return this._value;
    }
    /** Number of seconds since 1970-01-01 */
    get epoch() {
        return toOffset(this) * 86400;
    }
    /** The year, between 0 and 9999. */
    get year() {
        return Math.floor(this._value / 10000);
    }
    /** The month, between 1 and 12. */
    get month() {
        return Math.floor(this._value / 100) % 100;
    }
    /** The day of the month, between 1 and 31. */
    get day() {
        return this._value % 100;
    }
    /** The day of the week, between 0 (Sunday) and 6 (Saturday).  */
    get weekDay() {
        // Add Julian offset (+1 because Julian origin is Monday)
        // so that modulo is on positive value even if date is before 1970
        return (2_440_588 + 1 + toOffset(this)) % 7;
    }
    /** The day of the year, between 1 and 366. */
    get yearDay() {
        const month = this.month;
        const leap = month > 2 && isLeap(this.year) ? 1 : 0;
        return monthOffsets[month - 1] + leap + this.day;
    }
    /**
     * The week number, between 1 and 53, as defined by [ISO 8601](https://en.wikipedia.org/wiki/ISO_8601#Week_dates).
     *
     * Week 1 is the beginning of the week that contains January 4th.
     */
    get week() {
        return this.weekNumber(1);
    }
    valueOf() {
        return this.toString();
    }
    /**
     * Compares two dates.
     *
     * `date1.compare(date2)` returns a positive integer if `date1 > date2`,
     * a negative integer if `date1 < date2`, and 0 if the dates are equal.
     */
    compare(date) {
        return this._value - date._value;
    }
    /** Tests date equality */
    equals(date) {
        return this._value === date._value;
    }
    /** Is the date between `begin` and `end`? (inclusive) */
    isBetween(begin, end) {
        return begin._value <= this._value && this._value <= end._value;
    }
    /** Is the date in a leap year? */
    isLeapYear() {
        return isLeap(this.year);
    }
    /** Is the date a work day? (between Monday and Friday) */
    isWorkDay() {
        return this.weekDay !== 0 && this.weekDay !== 6;
    }
    /** The number of days in the date's month, between 28 and 31. */
    daysInMonth() {
        return daysInMonth(this.year, this.month);
    }
    /** The 1st of January of the date's year. */
    begOfYear() {
        return (0, exports.make)(this.year, 1, 1);
    }
    /** The 31st of December of the date's year. */
    endOfYear() {
        return (0, exports.make)(this.year, 12, 31);
    }
    /** The first day of the date's quarter. */
    begOfQuarter() {
        return (0, exports.make)(this.year, Math.floor((this.month - 1) / 3) * 3 + 1, 1);
    }
    /** The last day of the date's quarter. */
    endOfQuarter() {
        return this.begOfQuarter().addMonths(2).endOfMonth();
    }
    /** The first day of the date's month. */
    begOfMonth() {
        return (0, exports.make)(this.year, this.month, 1);
    }
    /** The last day of the date's month. */
    endOfMonth() {
        return (0, exports.make)(this.year, this.month, this.daysInMonth());
    }
    /** Date for a given day in the same month as this date. */
    sameMonth(day) {
        const d = Math.min(day, this.daysInMonth());
        return (0, exports.make)(this.year, this.month, d);
    }
    /** The last occurence of a given day of the month before this date. */
    pastDay(day, includeThis = false) {
        const delta = day - this.day;
        return delta > 0 || (delta === 0 && !includeThis) ? this.addMonths(-1).sameMonth(day) : this.sameMonth(day);
    }
    /** The next occurence of a given day of the month after this date. */
    futureDay(day, includeThis = false) {
        const delta = day - this.day;
        return delta < 0 || (delta === 0 && !includeThis) ? this.addMonths(1).sameMonth(day) : this.sameMonth(day);
    }
    /** The last occurence of the same day in a given month before this date. */
    pastMonth(month, includeThis = false) {
        let delta = this.month - month;
        delta = delta === 0 ? (includeThis ? 0 : 12) : delta > 0 ? delta : delta + 12;
        return this.addMonths(-delta);
    }
    /** The next occurence of the same day in a given month after this date. */
    futureMonth(month, includeThis = false) {
        let delta = month - this.month;
        delta = delta === 0 ? (includeThis ? 0 : 12) : delta > 0 ? delta : delta + 12;
        return this.addMonths(delta);
    }
    /** The beginning of the week which contains this date. */
    begOfWeek(startDay = WeekDay.sunday) {
        // 0: Sunday (default), 1: Monday
        const delta = this.weekDay - (startDay || 0);
        return delta === 0 ? this : delta > 0 ? this.addDays(-delta) : this.addDays(-delta - 7);
    }
    /** The end of the week which contains this date. */
    endOfWeek(startDay = WeekDay.sunday) {
        return this.begOfWeek(startDay).addDays(6);
    }
    /** The requested week day in the same week as this date. */
    sameWeek(weekDay, startDay = WeekDay.sunday) {
        return this.begOfWeek(startDay).futureWeekDay(weekDay, true);
    }
    /** The last requested week day before this date. */
    pastWeekDay(weekDay, includeThis = false) {
        const result = this.begOfWeek(weekDay);
        return !includeThis && result.equals(this) ? this.addWeeks(-1) : result;
    }
    /** The next requested week day after this date. */
    futureWeekDay(weekDay, includeThis = false) {
        return this.pastWeekDay(weekDay, !includeThis).addWeeks(1);
    }
    /** The week number between 0 and 53. */
    weekNumber(firstDayOfWeek) {
        const begOfWeek1 = (0, exports.make)(this.year, 1, 4).begOfWeek(firstDayOfWeek);
        return Math.floor((7 + toOffset(this) - toOffset(begOfWeek1)) / 7);
    }
    /** Adds `years` years to the date. */
    addYears(years) {
        if (years === 0)
            return this;
        const year = this.year + years;
        const month = this.month;
        const day = Math.min(this.day, daysInMonth(year, month));
        return (0, exports.make)(year, month, day);
    }
    /** Adds `months` months to the date. */
    addMonths(months) {
        if (months === 0)
            return this;
        let { day, year } = this;
        const month0 = this.month - 1 + months;
        year += Math.floor(month0 / 12);
        const month = ((month0 + 120000) % 12) + 1;
        const monthLen = daysInMonth(year, month);
        day = day < monthLen ? day : monthLen;
        return (0, exports.make)(year, month, day);
    }
    /** Adds `weeks` weeks to the date. */
    addWeeks(weeks) {
        return this.addDays(7 * weeks);
    }
    /** Adds `days` days to the date. */
    addDays(days) {
        if (days === 0)
            return this;
        return fromOffset(toOffset(this) + days);
    }
    /** The number of days between two dates. */
    daysDiff(date) {
        return toOffset(this) - toOffset(date);
    }
    /**
     * Adds a delta which may combine years, months and days to the date.
     *
     * @example `d.add({ months: 2 })` adds 2 months to `d`.
     */
    add(delta) {
        return this.addYears(delta.years || 0)
            .addMonths(delta.months || 0)
            .addWeeks(delta.weeks || 0)
            .addDays(delta.days || 0);
    }
    toString() {
        // RFC 3339 by default -- very fast
        const str = pad(this._value, 8);
        return `${str.substring(0, 4)}-${str.substring(4, 6)}-${str.substring(6, 8)}`;
    }
    /**
     * formats the date according to `format`
     * example: date.toString(format = 'YYYY-MM-DD')
     * @param locale LocalizeLocale
     * @param format string
     * @returns string
     */
    format(fmt, locale = 'base') {
        let result = '';
        const { day, weekDay, month, year } = this;
        (0, walkformat_1.walkFormat)(fmt, {
            literal(lit) {
                result += lit;
            },
            d(repeat) {
                switch (repeat) {
                    case 1:
                        result += day.toString();
                        break;
                    case 2:
                        result += pad(day, 2);
                        break;
                    case 3:
                        result += (0, exports.weekDayName)(weekDay, locale, true);
                        break;
                    case 4:
                        result += (0, exports.weekDayName)(weekDay, locale);
                        break;
                    default:
                        throw datetime_1.Datetime.formatDirectiveError(fmt, 'd', locale, repeat);
                }
            },
            D(repeat) {
                switch (repeat) {
                    case 1:
                        result += day.toString();
                        break;
                    case 2:
                        result += pad(day, 2);
                        break;
                    default:
                        throw datetime_1.Datetime.formatDirectiveError(fmt, 'D', locale, repeat);
                }
            },
            M(repeat) {
                switch (repeat) {
                    case 1:
                        result += month.toString();
                        break;
                    case 2:
                        result += pad(month, 2);
                        break;
                    case 3:
                        result += (0, exports.monthName)(month, locale, true);
                        break;
                    case 4:
                        result += (0, exports.monthName)(month, locale);
                        break;
                    default:
                        throw datetime_1.Datetime.formatDirectiveError(fmt, 'M', locale, repeat);
                }
            },
            Y(repeat) {
                switch (repeat) {
                    case 2:
                        result += pad(year % 100, 2);
                        break;
                    case 4:
                        result += pad(year, 4);
                        break;
                    default:
                        throw datetime_1.Datetime.formatDirectiveError(fmt, 'Y', locale, repeat);
                }
            },
        });
        return result;
    }
    ///
    /// #### Conversion to datetime and JavaScript Date
    /// * `date.at(time, millisecond = 0)`: combines the date with `time` and `millisecond` and returns a timestamp.
    at(t, timeZone, millisecond = 0) {
        return datetime.make(this.year, this.month, this.day, t.hour, t.minute, t.second, millisecond, timeZone);
    }
    /// * `date.toJsDate(timeZone)`: converts the date to a JavaScript `Date` object.
    toJsDate(timeZone) {
        if (!timeZone)
            return new Date(this.year, this.month - 1, this.day);
        if (timeZone === 'UTC')
            return new Date(Date.UTC(this.year, this.month - 1, this.day));
        // If timeZone, create a DateTime in the correct time zone, get its date and then apply toJsDate without arg
        const dt = datetime.make(this.year, this.month, this.day, 0, 0, 0, 0, timeZone);
        return dt.date.toJsDate();
    }
    /// * `date.toJSON()
    ///   converts to JSON format
    toJSON() {
        return this.toString();
    }
    isNull() {
        return this._value === 0;
    }
    ///
    /// ## Functions
    ///
    /// #### Localized names for months, week days, etc.
    /// * `dateApi.amDesignator()`: returns the AM designator in the provided locale.
    static amDesignator(locale = 'base') {
        return (0, localize_1.localizedText)('@sage/xtrem-date-time/datetime__amDesignator', 'AM', {}, locale);
    }
    /// * `dateApi.pmDesignator()`: returns the PM designator in the provided locale.
    static pmDesignator(locale = 'base') {
        return (0, localize_1.localizedText)('@sage/xtrem-date-time/datetime__pmDesignator', 'PM', {}, locale);
    }
    /// * `dateApi.monthName(month)`: returns the name of `month` in the provided locale.
    static monthName(month, locale = 'base', abbrev = false) {
        const mthIndex = month - 1;
        if (mthIndex < 0 || mthIndex > 11) {
            throw new Error((0, localize_1.localizedText)('@sage/xtrem-date-time/datetime__invMonth', 'invalid month value', {}, locale));
        }
        return abbrev
            ? (0, localize_1.localizedText)(`@sage/xtrem-date-time/datetime__abbreviatedMonthName${mthIndex}`, exports.abbreviatedMonthNames[mthIndex], {}, locale)
            : (0, localize_1.localizedText)(`@sage/xtrem-date-time/datetime__monthName${mthIndex}`, exports.monthNames[mthIndex], {}, locale);
    }
    /// * `dateApi.monthFromName(name)`: converts a month name to a month number, using the provided locale.
    static monthFromName(name, locale = 'base') {
        const localeMonthValue = (0, utils_1.localizedGroup)(exports.monthNames, '@sage/xtrem-date-time/datetime__monthName', {}, locale);
        const localeAbbrevMonthValue = (0, utils_1.localizedGroup)(exports.abbreviatedMonthNames, '@sage/xtrem-date-time/datetime__abbreviatedMonthName', {}, locale);
        const s = name.toLowerCase();
        for (let i = 0; i < localeMonthValue.length; i += 1) {
            if (localeMonthValue[i].toLowerCase() === s || localeAbbrevMonthValue[i].toLowerCase() === s) {
                return i + 1;
            }
        }
        return -1;
    }
    /// * `dateApi.weekDayName(weekDay)`: returns the name of `weekDay` in the provided locale.
    static weekDayName(weekDay, locale = 'base', abbrev = false) {
        if (weekDay < 0 || weekDay > 6) {
            throw new Error((0, localize_1.localizedText)('@sage/xtrem-date-time/datetime__invDay', 'invalid day value', {}, locale));
        }
        return abbrev
            ? (0, localize_1.localizedText)(`@sage/xtrem-date-time/datetime__abbreviatedDayName${weekDay}`, exports.abbreviatedDayNames[weekDay], {}, locale)
            : (0, localize_1.localizedText)(`@sage/xtrem-date-time/datetime__dayName${weekDay}`, exports.dayNames[weekDay], {}, locale);
    }
    /// * `dateApi.weekDayFromName(name)`: converts a day name to a weekDay number, using the provided locale setting.
    static weekDayFromName(name, locale = 'base') {
        const localeDayValue = (0, utils_1.localizedGroup)(exports.dayNames, '@sage/xtrem-date-time/datetime__dayName', {}, locale);
        const localeAbbrevDayValue = (0, utils_1.localizedGroup)(exports.abbreviatedDayNames, '@sage/xtrem-date-time/datetime__abbreviatedDayName', {}, locale);
        const s = name.toLowerCase();
        for (let i = 0; i < localeDayValue.length; i += 1) {
            if (localeDayValue[i].toLowerCase() === s || localeAbbrevDayValue[i].toLowerCase() === s) {
                return i;
            }
        }
        return -1;
    }
    ///
    /// #### Parsing
    /// * `dateApi.parse(str, format = 'YYYY-MM-DD')`: parses `str` according to `format`. Returns a date.
    static parse(str, locale = 'base', format) {
        if (str == null)
            throw new Error((0, localize_1.localizedText)('@sage/xtrem-date-time/datetime__dateNull', '"date is null"', {}, locale));
        if (format == null) {
            // RFC 3339 by default -- very fast
            // No need to check the whole validity of the date, it will be checked by the constructor
            if (!/^\d{4}-\d{2}-\d{2}$/.test(str))
                throw new Error((0, localize_1.localizedText)('@sage/xtrem-date-time/datetime__badFormat', "bad date format, expected 'YYYY-MM-DD', got '{{format}}'", { format: str }, locale));
            const value = parseInt(str.replace(/-/g, ''), 10);
            return new DateValue(value);
        }
        const dtObj = (0, parse_1.createDateTimeObj)();
        (0, parse_1.parseDateTime)(str, format, dtObj, locale);
        // ignore weekday
        return (0, exports.make)(dtObj.year, dtObj.month, dtObj.day);
    }
    ///
    /// #### Current date
    /// * `dateApi.today(timeZone)`: returns the current date.
    static today(timeZone) {
        return datetime_1.Datetime.now(false, timeZone).date;
    }
    ///
    /// #### Constructors
    /// * `dateApi.fromJsDate(jsDate, utc)`: convert a JavaScript `Date` to a date value.
    static fromJsDate(jsDate, timeZone) {
        if (!timeZone)
            return (0, exports.make)(jsDate.getFullYear(), jsDate.getMonth() + 1, jsDate.getDate());
        if (timeZone === 'UTC')
            return (0, exports.make)(jsDate.getUTCFullYear(), jsDate.getUTCMonth() + 1, jsDate.getUTCDate());
        // If timeZone, convert jsDate to a datetime in the time zone and then get its date.
        return datetime.fromJsDate(jsDate, timeZone).date;
    }
    static fromInternalValue(value) {
        return new DateValue(value);
    }
    /// * `dateApi.make(year, month, day)`: returns a date value for the specified year, month and day.
    static make(year, month, day) {
        return new DateValue(year * 10000 + month * 100 + day);
    }
    /// * `dateApi.makeInWeek(year, week, wday)`: returns a date value for the specified week and week day.
    static makeInWeek(year, week, wday) {
        const dday = wday ? wday - 1 : 6;
        return (0, exports.make)(year, 1, 4)
            .begOfWeek(1)
            .addDays(7 * (week - 1) + dday);
    }
    static getWeekDayNames(startDay = 'Sunday') {
        const startIndex = exports.dayNames.indexOf(startDay);
        if (startIndex === 0) {
            return exports.dayNames;
        }
        return exports.dayNames.slice(startIndex).concat(exports.dayNames.slice(0, startIndex));
    }
    static getWeekDayNumbers(startDay = WeekDay.sunday) {
        const days = [0, 1, 2, 3, 4, 5, 6];
        if (startDay === WeekDay.sunday) {
            return days;
        }
        return days.slice(startDay).concat(days.slice(0, startDay));
    }
    ///
    /// #### Instanceof
    /// * `dateApi.isDate(obj)`: returns true iff `obj` is a date value.
    static isDate(obj) {
        // instanceOf is broken if multiple copies of the module are loaded
        return obj && obj.constructorName === 'DateValue';
    }
}
exports.DateValue = DateValue;
exports.amDesignator = DateValue.amDesignator.bind(DateValue);
exports.pmDesignator = DateValue.pmDesignator.bind(DateValue);
exports.monthName = DateValue.monthName.bind(DateValue);
exports.monthFromName = DateValue.monthFromName.bind(DateValue);
exports.weekDayName = DateValue.weekDayName.bind(DateValue);
exports.weekDayFromName = DateValue.weekDayFromName.bind(DateValue);
exports.parse = DateValue.parse.bind(DateValue);
exports.today = DateValue.today.bind(DateValue);
exports.fromJsDate = DateValue.fromJsDate.bind(DateValue);
exports.fromInternalValue = DateValue.fromInternalValue.bind(DateValue);
exports.make = DateValue.make.bind(DateValue);
exports.makeInWeek = DateValue.makeInWeek.bind(DateValue);
exports.isDate = DateValue.isDate.bind(DateValue);
exports.getWeekDayNames = DateValue.getWeekDayNames.bind(DateValue);
exports.getWeekDayNumbers = DateValue.getWeekDayNumbers.bind(DateValue);
//# sourceMappingURL=date.js.map