"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DateRange = void 0;
exports.isDateRange = isDateRange;
exports.parse = parse;
exports.make = make;
exports.getDayRange = getDayRange;
exports.getMonthRange = getMonthRange;
exports.getYearRange = getYearRange;
exports.getPreviousWeekRange = getPreviousWeekRange;
exports.getSameWeekRange = getSameWeekRange;
exports.getNextWeekRange = getNextWeekRange;
exports.getDateRange = getDateRange;
const date_1 = require("./date");
const range_1 = require("./range");
const utils_1 = require("./utils");
class DateRange extends range_1.Range {
    static { this.parse = parse; }
    static { this.isDateRange = isDateRange; }
    static { this.make = make; }
    static { this.getDayRange = getDayRange; }
    static { this.getMonthRange = getMonthRange; }
    static { this.getYearRange = getYearRange; }
    static { this.getDateRange = getDateRange; }
    // eslint-disable-next-line class-methods-use-this
    get constructorName() {
        return 'DateRange';
    }
    get durationInDays() {
        if (this.excludedEnd instanceof date_1.DateValue && this.includedStart instanceof date_1.DateValue) {
            return this.excludedEnd.daysDiff(this.includedStart);
        }
        return Infinity;
    }
    /** @internal */
    // eslint-disable-next-line class-methods-use-this
    incrementsByRangeUnit(elm, step = 1) {
        return elm.addDays(step);
    }
    /** @internal */
    // eslint-disable-next-line class-methods-use-this
    getBoundValue(elm) {
        if (!elm)
            return null;
        return elm?.value;
    }
    /** @internal */
    // eslint-disable-next-line class-methods-use-this
    getStringBoundValue(elm) {
        if (!elm)
            return '';
        return elm.toString();
    }
}
exports.DateRange = DateRange;
function isDateRange(obj) {
    return obj && obj.constructorName === 'DateRange';
}
function parse(str) {
    const [start, end] = str.split(',');
    if (!end)
        throw new Error(`invalid date range format: ${str}.`);
    const excludesStart = start.length === 0 || start.startsWith('(');
    const excludesEnd = end.length === 0 || end.endsWith(')');
    if (!start && !end)
        return new DateRange(null, null);
    const parseHalf = (s) => {
        if (!s) {
            return null;
        }
        const clean = s.replace(/"?(\d{1,4})(?:-(\d{1,2})(?:-(\d{1,2})))"?/, (_all, year, month = '01', day = '01') => `${year.padStart(4, '0')}-${month.padStart(2, '0')}-${day.padStart(2, '0')}`);
        return date_1.DateValue.parse(clean);
    };
    return new DateRange(parseHalf(start && start.substring(1)), parseHalf(end && end.substring(0, end.length - 1)), excludesStart, excludesEnd);
}
function make(start, end, excludesStart = false, excludesEnd = true) {
    return new DateRange(start, end, excludesStart, excludesEnd);
}
function getDayRange(d) {
    return new DateRange(d, d, false, false);
}
function getMonthRange(d) {
    return new DateRange(d.begOfMonth(), d.endOfMonth(), false, false);
}
function getYearRange(d) {
    return new DateRange(d.begOfYear(), d.endOfYear(), false, false);
}
function getPreviousWeekRange(date, startOfWeek) {
    const rangeStart = date.begOfWeek(startOfWeek).addWeeks(-1);
    const rangeEnd = date.begOfWeek(startOfWeek).addDays(-1);
    return new DateRange(rangeStart, rangeEnd, false, false);
}
function getSameWeekRange(date, startOfWeek) {
    const rangeStart = date.begOfWeek(startOfWeek);
    const rangeEnd = date.endOfWeek(startOfWeek);
    return new DateRange(rangeStart, rangeEnd, false, false);
}
function getNextWeekRange(date, startOfWeek) {
    const rangeStart = date.endOfWeek(startOfWeek).addDays(1);
    const rangeEnd = date.endOfWeek(startOfWeek).addWeeks(1);
    return new DateRange(rangeStart, rangeEnd, false, false);
}
function getDateRange({ date: d, range, locale, }) {
    if (!(0, utils_1.isValidDatePropertyValue)(d)) {
        throw new Error(`Invalid date ${d}`);
    }
    const dateValue = date_1.DateValue.parse(d);
    let startOfWeek = date_1.WeekDay.monday;
    if (locale === 'en-US') {
        startOfWeek = date_1.WeekDay.sunday;
    }
    switch (range) {
        case 'same-day':
            return getDayRange(dateValue);
        case 'previous-day':
            return getDayRange(dateValue.addDays(-1));
        case 'same-month':
            return getMonthRange(dateValue);
        case 'previous-month':
            return getMonthRange(dateValue.addMonths(-1));
        case 'same-year':
            return getYearRange(dateValue);
        case 'previous-year':
            return getYearRange(dateValue.addYears(-1));
        case 'last-7-days':
            return new DateRange(dateValue.addDays(-7), dateValue);
        case 'last-30-days':
            return new DateRange(dateValue.addDays(-30), dateValue);
        case 'previous-week':
            return getPreviousWeekRange(dateValue, startOfWeek);
        case 'same-week':
            return getSameWeekRange(dateValue, startOfWeek);
        case 'next-day':
            return getDayRange(dateValue.addDays(1));
        case 'next-week':
            return getNextWeekRange(dateValue, startOfWeek);
        case 'next-month':
            return getMonthRange(dateValue.addMonths(1));
        case 'next-year':
            return getYearRange(dateValue.addYears(1));
        default:
            throw new Error("Invalid parameters supplied to \"getRange\": \"range\" must be one of the following: 'same-day', 'previous-day', 'same-month', 'previous-month', 'same-year', 'previous-year', 'last-7-days', 'last-30-days', 'previous-week', 'same-week', 'next-day', 'next-week', 'next-month', 'next-year'.");
    }
}
//# sourceMappingURL=date-range.js.map