import type { Location } from '@sage/wh-master-data-api';
import { stockControl } from '@sage/wh-master-data/build/lib/menu-items/stock-control';
import { getCurrentSiteDepositor } from '@sage/wh-master-data/lib/client-functions/get-selected-site-depositor';
import { getSelectedSiteDepositorByStorage } from '@sage/wh-master-data/lib/client-functions/storage-properties';
import type { GraphApi } from '@sage/wh-pages-api';
import { setApplicativePageCrudActions } from '@sage/wh-system/lib/client-functions/applicative-crud-actions';
import * as ui from '@sage/xtrem-ui';
import { StockByProductArgs } from '../interfaces/index';

/**
 * Page to view stock by store / location
 * using reverse order by location / store
 */
@ui.decorators.page<MobileViewStockByStoreLocation>({
    title: 'View stock by location',
    mode: 'default',
    menuItem: stockControl,
    priority: 100,
    subtitle: 'Select a location',
    isTitleHidden: false,
    node: '@sage/wh-master-data/Location',
    isTransient: false,
    createAction: undefined,
    objectTypeSingular: 'Location',
    objectTypePlural: 'Locations',
    authorizationCode: 'INQSTOPRO',
    // idField() {
    //     return this.code;
    // },
    // headerCard() {
    //     return {
    //         title: this.code,
    //     };
    // },
    headerDropDownActions() {
        return [
            // this.$standardSaveAction,
            // this.$standardCancelAction,
            // this.$standardOpenRecordHistoryAction,
        ];
    },
    async onLoad(): Promise<void> {
        if (!(await this._initSiteDepositor())) {
            this.disablePage();
            // Close the page
            this.$.finish();
        }
    },
    async onDirtyStateUpdated(isDirty: boolean) {
        setApplicativePageCrudActions({
            page: this,
            isDirty,
            save: undefined, // this.$standardSaveAction,
            cancel: undefined, // this.$standardCancelAction,
            remove: undefined, // this.$standardDeleteAction,
        });
    },
    businessActions() {
        return [
            // this.$standardSaveAction,
            // this.$standardCancelAction,
            // this.$standardNewAction,
            // this.$standardDeleteAction,
        ];
    },
    navigationPanel: {
        canFilter: false,
        isHeaderHidden: true,
        isAutoSelectEnabled: true,
        listItem: {
            title: ui.nestedFields.text({
                title: 'Location',
                bind: 'displayCode',
                canFilter: false,
            }),
            titleRight: ui.nestedFields.reference({
                title: 'Depositor',
                bind: 'depositor',
                node: '@sage/wh-master-data/Depositor',
                canFilter: true,
                valueField: 'code',
            }),
            line2: ui.nestedFields.text({
                title: 'Location type',
                bind: 'locationType',
                canFilter: true,
            }),
            line2Right: ui.nestedFields.reference({
                title: 'Product',
                bind: 'product',
                node: '@sage/wh-product-data/Product',
                canFilter: true,
                valueField: 'code',
            }),
            line3: ui.nestedFields.reference({
                title: 'Container',
                bind: 'referenceContainer',
                node: '@sage/wh-product-data/Container',
                canFilter: true,
                valueField: 'code',
            }),
            line3Right: ui.nestedFields.text({
                title: 'Location',
                bind: 'code',
                canFilter: true,
                isHidden: true,
            }),
            line4: ui.nestedFields.reference({
                title: 'Store',
                bind: 'store',
                node: '@sage/wh-master-data/Store',
                valueField: 'code',
                canFilter: true,
                isHidden: true,
            }),
            line4Right: ui.nestedFields.numeric({
                bind: '_id',
                canFilter: false,
                isHidden: true,
            }),
            line5: ui.nestedFields.text({
                title: 'Store location',
                bind: 'storeLocationCode',
                canFilter: true,
                isHidden: true,
            }),
            line5right: ui.nestedFields.text({
                title: 'Location',
                bind: 'storeLocationKey',
                canFilter: true,
                isHidden: true,
            }),
        },
        orderBy: {
            store: { code: 1 },
            code: 1,
        },
        async optionsMenu(_graph, storage, _queryParam, _username, _userCode) {
            const _selectedSiteDepositor = getSelectedSiteDepositorByStorage(storage);
            return [
                {
                    title: '',
                    graphQLFilter: {
                        _or: [
                            {
                                site: {
                                    code: _selectedSiteDepositor?.site,
                                },
                                depositor: { code: _selectedSiteDepositor?.depositor },
                            },
                            {
                                site: {
                                    code: _selectedSiteDepositor?.site,
                                },
                                depositor: null,
                            },
                        ],
                    },
                },
            ];
        },
        onSelect(storeLocation: any): boolean | void {
            if (storeLocation && this.siteCodeSelected && this.depositorCodeSelected) {
                // eslint-disable-next-line consistent-return
                this._countStockObjects(storeLocation).then(count => {
                    if (count > 0) {
                        this.$.router.goTo(
                            '@sage/wh-pages/MobileViewStockByStoreLocationSelectProduct',
                            this._getCurrentParameters(storeLocation),
                        );
                        return true;
                    }
                    this.$.showToast(
                        ui.localize(
                            '@sage/wh-pages/selected-store-location-no-results',
                            'Location {{ storeCode}} {{ locationCode }} has no stock records.',
                            {
                                storeCode: storeLocation.store?.code ?? '',
                                locationCode: storeLocation.code ?? '',
                            },
                        ),
                        { type: 'info' },
                    );
                });
            }
            return true;
        },
    },
})
export class MobileViewStockByStoreLocation extends ui.Page<GraphApi> {
    /** internal properties */

    private siteCodeSelected: string | undefined;

    private depositorCodeSelected: string | undefined;

    /*
     *
     *  Sections
     *
     */

    @ui.decorators.section<MobileViewStockByStoreLocation>({
        isTitleHidden: true,
        isHidden: true,
    })
    mainSection!: ui.containers.Section;

    /*
     *
     *  Blocks
     *
     */

    @ui.decorators.block<MobileViewStockByStoreLocation>({
        parent() {
            return this.mainSection;
        },
    })
    storeLocationBlock!: ui.containers.Block;
    /*
     *
     *  Page properties
     *
     */
    // -----------------------------

    /*
     *
     *  Technical Fields
     *
     */

    @ui.decorators.referenceField<MobileViewStockByStoreLocation>({
        title: 'Site',
        isReadOnly: true,
        isHidden: true,
    })
    site!: ui.fields.Reference;

    @ui.decorators.referenceField<MobileViewStockByStoreLocation>({
        title: 'Depositor',
        isReadOnly: true,
        isHidden: true,
    })
    depositor!: ui.fields.Reference;

    private async _initSiteDepositor(): Promise<boolean> {
        const siteDepositor = await getCurrentSiteDepositor(
            this,
            ui.localize('@sage/wh-pages/dialog-error-title', 'Error'),
            ui.localize(
                '@sage/wh-pages/dialog-error-location-inquiry-set-site-depositor',
                'Define a default site and depositor on the user function profile.',
            ),
        );
        if (siteDepositor && siteDepositor?.site && siteDepositor?.depositor) {
            this.siteCodeSelected = siteDepositor?.site;
            this.depositorCodeSelected = siteDepositor?.depositor;
            return true;
        }
        return false;
    }

    private async disablePage(): Promise<void> {
        this.mainSection.isDisabled = true;
        this.storeLocationBlock.isDisabled = true;
    }

    /**
     *  Count stock objects
     * @param storeLocation
     * @returns
     */
    private async _countStockObjects(storeLocation: any): Promise<number> {
        try {
            return Number(
                (
                    await this.$.graph
                        .node('@sage/wh-stock-data/StockObject')
                        .aggregate.read(
                            {
                                _id: {
                                    distinctCount: true,
                                },
                            },
                            {
                                filter: {
                                    site: {
                                        code: this.siteCodeSelected,
                                    },
                                    depositor: {
                                        code: this.depositorCodeSelected,
                                    },
                                    ...(storeLocation?.store?.code && { store: { code: storeLocation?.store?.code } }),
                                    ...(storeLocation?.code && {
                                        location: { code: storeLocation?.code },
                                    }),
                                    product: { isActive: true },
                                },
                            },
                        )
                        .execute()
                )?._id?.distinctCount || 0,
            );
        } catch (error) {
            ui.console.error(`Error counting stock objects:\n${JSON.stringify(error)}`);
        }

        return 0;
    }

    /**
     * get query parameters
     * @returns expected parameters or undefined
     */
    private _getQueryParameters(): StockByProductArgs | undefined {
        try {
            return JSON.parse(String(this.$.queryParameters.stockByProductArgs)) as StockByProductArgs;
        } catch (error) {
            ui.console.error(`Error retrieving query parameters:\n${JSON.stringify(error)}`);
            return undefined;
        }
    }

    /**
     * Return current parameter before chain another page
     * @return return the current parameters
     */
    private _getCurrentParameters(storeLocation: Location): {
        stockByProductArgs: string;
    } {
        return {
            stockByProductArgs: JSON.stringify(<StockByProductArgs>{
                isViewByStoreLocation: true,
                siteDepositorSelected: {
                    siteCode: this.siteCodeSelected,
                    depositorCode: this.depositorCodeSelected,
                },
                storeLocationSelected: {
                    storeCode: storeLocation?.store?.code,
                    locationCode: storeLocation?.code,
                    locationType: storeLocation?.locationType,
                    referenceContainerCode: storeLocation?.referenceContainer?.code,
                },
                _id: storeLocation._id,
            }),
        };
    }
}
