import { StockUnit } from '@sage/wh-master-data/lib/interfaces';
import type { GraphApi } from '@sage/wh-pages-api';
import type { Product } from '@sage/wh-product-data-api';
import { getConsumptionUnitCode } from '@sage/wh-product-data/lib/client-functions/get-consumption-unit';
import { ConsumptionUnit } from '@sage/wh-product-data/lib/interfaces';
import type { SerialNumber, StockObject } from '@sage/wh-stock-data-api';
import {
    generateAggregateSerialNumberRequest,
    generateAggregateStockObjectRequest,
} from '@sage/wh-stock-data/lib/client-functions/aggregators';
import { StockObjectSelected } from '@sage/wh-stock-data/lib/interfaces';
import { Dict, Edges, extractEdges } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import { StockByProductArgs } from '../interfaces';

interface AggregatedProductStoreLocation {
    _id: string;
    _key: string;
    productCode: string;
    localizedDescription: string;
    numberOfConsumptionUnit: number;
    distinctLot: number;
    distinctSerialNumber: number;
    isKeyInLotNumber: boolean;
    isKeyInSerialNumber: boolean;
    isLocalizedSerialNumberAllowed: boolean;
    isStockUnitManagement: boolean;
    stockUnit?: StockUnit;
}

interface ProductStoreLocationValue
    extends Omit<
        AggregatedProductStoreLocation,
        '_key' | 'distinctLot' | 'distinctSerialNumber' | 'isStockUnitManagement'
    > {
    distinctLot: string;
    distinctSerialNumber: string;
    consumptionUnitCode: string;
}

const _hideWhenEmptyValue = (value: any, _rowValue?: Dict<Product>) => {
    return typeof value !== 'number' && !value;
};

@ui.decorators.page<MobileViewStockByStoreLocationSelectProduct>({
    node: '@sage/wh-stock-data/StockObject',
    mode: 'default',
    isTitleHidden: true,
    title: 'Stock by location',
    subtitle: 'Select a product',
    skipDirtyCheck: true,
    objectTypeSingular: 'Product',
    objectTypePlural: 'Products',
    authorizationCode: 'INQSTOPRO',
    headerCard() {
        return {
            title: this.storeLocation,
            titleRight: this.distinctProducts,
            line2: this.locationType,
            line3: this.referenceContainerCode,
        };
    },
    navigationPanel: undefined,
    // navigationPanel: {
    //     menuType: 'toggle',
    //     canFilter: true,
    //     isFirstLetterSeparatorHidden: true,
    //     isHeaderHidden: false,
    //     listItem: {
    //         title: ui.nestedFields.reference({
    //             title: 'Product',
    //             node: '@sage/wh-product-data/Product',
    //             bind: 'product',
    //             valueField: 'code',
    //             canFilter: true,
    //         }),
    //         titleRight: ui.nestedFields.reference({
    //             title: 'Description',
    //             node: '@sage/wh-product-data/Product',
    //             bind: 'product',
    //             valueField: 'localizedDescription',
    //             canFilter: true,
    //         }),
    //         line2: ui.nestedFields.reference({
    //             title: 'Store',
    //             node: '@sage/wh-master-data/Store',
    //             bind: 'store',
    //             valueField: 'code',
    //             canFilter: false,
    //         }),
    //         line2Right: ui.nestedFields.reference({
    //             title: 'Location',
    //             node: '@sage/wh-master-data/Location',
    //             bind: 'location',
    //             valueField: 'code',
    //             canFilter: false,
    //         }),
    //         line3: ui.nestedFields.reference({
    //             title: 'Container',
    //             node: '@sage/wh-product-data/Container',
    //             bind: 'container',
    //             valueField: { container: { code: true } },
    //             canFilter: false,
    //         }),
    //         line3Right: ui.nestedFields.text({
    //             title: 'Stock object',
    //             bind: 'code',
    //             canFilter: false,
    //         }),
    //     },
    //     orderBy: {
    //         code: 1,
    //     },
    //     async optionsMenu(_graph, _storage, _queryParam, _username, userCode) {
    //         const _productParameters = JSON.parse(String(_queryParam['stockByProductArgs'])) as StockByProductArgs;
    //         return [
    //             {
    //                 title: '',
    //                 graphQLFilter: {
    //                     site: {
    //                         code: _productParameters?.depositorSiteSelected?.siteCodeSelected ?? '',
    //                     },
    //                     depositor: {
    //                         code: _productParameters?.depositorSiteSelected?.depositorCodeSelected ?? '',
    //                     },
    //                     product: {
    //                         code: _productParameters?.productCode ?? '',
    //                         isActive: true,
    //                     },
    //                 },
    //             },
    //         ];
    //     },
    // },
    async onLoad(): Promise<void> {
        // Requires a selected product in the query parameters.  This should not occur unless user manually
        // directs themselves to this page
        const _parameters = this._getQueryParameters();
        if (!_parameters) {
            this.$.showToast(ui.localize('@sage/wh-pages/product-required', 'Selected product is required'), {
                type: 'warning',
            });
            this.$.router.goTo('@sage/wh-pages/MobileViewStockByProduct');
            return;
        }

        this._parameters = _parameters;

        this.storeLocation.value = `${_parameters?.storeLocationSelected?.storeCode} ${_parameters?.storeLocationSelected?.locationCode}`;
        this.locationType.value = _parameters?.storeLocationSelected?.locationType ?? null;
        this.referenceContainerCode.value = _parameters?.storeLocationSelected?.referenceContainerCode ?? null;

        /**
         *  Read aggregated product data and populate the header with the results
         */
        const _aggregatedProducts = await this._readAggregatedProducts(_parameters);

        this.distinctProducts.value = _aggregatedProducts?.length ?? 0;
        /**
         * Read aggregated store and location data and populate the table with the results
         */
        // const _aggregatedStoreAndLocation = await this._readAggregatedStoreAndLocation(_parameters);

        this.mainBlock.title = ui.localize('@sage/wh-pages/number-of-products', '{{ numberOfProducts }} product(s)', {
            numberOfProducts: _aggregatedProducts?.length ?? 0,
        });

        // Redirect user to first page if selected product yields no records.  User should not get to this page
        // if no stock records unless manually entering page.
        if (!_aggregatedProducts?.length) {
            this.$.showToast(
                ui.localize(
                    '@sage/wh-pages/selected-store-location-no-results',
                    'Location {{ storeCode}} {{ locationCode }} has no stock records.',
                    {
                        storeCode: _parameters.storeLocationSelected?.storeCode ?? '',
                        locationCode: _parameters.storeLocationSelected?.locationCode ?? '',
                    },
                ),
            );
            this.$.router.goTo('@sage/wh-pages/MobileViewStockByStoreLocation');
            return;
        }

        this.$.setPageClean();

        // Populate the aggregated product data with consumption unit

        await this._initializeProductTable(_aggregatedProducts, _parameters);
    },
})
export class MobileViewStockByStoreLocationSelectProduct extends ui.Page<GraphApi> {
    /**
     * Internal properties
     */

    private _parameters: StockByProductArgs | undefined;

    @ui.decorators.section<MobileViewStockByStoreLocationSelectProduct>({
        isTitleHidden: true,
    })
    mainSection!: ui.containers.Section;

    @ui.decorators.block<MobileViewStockByStoreLocationSelectProduct>({
        parent() {
            return this.mainSection;
        },
        isTitleHidden: false,
    })
    mainBlock!: ui.containers.Block;

    @ui.decorators.textField<MobileViewStockByStoreLocationSelectProduct>({
        isTransient: true,
        isReadOnly: true,
    })
    storeLocation!: ui.fields.Text;

    @ui.decorators.numericField<MobileViewStockByStoreLocationSelectProduct>({
        isTransient: true,
        isReadOnly: true,
        prefix: ui.localize('@sage/wh-pages/products-prefix', 'Products:'),
    })
    distinctProducts!: ui.fields.Numeric;

    @ui.decorators.textField<MobileViewStockByStoreLocationSelectProduct>({
        isTransient: true,
        isReadOnly: true,
    })
    locationType!: ui.fields.Text;

    @ui.decorators.textField<MobileViewStockByStoreLocationSelectProduct>({
        isTransient: true,
        isReadOnly: true,
        isHidden: _hideWhenEmptyValue,
    })
    referenceContainerCode!: ui.fields.Text;

    @ui.decorators.tableField<MobileViewStockByStoreLocationSelectProduct>({
        parent() {
            return this.mainBlock;
        },
        canSelect: false,
        canUserHideColumns: false,
        canFilter: false,
        isTransient: true,
        displayMode: ui.fields.TableDisplayMode.comfortable,
        mobileCard: undefined,
        onRowClick(_id: string, rowItem) {
            if (this._parameters) {
                this.$.router.goTo(
                    '@sage/wh-pages/MobileViewStockByProductSelectStockObject',
                    this._getCurrentParameters(rowItem),
                );
            }
        },
        orderBy: { storeLocationCode: 1 },
        columns: [
            ui.nestedFields.text({
                bind: 'productCode',
                isReadOnly: true,
            }),
            ui.nestedFields.numeric({
                bind: 'numberOfConsumptionUnit',
                isReadOnly: true,
                postfix(value: any, _rowValue?: Dict<any>) {
                    return _rowValue?.consumptionUnitCode ?? '';
                },
                scale(value, rowValue?: any) {
                    return Number(rowValue?.stockUnit?.numberOfDecimals ?? 0);
                },
            }),
            ui.nestedFields.text({
                bind: 'localizedDescription',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: '_spacerColumn',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'distinctLot',
                isReadOnly: true,
                isTransient: true,
            }),
            ui.nestedFields.text({
                bind: 'distinctSerialNumber',
                isReadOnly: true,
                isTransient: true,
            }),
            ui.nestedFields.text({
                bind: 'consumptionUnitCode',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isKeyInLotNumber',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isLocalizedSerialNumberAllowed',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: '_id',
                isReadOnly: true,
                isHidden: true,
            }),
        ],
    })
    products!: ui.fields.Table<ProductStoreLocationValue>;

    private async _initializeProductTable(
        _aggregatedProducts: AggregatedProductStoreLocation[],
        _parameters: StockByProductArgs,
    ): Promise<void> {
        // Convert products to table values using functional approach
        const productValues = await Promise.all(
            _aggregatedProducts.map(async _product => {
                const _consumptionUnitCode = (await getConsumptionUnitCode(this, _product.productCode)) ?? '';
                return <ProductStoreLocationValue>{
                    _id: _product._id,
                    productCode: _product.productCode,
                    localizedDescription: _product.localizedDescription,
                    numberOfConsumptionUnit: _product.numberOfConsumptionUnit,
                    isKeyInLotNumber: _product.isKeyInLotNumber,
                    isKeyInSerialNumber: _product.isKeyInSerialNumber,
                    isLocalizedSerialNumberAllowed: _product.isLocalizedSerialNumberAllowed,
                    consumptionUnitCode: _consumptionUnitCode,
                    stockUnit: _product.stockUnit,
                    distinctLot: _product.isKeyInLotNumber
                        ? `${ui.localize('@sage/wh-pages/lots-numbers-prefix', 'Lots:')} ${_product.distinctLot}`
                        : '',
                    distinctSerialNumber: _product.isLocalizedSerialNumberAllowed
                        ? `${ui.localize('@sage/wh-pages/serial-numbers-prefix', 'Serial no.:')} ${_product.distinctSerialNumber}`
                        : '',
                };
            }),
        );

        this.products.value = productValues;
    }

    /**
     * get query parameters
     * @returns expected parameters or undefined
     */
    private _getQueryParameters(): StockByProductArgs | undefined {
        try {
            return JSON.parse(String(this.$.queryParameters.stockByProductArgs)) as StockByProductArgs;
        } catch (error) {
            ui.console.error(`Error retrieving query parameters:\n${JSON.stringify(error)}`);
            return undefined;
        }
    }

    /**
     * Return current parameter before chain another page
     * @return return the current parameters
     */
    private _getCurrentParameters(
        rowItem: any,
        _stockObject?: StockObjectSelected,
    ): {
        stockByProductArgs: string;
    } {
        return {
            stockByProductArgs: JSON.stringify(this._getUpdatedParameters(rowItem, _stockObject)),
        };
    }

    private _getUpdatedParameters(rowItem: any, _stockObject?: StockObjectSelected): StockByProductArgs {
        return <StockByProductArgs>{
            ...this._parameters,
            productCode: rowItem.productCode,
            localizedDescription: rowItem.localizedDescription,
            isKeyInSerialNumber: rowItem.isKeyInSerialNumber,
            isLocalizedSerialNumberAllowed: rowItem.isLocalizedSerialNumberAllowed,
            isKeyInLotNumber: rowItem.isKeyInLotNumber,
            consumptionUnit: <ConsumptionUnit>{ code: rowItem.consumptionUnitCode },
            stockUnit: rowItem.stockUnit,
        };
    }

    private async _readAggregatedProducts(
        _parameters: StockByProductArgs,
    ): Promise<AggregatedProductStoreLocation[] | null> {
        const _productStoreLocationList: AggregatedProductStoreLocation[] = [];
        const _filterByProduct = {
            ...(_parameters?.siteDepositorSelected?.siteCode && {
                site: _parameters?.siteDepositorSelected?.siteCode,
            }),
            ...(_parameters?.siteDepositorSelected?.depositorCode && {
                depositor: _parameters.siteDepositorSelected.depositorCode,
            }),
            ...(_parameters?.storeLocationSelected?.storeCode && {
                store: _parameters?.storeLocationSelected?.storeCode,
            }),
            ...(_parameters?.storeLocationSelected?.locationCode && {
                location: _parameters?.storeLocationSelected?.locationCode,
            }),
            product: { isActive: true },
        };

        const _filterBySerialNumber = {
            ...(_parameters?.siteDepositorSelected?.siteCode && {
                site: _parameters?.siteDepositorSelected?.siteCode,
            }),
            ...(_parameters?.siteDepositorSelected?.depositorCode && {
                depositor: _parameters.siteDepositorSelected.depositorCode,
            }),
            ...(_parameters?.storeLocationSelected?.storeCode && {
                stockObject: {
                    store: _parameters?.storeLocationSelected?.storeCode,
                    ...(_parameters?.storeLocationSelected?.locationCode && {
                        location: _parameters?.storeLocationSelected?.locationCode,
                    }),
                },
            }),
        };

        /**
         * The accumulation will be done by product by processing the stock items.
         */

        // Separate aggregate calls in order to exclude blank lots from distinct count
        const _requests = {
            aggregateProduct: generateAggregateStockObjectRequest<StockObject>(
                this,
                {
                    group: {
                        product: {
                            code: {
                                _by: 'value',
                            },
                            localizedDescription: {
                                _by: 'value',
                            },
                            isKeyInLotNumber: {
                                _by: 'value',
                            },
                            isKeyInSerialNumber: {
                                _by: 'value',
                            },
                            isStockUnitManagement: {
                                _by: 'value',
                            },
                            outputMode: {
                                isLocalizedSerialNumberAllowed: {
                                    _by: 'value',
                                },
                            },
                            stockUnit: {
                                code: {
                                    _by: 'value',
                                },
                                numberOfDecimals: {
                                    _by: 'value',
                                },
                            },
                            _id: {
                                _by: 'value',
                            },
                        },
                    },
                    values: {
                        numberOfConsumptionUnit: { sum: true },
                    },
                },
                {
                    filter: _filterByProduct,
                },
            ),
            aggregateLocationLot: generateAggregateStockObjectRequest<StockObject>(
                this,
                {
                    group: {
                        product: {
                            code: {
                                _by: 'value',
                            },
                            isKeyInLotNumber: {
                                _by: 'value',
                            },
                        },
                    },
                    values: {
                        lotNumber: { distinctCount: true },
                    },
                },
                {
                    filter: {
                        ..._filterByProduct,
                        product: { isActive: true, isKeyInLotNumber: true },
                        lotNumber: { _ne: null },
                    },
                },
            ),
            aggregateProductSerialNumber: generateAggregateSerialNumberRequest<SerialNumber>(
                this,
                {
                    group: {
                        product: {
                            code: {
                                _by: 'value',
                            },
                            isKeyInSerialNumber: {
                                _by: 'value',
                            },
                            outputMode: {
                                isLocalizedSerialNumberAllowed: {
                                    _by: 'value',
                                },
                            },
                        },
                    },
                    values: {
                        keyCodeForDuplicates: { distinctCount: true },
                    },
                },
                {
                    filter: {
                        ..._filterBySerialNumber,
                        product: {
                            isActive: true,
                            isKeyInSerialNumber: true,
                            outputMode: {
                                isLocalizedSerialNumberAllowed: true,
                            },
                        },
                        outputDate: { _eq: null },
                    },
                },
            ),
        };

        /**
         * Because result is a location aggregated, is necessary to count records returned after
         * the aggregation to get the number of distinct locations.
         */
        const _response = await new ui.queryUtils.BatchRequest(_requests).execute();

        interface AggregateByProduct {
            values: {
                numberOfConsumptionUnit: {
                    sum: string | number;
                };
            };
            group: {
                product: {
                    code: string;
                    localizedDescription: string;
                    isKeyInLotNumber: boolean;
                    isKeyInSerialNumber: boolean;
                    isStockUnitManagement: boolean;
                    outputMode: {
                        isLocalizedSerialNumberAllowed: boolean;
                    };
                    stockUnit: {
                        code: string;
                        numberOfDecimal: number | null;
                    };
                    _id: string;
                };
            };
        }

        const _aggregateProduct = extractEdges<AggregateByProduct>(_response?.aggregateProduct as Edges<any>);

        if (_aggregateProduct?.length) {
            _aggregateProduct.forEach(aggregation => {
                const _product = aggregation.group.product;
                _productStoreLocationList.push(<AggregatedProductStoreLocation>{
                    _key: _product.code,
                    _id: _product._id,
                    productCode: _product.code,
                    localizedDescription: _product.localizedDescription,
                    numberOfConsumptionUnit: Number(aggregation.values.numberOfConsumptionUnit.sum ?? 0),
                    distinctLot: 0,
                    distinctSerialNumber: 0,
                    isKeyInLotNumber: _product.isKeyInLotNumber,
                    isKeyInSerialNumber: _product.isKeyInSerialNumber,
                    isLocalizedSerialNumberAllowed: _product.outputMode.isLocalizedSerialNumberAllowed,
                    isStockUnitManagement: _product.isStockUnitManagement,
                    stockUnit: _product.isStockUnitManagement ? _product.stockUnit : undefined,
                });
            });
        }

        interface AggregateByLocationLot {
            values: {
                lotNumber: {
                    distinctCount: number;
                };
            };
            group: {
                product: {
                    code: string;
                };
            };
        }

        if (_response?.aggregateLocationLot) {
            const _aggregateProductLocationLot = extractEdges<AggregateByLocationLot>(
                _response.aggregateLocationLot as Edges<any>,
            );
            _aggregateProductLocationLot.forEach(aggregation => {
                const _id = aggregation.group.product.code;
                const _item = _productStoreLocationList.find(item => item._key === _id);
                if (_item) {
                    _item.distinctLot += aggregation.values.lotNumber.distinctCount;
                }
            });
        }

        if (_response?.aggregateProductSerialNumber) {
            interface ResultAggregateProductSerialNumber {
                values: {
                    keyCodeForDuplicates: {
                        distinctCount: number;
                    };
                };
                group: {
                    product: {
                        code: string;
                        isKeyInSerialNumber: boolean;
                        outputMode: {
                            isLocalizedSerialNumberAllowed: boolean;
                        };
                    };
                };
            }

            const _aggregateProductSerialNumber = extractEdges<ResultAggregateProductSerialNumber>(
                _response.aggregateProductSerialNumber as Edges<any>,
            );
            _aggregateProductSerialNumber.forEach(aggregation => {
                const _id = aggregation.group.product.code;
                const _item = _productStoreLocationList.find(item => item._key === _id);
                if (_item) {
                    // _item.
                    _item.distinctSerialNumber += aggregation.values.keyCodeForDuplicates.distinctCount;
                }
            });
        }

        return _productStoreLocationList ?? null;
    }
}
