import { stockControl } from '@sage/wh-master-data/build/lib/menu-items/stock-control';
import { getCurrentSiteDepositor } from '@sage/wh-master-data/lib/client-functions/get-selected-site-depositor';
import { getSelectedSiteDepositorByStorage } from '@sage/wh-master-data/lib/client-functions/storage-properties';
import type { GraphApi } from '@sage/wh-pages-api';
import type { Product } from '@sage/wh-product-data-api';
import { setApplicativePageCrudActions } from '@sage/wh-system/lib/client-functions/applicative-crud-actions';
import { Dict } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import { StockByProductArgs } from '../interfaces/index';

@ui.decorators.page<MobileViewStockByProduct>({
    title: 'View stock by product',
    mode: 'default',
    menuItem: stockControl,
    priority: 100,
    subtitle: 'Select a product',
    isTitleHidden: false,
    node: '@sage/wh-product-data/Product',
    isTransient: false,
    createAction: undefined,
    objectTypeSingular: 'Product',
    objectTypePlural: 'Products',
    authorizationCode: 'INQSTOPRO',
    // idField() {
    //     return this.code;
    // },
    // headerCard() {
    //     return {
    //         title: this.code,
    //     };
    // },
    headerDropDownActions() {
        return [
            // this.$standardSaveAction,
            // this.$standardCancelAction,
            // this.$standardOpenRecordHistoryAction,
        ];
    },
    async onLoad(): Promise<void> {
        if (!(await this._initSiteDepositor())) {
            this.disablePage();
            // Close the page
            this.$.finish();
        }
    },
    async onDirtyStateUpdated(isDirty: boolean) {
        setApplicativePageCrudActions({
            page: this,
            isDirty,
            save: undefined, // this.$standardSaveAction,
            cancel: undefined, // this.$standardCancelAction,
            remove: undefined, // this.$standardDeleteAction,
        });
    },
    businessActions() {
        return [
            // this.$standardSaveAction,
            // this.$standardCancelAction,
            // this.$standardNewAction,
            // this.$standardDeleteAction,
        ];
    },
    navigationPanel: {
        canFilter: false,
        isHeaderHidden: true,
        isAutoSelectEnabled: true,
        listItem: {
            title: ui.nestedFields.text({
                title: 'Product',
                bind: 'code',
                canFilter: true,
            }),
            titleRight: ui.nestedFields.checkbox({
                title: 'Lot',
                bind: 'isKeyInLotNumber',
                canFilter: false,
                isHidden: true,
            }),
            line2: ui.nestedFields.text({
                title: 'Description',
                bind: 'localizedDescription',
                canFilter: true,
            }),
            line2Right: ui.nestedFields.checkbox({
                title: 'Serial number',
                bind: 'isKeyInSerialNumber',
                canFilter: false,
                isHidden: true,
            }),
            line3: ui.nestedFields.reference({
                title: 'Stock unit',
                node: '@sage/wh-master-data/UnitOfMeasure',
                bind: 'stockUnit',
                valueField: 'code',
                canFilter: false,
                isHidden(value: any, rowValue?: Dict<Product>) {
                    return !rowValue?.isStockUnitManagement;
                },
                columns: [
                    ui.nestedFields.numeric({
                        bind: 'numberOfDecimals',
                        isHidden: true,
                    }),
                ],
            }),
            line3Right: ui.nestedFields.checkbox({
                title: 'Hazardous materials',
                bind: 'isHazardousMaterials',
                canFilter: false,
                isHidden: true,
            }),
            line4: ui.nestedFields.checkbox({
                title: 'Stock unit management',
                bind: 'isStockUnitManagement',
                canFilter: false,
                isHidden: true,
            }),
            line4RightTechnical: ui.nestedFields.technical({
                node: '@sage/wh-product-data/OutputMode',
                bind: 'outputMode',
                nestedFields: [
                    ui.nestedFields.checkbox({
                        bind: 'code',
                    }),
                    ui.nestedFields.checkbox({
                        bind: 'isSerialNumberInOutputAllowed',
                    }),
                    ui.nestedFields.checkbox({
                        bind: 'isLocalizedSerialNumberAllowed',
                    }),
                ],
            }),
            line5: ui.nestedFields.numeric({
                bind: '_id',
                canFilter: false,
                isHidden: true,
            }),
        },
        orderBy: {
            code: 1,
        },
        async optionsMenu(_graph, storage, _queryParam, _username, _userCode) {
            const _selectedSiteDepositor = getSelectedSiteDepositorByStorage(storage);
            return [
                {
                    title: '',
                    graphQLFilter: {
                        site: {
                            code: _selectedSiteDepositor?.site,
                        },
                        depositor: {
                            code: _selectedSiteDepositor?.depositor,
                        },
                        isActive: true,
                    },
                },
            ];
        },
        onSelect(product: any) {
            if (product?.isActive && this.siteCodeSelected && this.depositorCodeSelected) {
                // eslint-disable-next-line consistent-return
                this._countStockObjects(product).then(count => {
                    if (count > 0) {
                        this.$.router.goTo(
                            '@sage/wh-pages/MobileViewStockByProductSelectStoreLocation',
                            this._getCurrentParameters(product),
                        );
                        return true;
                    }
                    this.$.showToast(
                        ui.localize(
                            '@sage/wh-pages/selected-product-no-results',
                            'Product {{ code }} has no stock records.',
                            {
                                code: product?.code ?? '',
                            },
                        ),
                        { type: 'info' },
                    );
                });
            }
            return true;
        },
    },
})
export class MobileViewStockByProduct extends ui.Page<GraphApi> {
    /** internal properties */

    private siteCodeSelected: string | undefined;

    private depositorCodeSelected: string | undefined;

    /*
     *
     *  Sections
     *
     */

    @ui.decorators.section<MobileViewStockByProduct>({
        isTitleHidden: true,
        isHidden: true,
    })
    mainSection!: ui.containers.Section;

    /*
     *
     *  Blocks
     *
     */

    @ui.decorators.block<MobileViewStockByProduct>({
        parent() {
            return this.mainSection;
        },
    })
    productBlock!: ui.containers.Block;
    /*
     *
     *  Page properties
     *
     */
    // -----------------------------

    /*
     *
     *  Fields
     *
     */

    // -----------------------------------------------
    // Block: Product block
    // -----------------------------------------------

    @ui.decorators.checkboxField<MobileViewStockByProduct>({
        parent() {
            return this.productBlock;
        },
        isTransient: false,
        isMandatory: false,
        isDisabled: true,
        isHidden: true,
    })
    isStockUnitManagement!: ui.fields.Checkbox;

    @ui.decorators.referenceField<MobileViewStockByProduct>({
        parent() {
            return this.productBlock;
        },
        title: 'Stock unit',
        isTransient: false,
        isMandatory: false,
        isDisabled: true,
        isHidden(value) {
            return !value || !this.isStockUnitManagement.value;
        },
    })
    stockUnit!: ui.fields.Reference;

    @ui.decorators.checkboxField<MobileViewStockByProduct>({
        parent() {
            return this.productBlock;
        },
        title: 'Lot',
        isTransient: false,
        isMandatory: false,
        isDisabled: true,
        isHidden: true,
    })
    isKeyInLotNumber!: ui.fields.Checkbox;

    @ui.decorators.checkboxField<MobileViewStockByProduct>({
        parent() {
            return this.productBlock;
        },
        title: 'Serial number',
        isTransient: false,
        isMandatory: false,
        isDisabled: true,
        isHidden: true,
    })
    isKeyInSerialNumber!: ui.fields.Checkbox;

    @ui.decorators.checkboxField<MobileViewStockByProduct>({
        parent() {
            return this.productBlock;
        },
        title: 'Hazardous materials',
        isTransient: false,
        isMandatory: false,
        isDisabled: true,
        isHidden: true,
    })
    isHazardousMaterials!: ui.fields.Checkbox;

    @ui.decorators.referenceField<MobileViewStockByProduct>({
        parent() {
            return this.productBlock;
        },
        title: 'Site',
        isReadOnly: true,
        isHidden: true,
    })
    site!: ui.fields.Reference;

    @ui.decorators.referenceField<MobileViewStockByProduct>({
        parent() {
            return this.productBlock;
        },
        title: 'Depositor',
        isReadOnly: true,
        isHidden: true,
    })
    depositor!: ui.fields.Reference;

    private async _initSiteDepositor(): Promise<boolean> {
        const siteDepositor = await getCurrentSiteDepositor(
            this,
            ui.localize('@sage/wh-pages/dialog-error-title', 'Error'),
            ui.localize(
                '@sage/wh-pages/dialog-error-location-inquiry-set-site-depositor',
                'Define a default site and depositor on the user function profile.',
            ),
        );
        if (siteDepositor && siteDepositor?.site && siteDepositor?.depositor) {
            this.siteCodeSelected = siteDepositor?.site;
            this.depositorCodeSelected = siteDepositor?.depositor;
            return true;
        }
        return false;
    }

    private async disablePage(): Promise<void> {
        this.mainSection.isDisabled = true;
        this.productBlock.isDisabled = true;
    }

    /**
     *  Count stock objects for a product
     * @param storeLocation
     * @returns
     */
    private async _countStockObjects(product: any): Promise<number> {
        try {
            return Number(
                (
                    await this.$.graph
                        .node('@sage/wh-stock-data/StockObject')
                        .aggregate.read(
                            {
                                _id: {
                                    distinctCount: true,
                                },
                            },
                            {
                                filter: {
                                    site: {
                                        code: this.siteCodeSelected,
                                    },
                                    depositor: {
                                        code: this.depositorCodeSelected,
                                    },
                                    product: { isActive: true, code: product.code },
                                },
                            },
                        )
                        .execute()
                )?._id?.distinctCount ?? 0,
            );
        } catch (error) {
            ui.console.error(`Error counting stock objects:\n${JSON.stringify(error)}`);
        }

        return 0;
    }

    /**
     * get query parameters
     * @returns expected parameters or undefined
     */
    private _getQueryParameters(): StockByProductArgs | undefined {
        try {
            return JSON.parse(String(this.$.queryParameters.stockByProductArgs)) as StockByProductArgs;
        } catch (error) {
            ui.console.error(`Error retrieving query parameters:\n${JSON.stringify(error)}`);
            return undefined;
        }
    }

    /**
     * Return current parameter before chain another page
     * @return return the current parameters
     */
    private _getCurrentParameters(product: any): {
        stockByProductArgs: string;
    } {
        return {
            stockByProductArgs: JSON.stringify(<StockByProductArgs>{
                siteDepositorSelected: {
                    siteCode: this.siteCodeSelected,
                    depositorCode: this.depositorCodeSelected,
                },
                _id: product._id,
                productCode: product.code,
                localizedDescription: product.localizedDescription,
                isKeyInLotNumber: product.isKeyInLotNumber,
                isKeyInSerialNumber: product.isKeyInSerialNumber,
                isSerialNumberInOutputAllowed: product.outputMode?.isSerialNumberInOutputAllowed,
                isLocalizedSerialNumberAllowed: product.outputMode?.isLocalizedSerialNumberAllowed,
                isHazardousMaterials: product.isHazardousMaterials,
                isStockUnitManagement: product.isStockUnitManagement,
                ...(product.isStockUnitManagement && {
                    stockUnit: {
                        code: product.stockUnit?.code,
                        numberOfDecimals: product.stockUnit?.numberOfDecimals,
                    },
                }),
            }),
        };
    }
}
