import { hightestContainerLevel } from '@sage/wh-master-data/lib/interfaces';
import type { GraphApi } from '@sage/wh-pages-api';
import type { BlockingReason, UnitLength } from '@sage/wh-product-data-api';
import type { StockObject } from '@sage/wh-stock-data-api';
import { StockObjectSelected } from '@sage/wh-stock-data/lib/interfaces';
import { Dict } from '@sage/xtrem-client';
import { DateValue } from '@sage/xtrem-date-time';
import * as ui from '@sage/xtrem-ui';
import { LocalizedSerialNumbersArgs, StockByProductArgs, StockByProductArgsHeader } from '../interfaces';
import { StockObjectEntry } from './mobile-view-stock-by-product-select-stock-object';

const _hideWhenEmptyValue = (value: any, _rowValue?: Dict<StockObject>) => {
    return typeof value !== 'number' && !value;
};

interface StockObjectDetails
    extends Omit<
        StockObjectEntry,
        'isControlOutputFlow' | 'isBlockedInOutput' | 'isAuthorizedInOutput' | 'isBlockedInInventory' | 'stockCountCode'
    > {
    localizedSerialNumberLink?: string;
    blockingReasonByInventory?: string;
    blockingReasonByOutput?: string;
    blockingReasonByForbiddenInOutput?: string;
    blockingReasonByReservation?: string;
    blockingReasonByStockNature?: string;
    blockingReasonByDatesExpired?: string;
    numberOfDecimals?: number;
}

type BlockingReasons = { [key in BlockingReason]: string };

// TODO: use literals translation of enum blockingReason
const _blockingReasons = <BlockingReasons>{
    inventoryPending: ui.localize('@sage/wh-pages/stock-blocking-reason_inventory-pending', 'Inventory pending'),
    blockedInOutput: ui.localize('@sage/wh-pages/stock-blocking-reason_blocked-in-output', 'Blocked in output'),
    forbiddenInOutput: ui.localize('@sage/wh-pages/stock-blocking-reason_forbidden-in-output', 'Forbidden in output'),
    reserved: ui.localize('@sage/wh-pages/stock-blocking-reason_reserved', 'Reserved'),
    stockNature: ui.localize('@sage/wh-pages/stock-blocking-reason_stock-nature', 'Stock nature'),
    datesExpired: ui.localize('@sage/wh-pages/stock-blocking-reason_dates-expired', 'Dates expired'),
};

@ui.decorators.page<MobileViewStockByProductStockObjectDetails>({
    isTitleHidden: true,
    title: 'Stock by product',
    subtitle: 'Product details',
    isTransient: true,
    skipDirtyCheck: true,
    authorizationCode: 'INQSTOPRO',
    headerCard() {
        return {
            title: this.productCode,
            titleRight: this.numberOfConsumptionUnit,
            line2: this.stockObjectCode,
            line2Right: this.distinctSerialNumber,
            line3: this.numberOfDecimals,
            // image: this.image,
        };
    },
    async onLoad(): Promise<void> {
        // Requires a selected product in the query parameters.  This should not occur unless user manually
        // directs themselves to this page
        const _parameters = this._getQueryParameters();
        this._setTitle(_parameters);

        if (!_parameters || !_parameters.productCode || !_parameters?._id) {
            this.$.showToast(ui.localize('@sage/wh-pages/product-required', 'Selected product is required'), {
                type: 'warning',
            });

            this.$.router.goTo(
                _parameters && !_parameters?.isViewByStoreLocation
                    ? '@sage/wh-pages/MobileViewStockByProductSelectStockObject'
                    : '@sage/wh-pages/MobileViewStockByStoreLocationSelectProduct',
                this._getCurrentParameters(_parameters),
            );
            return;
        }

        const _selectedStockObject = _parameters?.stockObjectSelected?.details as StockObjectEntry | undefined;

        if (
            !_parameters.stockObjectSelected?._id ||
            !_parameters.stockObjectSelected?.code ||
            !_selectedStockObject ||
            !_selectedStockObject?.code
        ) {
            this.$.showToast(ui.localize('@sage/wh-pages/stock-object-required', 'Selected stock object is required'), {
                type: 'warning',
            });

            this.$.router.goTo(
                '@sage/wh-pages/MobileViewStockByProductSelectStockObject',
                this._getCurrentParameters(_parameters),
            );
            return;
        }

        this._parameters = _parameters;

        // Create details header
        this.distinctSerialNumber.isHidden = !_parameters.isLocalizedSerialNumberAllowed;
        this.productCode.value = _parameters.productCode;
        this.numberOfConsumptionUnit.value = Number(_selectedStockObject?.numberOfConsumptionUnit ?? 0);
        this.stockObjectCode.value = _parameters.stockObjectSelected.code;
        this.distinctSerialNumber.value = Number(_selectedStockObject?.distinctSerialNumber ?? 0);

        // Display product picture
        try {
            const _productPicture = await this.$.graph
                .node('@sage/wh-product-data/Product')
                .read(
                    { picture: { value: true } },
                    `${_parameters?.siteDepositorSelected.siteCode}|${_parameters?.siteDepositorSelected.depositorCode}|${_parameters?.productCode}`,
                )
                .execute();

            this.image.value = _productPicture?.picture ?? null;
        } catch (error) {
            this.image.value = null;
            ui.console.error(`Error reading picture :\n${JSON.stringify(error)}`);
        }

        this.$.setPageClean();

        this._initializeStockObjectDetail(_parameters);

        // Finalize details header
        this.numberOfDecimals.value = Number(this.stockObjectDetails.value[0]?.numberOfDecimals ?? 0);
    },
})
export class MobileViewStockByProductStockObjectDetails extends ui.Page<GraphApi> {
    /**
     * Internal properties
     */

    private _parameters: StockByProductArgs | undefined;

    /*
     *
     *  Sections
     *
     */

    @ui.decorators.section<MobileViewStockByProductStockObjectDetails>({
        isTitleHidden: true,
    })
    mainSection!: ui.containers.Section;

    /*
     *
     *  Blocks
     *
     */

    @ui.decorators.block<MobileViewStockByProductStockObjectDetails>({
        parent() {
            return this.mainSection;
        },
    })
    mainBlock!: ui.containers.Block;

    /*
     *
     *  Technical fields
     *
     */

    @ui.decorators.textField<MobileViewStockByProductStockObjectDetails>({
        isTransient: true,
        isReadOnly: true,
    })
    productCode!: ui.fields.Text;

    @ui.decorators.numericField<MobileViewStockByProductStockObjectDetails>({
        isTransient: true,
        isReadOnly: true,
        postfix(_value, __argsrowValue) {
            return this._parameters?.consumptionUnit?.code ?? '';
        },
        scale(_value, rowValue?: any) {
            return Number(rowValue?.numberOfDecimals ?? 0);
        },
    })
    numberOfConsumptionUnit!: ui.fields.Numeric;

    @ui.decorators.numericField<MobileViewStockByProductStockObjectDetails>({
        isTransient: true,
        isReadOnly: true,
        isHidden: true,
    })
    numberOfDecimals!: ui.fields.Numeric;

    @ui.decorators.textField<MobileViewStockByProductStockObjectDetails>({
        isTransient: true,
        isReadOnly: true,
        prefix: ui.localize('@sage/wh-pages/prefix-stock-object', 'SO:'),
    })
    stockObjectCode!: ui.fields.Text;

    @ui.decorators.numericField<MobileViewStockByProductStockObjectDetails>({
        isTransient: true,
        isReadOnly: true,
        prefix: ui.localize('@sage/wh-pages/serial-numbers-prefix', 'Serial no.:'),
    })
    distinctSerialNumber!: ui.fields.Numeric;

    /*
     *  Details fields
     */

    @ui.decorators.imageField<MobileViewStockByProductStockObjectDetails>({
        parent() {
            return this.mainBlock;
        },
        isTransient: true,
        isReadOnly: true,
        width: 'medium',
        height: 'medium',
    })
    image!: ui.fields.Image;

    @ui.decorators.detailListField<MobileViewStockByProductStockObjectDetails>({
        parent() {
            return this.mainBlock;
        },
        isTransient: true,
        fields: [
            // Blank header when no serial numbers
            ui.nestedFields.text({
                bind: '_spacerColumn',
                isReadOnly: true,
                isTransient: false,
                isHidden(_value: any, _rowValue?: Dict<StockObject>) {
                    return this._parameters?.isLocalizedSerialNumberAllowed ?? false;
                },
            }),
            ui.nestedFields.link({
                bind: 'localizedSerialNumberLink',
                title: 'Serial no.',
                isHidden(_value: any, _rowValue?: Dict<StockObject>) {
                    return !this._parameters?.isLocalizedSerialNumberAllowed;
                },
                isTransient: true,
                async onClick(_id, rowData: any) {
                    if (this._parameters) {
                        this.$.setPageClean();
                        this.$.router.goTo(
                            '@sage/wh-pages/MobileViewLocalizedSerialDetails',
                            this._getSerialsParameters(_id, rowData, this._parameters),
                        );
                    }
                },
            }),
            ui.nestedFields.text({
                bind: 'blockingReasonByInventory',
                title: 'Blocking reason',
                isReadOnly: true,
                isTransient: true,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.text({
                bind: 'blockingReasonByOutput',
                title: 'Blocking reason',
                isReadOnly: true,
                isTransient: true,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.text({
                bind: 'blockingReasonByForbiddenInOutput',
                title: 'Blocking reason',
                isReadOnly: true,
                isTransient: true,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.text({
                bind: 'blockingReasonByReservation',
                title: 'Blocking reason',
                isReadOnly: true,
                isTransient: true,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.text({
                bind: 'blockingReasonByStockNature',
                title: 'Blocking reason',
                isReadOnly: true,
                isTransient: true,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.text({
                bind: 'blockingReasonByDatesExpired',
                title: 'Blocking reason',
                isReadOnly: true,
                isTransient: true,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.numeric({
                bind: 'numberOfContainer',
                title: 'Container qty',
                isReadOnly: true,
                isTransient: true,
                postfix(value, rowValue) {
                    return rowValue?.containerCode ?? '';
                },
            }),
            ui.nestedFields.numeric({
                bind: 'homogeneousQuantity',
                title: 'Homogeneous qty',
                isReadOnly: true,
                isTransient: true,
                postfix(value, rowValue) {
                    return rowValue?.homogeneousContainerCode ?? '';
                },
                scale(_value, rowValue?: any) {
                    return Number(rowValue?.numberOfDecimals ?? 0);
                },
            }),
            ui.nestedFields.numeric({
                bind: 'numberOfConsumptionUnit',
                title: 'Qty of CU',
                isReadOnly: true,
                isTransient: true,
                postfix(value, rowValue) {
                    return rowValue?.consumptionUnitCode ?? '';
                },
                scale(_value, rowValue?: any) {
                    return Number(rowValue?.numberOfDecimals ?? 0);
                },
            }),
            ui.nestedFields.numeric({
                bind: 'numberOfConsumptionUnitInPendingInput',
                title: 'Pending input qty',
                isReadOnly: true,
                isTransient: true,
                isHidden: _hideWhenEmptyValue,
                postfix(value, rowValue) {
                    return rowValue?.consumptionUnitCode ?? '';
                },
                scale(_value, rowValue?: any) {
                    return Number(rowValue?.numberOfDecimals ?? 0);
                },
            }),
            ui.nestedFields.numeric({
                bind: 'numberOfConsumptionUnitInPendingOutput',
                title: 'Pending output qty',
                isReadOnly: true,
                isTransient: true,
                isHidden: _hideWhenEmptyValue,
                postfix(value, rowValue) {
                    return rowValue?.consumptionUnitCode ?? '';
                },
                scale(_value, rowValue?: any) {
                    return Number(rowValue?.numberOfDecimals ?? 0);
                },
            }),
            ui.nestedFields.text({
                bind: 'stockStatusCode',
                title: 'Stock nature',
                isReadOnly: true,
                isTransient: true,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.text({
                bind: 'lotNumber',
                title: 'Lot',
                isReadOnly: true,
                isTransient: false,
                isHidden(value: any, _rowValue?: Dict<StockObject>) {
                    return !this._parameters?.isKeyInLotNumber || _hideWhenEmptyValue(value, _rowValue);
                },
            }),
            ui.nestedFields.text({
                bind: 'supportNumber',
                title: 'Support',
                isReadOnly: true,
                isTransient: false,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.text({
                bind: 'reservationNumber',
                title: 'Reservation',
                isReadOnly: true,
                isTransient: false,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.date({
                bind: 'fifoDate',
                title: 'FIFO date',
                isReadOnly: true,
                isTransient: false,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.date({
                bind: 'receiptDate',
                title: 'Input date',
                isReadOnly: true,
                isTransient: false,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.date({
                bind: 'manufacturedDate',
                title: 'Manufacturing date',
                isReadOnly: true,
                isTransient: false,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.date({
                bind: 'detentionDate',
                title: 'Detention date',
                isReadOnly: true,
                isTransient: false,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.date({
                bind: 'sellByDate',
                title: 'Sell by date',
                isReadOnly: true,
                isTransient: false,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.date({
                bind: 'useByDate',
                title: 'Used by date',
                isReadOnly: true,
                isTransient: false,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.date({
                bind: 'shipByDate',
                title: 'Send by date',
                isReadOnly: true,
                isTransient: false,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.numeric({
                bind: '_id',
                isHidden: true,
                isTransient: true,
            }),
            ui.nestedFields.text({
                bind: 'stockStatus',
                isReadOnly: true,
                isTransient: true,
                isHidden: true,
            }),
        ],
    })
    stockObjectDetails!: ui.fields.DetailList;

    /**
     *  Move extracted data to stockObject transient table
     *  @param _stockObjects
     *  @param _aggregatedByStockObjects
     *  @param _parameters
     *  @returns true when stockObject table is not empty
     */
    private _initializeStockObjectDetail(_parameters: StockByProductArgs): boolean {
        const _stockObjectDetails: StockObjectDetails[] = [];
        const _stockObjectEntry = _parameters?.stockObjectSelected?.details as StockObjectEntry;

        if (_stockObjectEntry) {
            /*
                The number of decimal places is only determined if the container or
                homogeneous container is the highest level of container and there is
                an active SKU for the product.
            */
            const _numberOfDecimals =
                _parameters?.stockUnit?.code &&
                (_stockObjectEntry.containerLevel === hightestContainerLevel ||
                    _stockObjectEntry.homogeneousContainerLevel === hightestContainerLevel)
                    ? Number(_parameters.stockUnit?.numberOfDecimals ?? 0)
                    : 0;

            _stockObjectDetails.push(<StockObjectDetails>{
                ..._stockObjectEntry,
                numberOfDecimals: _numberOfDecimals,
                localizedSerialNumberLink: _parameters.isLocalizedSerialNumberAllowed
                    ? ui.localize('@sage/wh-pages/label-view-all', 'View list')
                    : undefined,
            });

            this._updateBlockingReasons(_stockObjectEntry, _stockObjectDetails[0]);
        }

        this.stockObjectDetails.value = _stockObjectDetails;

        return _stockObjectDetails.length > 0;
    }

    /**
     * Update all active blockings reasons messages
     * @param _stockObject
     * @returns
     */
    // eslint-disable-next-line class-methods-use-this
    private _updateBlockingReasons(_stockObject: StockObjectEntry, _stockObjectDetails: StockObjectDetails): void {
        const _today = DateValue.today();
        const _checkExpiredDate = (detentionUnit: string | undefined, detentionDate: string | undefined): boolean =>
            !!detentionUnit &&
            <UnitLength>detentionUnit !== 'no' &&
            !!detentionDate &&
            DateValue.parse(detentionDate) < _today;

        // The order of messages is the same as in the classic part.

        _stockObjectDetails.blockingReasonByInventory = _stockObject?.isBlockedInInventory
            ? _blockingReasons.inventoryPending +
              (_stockObject?.stockCountCode ? ` (${_stockObject?.stockCountCode})` : '')
            : undefined;

        _stockObjectDetails.blockingReasonByOutput =
            _stockObject?.isControlOutputFlow || _stockObject?.isBlockedInOutput
                ? _blockingReasons.blockedInOutput
                : undefined;

        _stockObjectDetails.blockingReasonByForbiddenInOutput = !_stockObject?.isAuthorizedInOutput
            ? _blockingReasons.forbiddenInOutput
            : undefined;

        _stockObjectDetails.blockingReasonByReservation = _stockObject?.reservationNumber
            ? _blockingReasons.reserved
            : undefined;

        _stockObjectDetails.blockingReasonByStockNature = _stockObject?.stockStatusCode
            ? _blockingReasons.stockNature
            : undefined;

        _stockObjectDetails.blockingReasonByDatesExpired =
            _checkExpiredDate(_stockObject?.detentionUnit, _stockObject?.detentionDate) ||
            _checkExpiredDate(_stockObject?.sellByDateUnit, _stockObject?.sellByDate) ||
            _checkExpiredDate(_stockObject?.useByDateUnit, _stockObject?.useByDate) ||
            _checkExpiredDate(_stockObject?.sendByDateUnit, _stockObject?.shipByDate)
                ? _blockingReasons.datesExpired
                : undefined;
    }

    /**
     * get query parameters
     * @returns expected parameters or undefined
     */
    private _getQueryParameters(): StockByProductArgs | undefined {
        try {
            return JSON.parse(String(this.$.queryParameters.stockByProductArgs)) as StockByProductArgs;
        } catch (_error) {
            return undefined;
        }
    }

    /**
     * Return current parameter before chain another page
     * @return return the current parameters
     */
    // eslint-disable-next-line class-methods-use-this
    private _getCurrentParameters(_parameters?: StockByProductArgs): { stockByProductArgs: string } | undefined {
        return _parameters
            ? <StockByProductArgsHeader>{
                  stockByProductArgs: JSON.stringify(<StockByProductArgs>{
                      ..._parameters,
                      stockObjectSelected: undefined,
                  }),
              }
            : undefined;
    }

    /**
     * get query parameters for localized serial numbers
     * @param _id current OS id
     * @param rowData current row (have same OS id)
     *  @param _parameters global parameters
     * @returns expected parameters or undefined
     */
    private _getSerialsParameters(
        _id: string,
        rowData: any,
        _parameters: StockByProductArgs,
    ): { localizedSerialNumbersArgs: string } {
        return {
            localizedSerialNumbersArgs: JSON.stringify(<LocalizedSerialNumbersArgs>{
                isViewByStoreLocation: _parameters?.isViewByStoreLocation,
                productCode: _parameters?.productCode ?? this.productCode.value ?? undefined,
                localizedDescription: _parameters?.localizedDescription,
                depositorSiteSelected: _parameters?.siteDepositorSelected,
                stockObjectSelected: <StockObjectSelected>{
                    _id: rowData._id,
                    code: rowData.code,
                },
            }),
        };
    }

    /**
     * Change title page
     * @param _parameters
     */
    private _setTitle(_parameters?: StockByProductArgs): void {
        this.$.page.title = _parameters?.isViewByStoreLocation
            ? ui.localize('@sage/wh-pages/title__stock-by-store-location', 'Stock by location')
            : ui.localize('@sage/wh-pages/title__stock-by-product', 'Stock by product');
    }
}
