import type { GraphApi } from '@sage/wh-pages-api';
import { getConsumptionUnit } from '@sage/wh-product-data/lib/client-functions/get-consumption-unit';
import type { SerialNumber, StockObject } from '@sage/wh-stock-data-api';
import {
    generateAggregateSerialNumberRequest,
    generateAggregateStockObjectRequest,
} from '@sage/wh-stock-data/lib/client-functions/aggregators';
import { StockObjectSelected } from '@sage/wh-stock-data/lib/interfaces';
import { AggregateGroupSelector, Edges, extractEdges } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import { StockByProductArgs } from '../interfaces/arguments';

interface AggregatedProduct {
    distinctLocations?: number;
    distinctLots?: number;
    distinctSerialNumbers?: number;
}
export interface AggregatedStoreLocation {
    _key?: string;
    storeCode: string;
    locationCode: string;
    locationType: string;
    numberOfConsumptionUnit: number;
    distinctStockObject: number;
    distinctLot: number;
    distinctSerialNumber: number;
}
export interface StoreLocationValue extends Omit<AggregatedStoreLocation, '_key' | 'distinctStockObject'> {
    _id: string;
    storeLocationCode: string;
    distinctStockObject?: number;
}

@ui.decorators.page<MobileViewStockByProductSelectStoreLocation>({
    node: '@sage/wh-stock-data/StockObject',
    mode: 'default',
    isTitleHidden: true,
    title: 'Stock by product',
    subtitle: 'Select a location',
    skipDirtyCheck: true,
    objectTypeSingular: 'Stock object',
    objectTypePlural: 'Stock objects',
    authorizationCode: 'INQSTOPRO',
    headerCard() {
        return {
            title: this.product,
            titleRight: this.distinctLocations,
            line2: this.localizedDescription,
            line2Right: this.distinctLots,
            line3: this.distinctSerialNumbers,
        };
    },
    navigationPanel: undefined,
    // navigationPanel: {
    //     menuType: 'toggle',
    //     canFilter: true,
    //     isFirstLetterSeparatorHidden: true,
    //     isHeaderHidden: false,
    //     listItem: {
    //         title: ui.nestedFields.reference({
    //             title: 'Product',
    //             node: '@sage/wh-product-data/Product',
    //             bind: 'product',
    //             valueField: 'code',
    //             canFilter: true,
    //         }),
    //         titleRight: ui.nestedFields.reference({
    //             title: 'Description',
    //             node: '@sage/wh-product-data/Product',
    //             bind: 'product',
    //             valueField: 'localizedDescription',
    //             canFilter: true,
    //         }),
    //         line2: ui.nestedFields.reference({
    //             title: 'Store',
    //             node: '@sage/wh-master-data/Store',
    //             bind: 'store',
    //             valueField: 'code',
    //             canFilter: false,
    //         }),
    //         line2Right: ui.nestedFields.reference({
    //             title: 'Location',
    //             node: '@sage/wh-master-data/Location',
    //             bind: 'location',
    //             valueField: 'code',
    //             canFilter: false,
    //         }),
    //         line3: ui.nestedFields.reference({
    //             title: 'Container',
    //             node: '@sage/wh-product-data/Container',
    //             bind: 'container',
    //             valueField: { container: { code: true } },
    //             canFilter: false,
    //         }),
    //         line3Right: ui.nestedFields.text({
    //             title: 'Stock object',
    //             bind: 'code',
    //             canFilter: false,
    //         }),
    //     },
    //     orderBy: {
    //         code: 1,
    //     },
    //     async optionsMenu(_graph, _storage, _queryParam, _username, userCode) {
    //         const _productParameters = JSON.parse(String(_queryParam['stockByProductArgs'])) as StockByProductArgs;
    //         return [
    //             {
    //                 title: '',
    //                 graphQLFilter: {
    //                     site: {
    //                         code: _productParameters?.depositorSiteSelected?.siteCodeSelected ?? '',
    //                     },
    //                     depositor: {
    //                         code: _productParameters?.depositorSiteSelected?.depositorCodeSelected ?? '',
    //                     },
    //                     product: {
    //                         code: _productParameters?.productCode ?? '',
    //                         isActive: true,
    //                     },
    //                 },
    //             },
    //         ];
    //     },
    // },
    async onLoad(): Promise<void> {
        // Requires a selected product in the query parameters.  This should not occur unless user manually
        // directs themselves to this page
        const _parameters = this._getQueryParameters();
        if (!_parameters) {
            this.$.showToast(ui.localize('@sage/wh-pages/product-required', 'Selected product is required'), {
                type: 'warning',
            });
            this.$.router.goTo('@sage/wh-pages/MobileViewStockByProduct');
            return;
        }

        /**
         * Read the product container with the highest level
         * and override default consumption unit (not currently assigned)
         */

        _parameters.consumptionUnit = (await getConsumptionUnit(this, _parameters?.productCode)) ?? undefined;

        _parameters.isLocalizedSerialNumberAllowed =
            (_parameters?.isKeyInSerialNumber && _parameters?.isLocalizedSerialNumberAllowed) ?? false;

        this._parameters = _parameters;

        this.product.value = _parameters.productCode;
        this.localizedDescription.value = _parameters.localizedDescription ?? null;

        /**
         *  Read aggregated product data and populate the header with the results
         */
        const _aggregatedProduct = await this._readAggregatedProduct(_parameters);

        this.distinctLocations.value = _aggregatedProduct?.distinctLocations ?? 0;
        this.distinctLots.value = _aggregatedProduct?.distinctLots ?? 0;
        this.distinctSerialNumbers.value = _aggregatedProduct?.distinctSerialNumbers ?? 0;

        /**
         * Read aggregated store and location data and populate the table with the results
         */
        const _aggregatedStoreAndLocation = await this._readAggregatedStoreAndLocation(_parameters);

        this.mainBlock.title = ui.localize(
            '@sage/wh-pages/number-of-locations',
            '{{ distinctLocations }} location(s)',
            {
                distinctLocations: _aggregatedProduct?.distinctLocations ?? 0,
            },
        );

        // Redirect user to first page if selected product yields no records.  User should not get to this page
        // if no stock records unless manually entering page.
        if (!_aggregatedStoreAndLocation?.length) {
            this.$.showToast(
                ui.localize('@sage/wh-pages/selected-product-no-results', 'Product {{ code }} has no stock records.', {
                    code: _parameters.productCode,
                }),
                { type: 'info' },
            );
            this.$.router.goTo('@sage/wh-pages/MobileViewStockByProduct');
            return;
        }

        const _storeLocationValues: StoreLocationValue[] = [];
        const _getKeyStoreLocation = (storeCode: string, locationCode: string): string =>
            `${storeCode} ${locationCode}`;

        // initialize data structure to consolidate and store aggregated values to the corresponding location
        const storeLocationValues = _aggregatedStoreAndLocation.map(storeLocation => {
            const _storeLocationCode = _getKeyStoreLocation(storeLocation.storeCode, storeLocation.locationCode);
            return <StoreLocationValue>{
                _id: _storeLocationCode, // FIXME: _id is not used in the table
                storeCode: storeLocation.storeCode,
                locationCode: storeLocation.locationCode,
                storeLocationCode: _storeLocationCode,
                locationType: storeLocation.locationType,
                containerCode: _parameters.consumptionUnit?.code ?? '',
                numberOfConsumptionUnit: storeLocation.numberOfConsumptionUnit,
                distinctSerialNumber: storeLocation?.distinctSerialNumber ?? 0,
                distinctLot: storeLocation?.distinctLot ?? 0,
            };
        });

        this.storesLocations.value = storeLocationValues;
    },
})
export class MobileViewStockByProductSelectStoreLocation extends ui.Page<GraphApi> {
    /**
     * Internal properties
     */

    private _parameters: StockByProductArgs | undefined;

    @ui.decorators.section<MobileViewStockByProductSelectStoreLocation>({
        isTitleHidden: true,
    })
    mainSection!: ui.containers.Section;

    @ui.decorators.block<MobileViewStockByProductSelectStoreLocation>({
        parent() {
            return this.mainSection;
        },
        isTitleHidden: false,
    })
    mainBlock!: ui.containers.Block;

    @ui.decorators.textField<MobileViewStockByProductSelectStoreLocation>({
        isTransient: true,
        isReadOnly: true,
    })
    product!: ui.fields.Text;

    @ui.decorators.numericField<MobileViewStockByProductSelectStoreLocation>({
        isTransient: true,
        isReadOnly: true,
        prefix: ui.localize('@sage/wh-pages/prefix-locations', 'Locations:'),
    })
    distinctLocations!: ui.fields.Numeric;

    @ui.decorators.textField<MobileViewStockByProductSelectStoreLocation>({
        isTransient: true,
        isReadOnly: true,
    })
    localizedDescription!: ui.fields.Text;

    @ui.decorators.numericField<MobileViewStockByProductSelectStoreLocation>({
        isTransient: true,
        isReadOnly: true,
        prefix: ui.localize('@sage/wh-pages/lots-numbers-prefix', 'Lots:'),
        isHidden(_value, _rowValue) {
            return !this._parameters?.isKeyInLotNumber;
        },
    })
    distinctLots!: ui.fields.Numeric;

    @ui.decorators.numericField<MobileViewStockByProductSelectStoreLocation>({
        isTransient: true,
        isReadOnly: true,
        prefix: ui.localize('@sage/wh-pages/serial-numbers-prefix', 'Serial no.:'),
        isHidden() {
            return !this._parameters?.isKeyInSerialNumber || !this._parameters?.isLocalizedSerialNumberAllowed;
        },
    })
    distinctSerialNumbers!: ui.fields.Numeric;

    @ui.decorators.tableField<MobileViewStockByProductSelectStoreLocation>({
        parent() {
            return this.mainBlock;
        },
        canSelect: false,
        canUserHideColumns: false,
        canFilter: false,
        isTransient: true,
        displayMode: ui.fields.TableDisplayMode.comfortable,
        mobileCard: undefined,
        async onRowClick(_id: string, rowItem) {
            if (this._parameters) {
                this.$.router.goTo(
                    '@sage/wh-pages/MobileViewStockByProductSelectStockObject',
                    this._getCurrentParameters(rowItem),
                );
            }
        },
        orderBy: { storeLocationCode: 1 },
        columns: [
            ui.nestedFields.text({
                bind: 'storeCode',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'locationCode',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'storeLocationCode',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'containerCode',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'numberOfConsumptionUnit',
                isReadOnly: true,
                postfix(_value, _rowValue) {
                    return this._parameters?.consumptionUnit?.code ?? '';
                },
                scale(_value, _rowValue) {
                    return this._parameters?.stockUnit?.numberOfDecimals ?? 0;
                },
            }),
            ui.nestedFields.text({
                bind: 'locationType',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'reserved',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'distinctLot',
                isReadOnly: true,
                isHidden() {
                    return !this._parameters?.isKeyInLotNumber;
                },
                prefix: ui.localize('@sage/wh-pages/lots-numbers-prefix', 'Lots:'),
            }),
            ui.nestedFields.text({
                bind: 'distinctSerialNumber',
                isReadOnly: true,
                prefix: ui.localize('@sage/wh-pages/serial-numbers-prefix', 'Serial no.:'),
                isHidden() {
                    return !this._parameters?.isKeyInSerialNumber || !this._parameters?.isLocalizedSerialNumberAllowed;
                },
            }),
        ],
    })
    storesLocations!: ui.fields.Table<any>;

    /**
     * get query parameters
     * @returns expected parameters or undefined
     */
    private _getQueryParameters(): StockByProductArgs | undefined {
        try {
            return JSON.parse(String(this.$.queryParameters.stockByProductArgs)) as StockByProductArgs;
        } catch (error) {
            ui.console.error(`Error retrieving query parameters:\n${JSON.stringify(error)}`);
            return undefined;
        }
    }

    /**
     * Return current parameter before chain another page
     * @return return the current parameters
     */
    private _getCurrentParameters(
        rowItem: any,
        _stockObject?: StockObjectSelected,
    ): {
        stockByProductArgs: string;
    } {
        return {
            stockByProductArgs: JSON.stringify(this._getUpdatedParameters(rowItem, _stockObject)),
        };
    }

    private _getUpdatedParameters(rowItem: any, _stockObject?: StockObjectSelected): StockByProductArgs {
        return <StockByProductArgs>{
            ...this._parameters,
            storeLocationSelected: {
                storeCode: rowItem?.storeCode,
                locationCode: rowItem?.locationCode,
            },
            stockObjectsSelected: _stockObject,
        };
    }

    private async _readAggregatedProduct(_parameters: StockByProductArgs): Promise<AggregatedProduct | null> {
        const _aggregatedProduct = <AggregatedProduct>{ distinctLocations: 0 };
        const _isLocalizedSerialNumberAllowed =
            _parameters?.isKeyInSerialNumber && _parameters?.isLocalizedSerialNumberAllowed;
        const _filterByProduct = {
            ...(_parameters?.siteDepositorSelected?.siteCode && {
                site: _parameters?.siteDepositorSelected?.siteCode,
            }),
            ...(_parameters?.siteDepositorSelected?.depositorCode && {
                depositor: _parameters.siteDepositorSelected.depositorCode,
            }),
            ...(_parameters?.productCode && { product: _parameters?.productCode }),
        };
        const _groupByProduct: AggregateGroupSelector<StockObject> = {
            product: {
                code: {
                    _by: 'value',
                },
            },
        };
        // Separate aggregate calls in order to exclude blank lots from distinct count
        const _requests = {
            aggregateLocation: generateAggregateStockObjectRequest<StockObject>(
                this,
                {
                    group: _groupByProduct,
                    values: {
                        location: { code: { distinctCount: true } },
                    },
                },
                {
                    filter: _filterByProduct,
                },
            ),
            ...(_parameters?.isKeyInLotNumber && {
                aggregateLocationLot: generateAggregateStockObjectRequest<StockObject>(
                    this,
                    {
                        group: _groupByProduct,
                        values: {
                            lotNumber: { distinctCount: true },
                        },
                    },
                    {
                        filter: { ..._filterByProduct, lotNumber: { _ne: null } },
                    },
                ),
            }),
            ...(_isLocalizedSerialNumberAllowed && {
                aggregateOsSerialNumber: generateAggregateSerialNumberRequest<SerialNumber>(
                    this,
                    {
                        group: {
                            product: { code: { _by: 'value' } },
                        },
                        values: {
                            //                            product: { code: { distinctCount: true } },
                            keyCodeForDuplicates: { distinctCount: true },
                        },
                    },
                    {
                        filter: { ..._filterByProduct, outputDate: { _eq: null } },
                    },
                ),
            }),
        };

        /**
         * Because result is a location aggregated, is necessary to count records returned after
         * the aggregation to get the number of distinct locations.
         */
        interface AggregateByLocation {
            values: {
                location: {
                    code: {
                        distinctCount: number;
                    };
                };
            };
        }

        const _response = await new ui.queryUtils.BatchRequest(_requests).execute();
        const _aggregateLocation = extractEdges<AggregateByLocation>(_response?.aggregateLocation as Edges<any>);

        if (_aggregateLocation?.length) {
            _aggregatedProduct.distinctLocations = _aggregateLocation[0].values.location.code.distinctCount;

            if (_parameters?.isKeyInLotNumber && _response?.aggregateLocationLot && _response?.aggregateLocationLot) {
                interface AggregateByLocationLot {
                    values: {
                        lotNumber: {
                            distinctCount: number;
                        };
                    };
                }

                const _aggregateLocationLot = extractEdges<AggregateByLocationLot>(
                    _response.aggregateLocationLot as Edges<any>,
                );
                _aggregateLocationLot.forEach(aggregation => {
                    if (_aggregatedProduct.distinctLots) {
                        _aggregatedProduct.distinctLots += aggregation.values.lotNumber.distinctCount;
                    } else {
                        _aggregatedProduct.distinctLots = aggregation.values.lotNumber.distinctCount;
                    }
                });
            }

            if (_isLocalizedSerialNumberAllowed && _response?.aggregateOsSerialNumber) {
                interface AggregateByOsSerialNumber {
                    values: {
                        keyCodeForDuplicates: {
                            distinctCount: number;
                        };
                    };
                }

                const _aggregateOsSerialNumber = extractEdges<AggregateByOsSerialNumber>(
                    _response.aggregateOsSerialNumber as Edges<any>,
                );
                _aggregateOsSerialNumber.forEach(aggregation => {
                    if (_aggregatedProduct.distinctSerialNumbers) {
                        _aggregatedProduct.distinctSerialNumbers +=
                            aggregation.values.keyCodeForDuplicates.distinctCount;
                    } else {
                        _aggregatedProduct.distinctSerialNumbers =
                            aggregation.values.keyCodeForDuplicates.distinctCount;
                    }
                });
            }
        }

        return _aggregatedProduct ?? null;
    }

    private async _readAggregatedStoreAndLocation(_parameters: StockByProductArgs): Promise<AggregatedStoreLocation[]> {
        const _storeLocationList: AggregatedStoreLocation[] = [];
        const _isLocalizedSerialNumberAllowed =
            _parameters?.isKeyInSerialNumber && _parameters?.isLocalizedSerialNumberAllowed;
        const _getIdStoreLocation = (storeCode: string, locationCode: string): string => `${storeCode} ${locationCode}`;
        const _filterByProduct = {
            ...(_parameters?.siteDepositorSelected.siteCode && {
                site: _parameters?.siteDepositorSelected.siteCode,
            }),
            ...(_parameters?.siteDepositorSelected.depositorCode && {
                depositor: _parameters?.siteDepositorSelected.depositorCode,
            }),
            ...(_parameters?.productCode && { product: _parameters?.productCode }),
        };
        const groupByLocation: AggregateGroupSelector<StockObject> = {
            store: {
                code: {
                    _by: 'value',
                },
            },
            location: {
                code: {
                    _by: 'value',
                },
            },
            locationType: { _by: 'value' },
        };

        // stockObject

        // Separate aggregate calls in order to exclude blank lots from distinct count
        const _requests = {
            aggregateLocation: generateAggregateStockObjectRequest<StockObject>(
                this,
                {
                    group: groupByLocation,
                    values: {
                        location: { code: { distinctCount: true } },
                        numberOfConsumptionUnit: { sum: true },
                    },
                },
                {
                    filter: _filterByProduct,
                },
            ),
            ...(_parameters?.isKeyInLotNumber && {
                aggregateLocationLot: generateAggregateStockObjectRequest<StockObject>(
                    this,
                    {
                        group: {
                            ...groupByLocation,

                            lotNumber: {
                                _by: 'value',
                            },
                        },
                        values: {
                            lotNumber: { distinctCount: true },
                        },
                    },
                    {
                        filter: { ..._filterByProduct, lotNumber: { _ne: null } },
                    },
                ),
            }),
            ...(_isLocalizedSerialNumberAllowed && {
                aggregateOsSerialNumber: generateAggregateSerialNumberRequest<SerialNumber>(
                    this,
                    {
                        group: {
                            stockObject: {
                                store: {
                                    code: {
                                        _by: 'value',
                                    },
                                },
                                location: {
                                    code: {
                                        _by: 'value',
                                    },
                                },
                            },
                            code: {
                                _by: 'value',
                            },
                        },
                        values: {
                            keyCodeForDuplicates: { distinctCount: true },
                        },
                    },
                    {
                        filter: { ..._filterByProduct, outputDate: { _eq: null } },
                    },
                ),
            }),
        };

        const _response = await new ui.queryUtils.BatchRequest(_requests).execute();
        interface AggregateByLocation {
            values: {
                location: {
                    code: {
                        distinctCount: number;
                    };
                };
                numberOfConsumptionUnit: {
                    sum: string | number;
                };
            };
            group: {
                store: {
                    code: string;
                };
                location: {
                    code: string;
                };
                locationType: string;
            };
        }
        const _aggregateLocation = extractEdges<AggregateByLocation>(_response?.aggregateLocation as Edges<any>);

        _aggregateLocation.forEach(aggregation => {
            const _id = _getIdStoreLocation(aggregation.group.store.code, aggregation.group.location.code);
            _storeLocationList.push(<AggregatedStoreLocation>{
                _key: _id,
                storeCode: aggregation.group.store.code,
                locationCode: aggregation.group.location.code,
                locationType: aggregation.group.locationType,
                numberOfConsumptionUnit: Number(aggregation.values.numberOfConsumptionUnit.sum ?? 0),
                distinctStockObject: aggregation.values.location.code.distinctCount,
                distinctLot: 0,
                distinctSerialNumber: 0,
            });
        });

        if (_parameters?.isKeyInLotNumber && _response?.aggregateLocationLot) {
            interface AggregateByLocationLot {
                values: {
                    lotNumber: {
                        distinctCount: number;
                    };
                };
                group: {
                    store: {
                        code: string;
                    };
                    location: {
                        code: string;
                    };
                };
            }

            const _aggregateLocationLot = extractEdges<AggregateByLocationLot>(
                _response.aggregateLocationLot as Edges<any>,
            );
            _aggregateLocationLot.forEach(aggregation => {
                const _id = _getIdStoreLocation(aggregation.group.store.code, aggregation.group.location.code);
                const _item = _storeLocationList.find(item => item._key === _id);
                if (_item) {
                    _item.distinctLot += aggregation.values.lotNumber.distinctCount;
                }
            });
        }

        if (_isLocalizedSerialNumberAllowed && _response?.aggregateOsSerialNumber) {
            interface ResultAggregateOsSerialNumber {
                values: {
                    keyCodeForDuplicates: {
                        distinctCount: number;
                    };
                };
                group: {
                    stockObject: {
                        store: {
                            code: string;
                        };
                        location: {
                            code: string;
                        };
                    };
                };
            }

            const _aggregateOsSerialNumber = extractEdges<ResultAggregateOsSerialNumber>(
                _response.aggregateOsSerialNumber as Edges<any>,
            );
            _aggregateOsSerialNumber.forEach(aggregation => {
                const _id = _getIdStoreLocation(
                    aggregation.group.stockObject.store.code,
                    aggregation.group.stockObject.location.code,
                );
                const _item = _storeLocationList.find(item => item._key === _id);
                if (_item) {
                    _item.distinctSerialNumber += aggregation.values.keyCodeForDuplicates.distinctCount;
                }
            });
        }

        return _storeLocationList;
    }
}
