import type { GraphApi } from '@sage/wh-pages-api';
import type { SerialNumber, StockObject } from '@sage/wh-stock-data-api';
import {
    generateAggregateSerialNumberRequest,
    generateAggregateStockObjectRequest,
} from '@sage/wh-stock-data/lib/client-functions/aggregators';
import { StockObjectSelected } from '@sage/wh-stock-data/lib/interfaces';
import { AggregateGroupSelector, Edges, ExtractEdges, extractEdges } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import type { PartialCollectionValueWithIds } from '@sage/xtrem-ui/build/lib/component/types';
import { StockByProductArgs, StockByProductArgsHeader } from '../interfaces';
import { AggregatedStoreLocation } from './mobile-view-stock-by-product-select-store-location';

// Aggregated values for many stock objects
interface AggregatedStockObjects extends AggregatedStoreLocation {
    _id?: string;
    stockObjectCode: string;
}

/**
 * Data structure for detail panel
 */
export interface StockObjectEntry {
    _id: string;
    code: string;
    containerCode: string;
    containerLevel: string;
    numberOfContainer: string;
    homogeneousContainerCode: string;
    homogeneousContainerLevel: string;
    homogeneousQuantity: string;
    consumptionUnitCode: string;
    numberOfConsumptionUnit: number | string;
    numberOfConsumptionUnitInPendingInput: number | string;
    numberOfConsumptionUnitInPendingOutput: number | string;
    lotNumber: string;
    supportNumber: string;
    reservationNumber: string;
    stockStatusCode: string;
    fifoDate?: string;
    receiptDate?: string;
    manufacturedDate?: string;
    detentionUnit?: string;
    detentionDate?: string;
    sellByDateUnit?: string;
    sellByDate?: string;
    useByDateUnit?: string;
    useByDate?: string;
    sendByDateUnit?: string;
    shipByDate?: string;
    isControlOutputFlow: boolean;
    isBlockedInOutput: boolean;
    isAuthorizedInOutput: boolean;
    isBlockedInInventory: boolean;
    stockCountCode?: string;
    distinctSerialNumber: number;
}

/**
 * Data structure for table
 */
interface StockObjectTable extends StockObjectEntry {
    numberOfConsumptionUnitExplained: string;
    distinctStockObject: number;
    distinctLot: number;
    storeCode: string;
    locationCode: string;
    storeLocation: string;
    productCode: string;
    locationType: string;
}

@ui.decorators.page<MobileViewStockByProductSelectStockObject>({
    title: 'Stock by product',
    subtitle: 'Select a stock object',
    isTitleHidden: true,
    isTransient: true,
    skipDirtyCheck: true,
    authorizationCode: 'INQSTOPRO',
    headerCard() {
        return {
            title: this.productCode,
            titleRight: this.numberOfConsumptionUnit,
            line2: this.localizedDescription,
            line3: this.storeLocation,
            line3Right: this.locationType,
        };
    },
    detailPanel() {
        return {
            isCloseButtonHidden: true,
            isTitleHidden: true,
            isHidden: true,
            isTransient: true,
            header: this.detailPanelSection,
            sections: [],
        };
    },
    async onLoad(): Promise<void> {
        // Requires a selected product in the query parameters.  This should not occur unless user manually
        // directs themselves to this page
        const _parameters = this._getQueryParameters();
        this._setTitle(_parameters);

        if (!_parameters || !_parameters.productCode) {
            this.$.showToast(ui.localize('@sage/wh-pages/product-required', 'Selected product is required'), {
                type: 'warning',
            });

            this.$.router.goTo(
                _parameters && !_parameters?.isViewByStoreLocation
                    ? '@sage/wh-pages/MobileViewStockByProduct'
                    : '@sage/wh-pages/MobileViewStockByStoreLocationSelectProduct',
                this._getCurrentParameters(_parameters),
            );
            return;
        }

        if (!_parameters.storeLocationSelected?.storeCode || !_parameters.storeLocationSelected?.locationCode) {
            this.$.showToast(
                ui.localize('@sage/wh-pages/store-location-required', 'Selected store / location is required'),
                {
                    type: 'warning',
                },
            );
            this.$.router.goTo(
                !_parameters?.isViewByStoreLocation
                    ? '@sage/wh-pages/MobileViewStockByProductSelectStoreLocation'
                    : '@sage/wh-pages/MobileViewStockByStoreLocation',
                this._getCurrentParameters(_parameters),
            );
            return;
        }

        this._parameters = _parameters;

        // Read stock objects (list in the store / location)
        const _stockObjects = await this._readStockObjects(_parameters);

        let _aggregatedLocationQuantity: AggregatedStoreLocation | null | undefined;

        if (_stockObjects?.length) {
            _aggregatedLocationQuantity = await this._getAggregatedLocationQuantities(_parameters);
        }

        // Should never happen
        if (!_stockObjects?.length || !_aggregatedLocationQuantity) {
            this.$.showToast(
                ui.localize('@sage/wh-pages/selected-product-no-results', 'Product {{ code }} has no stock records.', {
                    code: this.productCode.value ?? '',
                }),
                { type: 'info' },
            );
            this.$.router.goTo(
                !_parameters?.isViewByStoreLocation
                    ? '@sage/wh-pages/MobileViewStockByProduct'
                    : '@sage/wh-pages/MobileViewStockByStoreLocationSelectProduct',
                this._getCurrentParameters(_parameters),
            );
            return;
        }

        // Create table header
        this.productCode.value = _parameters.productCode;
        this.numberOfConsumptionUnit.value = Number(_aggregatedLocationQuantity?.numberOfConsumptionUnit ?? 0);
        this.numberOfConsumptionUnit.postfix = _parameters?.consumptionUnit?.code ?? '';
        this.localizedDescription.value = _parameters?.localizedDescription ?? null;
        this.storeCode.value = _parameters.storeLocationSelected?.storeCode;
        this.locationCode.value = _parameters.storeLocationSelected?.locationCode;
        this.storeLocation.prefix = ui.localize('@sage/wh-pages/store-location-prefix', 'Location');
        this.storeLocation.value = `${_parameters.storeLocationSelected?.storeCode} ${_parameters.storeLocationSelected?.locationCode}`;
        this.locationType.value = _aggregatedLocationQuantity?.locationType;
        this.mainBlock.title = ui.localize(
            '@sage/wh-pages/number-of-stock-objects',
            '{{ numberOfStockObjects }} stock object(s)',
            {
                numberOfStockObjects: _stockObjects?.length ?? 0,
            },
        );
        // aggregated value for store / location (one record)
        const _aggregatedByStockObjects = await this._readAggregatedStockObjects(_parameters);

        this.$.setPageClean();

        this._initializeStockObjectTable(_stockObjects, _aggregatedByStockObjects, _parameters);
    },
})
export class MobileViewStockByProductSelectStockObject extends ui.Page<GraphApi> {
    /**
     * Internal properties
     */

    private _parameters: StockByProductArgs | undefined;

    /*
     *
     *  Sections
     *
     */

    @ui.decorators.section<MobileViewStockByProductSelectStockObject>({
        isTitleHidden: true,
    })
    mainSection!: ui.containers.Section;

    @ui.decorators.section<MobileViewStockByProductSelectStockObject>({
        title: 'Select stock by product',
        isTitleHidden: true,
    })
    detailPanelSection!: ui.containers.Section;

    @ui.decorators.section<MobileViewStockByProductSelectStockObject>({
        isTitleHidden: true,
    })
    sectionHeader!: ui.containers.Section;

    /*
     *
     *  Blocks
     *
     */

    @ui.decorators.block<MobileViewStockByProductSelectStockObject>({
        parent() {
            return this.mainSection;
        },
    })
    mainBlock!: ui.containers.Block;

    @ui.decorators.block<MobileViewStockByProductSelectStockObject>({
        isTitleHidden: true,
        parent() {
            return this.detailPanelSection;
        },
    })
    detailsBlock!: ui.containers.Block;

    /*
     *
     *  Technical fields
     *
     */

    @ui.decorators.textField<MobileViewStockByProductSelectStockObject>({
        isTransient: true,
        isReadOnly: true,
        isHidden: true,
    })
    storeCode!: ui.fields.Text;

    @ui.decorators.textField<MobileViewStockByProductSelectStockObject>({
        isTransient: true,
        isReadOnly: true,
        isHidden: true,
    })
    locationCode!: ui.fields.Text;

    @ui.decorators.textField<MobileViewStockByProductSelectStockObject>({
        isTransient: true,
        isReadOnly: true,
    })
    productCode!: ui.fields.Text;

    @ui.decorators.numericField<MobileViewStockByProductSelectStockObject>({
        isTransient: true,
        isReadOnly: true,
        scale(_value, _rowValue) {
            return this._parameters?.stockUnit?.numberOfDecimals ?? 0;
        },
    })
    numberOfConsumptionUnit!: ui.fields.Numeric;

    @ui.decorators.textField<MobileViewStockByProductSelectStockObject>({
        isTransient: true,
        isReadOnly: true,
    })
    localizedDescription!: ui.fields.Text;

    @ui.decorators.textField<MobileViewStockByProductSelectStockObject>({
        isTransient: true,
        isReadOnly: true,
    })
    reserved!: ui.fields.Text;

    @ui.decorators.textField<MobileViewStockByProductSelectStockObject>({
        isTransient: true,
        isReadOnly: true,
        prefix: ui.localize('@sage/wh-pages/store-location-prefix', 'Location'),
    })
    storeLocation!: ui.fields.Text;

    @ui.decorators.textField<MobileViewStockByProductSelectStockObject>({
        isTransient: true,
        isReadOnly: true,
    })
    locationType!: ui.fields.Text;

    /*
     *
     *  Fields
     *
     */

    @ui.decorators.tableField<MobileViewStockByProductSelectStockObject>({
        parent() {
            return this.mainBlock;
        },
        node: '@sage/wh-stock-data/StockObject',
        isTransient: true,
        isFullWidth: true,
        isTitleHidden: true,
        canFilter: false,
        canSelect: false,
        canExport: false,
        canUserHideColumns: false,
        mobileCard: undefined,
        orderBy: { code: 1 },
        columns: [
            ui.nestedFields.text({
                bind: 'code',
                isReadOnly: true,
                prefix: ui.localize('@sage/wh-pages/prefix-stock-object', 'SO:'),
            }),
            ui.nestedFields.text({
                bind: 'dummy',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'numberOfConsumptionUnit',
                isReadOnly: true,
                postfix(value, rowValue) {
                    return rowValue?.consumptionUnit?.code ?? '';
                },
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'numberOfConsumptionUnitExplained',
                isReadOnly: true,
            }),
            ui.nestedFields.numeric({
                bind: 'distinctSerialNumber',
                isReadOnly: true,
                prefix: ui.localize('@sage/wh-pages/serial-numbers-prefix', 'Serial no.:'),
                isHidden() {
                    return !this._parameters?.isKeyInSerialNumber && !this._parameters?.isLocalizedSerialNumberAllowed;
                },
            }),
            ui.nestedFields.text({
                bind: 'consumptionUnitCode',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'containerCode',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'containerLevel',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'numberOfContainer',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'homogeneousContainerCode',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'homogeneousContainerLevel',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'homogeneousQuantity',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: '_id',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'distinctStockObject',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'distinctLot',
                isReadOnly: true,
                isHidden: true,
                prefix: ui.localize('@sage/wh-pages/lots-numbers-prefix', 'Lots:'),
            }),
            ui.nestedFields.text({
                bind: 'storeCode',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'locationCode',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'storeLocation',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'productCode',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'locationType',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'numberOfConsumptionUnitInPendingInput',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'numberOfConsumptionUnitInPendingOutput',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'stockStatus',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'supportNumber',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'reservationNumber',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.date({
                bind: 'fifoDate',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.date({
                bind: 'receiptDate',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.date({
                bind: 'manufacturedDate',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.date({
                bind: 'detentionDate',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.date({
                bind: 'sellByDate',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.date({
                bind: 'useByDate',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.date({
                bind: 'shipByDate',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isControlOutputFlow',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isBlockedInOutput',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isAuthorizedInOutput',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.checkbox({
                bind: 'isBlockedInInventory',
                isReadOnly: true,
                isHidden: true,
            }),
        ],
        onRowClick(recordId: string, rowItem: StockObjectTable) {
            this._onRowClick(this._parameters, recordId, rowItem);
        },
    })
    stockObject!: ui.fields.Table<StockObjectTable>;

    /**
     *  Move extracted data to stockObject transient table
     *  @param _stockObjects
     *  @param _aggregatedByStockObjects
     *  @param _parameters
     *  @returns true when stockObject table is not empty
     */
    private _initializeStockObjectTable(
        _stockObjects: ExtractEdges<StockObject>[],
        _aggregatedByStockObjects: AggregatedStockObjects[],
        _parameters: StockByProductArgs,
    ): boolean {
        const _stockObjectValues: PartialCollectionValueWithIds<StockObjectTable>[] = _stockObjects.map(
            _stockObject => {
                const _aggregatedStockObject = _aggregatedByStockObjects.find(_item => _item._id === _stockObject._id);
                return <StockObjectTable>{
                    _id: _stockObject._id,
                    code: _stockObject.code,
                    consumptionUnitCode: _parameters.consumptionUnit?.code,
                    numberOfConsumptionUnit: _stockObject.numberOfConsumptionUnit,
                    distinctStockObject: _aggregatedStockObject?.distinctStockObject ?? 0,
                    distinctSerialNumber: _aggregatedStockObject?.distinctSerialNumber ?? 0,
                    numberOfConsumptionUnitInPendingInput: _stockObject?.numberOfConsumptionUnitInPendingInput ?? 0,
                    numberOfConsumptionUnitInPendingOutput: _stockObject?.numberOfConsumptionUnitInPendingOutput ?? 0,
                    distinctLot: _aggregatedStockObject?.distinctLot ?? 0,
                    storeCode: _parameters.storeLocationSelected?.storeCode,
                    locationCode: _parameters.storeLocationSelected?.locationCode,
                    storeLocation: `${_parameters.storeLocationSelected?.storeCode} ${_parameters.storeLocationSelected?.locationCode}`,
                    productCode: _parameters.productCode,
                    locationType: _stockObject.locationType,
                    containerCode: _stockObject.container?.container?.code,
                    containerLevel: _stockObject.container?.containerLevel,
                    numberOfContainer: _stockObject.numberOfContainer,
                    homogeneousContainerCode: _stockObject.homogeneousContainer?.container?.code,
                    homogeneousContainerLevel: _stockObject.homogeneousContainer?.containerLevel,
                    homogeneousQuantity: _stockObject.homogeneousQuantity,
                    stockStatusCode: _stockObject?.stockStatus?.code,
                    supportNumber: _stockObject.supportNumber,
                    lotNumber: _stockObject.lotNumber,
                    reservationNumber: _stockObject.reservationNumber,
                    fifoDate: _stockObject?.fifoDate,
                    receiptDate: _stockObject?.receiptDate,
                    manufacturedDate: _stockObject?.manufacturedDate,
                    detentionDate: _stockObject?.detentionDate,
                    sellByDate: _stockObject?.sellByDate,
                    useByDate: _stockObject?.useByDate,
                    shipByDate: _stockObject?.shipByDate,
                    detentionUnit: _stockObject?.product?.detentionUnit,
                    sellByDateUnit: _stockObject?.product?.sellByDateUnit,
                    sendByDateUnit: _stockObject?.product?.sendByDateUnit,
                    useByDateUnit: _stockObject?.product?.useByDateUnit,
                    isControlOutputFlow: _stockObject?.depositor?.isControlOutputFlow,
                    isBlockedInOutput:
                        _stockObject?.location?.isBlockedInOutput ||
                        (_stockObject?.product?.isBlockedInOutput && _stockObject?.depositor?.isControlOutputFlow),
                    isAuthorizedInOutput:
                        _stockObject?.store?.isAuthorizedInOutput && _stockObject?.container?.isOutputAllowed,
                    isBlockedInInventory:
                        !!_stockObject?.location?.stockCount?.code ||
                        (!!_stockObject?.product?.stockCount?.code && _stockObject?.depositor?.isControlOutputFlow),
                    stockCountCode:
                        (_stockObject?.location?.stockCount?.code ?? _stockObject?.depositor?.isControlOutputFlow)
                            ? _stockObject?.product?.stockCount?.code
                            : undefined,
                    numberOfConsumptionUnitExplained: ui.localize(
                        '@sage/wh-pages/numberOfConsumptionUnit-explained',
                        '{{ numberOfContainer }} {{ containerCode }} of {{ homogeneousQuantity }} {{ homogeneousContainerCode }} = {{ numberOfConsumptionUnit }} {{ consumptionUnitCode }}',
                        {
                            containerCode: _stockObject.container?.container?.code,
                            numberOfContainer: _stockObject.numberOfContainer,
                            homogeneousContainerCode: _stockObject.homogeneousContainer?.container?.code,
                            homogeneousQuantity: _stockObject.homogeneousQuantity,
                            consumptionUnitCode: _parameters.consumptionUnit?.code,
                            numberOfConsumptionUnit: _stockObject.numberOfConsumptionUnit,
                        },
                    ),
                };
            },
        );

        this.stockObject.value = _stockObjectValues;

        return _stockObjectValues.length > 0;
    }

    /*
     *
     *  Fields management functions
     *
     */

    private _onRowClick(_parameters: any, recordId: string, rowItem: StockObjectTable): void {
        /**
         * TODO: Remove keys that are not needed in the next page
         * See if possible to use StockObjectEntry interface definition
         * to avoid this and verify is each property is defined.
         */
        const _keysToRemove: Array<keyof StockObjectTable> = [
            'numberOfConsumptionUnitExplained',
            'distinctStockObject',
            'distinctLot',
            'storeCode',
            'locationCode',
            'storeLocation',
            'productCode',
            'locationType',
        ];

        const _details = Object.keys(rowItem)
            .filter((key): key is keyof StockObjectEntry => !_keysToRemove.includes(key as keyof StockObjectTable))
            .reduce(
                (acc, key) => {
                    if (rowItem[key] !== undefined) {
                        acc[key] = rowItem[key];
                    }
                    return acc;
                },
                {} as Partial<Record<keyof StockObjectEntry, any>>,
            );

        this.$.router.goTo(
            '@sage/wh-pages/MobileViewStockByProductStockObjectDetails',
            this._getCurrentParameters(<StockByProductArgs>{
                ..._parameters,
                stockObjectSelected: <StockObjectSelected>{
                    _id: recordId,
                    code: rowItem.code,
                    details: _details,
                },
            }),
        );
    }

    /**
     * get query parameters
     * @returns expected parameters or undefined
     */
    private _getQueryParameters(): StockByProductArgs | undefined {
        try {
            return JSON.parse(String(this.$.queryParameters.stockByProductArgs)) as StockByProductArgs;
        } catch (_error) {
            return undefined;
        }
    }

    /**
     * Return current parameter before chain another page
     * @return return the current parameters
     */
    // eslint-disable-next-line class-methods-use-this
    private _getCurrentParameters(_parameters?: StockByProductArgs): { stockByProductArgs: string } | undefined {
        return _parameters
            ? <StockByProductArgsHeader>{
                  stockByProductArgs: JSON.stringify(_parameters),
              }
            : undefined;
    }

    /**
     * Return aggregated stockObject for lots / serial numbers
     * @returns
     */
    private async _readAggregatedStockObjects(_parameters: StockByProductArgs): Promise<AggregatedStockObjects[]> {
        const _stockObjects: AggregatedStockObjects[] = [];
        const _isLocalizedSerialNumberAllowed =
            _parameters?.isKeyInSerialNumber && _parameters?.isLocalizedSerialNumberAllowed;
        const _getKeyStockObjectStoreLocation = (storeCode: string, locationCode: string, code: string): string =>
            `${storeCode} ${locationCode} ${code}`;
        const _filterByProduct = {
            ...(_parameters?.siteDepositorSelected?.siteCode && {
                site: _parameters?.siteDepositorSelected?.siteCode,
            }),
            ...(_parameters?.siteDepositorSelected?.depositorCode && {
                depositor: _parameters?.siteDepositorSelected?.depositorCode,
            }),
            ...(_parameters?.productCode && { product: _parameters?.productCode }),
            ...(_parameters?.storeLocationSelected?.storeCode && {
                store: _parameters?.storeLocationSelected?.storeCode,
            }),
            ...(_parameters?.storeLocationSelected?.locationCode && {
                location: _parameters?.storeLocationSelected?.locationCode,
            }),
        };
        const _filterBySerialNumbers = {
            ...(_parameters?.siteDepositorSelected?.siteCode && {
                site: _parameters?.siteDepositorSelected?.siteCode,
            }),
            ...(_parameters?.siteDepositorSelected?.depositorCode && {
                depositor: _parameters?.siteDepositorSelected?.depositorCode,
            }),
            ...(_parameters?.productCode && { product: _parameters?.productCode }),
            ...(_parameters?.storeLocationSelected?.storeCode && {
                stockObject: {
                    store: _parameters?.storeLocationSelected?.storeCode,
                },
            }),
            ...(_parameters?.storeLocationSelected?.locationCode && {
                stockObject: {
                    location: _parameters?.storeLocationSelected?.locationCode,
                },
            }),
        };
        const groupByStockObject: AggregateGroupSelector<StockObject> = {
            store: { code: { _by: 'value' } },
            location: { code: { _by: 'value' } },
            locationType: { _by: 'value' },
            code: { _by: 'value' },
            _id: { _by: 'value' },
        };
        const groupBySerialNumbers: AggregateGroupSelector<SerialNumber> = {
            stockObject: {
                store: { code: { _by: 'value' } },
                location: { code: { _by: 'value' } },
                code: { _by: 'value' },
                _id: { _by: 'value' },
            },
        };

        // Separate aggregate calls in order to exclude blank lots from distinct count
        const _requests = {
            aggregateLocation: generateAggregateStockObjectRequest<StockObject>(
                this,
                {
                    group: groupByStockObject,
                    values: {
                        code: { distinctCount: true },
                        numberOfConsumptionUnit: { sum: true },
                    },
                },
                {
                    filter: _filterByProduct,
                },
            ),
            ...(_parameters?.isKeyInLotNumber && {
                aggregateLocationLot: generateAggregateStockObjectRequest<StockObject>(
                    this,
                    {
                        group: {
                            ...groupByStockObject,
                            lotNumber: {
                                _by: 'value',
                            },
                        },
                        values: {
                            lotNumber: { distinctCount: true },
                        },
                    },
                    {
                        filter: { ..._filterByProduct, lotNumber: { _ne: null } },
                    },
                ),
            }),
            ...(_isLocalizedSerialNumberAllowed && {
                aggregateOsSerialNumber: generateAggregateSerialNumberRequest<SerialNumber>(
                    this,
                    {
                        group: groupBySerialNumbers,
                        values: {
                            keyCodeForDuplicates: { distinctCount: true },
                        },
                    },
                    {
                        filter: { ..._filterBySerialNumbers, outputDate: { _eq: null } },
                    },
                ),
            }),
        };

        const _response = await new ui.queryUtils.BatchRequest(_requests).execute();
        interface AggregateByLocation {
            group: {
                store: {
                    code: string;
                };
                location: {
                    code: string;
                };
                locationType: string;
                code: string;
                _id: any;
            };
            values: {
                code: {
                    distinctCount: number;
                };
                numberOfConsumptionUnit: {
                    sum: string | number;
                };
            };
        }

        const _aggregateLocation = extractEdges<AggregateByLocation>(_response?.aggregateLocation as Edges<any>);

        _aggregateLocation.forEach(aggregation => {
            const _key = _getKeyStockObjectStoreLocation(
                aggregation.group.store.code,
                aggregation.group.location.code,
                aggregation.group.code,
            );
            _stockObjects.push(<AggregatedStockObjects>{
                _id: aggregation.group._id,
                _key,
                stockObjectCode: aggregation.group.code,
                storeCode: aggregation.group.store.code,
                locationCode: aggregation.group.location.code,
                locationType: aggregation.group.locationType,
                numberOfConsumptionUnit: Number(aggregation.values.numberOfConsumptionUnit.sum ?? 0),
                distinctStockObject: aggregation.values.code.distinctCount,
                distinctLot: 0,
                distinctSerialNumber: 0,
            });
        });

        if (_parameters?.isKeyInLotNumber && _response?.aggregateLocationLot) {
            interface AggregateByLocationLot {
                group: {
                    store: {
                        code: string;
                    };
                    location: {
                        code: string;
                    };
                    locationType: string;
                    code: string;
                };
                values: {
                    lotNumber: {
                        distinctCount: number;
                    };
                };
            }
            const _aggregateLocationLot = extractEdges<AggregateByLocationLot>(
                _response.aggregateLocationLot as Edges<any>,
            );
            _aggregateLocationLot.forEach(aggregation => {
                const _key = _getKeyStockObjectStoreLocation(
                    aggregation.group.store.code,
                    aggregation.group.location.code,
                    aggregation.group.code,
                );
                const _item = _stockObjects.find(item => item._key === _key);
                if (_item) {
                    _item.distinctLot = aggregation.values.lotNumber.distinctCount;
                }
            });
        }

        if (_isLocalizedSerialNumberAllowed && _response?.aggregateOsSerialNumber) {
            interface AggregateByOsSerialNumber {
                group: {
                    stockObject: {
                        store: {
                            code: string;
                        };
                        location: {
                            code: string;
                        };
                        code: string;
                        _id: any;
                    };
                };
                values: {
                    keyCodeForDuplicates: {
                        distinctCount: number;
                    };
                };
            }
            const _aggregateOsSerialNumber = extractEdges<AggregateByOsSerialNumber>(
                _response.aggregateOsSerialNumber as Edges<any>,
            );
            _aggregateOsSerialNumber.forEach(aggregation => {
                const _key = _getKeyStockObjectStoreLocation(
                    aggregation.group.stockObject.store.code,
                    aggregation.group.stockObject.location.code,
                    aggregation.group.stockObject.code,
                );
                const _item = _stockObjects.find(item => item._key === _key);
                if (_item) {
                    _item.distinctSerialNumber = aggregation.values.keyCodeForDuplicates.distinctCount;
                }
            });
        }

        return _stockObjects;
    }

    /**
     * Read stock objects for a given store / location
     * @param _parameters
     * @returns stock objects selected
     */
    private async _readStockObjects(
        _parameters: StockByProductArgs,
        numberOfRecords = 1000,
    ): Promise<ExtractEdges<StockObject>[]> {
        interface ResultStockObject {
            _id: string;
            code: string;
            lotNumber: string;
            locationType: string;
            numberOfConsumptionUnit: number | string;
            numberOfConsumptionUnitInPendingInput: number | string;
            numberOfConsumptionUnitInPendingOutput: number | string;
            container: {
                isOutputAllowed: boolean;
                container: { code: string };
                containerLevel: string;
            };
            numberOfContainer: number | string;
            homogeneousContainer: { container: { code: string }; containerLevel: string };
            homogeneousQuantity: number | string;
            stockStatus?: { code: string };
            supportNumber: string;
            reservationNumber: string;
            fifoDate?: string;
            receiptDate?: string;
            manufacturedDate?: string;
            detentionDate?: string;
            sellByDate?: string;
            useByDate?: string;
            shipByDate?: string;
            depositor: { isControlOutputFlow: boolean };
            product: {
                isBlockedInOutput: boolean;
                detentionUnit: string;
                sellByDateUnit: string;
                sendByDateUnit: string;
                useByDateUnit: string;
                stockCount?: { code: string };
            };
            store: { isAuthorizedInOutput: boolean };
            location: { isBlockedInOutput: boolean; stockCount?: { code: string } };
        }

        try {
            return <ExtractEdges<StockObject>[]>(<unknown>extractEdges<ResultStockObject>(
                    await this.$.graph
                        .node('@sage/wh-stock-data/StockObject')
                        .query(
                            ui.queryUtils.edgesSelector(
                                {
                                    _id: true,
                                    code: true,
                                    lotNumber: true,
                                    locationType: true,
                                    numberOfConsumptionUnit: true,
                                    numberOfConsumptionUnitInPendingInput: true,
                                    numberOfConsumptionUnitInPendingOutput: true,
                                    container: {
                                        isOutputAllowed: true,
                                        container: { code: true },
                                        containerLevel: true,
                                    },
                                    numberOfContainer: true,
                                    homogeneousContainer: {
                                        container: { code: true },
                                        containerLevel: true,
                                    },
                                    homogeneousQuantity: true,
                                    stockStatus: { code: true },
                                    supportNumber: true,
                                    reservationNumber: true,
                                    fifoDate: true,
                                    receiptDate: true,
                                    manufacturedDate: true,
                                    detentionDate: true,
                                    sellByDate: true,
                                    useByDate: true,
                                    shipByDate: true,
                                    depositor: {
                                        isControlOutputFlow: true,
                                    },
                                    product: {
                                        isBlockedInOutput: true,
                                        detentionUnit: true,
                                        sellByDateUnit: true,
                                        sendByDateUnit: true,
                                        useByDateUnit: true,
                                        stockCount: { code: true },
                                    },
                                    store: {
                                        isAuthorizedInOutput: true,
                                    },
                                    location: {
                                        isBlockedInOutput: true,
                                        stockCount: { code: true },
                                    },
                                },
                                {
                                    first: numberOfRecords,
                                    filter: {
                                        site: {
                                            code: _parameters?.siteDepositorSelected?.siteCode,
                                        },
                                        depositor: {
                                            code: _parameters?.siteDepositorSelected?.depositorCode,
                                        },
                                        store: {
                                            code: _parameters?.storeLocationSelected?.storeCode as string,
                                        },
                                        location: {
                                            code: _parameters?.storeLocationSelected?.locationCode as string,
                                        },
                                        product: {
                                            code: _parameters?.productCode,
                                            isActive: true,
                                        },
                                    },
                                    orderBy: {
                                        code: 1,
                                    },
                                },
                            ),
                        )
                        .execute(),
                )) ?? [];
        } catch (error) {
            ui.console.error(`Error reading stock object:\n${JSON.stringify(error)}`);
            return [];
        }
    }

    /**
     * Read aggregated quantities for give store / location
     * lots and serials numbers has not been aggregated
     * @param _parameters
     * @returns aggregated quantities
     */
    private async _getAggregatedLocationQuantities(
        _parameters: StockByProductArgs,
    ): Promise<AggregatedStoreLocation | null> {
        const _aggregatedLocationQuantities: AggregatedStoreLocation[] = [];
        const _getKeyStoreLocation = (storeCode: string, locationCode: string): string =>
            `${storeCode} ${locationCode}`;
        const _filterByProduct = {
            ...(_parameters?.siteDepositorSelected?.siteCode && {
                site: _parameters?.siteDepositorSelected?.siteCode,
            }),
            ...(_parameters?.siteDepositorSelected?.depositorCode && {
                depositor: _parameters?.siteDepositorSelected?.depositorCode,
            }),
            ...(_parameters?.productCode && { product: _parameters?.productCode }),
            ...(_parameters?.storeLocationSelected?.storeCode && {
                store: _parameters?.storeLocationSelected?.storeCode,
            }),
            ...(_parameters?.storeLocationSelected?.locationCode && {
                location: _parameters?.storeLocationSelected?.locationCode,
            }),
        };

        // Separate aggregate calls in order to exclude blank lots from distinct count
        const _requests = {
            aggregateLocation: generateAggregateStockObjectRequest<StockObject>(
                this,
                {
                    group: {
                        store: { code: { _by: 'value' } },
                        location: { code: { _by: 'value' } },
                        locationType: { _by: 'value' },
                    },
                    values: {
                        code: { distinctCount: true },
                        numberOfConsumptionUnit: { sum: true },
                    },
                },
                {
                    filter: _filterByProduct,
                },
            ),
        };

        const _response = await new ui.queryUtils.BatchRequest(_requests).execute();
        interface ReadAggregateLocation {
            group: {
                store: {
                    code: string;
                };
                location: {
                    code: string;
                };
                locationType: string;
            };
            values: {
                code: {
                    distinctCount: number;
                };
                numberOfConsumptionUnit: {
                    sum: string | number;
                };
            };
        }

        const _aggregateLocation = extractEdges<ReadAggregateLocation>(_response?.aggregateLocation as Edges<any>);

        _aggregateLocation.forEach(aggregation => {
            const _key = _getKeyStoreLocation(aggregation.group.store.code, aggregation.group.location.code);
            _aggregatedLocationQuantities.push(<AggregatedStoreLocation>{
                _id: undefined,
                _key,
                storeCode: aggregation.group.store.code,
                locationCode: aggregation.group.location.code,
                locationType: aggregation.group.locationType,
                numberOfConsumptionUnit: Number(aggregation.values.numberOfConsumptionUnit.sum ?? 0),
                distinctStockObject: aggregation.values.code.distinctCount,
                distinctLot: 0,
                distinctSerialNumber: 0,
            });
        });

        return _aggregatedLocationQuantities.length ? _aggregatedLocationQuantities[0] : null;
    }

    /**
     * Change title page
     * @param _parameters
     */
    private _setTitle(_parameters?: StockByProductArgs): void {
        this.$.page.title = _parameters?.isViewByStoreLocation
            ? ui.localize('@sage/wh-pages/title__stock-by-store-location', 'Stock by location')
            : ui.localize('@sage/wh-pages/title__stock-by-product', 'Stock by product');
    }
}
