import type { GraphApi, SerialNumber } from '@sage/wh-stock-data-api';
import { Dict, extractEdges } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import type { PartialCollectionValueWithIds } from '@sage/xtrem-ui/build/lib/component/types';
import { LocalizedSerialNumbersArgs } from '../interfaces';

const _hideWhenEmptyValue = (value: any, _rowValue?: Dict<SerialNumber>) => {
    return typeof value !== 'number' && !value;
};
interface ReturnSerialNumber {
    _id: string;
    _key: string;
    _uniqueCode: string;
    code: string;
    double: number;
    occurrence: number;
    serialGroupCode: string;
    serialParentGroupCode: string;
    free1: string;
    free2: string;
    free3: string;
    free4: string;
    free5: string;
}

@ui.decorators.page<MobileViewLocalizedSerialDetails>({
    title: 'Stock by product',
    subtitle: 'Serial number details',
    isTitleHidden: true, // hide the page's title and render that title only in the feature header instead of in both places (see X3-177000 & https://github.com/Sage-ERP-X3/etna/pull/1785)
    node: '@sage/wh-stock-data/SerialNumber',
    mode: 'default',
    isTransient: true,
    navigationPanel: undefined,
    authorizationCode: 'INQSTOPRO',
    headerCard() {
        return {
            title: this.productCode,
            titleRight: this.distinctSerialNumbers,
            line2: this.localizedDescription,
            line2Right: this.stockObjectCode,
        };
    },
    detailPanel() {
        return {
            isCloseButtonHidden: true,
            isTitleHidden: true,
            isHidden: true,
            isTransient: true,
            header: this.detailPanelSection,
            sections: [],
        };
    },
    async onLoad() {
        // Requires a selected product in the query parameters.  This should not occur unless user manually
        // directs themselves to this page
        const _parameters = this._getQueryParameters();
        this._setTitle(_parameters);

        this._parameters = _parameters;

        // Requires product and stock id
        if (
            _parameters &&
            _parameters.productCode &&
            _parameters.depositorSiteSelected &&
            _parameters.stockObjectSelected?._id &&
            _parameters.stockObjectSelected?.code
        ) {
            // Fill out header card
            this._id.value = _parameters.stockObjectSelected._id;
            this.productCode.value = _parameters.productCode;
            this.localizedDescription.value = _parameters?.localizedDescription ?? null;
            this.stockObjectCode.value = _parameters.stockObjectSelected?.code ?? null;
            const _localizedSerialNumbers = await this._readLocalizedSerialNumbers(_parameters);
            this.localizedSerialNumbers.value = _localizedSerialNumbers;

            this.distinctSerialNumbers.value = _localizedSerialNumbers.length;

            this.localizedSerialNumbers.value = _localizedSerialNumbers;
        } else {
            this.$.showToast(
                ui.localize('@sage/wh-pages/notification-error-missing-params', 'Missing required parameters'),
                { type: 'error' },
            );
        }
    },
})
export class MobileViewLocalizedSerialDetails extends ui.Page<GraphApi> {
    /**
     * Internal properties
     */

    private _parameters: LocalizedSerialNumbersArgs | undefined;
    /*
     * sections
     */

    @ui.decorators.section<MobileViewLocalizedSerialDetails>({
        isTitleHidden: true,
        // title: 'Serial Numbers',
    })
    mainSection!: ui.containers.Section;

    @ui.decorators.section<MobileViewLocalizedSerialDetails>({
        isTitleHidden: true,
        // title: 'Serial Numbers',
    })
    detailPanelSection!: ui.containers.Section;

    /*
     * blocks
     */

    @ui.decorators.block<MobileViewLocalizedSerialDetails>({
        parent() {
            return this.mainSection;
        },
    })
    mainBlock!: ui.containers.Block;

    @ui.decorators.block<MobileViewLocalizedSerialDetails>({
        parent() {
            return this.detailPanelSection;
        },
    })
    detailsPanelBlock!: ui.containers.Block;

    /**
     * technical fields
     * */

    @ui.decorators.textField<MobileViewLocalizedSerialDetails>({
        isTransient: true,
        isReadOnly: true,
    })
    productCode!: ui.fields.Text;

    @ui.decorators.numericField<MobileViewLocalizedSerialDetails>({
        isTransient: true,
        isReadOnly: true,
        prefix: ui.localize('@sage/wh-pages/serial-numbers-prefix', 'Serial no.:'),
    })
    distinctSerialNumbers!: ui.fields.Numeric;

    @ui.decorators.textField<MobileViewLocalizedSerialDetails>({
        isTransient: true,
        isReadOnly: true,
    })
    localizedDescription!: ui.fields.Text;

    @ui.decorators.textField<MobileViewLocalizedSerialDetails>({
        isTransient: true,
        isReadOnly: true,
        prefix: ui.localize('@sage/wh-pages/prefix-stock-object', 'SO:'),
    })
    stockObjectCode!: ui.fields.Text;

    @ui.decorators.textField<MobileViewLocalizedSerialDetails>({
        isTransient: true,
        isReadOnly: true,
    })
    _id!: ui.fields.Text;

    @ui.decorators.tableField<MobileViewLocalizedSerialDetails>({
        parent() {
            return this.mainBlock;
        },
        node: '@sage/wh-stock-data/SerialNumber',
        isTransient: true,
        isFullWidth: true,
        isTitleHidden: true,
        canFilter: false,
        canSelect: false,
        canExport: false,
        canUserHideColumns: false,
        mobileCard: undefined,
        orderBy: { _key: 1 },
        columns: [
            ui.nestedFields.text({
                title: 'Serial number',
                bind: '_uniqueCode',
                isReadOnly: true,
                isTransient: true,
            }),
            ui.nestedFields.text({
                title: 'Serial number',
                bind: 'code',
                isReadOnly: true,
                isTransient: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                title: 'Double',
                bind: 'double',
                isReadOnly: true,
                isTransient: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                title: 'Occurrence',
                bind: 'occurrence',
                isReadOnly: true,
                isTransient: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                title: 'Serial group',
                bind: 'serialGroupCode',
                isReadOnly: true,
                isTransient: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                title: 'Parent group',
                bind: 'serialParentGroupCode',
                isReadOnly: true,
                isTransient: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                title: 'Free 1',
                bind: 'free1',
                isReadOnly: true,
                isTransient: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                title: 'Free 2',
                bind: 'free2',
                isReadOnly: true,
                isTransient: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                title: 'Free 3',
                bind: 'free3',
                isReadOnly: true,
                isTransient: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                title: 'Free 4',
                bind: 'free4',
                isReadOnly: true,
                isTransient: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                title: 'Free 5',
                bind: 'free5',
                isReadOnly: true,
                isTransient: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: '_id',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: '_key',
                isReadOnly: true,
                isHidden: true,
            }),
        ],
        async onRowClick(recordId: string, rowItem: SerialNumber) {
            await this._onRowClick(recordId, rowItem);
        },
    })
    localizedSerialNumbers!: ui.fields.Table<ReturnSerialNumber>;

    @ui.decorators.detailListField<MobileViewLocalizedSerialDetails>({
        parent() {
            return this.detailsPanelBlock;
        },
        isTransient: true,
        fields: [
            // display a blank line at the top
            ui.nestedFields.text({
                bind: '_spacerColumn',
                isReadOnly: true,
                isTransient: true,
            }),
            ui.nestedFields.text({
                title: 'Serial number',
                bind: 'code',
                isReadOnly: true,
                isTransient: true,
            }),
            ui.nestedFields.numeric({
                title: 'Double',
                bind: 'double',
                isReadOnly: true,
                isTransient: true,
            }),
            ui.nestedFields.numeric({
                title: 'Occurrence',
                bind: 'occurrence',
                isReadOnly: true,
                isTransient: true,
            }),
            ui.nestedFields.text({
                title: 'Serial group',
                bind: 'serialGroupCode',
                isReadOnly: true,
                isTransient: true,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.text({
                title: 'Parent group',
                bind: 'serialParentGroupCode',
                isReadOnly: true,
                isTransient: true,
                isHidden: _hideWhenEmptyValue,
            }),
            ui.nestedFields.text({
                title: 'Free 1',
                bind: 'free1',
                isReadOnly: true,
                isTransient: true,
            }),
            ui.nestedFields.text({
                title: 'Free 2',
                bind: 'free2',
                isReadOnly: true,
                isTransient: true,
            }),
            ui.nestedFields.text({
                title: 'Free 3',
                bind: 'free3',
                isReadOnly: true,
                isTransient: true,
            }),
            ui.nestedFields.text({
                title: 'Free 4',
                bind: 'free4',
                isReadOnly: true,
                isTransient: true,
            }),
            ui.nestedFields.text({
                title: 'Free 5',
                bind: 'free5',
                isReadOnly: true,
                isTransient: true,
            }),
            ui.nestedFields.text({
                bind: '_id',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: '_key',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: '_uniqueCode',
                isReadOnly: true,
                isTransient: true,
                isHidden: true,
            }),
        ],
    })
    detailsPanelList!: ui.fields.DetailList;

    /*
     *
     *  Fields management functions
     *
     */

    private async _onRowClick(recordId: string, rowItem: SerialNumber): Promise<void> {
        this.detailsPanelList.value = [rowItem]; // populate details list

        await this.localizedSerialNumbers.validateWithDetails();

        if (this.$?.detailPanel) {
            this.$.detailPanel.isHidden = false;
        }
    }

    /**
     * get query parameters
     * @returns expected parameters or undefined
     */
    private _getQueryParameters(): LocalizedSerialNumbersArgs | undefined {
        try {
            return JSON.parse(String(this.$.queryParameters.localizedSerialNumbersArgs)) as LocalizedSerialNumbersArgs;
        } catch (error) {
            ui.console.error(`Error retrieving query parameters:\n${JSON.stringify(error)}`);
            return undefined;
        }
    }

    /**
     * Read serialized serial numbers
     * @param _parameters arguments parameters
     * @returns serialized serial numbers
     */

    private async _readLocalizedSerialNumbers(
        _parameters: LocalizedSerialNumbersArgs,
    ): Promise<PartialCollectionValueWithIds<ReturnSerialNumber>[]> {
        try {
            const _returnSerialNumbers: ReturnSerialNumber[] = [];
            interface EdgesSerialNumber {
                _id: string;
                keyCodeForDuplicates: string;
                code: string;
                double: number;
                occurrence: number;
                serialGroup: { code: string };
                serialParentGroup: { code: string };
                free1: string;
                free2: string;
                free3: string;
                free4: string;
                free5: string;
            }
            const _response =
                (extractEdges<unknown>(
                    await this.$.graph
                        .node('@sage/wh-stock-data/SerialNumber')
                        .query(
                            ui.queryUtils.edgesSelector<SerialNumber>(
                                {
                                    _id: true,
                                    code: true,
                                    double: true,
                                    occurrence: true,
                                    serialGroup: { code: true },
                                    serialParentGroup: { code: true },
                                    free1: true,
                                    free2: true,
                                    free3: true,
                                    free4: true,
                                    free5: true,
                                },
                                {
                                    filter: {
                                        site: {
                                            code: _parameters?.depositorSiteSelected?.siteCode,
                                        },
                                        depositor: {
                                            code: _parameters?.depositorSiteSelected?.depositorCode,
                                        },
                                        product: { code: _parameters.productCode },
                                        stockObject: { code: _parameters.stockObjectSelected.code },
                                    },
                                    first: 500,
                                    orderBy: { keyCodeForDuplicates: 1 },
                                },
                            ),
                        )
                        .execute(),
                ) as EdgesSerialNumber[]) ?? [];

            // Convert serial numbers using functional approach
            const serialNumbers = _response.map(
                _serialNumber =>
                    <ReturnSerialNumber>{
                        _id: _serialNumber._id,
                        _key: _serialNumber.keyCodeForDuplicates,
                        code: _serialNumber.code,
                        double: _serialNumber.double,
                        occurrence: _serialNumber.occurrence,
                        _uniqueCode: `${_serialNumber.code} - ${_serialNumber.double} - ${_serialNumber.occurrence}`,
                        serialGroupCode: _serialNumber?.serialGroup?.code ?? '',
                        serialParentGroupCode: _serialNumber?.serialParentGroup?.code ?? '',
                        free1: _serialNumber.free1 ?? '',
                        free2: _serialNumber.free2 ?? '',
                        free3: _serialNumber.free3 ?? '',
                        free4: _serialNumber.free4 ?? '',
                        free5: _serialNumber.free5 ?? '',
                    },
            );

            _returnSerialNumbers.push(...serialNumbers);

            return _returnSerialNumbers;

            // return (extractEdges<unknown>(_response) as PartialCollectionValueWithIds<SerialNumber>[]) ?? [];
        } catch (error) {
            ui.console.error(`Error reading serial numbers:\n${JSON.stringify(error)}`);
            return [];
        }
    }

    /**
     * Change title page
     * @param _parameters
     */
    private _setTitle(_parameters?: LocalizedSerialNumbersArgs): void {
        this.$.page.title = _parameters?.isViewByStoreLocation
            ? ui.localize('@sage/wh-pages/title__stock-by-store-location', 'Stock by location')
            : ui.localize('@sage/wh-pages/title__stock-by-product', 'Stock by product');
    }
}
